#!/bin/bash

function doImportGpo()
{
    # shellcheck disable=SC1091,SC1090
    . /etc/eole/samba4-vars.conf
    
    # shellcheck disable=SC1091,SC1090
    . /usr/lib/eole/samba4.sh
    
    if [ "${AD_SERVER_ROLE}" != "controleur de domaine" ]
    then
        echo "Pas de GPO sur les serveurs membres"
        return 1
    fi
    
    if [ "${AD_ADDITIONAL_DC}" != "non" ]
    then
        echo "Cette commande ne doit pas être éxecutée sur les Dc Secondaires."
        return 2
    fi
    
    GPO_ADMIN="gpo-${AD_HOST_NAME}"
    GPO_ADMIN_DN="${GPO_ADMIN}@${AD_REALM^^}"
    GPO_ADMIN_PWD_FILE=$(get_passwordfile_for_account "${GPO_ADMIN}")
    ADMIN_PWD="$(cat "${GPO_ADMIN_PWD_FILE}")"
    CREDENTIAL="${GPO_ADMIN_DN}%${ADMIN_PWD}"
    GPO_ADMIN_KEYTAB_FILE=$(get_keytabfile_for_account "${GPO_ADMIN}")
    if [ ! -f "${GPO_ADMIN_PWD_FILE}" ]
    then
        echo "Warning:Le fichier ${GPO_ADMIN_PWD_FILE} est manquant"
        return 3
    fi
    if [ ! -f "${GPO_ADMIN_KEYTAB_FILE}" ]
    then
        echo "Warning:Le fichier ${GPO_ADMIN_KEYTAB_FILE} est manquant"
        return 4
    fi
    
    if ! kinit "${GPO_ADMIN_DN}" -k -t "${GPO_ADMIN_KEYTAB_FILE}"
    then
        echo "Impossible de créer une session kerberos."
        return 5
    fi
    
    echo "* Fix netlogon"
    gpo-tool importation fix_netlogon_scripts_acl -k 1 -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -U"$CREDENTIAL" 
    
    SOURCE="/var/tmp/$GPONAME"
    if [ ! -d "$SOURCE" ]
    then
       mkdir "$SOURCE"
    fi
    cd "$SOURCE" || return 6
    if [ -f "$EXPORT_TAR_GZ" ]
    then
        tar xf "$EXPORT_TAR_GZ"
    else
        if [ -d "$EXPORT_TAR_GZ" ]
        then
            cp -rf "$EXPORT_TAR_GZ" .
        else
            echo "La source '$EXPORT_TAR_GZ' n'est pas un tar, ni un répertoire."
            return 5
        fi 
    fi 
    if [ ! -d "$SOURCE/policy" ]
    then
        echo "Le répertoire 'policy' n'existe pas dans '$EXPORT_TAR_GZ'."
        return 7
    fi
    
    echo "* Import GPO"
    if ! GPOID=$(gpo-tool importation show_by_name "$GPONAME" --attribut name -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"$CREDENTIAL" 2>/dev/null)
    then
        echo "* $GPONAME $GPOID absente" 
    else
        if [ ! -f /var/tmp/gpo-script/update_eole_script ]
        then
            echo "* $GPONAME $GPOID existe, stop."
            return 0
        fi
        echo "* $GPONAME $GPOID existe, rebuild demandé"
        /bin/rm /var/tmp/gpo-script/update_eole_script
        
        if ! gpo-tool importation delete_by_name "$GPONAME" -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"$CREDENTIAL" 2>/dev/null
        then
            echo "* Delete '$GPONAME' Erreur"
            # j'ignore si elle a déjà été supprimée !
        else
            echo "* Delete '$GPONAME' OK"
        fi
        # un délai pour prise en compte !
        sleep 5
    fi
    
    if ! samba-tool gpo restore -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"${CREDENTIAL}" "$GPONAME" "$SOURCE/policy" --restore-metadata
    then
        echo "* $GPONAME : restauration impossible !"
        kdestroy || /bin/true
        return 8
    fi
    # check
    if ! GPOID=$(gpo-tool importation show_by_name "$GPONAME" --attribut name -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"$CREDENTIAL" )
    then
        echo "* $GPONAME impossible de créer la GPO !"
        kdestroy || /bin/true
        return 9 
    fi
    
    if [ -f "$SOURCE/ldif" ]
    then
        VERSION_AD=$(gpo-tool importation show_by_name "$GPONAME" --attribut versionNumber -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"$CREDENTIAL" )
        VERSION_EXPORT=$(awk -F": " '/versionNumber/ {print $2;}' <"$SOURCE/ldif")
        if [ "$VERSION_AD" == "$VERSION_EXPORT" ]
        then
            echo "* Version ok"
        else
            echo "* Version différente, positionne à ${VERSION_EXPORT}"
            cat >/tmp/gpoUpdate.ldif <<EOF
dn: CN=$GPOID,CN=Policies,CN=System,$BASEDN
changetype: modify
replace: versionNumber
versionNumber: ${VERSION_EXPORT}
EOF
    
            ldbmodify -v -H "/var/lib/samba/private/sam.ldb" -U"${CREDENTIAL}" /tmp/gpoUpdate.ldif
        fi 
    fi
    
    if [ -f "$SOURCE/attr" ]
    then
        echo "* Restauration xattr"
        cd "/home/sysvol/${AD_REALM}/Policies/$GPOID" || return 1
        setfattr --restore "$SOURCE/attr" . 
    fi
    
    echo "* Fix GPO"
    gpo-tool importation fix_gpo_acl "$GPONAME" -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"$CREDENTIAL" 
    
    # link ?
    if [ -n "$DN_TO_LINK" ]
    then
        if ! samba-tool gpo setlink "$DN_TO_LINK" "$GPOID" -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"${CREDENTIAL}" >/dev/null
        then
            echo "* $GPONAME impossible de lier la GPO à $DN_TO_LINK !"
            kdestroy || /bin/true
            return 10 
        fi
    fi
    
    # nettoyage Policy non détruites
    cd "/home/sysvol/${AD_REALM}/Policies" || return 11
    for ID in "{"*
    do
        if [ "$ID" == "{6AC1786C-016F-11D2-945F-00C04FB984F9}" ] || [ "$ID" == "{31B2F340-016D-11D2-945F-00C04FB984F9}" ] || [ "$GPOID" == "$ID" ]
        then
            # protection les 2 par défault
            continue
        fi
        if [ "$GPOID" == "$ID" ]
        then
            # protection celle que l'on vient de créer...
            continue
        fi
        if ! samba-tool gpo show  "$ID" -H "ldap://${AD_HOST_NAME}.${AD_REALM}" -k 1 -U"${CREDENTIAL}" 1>/dev/null 2>&1
        then
            if [ -n "$ID" ] # protection avant rm !
            then
                echo "* Suppression GPO '$ID' inconnu"
                rm -rf "/home/sysvol/${AD_REALM}/Policies/$ID/"
            fi
        fi
    done 
    
    ## si le SYSVOL est présent ==> check + reset si besoin
    SYSVOL_PATH=$(samba-tool testparm --suppress-prompt --section-name sysvol --parameter-name='path' 2>/dev/null)
    if [ -n "$SYSVOL_PATH" ]
    then
        if [ -d "$SYSVOL_PATH" ]
        then
            if samba-tool ntacl sysvolcheck 2>/dev/null
            then
                echo "Check sysvol ACL : Ok"
            else
                echo "Check sysvol ACL NOK, do sysvolreset, please wait ..."
                if samba-tool ntacl sysvolreset 2>/dev/null
                then
                    echo "Reset sysvol ACL OK"
            else
                    echo "Reset sysvol ACL NOK"
                fi
            fi
        fi
    fi
    #samba-tool ntacl get "/home/sysvol/${AD_REALM}/Policies/$GPOID/User" --as-sddl
    #samba-tool ntacl get "/home/sysvol/${AD_REALM}/Policies/$GPOID/Machine" --as-sddl
    
    kdestroy || /bin/true
    return 0
}

GPONAME="$1"
EXPORT_TAR_GZ="$2"
DN_TO_LINK="$3"
echo "* Import GPO $GPONAME from export $EXPORT_TAR_GZ"
if [ -z "$GPONAME" ]
then
    echo "Usage: $0 <gpo_name> <export_gpos.tar.gz> [<dn_to_link>]]"
    exit 1
fi
if [ ! -f "$EXPORT_TAR_GZ" ]
then
    echo "Usage: $0 <gpo_name> <export_gpos.tar.gz> [<dn_to_link>]"
    exit 1
fi
if doImportGpo >>/var/log/samba/import-gpo.log
then
    echo "* Import GPO $GPONAME : OK"
else
    echo "* Import GPO $GPONAME : Erreur ($?)"
fi
exit 0
