# -*- coding: utf-8 -*-
#
##########################################################################
# eoleauth - client.py : base class for authentication plugins
# Copyright © 2013 Pôle de compétences EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
##########################################################################
from flask import request, session, redirect, current_app
from eoleflask.util import make_error_response, get_proxy_url

from eoleauthlib.i18n import _
from eoleauthlib.errors import AuthenticationNeeded, UnauthorizedError

class BaseClient:
    """Base Authentication Client class
    """
    # session mapper_attr: redefine in plugin if needed
    map_attr = None

    def __init__(self, active=True, mode='GLOBAL'):
        """Contructeur
        active: invalidate_session if False
        """
        self.allowed_users = current_app.config.get('ALLOWED_USERS', [])
        self.active = active
        self.mode = mode
        # self.active = False
        if not active:
            self.invalidate_session()
        self._init_client()

    def _init_client(self):
        """additional initialization actions, redefine in plugin
        """
        pass

    def check_authsource(self):
        """checks that authentication source is available, redefine in plugin
        """
        pass

    def invalidate_session(self):
        """invalidate current user session
        """
        session.pop('username', None)
        session_attrs = session.keys()
        if self.map_attr:
            # if mapping attr is defined, we need to destroy both mapping
            # and session from storage
            session_map = session.get(self.map_attr, None)
            if session_map:
                current_app.session_interface.remove_session(session_map)
        # empty session attributes
        for attr in session_attrs:
            session.pop(attr, None)

    def authenticate(self):
        """Ensures authentication before redirecting to application
        """
        try:
            self.is_authenticated()
        except UnauthorizedError, e:
            return make_error_response(unicode(e), 401)
        except AuthenticationNeeded, e:
            return self._authenticate()
        # redirects to calling app if already authenticated
        return redirect(request.args.get("app_url"))

    def _authenticate(self):
        """performs specific authentication, redefine in plugin
        """
        pass

    def logout(self):
        """Logout du client
        """
        if 'username' in session:
            self.invalidate_session()
        return self._logout()

    def _logout(self):
        """performs specific logout actions, redefine in plugin
        """
        return redirect(get_proxy_url(request, request.url))

    def check_allowed_users(self, user_data):
        """checks if current user is in the list of allowed users for this app
         ( if ALLOWED_USERS is defined in application configuration)
        """
        if self.allowed_users and user_data['username'] not in self.allowed_users:
            current_app.logger.error(_("Access denied to {0} (check configuration for allowed users)").format(current_app.name))
            raise UnauthorizedError(_("You are not allowed to access this application"))

    def is_authenticated(self):
        """Returns username if current session is authenticated
        if not, ask plugin to check for authentication source reponse
        and update session data if ok.
        """
        if session.has_key('username') and self.active:
            self.check_allowed_users(session)
            return True
        # not authenticated, or new authentication forced (active=False),
        # always destroy session and check for credentials
        self.invalidate_session()
        user_data = self._check_authentication()
        if user_data and 'username' in user_data:
            self.check_allowed_users(user_data)
            # utilisateur authentifié, validation de la session
            for attr_key, attr_data in user_data.items():
                session[attr_key] = attr_data
            return True
        raise AuthenticationNeeded, _("No user session detected")

    def _check_authentication(self):
        """checks authentication, redefine in plugin.
        returns user data in dictionnary form.
        if user is authenticated, 'username' must be in user data
        if plugin has a mapping attribute, return its value in user data
        """
        return None

