#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""CAS authentication module
"""
import urllib, urlparse, re, socket
from functools import wraps
from flask import request, session, redirect, current_app, url_for
from eoleauthlib.client import BaseClient
from eoleflask.util import get_proxy_url
TEST_SOCKET_TIMEOUT = 3

def parse_tag(chaine, tag):
    """Parses XML response and returns specified tag (or nothing)
    """
    tag1_pos1 = chaine.find("<" + tag)
    #  No tag found, return empty string.
    if tag1_pos1 == -1:
        return ""
    tag1_pos2 = chaine.find(">", tag1_pos1)
    if tag1_pos2 == -1:
        return ""
    tag2_pos1 = chaine.find("</" + tag, tag1_pos2)
    if tag2_pos1 == -1:
        return ""
    return chaine[tag1_pos2+1:tag2_pos1].strip()

def is_server_alive(server_ip, server_port):
    """
    checks that server responds on specified ip port
    """
    socket.setdefaulttimeout(TEST_SOCKET_TIMEOUT)
    try:
        socket.socket().connect((server_ip, int(server_port)))
        return True
    except socket.error:
        return False
    except:
        return False

def serviceURL(orig_url = "", cas_only=False):
    """strips arguments from orig_url
    ticket_only : only strips cas_ticket
    """
    # si pas d'url fournie, on prend celle de la requête actuelle
    ret = orig_url or get_proxy_url(request, request.url)
    ret = re.sub(r'ticket=[^&]*&?', '', ret)
    if not cas_only:
        ret = re.sub(r'\?&?$|&$', '', ret)
    return ret

class CASClient(BaseClient):
    """Client CAS
    """

    map_attr = "cas_ticket"

    def _init_client(self, active=True):
        """Contructeur
        """
        self.cas_url = current_app.config.get('CAS_URL', 'UNDEFINED')

    def check_authsource(self):
        """checks that authentication source is available
        """
        try:
            assert self.cas_url.startswith('https')
            # secure channel is mandatory for CAS server
            url_info = urlparse.urlparse(self.cas_url)
            return is_server_alive(url_info.hostname, url_info.port)
        except:
            current_app.logger.error(_("CAS plugin, server url undefined or doesn't use https"))
        return False

    def _authenticate(self):
        """Asks CAS server for Authentication
        """
        # vérifier accès CAS et rediriger sur login local si besoin ?
        # login_url = self.cas_url + 'login' + '?service=' + urllib.quote(serviceURL())
        login_url = urlparse.urljoin(self.cas_url, 'login') + '?service=' + urllib.quote(get_proxy_url(request, request.url))
        return redirect(login_url)

    def _check_authentication(self):
        """Checks response from CAS server.
        Performs logout if logoutRequest sent.
        """
        if request.method == 'POST' and request.form.has_key('logoutRequest'):
            self._process_logout_request()
        return self._validate()

    def _validate(self):
        """Validate ticket against CAS server and returns user attributes if any
        """
        # FIXME : contruire une url de service sur /login/app_url pour permettre d'avoir un filtre différent par application ?
        # 2ème option : proposer une fonction get_attributes utilisable dans les applications ?
        # Dans le cas d'applis ayant besoin d'attributs supplémentaires, utiliser une session
        # séparée pour éviter de les rendre accessibles
        ticket = request.args.get('ticket', '')
        if ticket:
            cas_validate = urlparse.urljoin(self.cas_url, 'validate') + '?service=' + urllib.quote(serviceURL()) + '&ticket=' + urllib.quote(ticket)
            f_validate   = urllib.urlopen(cas_validate)
            response = f_validate.read()
            if parse_tag(response, "cas:authenticationSuccess") != "":
                infos = {}
                infos['username'] = parse_tag(response,"cas:user")
                # stockage du ticket original pour vérification en cas de logout
                infos['cas_ticket'] = ticket
                # TODO : parser les attributs supplémentaires ici
                return infos
        return None

    def _process_logout_request(self):
        # Gestion du SingleLogout: Requete POST lancée par le SSO lors du logout
        id_logout = parse_tag(request.form['logoutRequest'], "samlp:SessionIndex")
        current_app.logger.debug(_('logout requested by CAS server'))
        # dans ce mode, la session n'est pas disponible (appel en backend)
        # on vérifie que la session vient bien du serveur CAS configuré
        try:
            cas_ip = socket.gethostbyname(urlparse.urlparse(self.cas_url).hostname)
            assert cas_ip == socket.gethostbyname(request.remote_addr)
            # Suppression de la session au niveau serveur en fonction de cas_ticket
            if not current_app.session_interface.remove_session(id_logout):
                current_app.logger.debug(_('could not remove session matching {0} = {1}').format(self.map_attr, id_logout))
        except AssertionError, e:
            current_app.logger.warning(_('Error, logout request does not match client configuration'))

    def _logout(self):
        """Redirects to CAS logout URL
        """
        # déclenchement du logout SSO global
        # logout_url = self.cas_url + 'logout' + '?service=' + urllib.quote(request.args.get(serviceURL()))
        logout_url = urlparse.urljoin(self.cas_url, 'logout') + '?service=' + urllib.quote(serviceURL())
        current_app.logger.debug('CAS LOGOUT : redirect to {0}'.format(logout_url))
        return redirect(logout_url)
