import sys
import termios
import fcntl
import struct
from os import path, mkdir, curdir


def get_current_column():
    """Get current terminal column number.

        :return: number of columns
        :rtype: `list`
        :raises: if cannot detect columns number: IOError(25,
                 'Inappropriate ioctl for device')

    """
    try:
        data = fcntl.ioctl(sys.stdout.fileno(),
                termios.TIOCGWINSZ, u'1234')
        return struct.unpack('hh', data)[1]
    except (IOError, ValueError):
        return 80


# fork from /usr/lib/python3.8/os.py
# mode is recursive now
def recurive_mode_makedirs(name, mode=0o777, exist_ok=False):
    """makedirs(name [, mode=0o777][, exist_ok=False])

    Super-mkdir; create a leaf directory and all intermediate ones.  Works like
    mkdir, except that any intermediate path segment (not just the rightmost)
    will be created if it does not exist. If the target directory already
    exists, raise an OSError if exist_ok is False. Otherwise no exception is
    raised.  This is recursive.

    """
    head, tail = path.split(name)
    if not tail:
        head, tail = path.split(head)
    if head and tail and not path.exists(head):
        try:
            recurive_mode_makedirs(head, exist_ok=exist_ok, mode=mode)
        except FileExistsError:
            # Defeats race condition when another thread created the path
            pass
        cdir = curdir
        if isinstance(tail, bytes):
            cdir = bytes(curdir, 'ASCII')
        if tail == cdir:           # xxx/newdir/. exists if xxx/newdir exists
            return
    try:
        mkdir(name, mode)
    except OSError:
        # Cannot rely on checking for EEXIST, since the operating system
        # could give priority to other errors like EACCES or EROFS
        if not exist_ok or not path.isdir(name):
            raise
