#! /usr/bin/env python
# -*- coding: UTF-8 -*-

###########################################################################
#
# Eole NG - 2008
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
###########################################################################

"""
Utilitaire de génération de fichiers netlogon depuis LDAP
"""
import time
import sys
from getopt import getopt, GetoptError
from commands import getstatusoutput
from os.path import basename, join, isfile
from os import symlink, listdir, system, path
from scribe.ldapconf import SMB_SERVEUR
from scribe.eoletools import my_escape
from scribe.linker import user_factory

def test_esu():
    """
    vérifie si Esu est utilisé
    """
    for fic in listdir('/home/esu/Base'):
        if fic.lower() == 'esuclnt.exe':
            return True
    return False

#############################################################
## Génération des fichiers de logon
#############################################################
class Logon:
    """
    Objet de génération des scripts de logon
    """

    def __init__(self, user, ostype, machine, adresse_ip, pid):
        self.user = user
        self.os_type = ostype
        self.machine = machine
        self.adresse_ip = adresse_ip
        self.pid = pid
        try:
            self.ldapuser = user_factory(user)
        except Exception, err:
            sys.exit(err)
        if not self.ldapuser.has_samba:
            sys.exit("l'utilisateur %s n'est pas autorisé à se connecter" % user)

    def build_netlogon(self):
        """ fonction de construction des scripts de connexion windows
        user : utilisateur,
        os_type : type de l'OS selon Samba,
        machine : nom de la machine,
        adresse_ip : IP
        le fichier est du format .bat pour Win95 et .txt pour le reste
        """
        self.ldapuser.ldap_admin.connect()
        homedir = self.ldapuser._get_attr(self.user, 'homeDirectory')[0].strip()
        homedrive = self.ldapuser._get_attr(self.user, 'sambaHomeDrive')[0].strip()
        # enregistrement de la connexion
        self.log_connexion()
        # mise à jour de /home/u/user/.ftp/
        self.ldapuser._gen_ftpdir(self.user, homedir)
        # mise à jour de /home/u/user/groupes/
        self.ldapuser._gen_groupedir(self.user, homedir)
        # répertoire devoirs
        self.ldapuser._gen_devdir(self.user, homedir)
        # génération du fichier client
        self.gen_fich(homedrive, homedir)
        # fin
        self.ldapuser.ldap_admin.close()

    def log_connexion(self):
        """
        Enregistrement de la connexion
        """
        code, primgrp = getstatusoutput('id -gn %s' % self.user)
        if code != 0:
            primgrp = None
        try:
            code, sdate = getstatusoutput('date +%a" "%d" "%b" "%Y" "%H":"%M')
            sdate = sdate.capitalize()
            cmd = 'echo CONNECTION %s %s %s %s %s %s %s' % (sdate, self.user,
                    primgrp, self.machine, self.os_type, self.adresse_ip,
                    self.pid)
            system('%s >> /var/log/samba/%s.log' % (cmd, self.machine))
            system('%s >> /var/log/samba/connexions.log' % (cmd))
        except:
            pass

    def gen_fich(self, homedrive, homedir):
        """
        Génération du fichier lu par le client
        """
        dir_groupe = join(homedir, 'groupes')
        script = ""
        # gestion des groupes et des partages
        my_groups = self.ldapuser._get_user_groups(self.user)
        for group in my_groups:
            for share in self.ldapuser._get_group_logon_shares(group):
                if share['drive'] != "":
                    # partage avec lettre de lecteur
                    script += self.gen_lecteur_bloc(share['drive'], share['uri'])
                else:
                    # lien dans le répertoire "groupes"
                    symlink(share['path'], join(dir_groupe, share['name']))
        # montage du répertoire personnel pour 9x et Samba
        if self.os_type in ['Win95', 'Samba']:
            script += self.gen_lecteur_bloc(homedrive, '\\\\%s\\PERSO' % SMB_SERVEUR)

        # scripts spécifiques 9x
        if self.os_type == 'Win95':
            script += self.gest_win95(my_groups)

        # gestion des scripts additionnels
        debut, fin = self.get_scripts(my_groups)
        # écriture du fichier
        self.write_fich(debut, script, fin)

    def gen_lecteur_bloc(self, lettre, share):
        """Génère un bloc de ligne pour le montage
        d'un lecteur réseau "share" sur la lettre "lettre"
        ou sur * sinon
        """
        lettre = lettre[:1] # juste la lettre, sans les ":"
        while share[-1] == '\\':
            share = share[:-1] # le partage sans \ à la fin
        sharename = share.split('\\')[-1].replace('$', '')
        if self.os_type == 'Win95':
            # on définit la commande net use selon le type d'os (NT ou 9X)
            label = '%s%s' % (lettre, sharename)
            chaine_net_use = 'NET USE '
            bloc = self.gen_cmd_line('IF EXIST %s:\\nul goto err%s' % (lettre, label))
            bloc += self.gen_cmd_line('%s %s: %s'%(chaine_net_use, lettre, share))
            bloc += self.gen_cmd_line('goto suit%s'%(label))
            bloc += self.gen_cmd_line(':err%s'%(label))
            bloc += self.gen_cmd_line('%s * %s'%(chaine_net_use, share))
            bloc += self.gen_cmd_line(':suit%s'%(label))
        else:
            bloc = 'lecteur,%s:,%s\n' % (lettre, share)
        return bloc

    def gen_cmd_line(self, line, hide=False, nowait=False):
        """génère une ligne DOS (avec le bon retour chariot)
        """
        if self.os_type == 'Win95':
            saut_ligne = chr(13)+chr(10)
            return '%s%s' % (line, saut_ligne)
        else:
            chaine = 'cmd,%s'
            if hide:
                chaine += ',HIDDEN'
            if nowait:
                chaine += ',NOWAIT'
            chaine += '\n'
            return chaine % line
        return ''

    def get_scripts(self, user_groups):
        """Gestion des scripts externes
        """
        # motif à rechercher dans les fichiers externes
        # (debut_script %motif% fin_script)
        motif_include = [ "%%NetUse%%", "%NetUse%" ]
        # chemin par défaut des scripts externes
        buffer_debut,  buffer_fin = '', ''
        # on traite ceux qui sont présents
        for chemin in self.get_scripts_list(user_groups):
            if isfile(chemin):
                debut = 1
                for line in file(chemin,"r").read().splitlines():
                    if line.strip() in motif_include:
                        debut = 0
                    else:
                        if debut == 1:
                            buffer_debut += self.parse_line(line.strip())
                        else: buffer_fin += self.parse_line(line.strip())
        return (buffer_debut, buffer_fin)

    def get_scripts_list(self, user_groups):
        """On créé la liste des fichiers possibles
        +--path_scripts/users/<user>.txt
        +--path_scripts/groups/<group>.txt
        +--path_scripts/machines/<machine>.txt
        +--path_scripts/os/<os>.txt
        +--path_scripts/os/<os>/<group>.txt
        +--path_scripts/os/<os>/<user>.txt
        """
        path_scripts = '/home/netlogon/scripts'
        scripts_ext = '.txt'
        chemins = [join(path_scripts, 'users', '%s%s' % (self.user, scripts_ext))]
        chemins.append(join(path_scripts, 'machines', '%s%s' % (self.machine, scripts_ext)))
        for group in user_groups:
            chemins.append(join(path_scripts, 'groups', '%s%s' % (group, scripts_ext)))
            chemins.append(join(path_scripts, 'os', '%s' % self.os_type, '%s%s' % (group, scripts_ext)))
        chemins.append(join(path_scripts, 'os', '%s%s' % (self.os_type, scripts_ext)))
        chemins.append(join(path_scripts, 'os', self.os_type, '%s%s' % (self.user, scripts_ext)))
        return chemins

    def parse_line(self, line):
        items = [ i.strip() for i in line.split(',') ]
        if items[0] == 'cmd':
            # exécuter une commande
            hide, nowait = False, False
            if 'HIDDEN' in items:
                hide = True
            if 'NOWAIT' in items:
                nowait = True
            return self.gen_cmd_line(items[1], hide, nowait)
        elif items[0] == 'lecteur':
            # monter un partage
            lettre, partage = items[1], items[2]
            return self.gen_lecteur_bloc(lettre, partage)
        else:
            return ''

    def write_fich(self, debut, milieu, fin):
        """Ecrit le fichier
        """
        bat_file = '/home/netlogon/%s%s.txt' % (self.user, self.os_type)
        try:
            fic = file(bat_file, 'w')
            fic.writelines(debut)
            fic.writelines(milieu)
            fic.writelines(fin)
            fic.close()

            system('/usr/bin/todos -u %s' % bat_file)
            system('chmod 644 %s' % bat_file)

        except:
            sys.exit("erreur d'écriture du fichier %s"%bat_file)


    #########
    # Win9x #
    #########
    def gest_win95(self, user_groups):
        """Gestion "à l'ancienne" pour Win9x/Me
        exécuté par le service et logon.py(.exe) sur les clients
        """
        contenu = self.gen_vnc_line(user_groups)
        contenu += self.gen_esu()
        return contenu

    def gen_esu(self):
        """Ajoute la ligne pour Esu
        """
        if test_esu():
            esuline = '\\\\%s\\ESU\\Base\\esuclnt.exe' % SMB_SERVEUR
            return self.gen_cmd_line(esuline, hide=True)
        return ''

    def gen_vnc_line(self, user_groups):
        """VNC utilise maintenant login/pass du domaine
        ce .reg spécifie le type de controle possible$
        (desactive, simple, controle) lors de l'observation pour les Win95
        """
        vnccontrole = 'regedit /s \\\\%s\\netlogon\\scripts\\os\\Win95\\vnc_controle.reg' % SMB_SERVEUR
        vnckill = '%windir%\\eole\\ultravnc\\winvnc -kill'
        vncservice = '%windir%\\eole\\ultravnc\\winvnc -service'
        # FIXME : les administratifs ?
        if 'professeurs' in user_groups:
            bloc = self.gen_cmd_line(vnccontrole)
            bloc += self.gen_cmd_line(vnckill)
        elif 'eleves' in user_groups:
            bloc = self.gen_cmd_line(vnccontrole)
            bloc += self.gen_cmd_line(vnckill)
            bloc += self.gen_cmd_line(vncservice)
        return bloc

#############################################################
## Boucle principale du programme - gestion des options
#############################################################
def mhelp():
    """
    affichage de l'aide
    """
    print ("""
    Utilisation: %s

    -u --user <nom de login>
    -o --os <type os>
    -m --machine <nom de machine>
    -i --ip <adresse ip>
    -p --pid <smbd pid>
    """ % basename(sys.argv[0]))

def main():
    """
    boucle principale (gestion des options)
    """
    user = None
    ostype = None
    machine = None
    #groups = None
    adresse_ip = None
    pid = 'unknown'

    try:
        options, args = getopt(sys.argv[1:], 'u:o:m:i:p:h',
                ['user=',
                'os=',
                'machine=',
                'ip=',
                'pid=',
                'help'])

    except GetoptError:
        er_type, er_value, er_traceback = sys.exc_info()
        er_msg = str(er_value)
        print "L'option %s est inconnue ou nécessite un argument" % er_msg.split(' ')[1]
        mhelp()
        sys.exit(1)

    for (opt, val) in options:
        # uid de l'utilisateur
        if (opt == '-u') or (opt == '--user'):
            user = my_escape(val.lower())
            if not user[0].isalnum():
                user = None

        # système d'exploitation
        elif (opt == '-o') or (opt == '--os'):
            ostype = my_escape(val)

        # nom de la machine
        elif (opt == '-m') or (opt == '--machine'):
            machine = my_escape(val.lower())

        # adresse ip client
        elif (opt == '-i') or (opt == '--ip'):
            adresse_ip = my_escape(val)

        # pid smbd
        elif (opt == '-p') or (opt == '--pid'):
            pid = my_escape(val)

        elif (opt == '-h') or (opt == '--help'):
            mhelp()
            sys.exit(0)
        else:
            mhelp()
            sys.exit(1)

    if (not user or not ostype or not machine or not adresse_ip):
        print "Il manque des arguments"
        mhelp()
        sys.exit(1)

    if user == "nobody":
        sys.exit("utilisateur nobody : ne rien faire")

    # construction du script de connexion ou pas
    netlogon = '/home/netlogon/%s%s.txt' % (user, ostype)
    logon = Logon(user, ostype, machine, adresse_ip, pid)
    #si le fichier existe déjà et qu'il a moins de 1min on loggue juste la connexion
    if path.isfile(netlogon):
        diff = time.time() - path.getmtime(netlogon)
        if diff >= 0 and diff <= 60:
            logon.log_connexion()
        else:
            logon.build_netlogon()
    else:
        logon.build_netlogon()


if __name__ == '__main__':
    main()

