# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
    définition des formats de données internes pour l'importation
"""
from axiom.item import Item
from axiom.attributes import text, reference, boolean
from axiom.store import Store
from os.path import isdir
from shutil import rmtree

def init_store(dbdir=None, delete=True):
    """
    Initialisation de la base
    """
    if dbdir is not None and isdir(dbdir) and delete:
        rmtree(dbdir)
    return Store(dbdir)

class Eleve(Item):
    """
    Eleve
    """
    typeName = 'eleve'
    schemaVersion = 1
    # arguments obligatoires
    int_id = text(allowNone=False, indexed=True) # id interne
    nom = text(allowNone=False)
    prenom = text(allowNone=False)
    date = text(allowNone=False)
    numero = text(allowNone=False)
    civilite = text(allowNone=False) # attendu 1/2/3
    # arguments facultatifs
    classe = reference()
    niveau = reference()
    prenom2 = text(default=u'')
    nom_patronymique = text(default=u'')
    ine = text(default=u'')
    rattachid = text(default=u'')
    regime = text(default=u'')
    enseignements = text(default=u'[]')
    # login ldap attribué après traitement
    login = text(default=u'')
    # login forcé (import CSV)
    force_login = text()
    force_password = text()

    def __repr__(self):
        return '<Eleve prenom="%s" nom="%s">' % \
                (str(self.prenom), str(self.nom))

    def get_groupes(self):
        """
            :return: les groupes correspondant à cet élève
        """
        jointures = self.store.query(JointureGroupeUser,
            JointureGroupeUser.user == self)
        return [joint.groupe for joint in jointures]

class Responsable(Item):
    """
    Responsable
    """
    typeName = 'responsable'
    schemaVersion = 1
    # arguments obligatoires
    int_id = text(allowNone=False, indexed=True) # id interne
    nom = text(allowNone=False)
    prenom = text(allowNone=False)
    # arguments facultatifs
    civilite = text(default=u'')
    date = text(default=u'')
    mail = text(default=u'')
    telephone = text(default=u'')
    tel_portable = text(default=u'')
    tel_pro = text(default=u'')
    # -> class Adress
    id_adresse = text(default=u'')
    adresse = reference()
    # login ldap attribué après traitement
    login = text(default=u'') # initialisé au traitement

    def get_eleves(self):
        """
            :return: les élèves correspondant à ce responsable
        """
        jointures = list(self.store.query(JointureResponsableEleve,
            JointureResponsableEleve.responsable == self))
        return [joint.eleve for joint in jointures]


    def __repr__(self):
        return '<Parent prenom="%s" nom="%s">' % \
                (str(self.prenom), str(self.nom))

class Adresse(Item):
    """
    Adresse
    """
    typeName = 'adresse'
    schemaVersion = 1
    int_id = text(allowNone=False, indexed=True) # id interne
    adresse = text(default=u'')
    code_postal = text(default=u'')
    ville = text(default=u'')
    pays = text(default=u'')

    def __repr__(self):
        return """<Adresse id="%s">
%s
%s
%s
%s
</Adresse>""" % (self.int_id, self.adresse, self.code_postal,
        self.ville, self.pays)

class JointureResponsableEleve(Item):
    """
    Jointure Responsable - Eleve
    """
    typeName = 'jointure_responsable_eleve'
    schemaVersion = 1
    eleve = reference(allowNone=False)
    responsable = reference(allowNone=False)
    # utilité de ce qui suit ?
    resp_legal = text(default=u'')
    resp_financier = text(default=u'')
    resp_paiement = text(default=u'')
    pers_contact = text(default=u'')
    code_parente = text(default=u'')

class Classe(Item):
    """
        classe
    """
    typeName = 'classe'
    schemaVersion = 1
#    int_id = text(allowNone=False, indexed=True)
    nom = text(allowNone=False, indexed=True) # modifiable au traitement
    niveau = reference(allowNone=False)

    def get_eleves(self):
        """
            :return: les élèves correspondant à cette classe
        """
        return list(self.store.query(Eleve,
            Eleve.classe == self))

class Niveau(Item):
    """
        niveau
    """
    typeName = 'niveau'
    schemaVersion = 1
    #int_id = text(allowNone=False, indexed=True)
    nom = text(allowNone=False, indexed=True) # modifiable au traitement

    def get_eleves(self):
        """
            :return: les élèves correspondant à cette classe
        """
        return list(self.store.query(Eleve,
            Eleve.niveau == self))


class Enseignant(Item):
    """
    Enseignant
    """
    typeName = 'enseignant'
    schemaVersion = 1
    # arguments obligatoires
    int_id = text(allowNone=False, indexed=True) # id interne
    nom = text(allowNone=False)
    prenom = text(allowNone=False)
    date = text(allowNone=False)
    # arguments facultatifs
    civilite = text(default=u'')
    mail = text(default=u'')
    classe = text(default=u'')
    nom_patronymique = text(default=u'')
    disciplines = text(default=u'[]')
    # login ldap attribué après traitement
    login = text(default=u'') # initialisé au traitement
    # login forcé (import CSV)
    force_login = text()
    force_password = text()

    def get_classes(self):
        """
            :return: les classes correspondant à cet enseignant
        """
        jointures = list(self.store.query(JointureClasseEnseignant,
            JointureClasseEnseignant.enseignant == self))
        return jointures
        #return [joint.classe for joint in jointures]

    def get_matieres(self):
        """
            :return: les matières correspondant à cet enseignant
        """
        jointures = self.store.query(JointureMatiereEnseignant,
            JointureMatiereEnseignant.enseignant == self)
        return [joint.matiere for joint in jointures]

    def get_groupes(self):
        """
            :return: les groupes correspondant à cet élève
        """
        jointures = self.store.query(JointureGroupeUser,
            JointureGroupeUser.user == self)
        return [joint.groupe for joint in jointures]

class Administratif(Item):
    """
    Personnel administratif
    """
    typeName = 'administratif'
    schemaVersion = 1
    # arguments obligatoires
    int_id = text(allowNone=False, indexed=True) # id interne
    nom = text(allowNone=False)
    prenom = text(allowNone=False)
    date = text(allowNone=False)
    # arguments facultatifs
    civilite = text(default=u'')
    mail = text(default=u'')
    nom_patronymique = text(default=u'')
    groupe = reference()
    # login ldap attribué après traitement
    login = text(default=u'') # initialisé au traitement
    # login forcé (import CSV v2)
    force_login = text()
    force_password = text()

class Invite(Item):
    """
    Compte invité
    """
    typeName = 'invite'
    schemaVersion = 1
    # arguments obligatoires
    #int_id = text(allowNone=False, indexed=True) # id interne
    nom = text(allowNone=False)
    prenom = text(allowNone=False)
    date = text(allowNone=False)
    # arguments facultatifs
    civilite = text(default=u'')
    mail = text(default=u'')
    # login ldap attribué après traitement
    login = text(default=u'') # initialisé au traitement
    # login forcé (import CSV v2)
    force_login = text()
    force_password = text()

class JointureClasseEnseignant(Item):
    """
    Jointure Classe - Enseignant
    """
    typeName = 'jointure_classe_enseignant'
    schemaVersion = 1
    classe = reference(allowNone=False)
    enseignant = reference(allowNone=False)
    profprincipal = boolean(default=False)

class EnsClasse(Item):
    """
    Classe (fournie par un enseignant)
    """
    typeName = 'classe2'
    schemaVersion = 1
    nom = text(allowNone=False, indexed=True) # modifiable au traitement

class JointureMatiereEnseignant(Item):
    """
    Jointure Matiere - Enseignant
    """
    typeName = 'jointure_matiere_enseignant'
    schemaVersion = 1
    matiere = reference(allowNone=False)
    enseignant = reference(allowNone=False)

class Matiere(Item):
    """
    Matiere
    """
    typeName = 'matiere'
    schemaVersion = 1
    nom = text(allowNone=False, indexed=True) # modifiable au traitement
    description = text(default=u'')

class Groupe(Item):
    """
    Groupe
    """
    typeName = 'groupe'
    schemaVersion = 1
    nom = text(allowNone=False, indexed=True) # modifiable au traitement
    description = text(default=u'')

    def count_members(self):
        """
            :return: nombre d'utilisateurs inscrits au groupe
        """
        return len(list(self.store.query(JointureGroupeUser,
                                    JointureGroupeUser.groupe == self)))


class Service(Item):
    """
    Service administratif
    """
    typeName = 'service'
    schemaVersion = 1
    nom = text(allowNone=False, indexed=True) # modifiable au traitement
    description = text(default=u'')

class JointureGroupeUser(Item):
    """
    Jointure Groupe - Eleve
    ou       Groupe - Enseignant
    """
    typeName = 'jointure_groupe_user'
    schemaVersion = 1
    groupe = reference(allowNone=False)
    user = reference(allowNone=False)

