# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG - 2011
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################
"""
 librairie pour le parsing des fichiers de données AAF

 - parse_aaf_eleves : parsing des fichiers XML issus de AAF
 - parse_aaf_prof   : parsing des fichiers XML issus de AAF

"""
from scribe.eoletools import replace_cars, replace_more_cars, \
not_empty, is_empty, formate_civilite
from scribe.importation import log
from scribe.importation.config import DEBUG_NUM
from scribe.storage import Eleve, \
Responsable, Adresse, JointureResponsableEleve, \
Classe, Niveau, Enseignant, Administratif, \
EnsClasse, JointureClasseEnseignant, \
Matiere, JointureMatiereEnseignant, \
Groupe, JointureGroupeUser, Service
from scribe.parsing.tools import parse_xml
from scribe.parsing.nomenclature import MATIERES

##########################################
    # Extraction AAF Elèves & Responsables
# Fichiers :
# - ENT_$RNE_Complet_$DATE_Eleve_0000.xml
# - ENT_$RNE_Complet_$DATE_PersRelEleve_0000.xml
##########################################
def parse_aaf_eleves(storage, eleve_file):
    """
    parsing des élèves depuis AAF
    """
    num = 0
    log.infolog("Lecture des élèves...", title=True)
    context, _ = parse_xml(eleve_file)
    # nb : la date de naissance est au format jj/mm/aaaa
    mapping = {'sn':'nom',
               'givenName':'prenom',
               'ENTPersonDateNaissance':'date',
               'personalTitle':'civilite',
               # FIXME XXX elenoet & Ine
               'ENTPersonJointure':'numero',
               'ENTPersonAutresPrenoms':'prenom2',
               'ENTPersonNomPatro':'nom_patronymique',
               # nouveaux attributs
               'ENTEleveStructRattachId':'rattachid',
               'ENTEleveRegime':'regime',
    }
    groupe_mapping = {'ENTEleveMEF':'mef_id',
                      'ENTEleveLibelleMEF':'mef',
                      "ENTEleveNivFormation":'niveau',
                      "ENTEleveFiliere":'filiere',
                      "ENTEleveClasses":'classe',
    }

    for _, televe in context:
        _type = televe.find('operationalAttributes/attr/value').text
        if _type != 'Eleve':
            log.infolog("%s pas un élève :(" % _type)
            continue
        #Attention : identifier/id == ENTPersonJointure
        eleid = televe.find('identifier/id').text
        eleve = {'int_id':unicode(eleid)}
        groupe = {}
        responsables = []
        relations = []
        options = []
        for attr in televe.findall('attributes/attr'):
            balise = attr.attrib['name']
            if mapping.has_key(balise):
                # WARNING : que des mono-valuées ?
                clean_text = replace_more_cars(attr.find('value').text)
                cle = mapping[balise]
                eleve[cle] = unicode(clean_text)
            elif balise == 'ENTEleveAutoriteParentale':
                for resp in attr.findall('value'):
                    responsables.append(unicode(resp.text))
            elif balise == 'ENTElevePersRelEleve':
                # stockage des relations pour #24829
                relations = attr.findall('value')
            elif groupe_mapping.has_key(balise):
                clean_text = replace_cars(attr.find('value').text)
                cle = groupe_mapping[balise]
                groupe[cle] = unicode(clean_text)
            elif balise == 'ENTEleveGroupes':
                # options enseignées
                for tgrp in attr.findall('value'):
                    if tgrp.text is not None and '$' in tgrp.text:
                        options.append(unicode(replace_cars(tgrp.text.split('$')[1])))
            elif balise == 'ENTEleveEnseignements':
                # enseignements suivis (multi)
                enst = []
                for ens in attr.findall('value'):
                    if ens.text is not None:
                        enst.append(replace_more_cars(ens.text))
                eleve['enseignements'] = unicode(enst)

        if len(responsables) == 0:
            # aucun responsable trouvé avec "ENTEleveAutoriteParentale" #24829
            for rel in relations:
                if rel.text is not None:
                    relation = rel.text.split('$')
                    # ENTElevePersRelEleve : 4ème valeur = code responsable légal
                    if relation[3] == '1':
                        responsables.append(unicode(relation[0]))

        # ENTPersonJointure #10027
        eleve['entpersonjointure'] = "AAF$%s" % eleve['int_id']
        #pré-traitement
        if not_empty(eleve, 'civilite'):
            eleve['civilite'] = unicode(formate_civilite(str(eleve['civilite'])))
        else:
            log.infolog("Attribution arbitraire d'une civilité à l'élève : %s %s" % (
                         str(eleve['prenom']), str(eleve['nom'])))
            eleve['civilite'] = u'1'
        # création de l'élève
        try:
            my_eleve = Eleve(store=storage, **eleve)
            num += 1
            if num % DEBUG_NUM == 0:
                log.debuglog("%d élèves lus..." % num)
        except TypeError, msg:
            log.errorlog("Erreur sur l'élève %s : %s" % (eleid, msg))
            continue
        # jointure élèves-responsables
        for id_resp in responsables:
            try:
                my_resp = storage.findFirst(Responsable, Responsable.int_id==id_resp)
                if my_resp is not None:
                    # FIXME : attributs supplémentaires ?
                    JointureResponsableEleve(store=storage, eleve=my_eleve, responsable=my_resp)
            except Exception, err:
                log.infolog("erreur jointure eleve-resp", err)
        # niveau (MEF)
        niveau = groupe['filiere']
        my_niveau = storage.findOrCreate(Niveau, nom=niveau)
        # classe (format AAF : "numero$classe")
        classe = groupe['classe'].split('$')[1]
        my_classe = storage.findOrCreate(Classe, nom=classe,
                                         niveau=my_niveau)
        # affectation de l'élève
        my_eleve.classe = my_classe
        my_eleve.niveau = my_niveau

        # inscription aux options
        for option in options:
            my_option = storage.findOrCreate(Groupe, nom=option)
            JointureGroupeUser(store=storage, groupe=my_option, user=my_eleve)

    log.infolog("TOTAL : %d élèves" % num)

def parse_aaf_responsables(storage, responsables_file):
    """
    parsing des responsables depuis AAF
    """
    num = 0
    log.infolog("Lecture des responsables...", title=True)
    context, _ = parse_xml(responsables_file)
    mapping = {'sn':'nom',
               'givenName':'prenom',
               'ENTPersonDateNaissance':'date',
               'personalTitle':'civilite',
               #'ENTPersonNomPatro':'nom_patronymique', #FIXME : différent de sn ?
               'homePhone':'telephone',
               'telephoneNumber':'tel_pro', #à vérifier

    }
    mapping_adresse = {'ENTPersonAdresse':'adresse',
                       'ENTPersonCodePostal':'code_postal',
                       'ENTPersonVille':'ville',
                       'ENTPersonPays':'pays',
    }
    for _, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'PersRelEleve':
            log.infolog("%s n'est pas un responsable :(" % _type)
            continue
        #Attention : identifier/id == ENTPersonJointure
        respid = tresp.find('identifier/id').text
        responsable = {'int_id':unicode(respid)}
        adresse = {'int_id':unicode(respid)}
        for attr in tresp.findall('attributes/attr'):
            balise = attr.attrib['name']
            if mapping.has_key(balise):
                clean_text = replace_more_cars(attr.find('value').text)
                cle = mapping[balise]
                responsable[cle] = unicode(clean_text)
            elif mapping_adresse.has_key(balise):
                clean_text = replace_cars(attr.find('value').text)
                cle = mapping_adresse[balise]
                adresse[cle] = unicode(clean_text)
        # ENTPersonJointure #10027
        responsable['entpersonjointure'] = "AAF$%s" % responsable['int_id']
        if not_empty(responsable, 'civilite'):
            responsable['civilite'] = unicode(formate_civilite(str(responsable['civilite'])))
        try:
            resp = Responsable(store=storage, **responsable)
            num += 1
            if num % DEBUG_NUM == 0:
                log.debuglog("%d responsables lus..." % num)
        except TypeError, msg:
            log.infolog("Erreur sur le responsable %s : %s" % (respid, msg))
        try:
            addr = Adresse(store=storage, **adresse)
            # affectation de l'adresse au responsable
            resp.adresse = addr
        except TypeError, msg:
            log.infolog("Erreur sur l'adresse de %s : %s" % (respid, msg))
    log.infolog("TOTAL : %d responsables" % num)


##########################################
# Extraction AAF Professeurs + Administratifs ?
# Fichier :
# - ENT_$RNE_Complet_$DATE_PersEducNat_0000.xml
##########################################
def parse_aaf_profs(storage, profs_file):
    """
    parsing des professeurs depuis AAF
    """
    num = 0
    log.infolog("Lecture des personnels...", title=True)
    context, _ = parse_xml(profs_file)
    mapping = {'sn':'nom',
               'givenName':'prenom',
               'ENTPersonDateNaissance':'date',
               'personalTitle':'civilite',
               'ENTPersonNomPatro':'nom_patronymique',
               'mail':'mail',
    }
    for _, tprof in context:
        _type = tprof.find('operationalAttributes/attr/value').text
        if _type != 'PersEducNat':
            log.infolog("%s n'est pas un personnel :(" % _type)
            continue
        #Attention : identifier/id == ENTPersonJointure
        administratif = False
        profid = tprof.find('identifier/id').text
        professeur = {'int_id':unicode(profid)}
        classes = []
        matieres = []
        options = []
        principal = []
        groupe = None
        for attr in tprof.findall('attributes/attr'):
            balise = attr.attrib['name']
            if mapping.has_key(balise):
                clean_text = replace_more_cars(attr.find('value').text)
                cle = mapping[balise]
                professeur[cle] = unicode(clean_text)
            elif balise == 'ENTAuxEnsClasses':
                # classes enseignées
                for tclasse in attr.findall('value'):
                    if tclasse.text is not None and '$' in tclasse.text:
                        nom = unicode(replace_cars(tclasse.text.split('$')[1]))
                        my_classe = storage.findOrCreate(EnsClasse, nom=nom)
                        classes.append(my_classe)
            elif balise == 'ENTAuxEnsClassesPrincipal':
                # professeur principal
                for tclasse in attr.findall('value'):
                    if tclasse.text is not None and '$' in tclasse.text:
                        nom = unicode(replace_cars(tclasse.text.split('$')[1]))
                        my_classe = storage.findOrCreate(EnsClasse, nom=nom)
                        principal.append(my_classe)
            elif balise == 'ENTAuxEnsMatiereEnseignEtab':
                # matieres enseignees
                for tmat in attr.findall('value'):
                    if tmat.text is not None and '$' in tmat.text:
                        desc = tmat.text.split('$')[1]
                        # mapping des matières selon la nomenclature Sconet
                        if desc in MATIERES:
                            mat = {'nom':unicode(replace_cars(MATIERES[desc])),
                                   'description':unicode(replace_cars(desc))}
                        else:
                            mat = {'nom':unicode(replace_cars(desc))}
                        my_matiere = storage.findOrCreate(Matiere, **mat)
                        matieres.append(my_matiere)
            elif balise == 'ENTAuxEnsGroupes':
                # options enseignées
                for tgrp in attr.findall('value'):
                    if tgrp.text is not None and '$' in tgrp.text:
                        nom = unicode(replace_cars(tgrp.text.split('$')[1]))
                        my_groupe = storage.findOrCreate(Groupe, nom=nom)
                        options.append(my_groupe)
            elif balise == 'ENTAuxEnsCategoDiscipline':
                # Catégories de disciplines de poste (multi)
                enst = []
                for ens in attr.findall('value'):
                    if ens.text is not None and '$' in ens.text:
                        enst.append(replace_more_cars(ens.text))
                professeur['disciplines'] = unicode(enst)
            elif balise == 'ENTPersonFonctions':
                # groupes pour les administratifs
                for tgrp in attr.findall('value'):
                    if tgrp.text is not None and '$' in tgrp.text:
                        fonctions = tgrp.text.split('$')
                        # FIXME: "sans objet" #3926
                        if fonctions[1] not in ['-', 'ENS']:
                            administratif = True
                            nom = unicode(replace_cars(fonctions[1]))
                            desc = unicode(replace_cars(fonctions[2]))
                            groupe = storage.findOrCreate(Service, nom=nom, description=desc)
            #elif balise == 'PersEducNatPresenceDevantEleves':
            #    if attr.find('value').text == 'N':
            #        administratif = True
        # ENTPersonJointure #10027
        professeur['entpersonjointure'] = "AAF$%s" % professeur['int_id']
        if not_empty(professeur, 'civilite'):
            professeur['civilite'] = unicode(formate_civilite(str(professeur['civilite'])))
        else:
            # civilité arbitraire #2599
            log.infolog("Attribution arbitraire d'une civilité au personnel : %s %s" % (
                         str(professeur['prenom']), str(professeur['nom'])))
            professeur['civilite'] = u'1'
        if is_empty(professeur, 'date'):
            # date de naissance arbitraire #1730
            professeur['date'] = u'01/01/0001'
        if not administratif:
            # c'est un enseignant
            try:
                prof = Enseignant(store=storage, **professeur)
                num += 1
            except Exception, msg:
                log.infolog("Erreur sur l'enseignant %s : %s" % (profid, msg))
            # inscription aux classes
            for classe in classes:
                if classe in principal:
                    is_principal = True
                else:
                    is_principal = False
                JointureClasseEnseignant(store=storage, classe=classe,
                        enseignant=prof, profprincipal=is_principal)
            # inscription aux matieres
            for matiere in matieres:
                JointureMatiereEnseignant(store=storage, matiere=matiere,
                        enseignant=prof)
            # inscription aux options
            for option in options:
                JointureGroupeUser(store=storage, groupe=option,
                        user=prof)

        else:
            # c'est un administratif
            if professeur.has_key('disciplines'):
                del(professeur['disciplines'])
            try:
                admin = Administratif(store=storage, **professeur)
                num += 1
            except Exception, msg:
                log.infolog("Erreur sur le personnel %s : %s" % (profid, msg))
            if groupe is not None:
                admin.groupe = groupe
        if num % DEBUG_NUM == 0:
            log.debuglog("%d personnels lus..." % num)
    log.infolog("TOTAL : %d personnels" % num)

