# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# backend.py
#
# librairie pour la gestion des utilisateurs scribe
#
###########################################################################

from creole import parsedico
import backend_conf
from templates import *
from os import system, makedirs, symlink, chmod, popen
from os.path import join, isdir, isfile
import ldap

# nouveau backend #
from scribe.eleves import Eleve
from scribe.enseignants import Enseignant

dico = parsedico.parse_dico()
smb_serveur = dico["smb_netbios_name"]
conf_dir = "/usr/share/eole/backend/conf"
# filtres pour les differentes branches
branche = "ou=%s,ou=%s,ou=education,o=gouv,c=fr" % (dico["numero_etab"], dico["nom_academie"])
user_filters = "(objectclass=sambaSamAccount)(objectclass=inetOrgPerson)(!(description=Computer))"
group_filters = "(objectclass=sambaGroupMapping)(objectclass=posixGroup)"
station_filters = "(description=Computer)(objectclass=posixAccount)"
share_filters = "(objectClass=sambaFileShare)"
# filtrage sur typeadmin, sinon autre filtre possible :
#       (objectClass=Administrateur) pour prof,
#       (objectClass=Eleves) pour les élèves
eleve_filters = "%s(!(typeadmin=*))" % user_filters
prof_filters = "%s(typeadmin=*)" % user_filters
domain = {'restreint' : 'i-%s' % dico['domaine_messagerie_etab'],
          'internet'  : dico['domaine_messagerie_etab']}
mail_admin = "admin@%s" % domain['internet']


#####################
##   Utilitaires   ##
#####################

def formate_date(date, sep='/'):
    """
    formatte une date issue de l'annuaire
    retourne jj/mm/aaaa ou jj/mm/aa selon la date initiale
    """
    if len(date) == 8:
        return "%s%s%s%s%s" % (date[6:8], sep, date[4:6], sep, date[0:4])
    if len(date) == 6:
        return "%s%s%s%s%s" % (date[4:6], sep, date[2:4], sep, date[0:2])
    return date

def deformate_date(date):
    """
    formatte une date en aaaammjj pour l'annuaire
    (historique)
    formats acceptés : jj/mm/aa[aa], jj-mm-aa[aa], jjmmaa[aa]
    """
    date = date.replace('/','').replace('-','')
    if len(date) == 8:
        return date[4:8]+date[2:4]+date[0:2]
    if len(date) == 6:
        return date[4:6]+date[2:4]+date[0:2]
    else:
        # FIXME : format incorrect
        return date

def gen_ftpdir(user):
    """
    Gestion du répertoire ".ftp"
    """
    homedir = get_attr(user, 'homeDirectory')[0].strip()
    ftpdir = join(homedir, '.ftp')
    system("rm -rf %s" % ftpdir)
    makedirs(ftpdir)
    system('/bin/chown %s %s' % (user, ftpdir))
    system('setfacl -bk %s' % ftpdir)
    chmod(ftpdir, 0500)
    homedir = join(dico['home_path'], homedir[6:])
#    if dico['home_path'] != '/home':
#        homedir = homedir.replace('/home',dico['home_path'])
    symlink(join(homedir, 'perso'), join(ftpdir, 'perso'))
    user_groups = get_user_groups(user)
    for group in user_groups:
        for share, sharedir in get_group_sharedirs(group):
            if share not in ['icones$', 'groupes']:
                if dico['home_path'] != '/home':
                    sharedir = sharedir.replace('/home', dico['home_path'])
                symlink(sharedir, join(ftpdir, share))

def touch(fichier):
    """
    Création d'un fichier vide
    """
    ficp = open(fichier, 'a')
    ficp.close()

def gen_profil(profil, login):
    """
    Génération du sambaProfilePath selon le profil demandé
    """
    profil = int(profil)
    if profil == 2:
        # profil obligatoire #1
        return "\\\\%s\\netlogon\\profil" % smb_serveur
    if profil == 3:
        # profil obligatoire #2
        return "\\\\%s\\netlogon\\profil2" % smb_serveur
    if profil == 4:
        # profil itinérant
        return "\\\\%s\\%s\\profil" % (smb_serveur, login)
    # profil local à la station
    return ""

#####################
## Primitives LDAP ##
#####################

def ldapsearch(filtre, attrlist=None):
    """
    recherche dans l'annuaire
    """
    query = ldap.open(backend_conf.serveur)
    reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", backend_conf.ldap_passwd)
    result = query.search_s("o=gouv,c=fr", ldap.SCOPE_SUBTREE, filtre, attrlist)
    query.unbind()
    return result

def ldapadd(dn, data):
    """
    ajout d'une entrée dans l'annuaire
    """
    query = ldap.open(backend_conf.serveur)
    reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", backend_conf.ldap_passwd)
    result = query.add_s(dn, data)
    query.unbind()
    return result

def ldapdelete(dn):
    """
    suppression d'une entrée de l'annuaire
    """
    query = ldap.open(backend_conf.serveur)
    reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", backend_conf.ldap_passwd)
    result = query.delete(dn)
    query.unbind()
    return result

def ldapmodify(dn, attribute, value):
    """
    modification d'une entrée de l'annuaire
    """
    if value == '':
        value = None
    query = ldap.open(backend_conf.serveur)
    reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", backend_conf.ldap_passwd)
    try:
        query.modify_s(dn, [(ldap.MOD_REPLACE, attribute, value)])
    except:
        try:
            query.modify_s(dn, [(ldap.MOD_ADD, attribute, value)])
        except:
            raise Exception, "Erreur : L'attribut %s n'a pas pu être modifié" % attribute
    return True

def authenticate(user, password):
    """
    tentative de validation d'un couple login/mdp
    eleve ou prof
    """
    if user == '' or password == '':
        return False
    query = ldap.open(backend_conf.serveur)
    try:
        query.simple_bind_s("uid=%s,ou=local,ou=eleves,ou=utilisateurs,%s" % (user, branche), password)
        query.unbind()
        return True
    except ldap.INVALID_CREDENTIALS:
        try:
            query.simple_bind_s("uid=%s,ou=local,ou=personnels,ou=utilisateurs,%s" % (user, branche), password)
            query.unbind()
            return True
        except ldap.INVALID_CREDENTIALS:
            return False

def add_ou_responsables():
    """
    ajout de l'unité organisationnelle "responsables"
    """
    dn = "ou=responsables,ou=utilisateurs,%s" % (branche)
    data = [("objectClass","organizationalUnit"),
            ("ou", "responsables"),]
    try:
        ldapadd(dn, data)
    except:
        return False
    return True

############################
## fonctions de controle ##
############################

def is_user(user):
    """
    vérification de l'existance d'un utilisateur
    """
    result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), ['uid'])
    if result == []:
        return False
    else:
        return True

def is_system_user(user):
    """
    indique si le login proposé est déjà un utilisateur système
    """
    user = user.lower()
    passfile = open('/etc/passwd','r')
    passbuffer = passfile.readlines()
    passfile.close()
    for ligne in passbuffer:
        if user == ligne.split(':')[0]:
            return True
    return False

def is_admin(user):
    """ renvoi True si l'utilisateur fait partie de l'équipe pédagogique """
    if get_user_data(user).has_key('typeadmin'):
        return True
    return False

def is_group(group):
    """ renvoi True si le groupe existe """
    result = ldapsearch("(&%s(cn=%s))" % (group_filters, group), ['cn'])
    return result != []

############################
## fonctions de recherche ##
############################

def get_gid(user):
    """
    gid de l'utilisateur
    """
    result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), ['gidNumber'])
    try:
        result = ldapsearch("(&%s(gidNumber=%s))" % (group_filters, result[0][1]['gidNumber'][0]), ['cn'])
        return result[0][1]['cn'][0]
    except:
        return ''

def get_attr(user, attr):
    """
    renvoi le contenu d'un attribut utilisateur
    """
    result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), [attr])
    try:
        return result[0][1][attr]
    except:
        return []

def get_users():
    """
    liste des utilisateurs
    """
    result = ldapsearch("(&%s)" % user_filters, ['uid'])
    users = []
    for user in result:
        users.append(user[1]['uid'][0])
    users.sort()
    return users

def get_user_no_groups(user):
    """
    liste des groupes dans lesquels l'utilisateur est absent
    """
    result = ldapsearch("(&%s(!(memberUid=%s)))" % (group_filters, user), ['cn'])
    groups = []
    for group in result:
        groups.append(group[1]['cn'][0])
    groups.sort()
    return groups

def get_user_data(user):
    """
    renvoit tout les attributs de l'utilisateur en vrac
    """
    return ldapsearch("(&%s(uid=%s))" % (user_filters, user))[0][1]

def get_active_users(active=True):
    """
    liste des utilisateurs actifs (True)
    ou inactifs (False)
    """
    result = ldapsearch("(&%s)" % (user_filters), ['uid', 'sambaAcctFlags'])
    users = []
    for user in result:
        if user[1]['sambaAcctFlags'][0].count('DU'):
            if not active:
                users.append(user[1]['uid'][0])
        elif active:
            users.append(user[1]['uid'][0])
    return users

def get_admin_classes(user):
    """
    liste des classes administrées par un professeur
    """
    return get_attr(user, 'Divcod')

def get_workstations():
    """
    liste des postes de travail
    """
    result = ldapsearch("(&%s)" % station_filters, ['uid'])
    stats = []
    for stat in result:
        # les postes se terminent par $
        stats.append(stat[1]['uid'][0].replace("$",""))
    stats.sort()
    return stats

def get_user_groups(uid=''):
    """
    liste des groupes d'un utilisateur tous par défaut
    """
    if uid != "":
        filtre = "(&(memberUid=%s)%s)" % (uid, group_filters)
    else:
        filtre = group_filters
    result = ldapsearch(filtre, ['cn'])
    groups = []
    for group in result:
        groups.append(group[1]['cn'][0])
    groups.sort()
    return groups

def get_members(group):
    """
    liste des membres d'un groupe
    """
    result = ldapsearch("(&%s(cn=%s))" % (group_filters, group), ['memberUid'])
    try:
        res = result[0][1]['memberUid']
        res.sort()
        return res
    except:
        return []

def list_groups(_type=''):
    """ liste des groupes particuliers (par type)
    @param _type: Classe / Niveau / Groupe / Equipe / Matiere / Administrateurs
                  (tous sinon)
    renvoit les cn des groupes
    """
    filtre = ''
    if _type:
        filtre += "(description=%s *)" % _type
    result = ldapsearch("(&%s%s)" % (group_filters, filtre), ['cn'])
    groups = [res[1]['cn'][0] for res in result]
    return groups

def get_group_shares(group):
    """
    liste des partages liés à un groupe
    """
    result = ldapsearch("(&%s(sambaShareGroup=%s))" % (share_filters, group), ['sambaShareName'])
    list_shares = []
    for share in result:
        list_shares.append(share[1]['sambaShareName'][0])
    return list_shares

def get_group_maillist(group):
    """
    renvoie la liste de diffusion d'un groupe
    """
    result = ldapsearch("(&%s(cn=%s))" % (group_filters, group), ['mail'])
    if result != []:
        if result[0][1].has_key('mail'):
            return result[0][1]['mail']
    return []

def get_group_sharedirs(group):
    """
    liste des répertoires partagés liés à un groupe
    """
    result = ldapsearch("(&%s(sambaShareGroup=%s))" % (share_filters, group), ['sambaShareName', 'sambaFilePath'])
    list_shares = []
    for share in result:
        list_shares.append([share[1]['sambaShareName'][0], share[1]['sambaFilePath'][0]])
    return list_shares

# FIXME : à recoder ?
#sys.path.append('/usr/share/eole/backend/')
#import manage_special_shares
def get_shares():
    """ renvoie les partages non-eole sous la forme d'un dico
    de la forme {nom du partage:lettre associée (''par def)}
        partage eole est formé de la même manière
    """
    spe = manage_special_shares.specialShares()
    shares = spe.get_reserved_letters()
    ## on gère les partages eole
    for key in shares.keys():
        if key in get_eole_shares().keys():
            shares.pop(key)
    return shares

def get_eole_shares():
    """ renvoie les partages eole sous la forme d'un dico
        de la forme {nom du partage:lettre associée (''par def)}
    """
    shares = manage_special_shares.shares.eole_shares
    shares.update({'perso':'U:'})
    return shares

def set_share_letter(share_name, drive=''):
    """ modifie l'attribution d'une lettre de lecteur à un partage """
    if not share_name in get_shares().keys():
        return False
    spe = manage_special_shares.specialShares()
    if drive:
        return spe.add_special_share(share_name, drive)
    else:
        return spe.del_special_share(share_name)

def get_niveau(classe):
    """
    renvoie le niveau associé à une classe (ex : 3e1 -> 3eme)
    """
    result = ldapsearch("(&(cn=%s)%s)" % (classe, group_filters), ['niveau'])
    return result[0][1]["niveau"][0]

def get_classes(niveau='*'):
    """ renvoie la liste des cn des classes de l'annuaire rangées par niveau
    @niveau: niveau des classes demandées
    renvoie un dico {'nivo':[classes]}
    """
    result = ldapsearch("(&%s(niveau=%s))" % (group_filters, niveau), ['cn'])
    classes = {}
    for res in result:
        nivo = get_niveau(res[1]['cn'][0])
        if not classes.has_key(nivo):
            classes[nivo] = []
        classes[nivo].append(res[1]['cn'][0])
    return classes

def get_domains():
    """ renvoie la liste des domaines mails disponibles
        @monetab.monacad.fr et @i-monetab.monacad.fr
    """
    return [domain['restreint'], domain['internet']]

def get_eleves():
    """ renvoie la liste des élèves de l'annuaire """
    result = ldapsearch("(&%s)" % eleve_filters)
    eleves = [res[1]['uid'][0]for res in result]
    return eleves

def get_profs():
    """ renvoie la liste des profs de l'annuaire"""
    result = ldapsearch("(&%s)" % prof_filters)
    profs = [res[1]['uid'][0]for res in result]
    return profs

def get_mails():
    """ renvoie la liste des utilisateurs mail """
    result = ldapsearch('objectClass=eolemail', ['uid', 'mail'])
    mails = [(res[1]['uid'][0], res[1]['mail'][0]) for res in result]
    return mails

def get_quota(login):
    """ renvoit le quota disque pour un utilisateur """
    cmd = "/usr/bin/quota -w %s" % login
    output = popen(cmd)
    lines = output.readlines()
    output.close()
    try:
        nb_blocks = lines[2].split()[2]
    except IndexError:
        # pas de quotas pour cet utilisateur
        return 0
    else:
        # on retourne la limite en Mo
        return (int(nb_blocks))/1024

def get_profil(user):
    """ renvoit le profil d'un utilisateur
        1 local
        2 obligatoire-1
        3 obligatoire-2
        4 itinérant
    """
    profil = get_attr(user, 'sambaProfilePath')
    if profil == []:
        return 1
    if "profil2" in profil[0]:
        return 3
    if "netlogon" in profil[0]:
        return 2
    return 4

#def get_share_templates():
#    """
#    retourne la liste des modèles de partage disponibles
#    """
#    templates = []
#    files = listdir('/usr/share/horus/models/')
#    for file in files:
#        if file.endswith('.tmpl'):
#            templates.append(file[0:-5])
#    if 'standard' not in templates:
#        # il en faut au moins un
#        templates.append('standard')
#    return templates


#######################
## fonctions d'ajout ##
#######################


def inscription(user, group):
    """
    inscription d'un utilisateur dans un groupe
    + regénération du .ftp
    """
    res = system("/usr/sbin/smbldap-groupmod -m %s %s" % (user, group))
    gen_ftpdir(user)
    return res == 0


def ajout_eleve(login, password, nom, prenom, date, classe, numero, civilite='1',
        domaine='restreint', quota='10', profil='1', shell=False):
    """
    ajout d'un élève
    @param login  : login élève
    @param password : mot de passe
    @param nom : nom de famille
    @param prenom : prénom
    @param date : date de naissance (jj/mm/aa[aa] ou jjmmaa[aa])
    @param classe : classe de l'élève
    @param numero : numéro élève
    @param civilite : civilité (1=M., 2=Mme, 3=Mlle)
    @param domaine : domaine mail (restreint ou internet)
    @param quota : quota éléve
    @param profil : profil (1=local, 2=obligatoire-1, 3=obligatoire-2, 4=itinérant)
    @param shell : activation du shell
    """
    if domaine.startswith('i-'):
        domaine = 'restreint'

    eleve = {}
    eleve['login'] = login
    eleve['password'] = password
    eleve['nom'] = nom
    eleve['prenom'] = prenom
    eleve['date'] = date
    eleve['classe'] = classe
    eleve['numero'] = numero
    eleve['civilite'] = civilite
    eleve['domaine'] = domaine
    eleve['quota'] = quota
    eleve['profil'] = profil
    eleve['shell'] = shell

    ldapuser = Eleve()
    ldapuser.add_one(**eleve)

    return True


def ajout_prof(login, password, nom, prenom, date='', classe='', civilite='1',
    mail='local', quota='0', profil='1', shell=False):
    """
    ajout d'un professeur
    @param login  : login professeur
    @param password : mot de passe
    @param nom : nom de famille
    @param prenom : prénom
    @param date : date de naissance (jj/mm/aa[aa] ou jjmmaa[aa])
    @param classe : classe si professeur principal
    @param civilite : civilité (1=M., 2=Mme, 3=Mlle)
    @param mail : adresse mail (si "local" -> création d'une boîte sur le Scribe)
    @param quota : quota professeur
    @param profil : profil (1=local, 2=obligatoire-1, 3=obligatoire-2, 4=itinérant)
    """
    if date == '':
        date = '01/01/0000'

    prof = {}
    prof['login'] = login
    prof['password'] = password
    prof['prenom'] = prenom
    prof['nom'] = nom
    prof['civilite'] = civilite
    prof['date'] = date
    prof['mail'] = mail
    prof['quota'] = quota
    prof['profil'] = profil
    prof['shell'] = shell
    if classe != '':
        prof['classe'] = classe

    ldapuser = Enseignant()
    ldapuser.add_one(**prof)

    return True

def add_workstation(workstation):
    """
    ajoute une station de travail
    [ ne devrait pas être utilisé ]
    """
    ret = system("/usr/sbin/smbldap-useradd -w %s" % workstation)
    return ret == 0

def ajout_classe(classe, niveau, domaine='restreint'):
    """
    @param classe : nom de la classe
    @param niveau : niveau associé
    @param domaine : domaine de la liste de diffusion (restreint/internet)
    """
    if is_group(classe):
        return False
    try:
        system("/usr/share/eole/backend/creation-groupe.py -p -g '%s' -l '%s' -t 'Classe' -n '%s' -m '%s'" %
                (classe, domaine, niveau, mail_admin))
        return True
    except:
        return False

def ajout_groupe(nom, _type, domaine='', optrw=''):
    """
    Ajoute un groupe de type  Niveau, Matière ou Groupe
    @param nom : nom du groupe
    @param _type : type de groupe Niveau|Matiere|Groupe
    @param domaine : domaine de la liste de diffusion (restreint/internet), vide = pas de liste de diffusion
    @param optrw : statut du partage (''/'rw'/'ro'/'dt')
    """
    if is_group(nom):
        return False
    if is_user(nom) or is_group(nom):
        return False
    options = "-g '%s' -t '%s'" % (nom, _type)
    if optrw: # avec partage
        if optrw == 'rw':
            options += ' -p'
        elif optrw == 'ro':
            options += ' -r'
        elif optrw == 'dt':
            options += ' -d'
    if domaine: # avec liste de diffusion
        options += " -l '%s'" % domaine
        options += " -m '%s'" % mail_admin
    try:
        system("/usr/share/eole/backend/creation-groupe.py %s" % options)
        return True
    except:
        return False

def add_share(sharename, groupe, filepath=None, drive=None):
    """
    ajout d'un partage samba associé à un groupe donné
    """
    if filepath is None:
        filepath = "/home/workgroups/"+sharename
    # construction des données ldap
    dn = "cn=smb://%s/%s,ou=local,ou=partages,%s" % (smb_serveur, sharename, branche)
    data = [("objectClass","sambaFileShare"),
    ( "cn", "smb://"+smb_serveur+"/"+sharename),
    ("sambaShareName", sharename),
    ("description", sharename),
    ("sambaShareGroup", groupe),
    ("sambaFilePath", filepath),
    ("sambaShareURI", "\\\\%s\\%s" % (smb_serveur, sharename)),
    # ("sambaShareAdmin", "Administrateurs"),
    # ("location", "\\"),
    # ("server", smb_serveur)
    ]
    if drive is not None:
        if not drive.endswith(":"):
            drive += ':'
        if len(drive) != 2:
            return False
        data.append(("sambaShareDrive", drive))
    try:
        ldapadd(dn, data)
    except:
        return False
    for user in get_members(groupe):
        gen_ftpdir(user)
    return True

# FIXME : ne devrait plus être utilisé !
#def add_maillist(groupe, domaine='restreint'):
#    """
#    ajout d'une liste de diffusion à un groupe existant
#    """
#    ldico = {}
#    try:
#        res = ldapsearch("(&%s(cn=%s))" % (group_filters, groupe), ['description'])[0]
#        dn = res[0]
#        ldico['type'] = res[1]['description'][0].split()[0]
#        if ldico['type'] not in ['Classe', 'Niveau', 'Equipe', 'Matiere', 'Groupe']:
#            ldico['type'] = 'Groupe'
#    except:
#        print "Groupe non trouvé"
#        return False
#    if domaine == 'restreint':
#        path = join('/var/lib/sympa/expl', domain['restreint'], groupe)
#        ldico['ldomaine'] = domain['restreint']
#    else:
#        path = join('/var/lib/sympa/expl', groupe)
#        ldico['ldomaine'] = domain['internet']
#    if isdir(path):
#        print "la liste %s existe déjà" % groupe
#        return True
#    ldico['groupe'] = groupe
#    ldico['date'] = time.ctime()
#    ldico['branche'] = branche
#    ldico['email'] = mail_admin
#    config = liste_tmpl % ldico
#    makedirs(path)
#    ficp = open(join(path, 'config'), 'w')
#    ficp.write(config)
#    ficp.close()
#    touch(join(path, 'info'))
#    alias = alias_tmpl % ldico
#    ficp = open("/etc/mail/sympa.aliases", 'a')
#    ficp.write(alias)
#    ficp.close()
#    system('/bin/chown -R sympa:sympa %s' % path)
#    ldapmodify(dn, 'mail', "%(groupe)s@%(ldomaine)s" % ldico)
#    return True


##############################
## fonctions de modification ##
###############################

def set_profil(login, profile=None):
    """ modifie le sambaProfilePath d'un utilisateur 1-2-3-4"""
    if profile is not None:
        return set_attr(login, 'sambaProfilePath', gen_profil(profile, login))

def set_quota(login, quota=''):
    """ applique de nouveau quota disque pour login
        @quota: int (str admis)
    """
    if type(quota) == str:
        try:
            quota = int(quota)
        except:
            return False
    # test de la partition (/home ou /)
    ret = system('mount | grep -q " /home "')
    if ret == 0:
        mnt = '/home'
    else:
        mnt = '/'

    cmd = "/usr/bin/sudo /usr/sbin/quotatool -b -u %s -q %sM -l %dM %s"
    retour = system(cmd % (login, quota, (int(quota)*2), mnt))
    return not retour

def mod_user(login, homedrive=None, enable=None, profile=None, gid=None):
    """
    modification d'un utilisateur existant.
    """
    # les élément à ne pas modifier sont à None
    args = ''
    if homedrive is not None:
        args += '-D %s ' % homedrive

    if profile is not None:
        args += '-F "%s" ' % gen_profil(profile, login)

    if enable is not None:
        if enable == True:
            args += "-J "
        else:
            args += "-I "
    if gid is not None:
        args += '-g %s' % gid

    if args != '':
        result = system('/usr/sbin/smbldap-usermod %s %s' % (args, login))
        if result != 0:
            return False
        if gid is not None:
            # le changement de groupe principal ne met pas à jour l'appartenance au groupe
            inscription(login, gid)
    return True

def mod_password(login, passwd):
    """
    changement d'un mot de passe
    """
    passwd = passwd.replace('"', '\\"')
    result = system('/usr/share/eole/backend/passwd-eole.pl -u "%s" -p "%s"' % (login, passwd))
    return True

def password_must_change(login):
    """ changement d'un mot de passe """
    result = system('/usr/bin/net sam set pwdmustchangenow %s yes &>/dev/null' % (login))
    return result == 0

def set_active_users(users, active=True):
    """
    active/désactive les utilisateurs passés en paramètre
    """
    if type(users) != list:
        users = [users]
    if active:
        value = 'U'
    else:
        value = '[DU]'
    for user in users:
        try:
            set_attr(user, 'sambaAcctFlags', value)
        except:
            return False
    return True


def set_attr(user, attr, content):
    """
    modifie un attribut utilisateur
    """
    result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), [''])
    ldapmodify(result[0][0], attr, content)
    return True

def change_classe(user, new_classe):
    """
    Changement de classe
    @param user : login de l'élève
    @param new_classe : nouvelle classe de l'élève
    """
    old_classe = get_attr(user, 'Divcod')[0]
    old_niveau = get_attr(user, 'Meflcf')[0]
    new_niveau = get_niveau(new_classe)
    # modification utilisateur
    set_attr(user, 'Divcod', new_classe)
    set_attr(user, 'Meflcf', new_niveau)
    # désinscriptions
    system("/usr/sbin/smbldap-groupmod -x %s %s > /dev/null" % (user, old_classe))
    system("/usr/sbin/smbldap-groupmod -x %s %s > /dev/null" % (user, old_niveau))
    # inscriptions
    system("/usr/sbin/smbldap-groupmod -m %s %s > /dev/null" % (user, new_classe))
    system("/usr/sbin/smbldap-groupmod -m %s %s > /dev/null" % (user, new_niveau))
    # déplacement du lien sur le rép de l'élève
    system("rm /home/workgroups/profs-%s/eleves/%s > /dev/null" % (old_classe, user))
    system("ln -s %s/%s/%s/perso /home/workgroups/profs-%s/eleves/%s" % (dico['home_path'], user[0], user, new_classe, user))
    gen_ftpdir(user)
    return True


def set_admin_classes(user, classes=[]):
    """
    initialisation de la liste des classes administrées par un professeur
    si aucune classe n'est fournie, c'est que le prof n'est plus admin
    """
    if classes == []:
        set_attr(user, 'typeadmin', '0')
        set_attr(user, 'Divcod', 'autre')
    else:
        set_attr(user, 'typeadmin', '2')
        set_attr(user, 'Divcod', classes)
    return True

##############################
## fonctions de suppression ##
##############################

def desinscription(user, group):
    """
    désinscription d'un utilisateur d'un groupe
    + regénération du .ftp
    """
    res = system("/usr/sbin/smbldap-groupmod -x %s %s" % (user, group))
    gen_ftpdir(user)
    return res == 0


def del_user(user, remove_data=False):
    """
    suppression d'un utilisateur avec ou sans son répertoire personnel
    """
    if remove_data == True:
        res = system("/usr/sbin/smbldap-userdel -r "+user)
    else:
        res = system("/usr/sbin/smbldap-userdel "+user)
    return res == 0

def del_workstation(workstation):
    """
    suppression d'une station de travail
    """
    if not workstation.endswith('$'):
        workstation = workstation+'$'
    res = system("/usr/sbin/smbldap-userdel %s" % workstation)
    return res == 0


def del_group(group):
    """
    suppression d'un groupe
    Attention il faut au préalable avoir fait les vérifications d'usage :
       - il reste des membres
       - il reste des partages liés
    """
    res = system("/usr/sbin/smbldap-groupdel %s" % group)
    return res == 0


def del_share(share, rmdir=False, sync=True):
    """
    suppression d'un partage
    @rmdir: supprime les repertoires ?
    @sync: synchronisé avec samba ?
    """
    result = ldapsearch("(&%s(sambaShareName=%s))" % (share_filters, share), ['sambaFilePath'])
    try:
        if rmdir:
            sharedir = result[0][1]['sambaFilePath'][0]
            system("rm -rf %s"%sharedir)
        ldapdelete(result[0][0])
    except:
        return False
    if sync:
        synchronize()
    return True


##########################
## gestion des fichiers ##
##########################

def synchronize(restart=True):
    """
    ajout des partages dans le fichier smb.conf
    """
    fic = open('/etc/sysconfig/eole/global_smb.tmpl')
    smb = [fic.read()]
    fic.close()

    result = ldapsearch(share_filters)
    for share in result:
        dico_share = {}
        dico_share['name'] = share[1]['sambaShareName'][0]
        dico_share['path'] = share[1]['sambaFilePath'][0]
        dico_share['group'] = share[1]['sambaShareGroup'][0]
        dico_share['desc'] = share[1]['description'][0]
        # création du répertoire du partage si nécessaire
        if not dico_share['path'].startswith('%') and not isdir(dico_share['path']):
            makedirs(dico_share['path'])
            chmod(dico_share['path'], 0700)
            system('setfacl -Rm g:%(group)s:rwx %(path)s' % dico_share)
            system('setfacl -Rdm g:%(group)s:rwx %(path)s' % dico_share)
        # génération des entrées partages pour le smb.conf
        if isfile('%s/%s.conf' % (conf_dir, dico_share['name'])):
            print "partage personnalisé : %s" % dico_share['name']
            fic = open('%s/%s.conf' % (conf_dir, dico_share['name']))
            smb.append(fic.read())
            fic.close()
        else:
            smb.append(share_tmpl % dico_share)

    # écriture du fichier final
    fic = open('/etc/samba/smb.conf', 'w')
    fic.writelines(smb)
    fic.close()

    if restart:
        system('/etc/init.d/samba restart')

