# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################
"""
 librairie pour le parsing des fichiers de données

 - parse_be1d_eleves : parsing des fichiers CSV issus de BE1D

"""
from csv import DictReader, reader
from axiom.attributes import AND
from scribe.eoletools import replace_cars, convert_file, formate_date, \
clean_date, formate_civilite, replace_more_cars
from scribe.importation import log
from scribe.importation.config import DEBUG_NUM
from scribe.storage import Eleve, Niveau, Classe, Responsable, Adresse, \
JointureResponsableEleve

##########################################
# Extraction Be1d Elèves
# Fichier :
# - ecole.csv
##########################################
def parse_be1d_eleves(storage, csvfile):
    """
    parsing des élèves depuis Be1d
    """
    num = 0
    log.infolog("Lecture des élèves...", title=True)
    # fichier en UTF-8 ?
    convert_file(csvfile)
    # champs obligatoires
    fields = ['Nom Elève', 'Prénom Elève', 'Date naissance', 'Sexe',
            'Niveau','Classe']
    reader = DictReader(file(csvfile), delimiter=';')
    # validation en-tête
    #ATTENTION : ça commence direct par le 1er élève
    data = reader.next()
    for field in fields:
        if not data.has_key(field):
            msg = "Le champ \"%s\" n'est pas présent dans l'en-tête du fichier !" % field
            log.errorlog(msg)
            raise Exception(msg)
    while True:
        eleve = {}
        eleve['prenom'] = unicode(replace_more_cars(data['Prénom Elève']))
        eleve['nom'] = unicode(replace_more_cars(data['Nom Elève']))
        eleve['date'] = unicode(formate_date(data['Date naissance']))
        #FIXME : trouver mieux comme numéro élève ?
        eleve['numero'] = unicode(formate_date(data['Date naissance'], ''))
        # pas d'identifiant interne au logiciel
        eleve['int_id'] = u''
        if data['Sexe'] == 'F':
            eleve['civilite'] = u'3' #Mlle
        else:
            eleve['civilite'] = u'1' #M.
        try:
            my_eleve = Eleve(store=storage, **eleve)
            niveau = unicode(replace_cars(data['Niveau']))
            if not niveau:
                raise TypeError("niveau invalide")
            classe = unicode(replace_cars(data['Classe']))
            if not classe:
                raise TypeError("classe invalide")
            my_niveau = storage.findOrCreate(Niveau, nom=niveau)
            my_classe = storage.findOrCreate(Classe, nom=classe,
                                             niveau=my_niveau)
            # affectation de l'élève
            my_eleve.classe = my_classe
            my_eleve.niveau = my_niveau
            num += 1
            if num % DEBUG_NUM == 0:
                log.debuglog("%d élèves lus..." % num)
        except TypeError, msg:
            log.errorlog("Erreur sur l'élève %s : %s" % (data['Nom Elève'],
                                                         msg))
        try:
            data = reader.next()
        except StopIteration:
            break
    log.infolog("TOTAL : %d élèves" % num)


##########################################
# Extraction Be1d Responsables
# Fichier :
# - parents.csv
##########################################
def parse_be1d_responsables(storage, csvfile):
    """
    parsing des responsables depuis Be1d
    """
    num = 0
    log.infolog("Lecture des responsables...", title=True)
    # fichier en UTF-8 ?
    convert_file(csvfile)
    # champs obligatoires
    fields = ["Nom responsable", "Prénom responsable", "Civilité Responsable",
              "Adresse responsable", "CP responsable", "Commune responsable",
              "Pays", "Courriel", "Téléphone domicile", "Téléphone travail",
              "Téléphone portable"]
    multi_fields = ["Nom de famille enfant", "Prénom enfant", "Classes enfants"]
    # champs inutilisés
    #"Nom usage responsable", "Nom d'usage enfant", "Numéro de poste"

    # on a des champs qui ont le même nom -> on ne peut pas utiliser DictReader :(
    csvreader = reader(open(csvfile), delimiter=';')
    # validation en-tête
    indice = {}
    header = csvreader.next()
    for field in fields:
        if field not in header:
            msg = "Le champ \"%s\" n'est pas présent dans l'en-tête du fichier !" % field
            log.errorlog(msg)
            raise Exception(msg)
        indice[field] = header.index(field)
    # indice le plus élevé parmi les champs obligatoires
    top = max(indice.values())
    for field in multi_fields:
        if field not in header:
            msg = "Le champ \"%s\" n'est pas présent dans l'en-tête du fichier !" % field
            log.errorlog(msg)
            raise Exception(msg)
        indice[field] = []
        for ind in range(len(header)):
            if header[ind] == field:
                indice[field].append(ind)
    # nombre de champs élève prévus dans le CSV
    num_eleves = len(indice["Nom de famille enfant"])

    mapping = { 'Nom responsable':'nom',
                'Prénom responsable':'prenom',
                'Civilité Responsable':'civilite',
                "Courriel":'mail',
                "Téléphone domicile":'telephone',
                "Téléphone travail":'tel_pro',
                "Téléphone portable":'tel_portable',
    }
    mapping_adresse = { "Adresse responsable":'adresse',
                        "CP responsable":'code_postal',
                        "Commune responsable":'ville',
                        "Pays":'pays',
    }

    for data in csvreader:
        ## entrée responsable
        if len(data) <= top:
            # ligne invalide
            continue
        responsable = {'int_id':u''}
        for attr in mapping:
            clean_text = replace_more_cars(data[indice[attr]])
            cle = mapping[attr]
            responsable[cle] = unicode(clean_text)
        if not responsable['nom']:
            continue
        responsable['civilite'] = unicode(formate_civilite(responsable['civilite']))
        try:
            resp = Responsable(store=storage, **responsable)
            num += 1
            if num % DEBUG_NUM == 0:
                log.debuglog("%d responsables lus..." % num)
        except TypeError, msg:
            log.infolog("Erreur sur le responsable %s %s : %s" % (responsable['prenom'],
                                                      responsable['nom'], msg))
            continue

        ## liens responsable - élève
        for num_eleve in range(num_eleves):
            try:
                nom = unicode(replace_more_cars(data[indice["Nom de famille enfant"][num_eleve]]))
                if not nom:
                    continue
                prenom = unicode(replace_more_cars(data[indice["Prénom enfant"][num_eleve]]))
                classe = unicode(replace_cars(data[indice["Classes enfants"][num_eleve]]))
            except IndexError:
                # cf. http://dev-eole.ac-dijon.fr/issues/638
                continue
            # recherche de l'élève :
            my_eleve = None
            my_eleves = list(storage.query(Eleve, AND(Eleve.nom == nom,
                                                     Eleve.prenom == prenom)))
            if len(my_eleves) == 0:
                # élève non trouvé
                log.infolog("Elève %s %s non touvé !" % (str(prenom), str(nom)))
                continue
            elif len(my_eleves) == 1:
                my_eleve = my_eleves[0]
            else:
                # homonymes -> on vérifie la classe
                for eleve in my_eleves:
                    if eleve.classe.nom == classe:
                        my_eleve = eleve
                        break
            if my_eleve is not None:
                JointureResponsableEleve(store=storage, eleve=my_eleve,
                                         responsable=resp)

        ## entrée adresse
        adresse = {'int_id':u''}
        for attr in mapping_adresse:
            clean_text = replace_cars(data[indice[attr]])
            cle = mapping_adresse[attr]
            adresse[cle] = unicode(clean_text)
        try:
            addr = Adresse(store=storage, **adresse)
            # affectation de l'adresse au responsable
            resp.adresse = addr
        except TypeError, msg:
            log.infolog("Erreur sur l'adresse de %s %s : %s" % (responsable['prenom'],
                                                      responsable['nom'], msg))
    log.infolog("TOTAL : %d responsables" % num)
