#-*-coding:utf-8-*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# eoleldap.py
#
# librairie pour la connexion à un serveur ldap
#
###########################################################################
"""
Librairie Ldap pour Scribe
"""
from ldapconf import SUFFIX, ROOT_DN, USER_FILTER, GROUP_FILTER, SHARE_FILTER, \
    SUPPORT_ETAB, ldap_server, ldap_passwd, num_etab, BRANCHE_GROUP_ETAB
from scribe.errors import LdapExistingGroup, LdapExistingUser, \
SystemExistingUser, NiveauNotFound
from eoletools import to_list
import ldap

def is_system_user(user):
    """
        indique si le login proposé est déjà un utilisateur système
    """
    user = user.lower()
    passfile = open('/etc/passwd','r')
    passbuffer = passfile.readlines()
    passfile.close()
    for ligne in passbuffer:
        if user == ligne.split(':')[0]:
            return True
    return False

class Ldap(object):
    """
        Interface de connection à un serveur ldap
    """

    def __init__(self, serveur=None, passwd=None, binddn=None):
        if serveur is None:
            self.serveur = ldap_server
        else:
            self.serveur = serveur
        if passwd is None:
            self.passwd = ldap_passwd
            self.local_passwd = True
        else:
            self.passwd = passwd
            self.local_passwd = False
        if binddn == None:
            self.binddn = ROOT_DN
        else:
            self.binddn = binddn
        self.connexion = None

    def reload_pwd(self):
        """
            recharge le mot de passe
        """
        if self.local_passwd:
            #reload(backend_conf)
            self.passwd = ldap_passwd

    def connect(self):
        """
            binding ldap si nécessaire
        """
        if not self.connexion:
            # FIXME : inutile avec execfile
            #self.reload_pwd()
            self.connexion = ldap.open(self.serveur)
            self.connexion.simple_bind_s(self.binddn, self.passwd)

    def close(self):
        """
            fermeture de la connexion ldap
        """
        if self.connexion:
            self.connexion.unbind()
            self.connexion = None

    def _add(self, dn , data):
        """
            ajout d'une entrée ldap
        """
        self.connexion.add_s(dn, data)

    def _delete(self, dn):
        """
            suppression d'une entrée ldap
        """
        self.connexion.delete(dn)

    def _modify(self, dn, data):
        """
            modification d'une entrée ldap
        """
        self.connexion.modify_s(dn, data)

    def _search(self, filtre, attrlist=None, suffix=None):
        """
            recherche dans l'annuaire
        """
        if suffix == None:
            suffix = SUFFIX
        attrlist = to_list(attrlist)
        return self.connexion.search_s(suffix, ldap.SCOPE_SUBTREE,
                                filtre, attrlist)

    def _search_one(self, filtre, attrlist=None, suffix=None):
        """
           recherche une entrée dans l'annuaire
        """
        result = self._search(filtre, attrlist, suffix=suffix)
        if len(result) > 0 and len(result[0]) == 2:
            return result[0][1]
        else:
            return {}

class LdapEntry(object):
    """
        classe de base pour gérer les entrées ldap
    """
    def __init__(self, serveur=None, passwd=None):
        self.serveur = serveur
        self.passwd = passwd
        self.ldap_admin = Ldap(serveur, passwd)
        self.cache_etab = {'login': {}, 'group': {}}

    def _is_group(self, name):
        """
            test si le groupe existe dans l'annuaire
        """
        cnfilter = "(&%s(cn=%s))" % (GROUP_FILTER, name)
        if self.ldap_admin._search_one(cnfilter):
            return True
        return False

    def _is_user(self, name):
        """
            test si l'utilisateur existe dans l'annuaire
        """
        uidfilter = "(&%s(uid=%s))" % (USER_FILTER, name)
        if self.ldap_admin._search_one(uidfilter):
            return True
        return False

    def _is_share(self, name):
        """
            test si le partage existe dans l'annuaire
        """
        shfilter = "(&%s(sambaShareName=%s))" % (SHARE_FILTER, name)
        if self.ldap_admin._search_one(shfilter):
            return True
        return False

    def is_available_name(self, name):
        self.ldap_admin.connect()
        res = self._is_available_name(name)
        self.ldap_admin.close()
        return res

    def _is_available_name(self, name):
        """
            teste la disponibilité d'un uid ou un cn
        """
        if self._is_group(name):
            return False
        elif self._is_user(name):
            return False
        elif is_system_user(name):
            return False
        return True

    def _test_available_name(self, name):
        """
            Test la disponibilité d'un nom
            raise une exception si pas disponible
        """
        if self._is_group(name):
            raise LdapExistingGroup
        elif self._is_user(name):
            raise LdapExistingUser
        elif is_system_user(name):
            raise SystemExistingUser
        return True

    def get_niveau(self, classe):
        """
            Retourne le niveau associé à la classe
        """
        self.ldap_admin.connect()
        res = self._get_niveau(classe)
        self.ldap_admin.close()
        return res

    def _get_niveau(self, classe):
        res = self.ldap_admin._search_one("(&%s(cn=%s))" % (GROUP_FILTER,
            classe), ['niveau'])
        try:
            niveau = res['niveau'][0]
        except KeyError:
            raise NiveauNotFound("Impossible de trouver le niveau associé à la classe %s" % classe)
        return niveau

    def get_group_sharedirs(self, group):
        """
            renvoie la liste des partages associés au groupe (mode déconnecté)
        """
        self.ldap_admin.connect()
        res = self._get_group_sharedirs(group)
        self.ldap_admin.close()
        return res

    def _get_group_sharedirs(self, group):
        """
            renvoie la liste des partages associés au groupe
        """
        res = self.ldap_admin._search("(&%s(sambaShareGroup=%s))" % (SHARE_FILTER, group),
                                ['sambaShareName', 'sambaFilePath'])
        list_shares = []
        for share in res:
            list_shares.append([share[1]['sambaShareName'][0],
                                share[1]['sambaFilePath'][0],])
        return list_shares

    def _get_group_logon_shares(self, group):
        """
            renvoie la liste des partages associés au groupe
            - pour la génération du script de logon -
        """
        res = self.ldap_admin._search("(&%s(sambaShareGroup=%s))" % (SHARE_FILTER, group),
                ['sambaShareName', 'sambaFilePath', 'sambaShareURI', 'sambaShareDrive', ])
        list_shares = []
        for share in res:
            if share[1].has_key('sambaShareDrive'):
                drive = share[1]['sambaShareDrive'][0]
            else:
                drive = ''
            list_shares.append({'name':share[1]['sambaShareName'][0],
                                'path':share[1]['sambaFilePath'][0],
                                'uri':share[1]['sambaShareURI'][0],
                                'drive':drive,
                              })
        return list_shares

    def get_members(self, group, etab=None):
        """
        renvoie la listes des membres d'un groupe
        """
        self.ldap_admin.connect()
        res = self._get_members(group, etab=etab)
        self.ldap_admin.close()
        return res

    def _get_members(self, group, etab=None):
        """
        renvoie la listes des membres d'un groupe
        """
        if etab is None:
            suffix = None
        else:
            suffix = BRANCHE_GROUP_ETAB % {'etab': etab}
        res = self.ldap_admin._search_one("(&%s(cn=%s))" % (GROUP_FILTER,
                    group), ['memberUid'], suffix=suffix)
        if res.has_key('memberUid'):
            res = res['memberUid']
            res.sort()
            return res
        else:
            return []

    def _get_user_groups(self, login):
        """
        renvoit la liste des groupes d'un utilisateur
        """
        user_groups = []
        res = self.ldap_admin._search("(&%s(memberUid=%s))" % (GROUP_FILTER,
                     login), ['cn'])
        for grp in res:
            user_groups.append(grp[1]['cn'][0])
        return user_groups

    def _get_users(self, filtre='', attrs=['uid']):
        """
        recherche d'utilisateurs
        """
        users = []
        res = self.ldap_admin._search("(&%s%s)" % (USER_FILTER, filtre), attrs)
        for user in res:
            if len(attrs) == 1:
                users.append(user[1][attrs[0]][0])
            else:
                users.append(user[1])
        return users

    def _get_etab(self, _type, name, ldap_filter, attr):
        """
            récupère le numéro d'etab du cache ou requète LDAP
        """
        if not SUPPORT_ETAB:
            return None
        if name in self.cache_etab[_type]:
            return self.cache_etab[_type][name]
        need_close = False
        if self.ldap_admin.connexion == None:
            self.ldap_admin.connect()
            need_close = True
        dn = self.ldap_admin._search(ldap_filter, [attr])[0][0]
        if need_close:
            self.ldap_admin.close()
        etab = dn.split(',ou=')[-3]
        self.cache_etab[_type][name] = etab
        return etab

    def get_etab_from_group(self, group):
        try:
            return self._get_etab('group', group, "(&%s(cn=%s))" % (GROUP_FILTER,
                    group), 'cn')
        except:
            return num_etab
