#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys
import argparse
import logging
import os.path
import cPickle as pickle

# Find right direction when running from source tree
sys.path.insert(0, "/usr/local/samba/lib/python2.7/site-packages")

from regex import compile as recomp
from regex import search

from yaml import load, dump, YAMLObject, safe_load, safe_dump
try:
    from yaml import CLoader as Loader, CDumper as Dumper
except ImportError:
    from yaml import Loader, Dumper

from samba.samdb import SamDB
from samba import ldb
from samba import dsdb
from samba.param import LoadParm
from samba.auth import system_session
from samba.credentials import Credentials, AUTO_USE_KERBEROS, MUST_USE_KERBEROS

import getpass
import unicodedata

global logger


# Construction d’une liste ordonnée des règles de substitution
def pattern_from_list(patterns):
    return recomp(r"|".join([r"({})".format(p) for p in patterns]))


SUBSTITUTION_RULES = []

SUBSTITUTION_RULES.append(('', pattern_from_list([
                                                  r"^\s+",
                                                  r"\s+$",
                                                  ])))

SUBSTITUTION_RULES.append(('-', pattern_from_list([
                                                   r"\.-\.",
                                                   ])))

#SUBSTITUTION_RULES.append(('-', pattern_from_list([
#                                                   r"\s+",
#                                                   ])))

#SUBSTITUTION_RULES.append(('-', pattern_from_list([
#                                                   r"---",
#                                                   ])))

SUBSTITUTION_RULES.append(('-', pattern_from_list([
                                                  r" - ",
                                                  ])))

SUBSTITUTION_RULES.append(('-', pattern_from_list([
                                                  r"\s+",
                                                  ])))

SUBSTITUTION_RULES.append(('', pattern_from_list([
                                                  r"[^A-Za-z0-9\.'-]",
                                                  ])))

SUBSTITUTION_RULES.append(('', pattern_from_list([
                                                  r"\.\.",
                                                  ])))

SUBSTITUTION_RULES.append(('', pattern_from_list([
                                                  r"\s+",
                                                  ])))

SUBSTITUTION_RULES.append(('', pattern_from_list([
                                                  r"^\.",
                                                  r"\.$",
                                                  ])))

SUBSTITUTION_RULES.append(('-', pattern_from_list([
                                                   r"'",
                                                   ])))


DIV_SEP = recomp(r"(?:(?<!\\),)?(?:(?:ou=)|(?:dc=))")
DIV_ELEMENT = recomp(r'(?:((ou=)|(dc=))(?P<div>[^,]+)(?:,)?)+')


def filter_content(content, substitution_rules=SUBSTITUTION_RULES):
    """Return content with forbidden patterns substituted.
    :param content: value that may content forbidden patterns
    :type content: str
    """
    for substitute, pattern in substitution_rules:
        content = pattern.sub(substitute, content)
    return content


def normalize(div):
    uni_div = div.decode('utf-8')
    nfkd_form = unicodedata.normalize('NFKD', uni_div)
    div = nfkd_form.encode('ASCII', 'ignore')
    return div


def extract_division_elements(raw_division):
    elements = DIV_ELEMENT.search(raw_division).captures('div')
    return elements


class Configuration:
    """
    Handle configuration for the script
    Load rules and configurations from a file.
    """
    def __init__(self, cfile):
        self.cfile = cfile
        self.rules = []
        self.general = {}

    def rules(self, rules):
        """ Define the loaded rules """
        self.rules = rules

    def general(self, general):
        """ Define the loaded general configuration """
        self.general = general

    def load(self):
        """ Load the configuration from a file"""
        data = None
        with open(self.cfile, 'r') as infile:
            data = load(infile)
            for rule in data['ExtractRules']:
                params = "ExtractRule("
                indx = len(rule.keys())
                for key in rule.keys():
                    params += "{0}=rule['{0}']".format(key)
                    if indx != 1:
                        params += ', '
                    else:
                        params += ')'
                    indx = indx - 1

                self.rules.append(eval(params))
            if 'General' in data.keys():
                self.general = data['General']
        return True

    def dump(self):
        """ Write configuration to a file only use for devel matters"""
        with open(self.cfile, 'w') as outfile:
            data = { 'ExtractRules': [] }
            for rl in self.rules:
                data['ExtractRules'].append(rl.__dict__)
            dump(data, outfile, default_flow_style=False)
            for gn in self.general:
                dump(gn, outfile, default_flow_style=False)
        return True


class ExtractRule(object):
    """ Define the group extraction rules
    :param name: The rule name
    :type name: `String`
    :param separator: Maker to identify if the rule applies
    :type separator: `String`
    :param offset: If you want to ignore fileds befor the "seaprator"
    :type offset: `Int`
    :param ignore: List of strings to ignore (ac, gouv, fr)
    :type ignore: `List`
    :param match: Regexp to match group names
    :type match: `String`
    :param code: Regexp to create group names
    :type code: `String`
    :param suffix: List of group suffixies
    :type suffix: `List`
    :param affectRule: Rules to put users in created groups
    :type affectRule: `Dict`
    :param destOU: Default destination OU
    :type affectRule: `String`
    """
    def __new__(cls, name="", separator="", offset=0, ignore=[],
                match=None, code=None, suffix=[], affectRule=[],
                destOU=None):
        instance = super(ExtractRule, cls).__new__(cls)
        instance.name = name
        instance.separator = separator
        instance.offset = offset
        instance.ignore = ignore
        instance.match = match
        instance.code = code
        instance.suffix = suffix
        instance.affectRule = affectRule
        instance.destOU = destOU
        return instance

    def show(self):
        print("  Rule Separator : {0}".format(self.separator))
        print("  Rule Offset    : {0}".format(self.offset))
        print("  Rule To remove : {0}".format(self.ignore))


class DirectoryUser(object):
    """ User in directory
    :param samaccountname: the samAccountName attribute in directory
    :type samAccountName: `String`
    :param rawDivision: the division attribute in directory
    :type rawDivision: `String`
    :param mail: The mail attribute in directory
    :type mail: `String`
    :param rule: The group generation rule based on 'separator' identification
    :type rule: `ExtractRule`
    :param divisions: The divisions extracted with the "rule"
    :type divisions: `List`
    :param groups: The list of generated groups for user
    :type groups: `List`
    :param config; General configuration object
    :type config: `Dict`
    """
    def __init__(self, name, division, mail, rdiv, rule):
        self.samaccountname = name
        self.rawDivision = rdiv
        self.mail = mail
        self.rule = rule
        self.divisions = self.__extractDivision__(division)
        self.groups = self.__genGroups__()

    def __extractDivision__(self, div):
        """ Extract divisions from the division attribute collected in the
        directory

        The filed is splitted on the ',' then we try to igonre listed fileds
        if it's the case. If it's not we try to match the regexp.
        If we don't have any ignore or any match we use the separator and the
        offset to create the 'division list'. We take all the fileds before
        separator remove the offset and we have a list.
        """
        if self.rule:
           if self.rule.ignore:
               div = [filter_content(el) for el in reversed(div) if el not in self.rule.ignore]
           elif self.rule.match:
               match = search(self.rule.match, self.rawDivision.lower())
               if match:
                   div = eval(self.rule.code)
                   div = [filter_content(d) for d in div]
               else:
                   div = []
           elif self.rule.separator:
               if self.rule.separator in division:
                   indx = div.index(self.rule.separator) - self.rule.offset
                   div = div[:indx]
                   div.pop(0)
                   div.reverse()
                   div = [filter_content.sub(d) for d in div]
               else:
                   div = []
           else:
               div = []
           return div

    def debug(self,section,message):
        """ Debug function for debug purpose """
        print("[DEBUG][{0}][{1}]".format(section, message))

    def __isMemberOf__(self, rule, group):
        """ Check if a user is memeber of a group according to the affect
        rule.

        Only supports "contain" key word for now
        """
        if rule['cond'] == "contain":
            if rule['value'] == '*':
                #self.debug("   MATCH FOR",group)
                return True

            if rule['value'] in self.__getattribute__(rule['attr']):
                #self.debug("   MATCH FOR",group)
                return True
            else:
                #self.debug("   NO MATCH FOR",group)
                return False

    def __cleanMembership__(self, groups):
        """Return group list purged leaving only leaves for affectRules
        with recursive == False
        """
        # lister les suffixes avec la propriété leave_only à Vrai
        # pour chacun de ces suffixes, lister les groupes qui ne sont pas des extrêmités
        # la liste de groupes finale est la différence entre la liste initiale et l’ensemble des groupes listés pour chaque suffixe.
        def re_in_list(re, list_of_str):
            truth = False
            for el in list_of_str:
                if search(re, el) and re != el:
                    truth = True
                    break
            return truth

        superfluous_groups = []
        leaves_only_suffixes = [rule['suffix'] for rule in self.rule.affectRule
                                if rule['leaves_only'] == True]
        for leaves_only_suffix in leaves_only_suffixes:
            suffixed_groups = [group[0].rpartition(leaves_only_suffix)[0] for group in groups
                               if group[0].endswith(leaves_only_suffix if leaves_only_suffix is not None else '')]
            for suffixed_group in suffixed_groups:
                if re_in_list(suffixed_group, suffixed_groups):
                    superfluous_groups.extend([group for group in groups
                                               if group[0] == suffixed_group + leaves_only_suffix])
        return list(set(groups).difference(set(superfluous_groups)))

    def __genGroups__(self):
        """ Generate group list for a user
        If the user match the "membership" rule the group is added to his
        group list.

        All the groups are generated, the root group dans the subgroups.
        for exemple for the divisions [ 'ac', 'cpii', 'pne', 'sys' ]
        we genearte this groups :
            - ac
            - ac.cpii
            - ac.cpii.pne
            - ac.cpii.pne.sys

        If we provide suffixes ([-gouv, -id2 ]) we alsot generate groups
        with suffixes like :
            - ac
            - ac-gouv
            - ac-i2
            - ac.cpii
            - ac.cpii-gouv
            - ac.cpii-i2
            - ac.cpii.pne
            - ac.cpii.pne-gouv
            - ac.cpii.pne-i2
            - ac.cpii.pne.sys
            - ac.cpii.pne.sys-gouv
            - ac.cpii.pne.sys-i2
        """
        grps = []
        root = None
        if self.divisions:
            for grp in self.divisions:
                #self.debug("FOR", grp)
                if root:
                    grp_name = root + "." + grp
                    #self.debug("grp_name", grp_name)
                else:
                    grp_name = grp
                    #self.debug("grp_name", grp_name)

                root=grp_name
                #self.debug("root", grp_name)

                if self.rule.suffix:
                    for sf in self.rule.suffix:
                        #self.debug("sf",sf)
                        if sf is None:
                            group = grp_name
                            #self.debug("group", group)
                        else:
                            group = "{0}{1}".format(grp_name, sf)
                            #self.debug("group", group)

                        for rule in self.rule.affectRule:
                            #self.debug("RULE", rule)
                            if rule['suffix'] == sf:
                                  if self.__isMemberOf__(rule,group):
                                      grps.append((group,self.rule.destOU))
                                      #self.debug("APPEND TO GROUPS", grps)

                            else:
                                continue
                else:
                    grps.append((grp_name, self.rule.destOU))
                    #self.debug("NOSUFFIXAPPEND",grps)

            return self.__cleanMembership__(grps)
        else:
            return []


    def show(self):
        """ Print the obejct """
        print("================================================================================================")
        print("Name: {0}\nDivision: {1}\nMail:{2}".format(
            self.samaccountname,
            self.divisions,
            self.mail))
        print("Groups :")
        for grp in self.groups:
            print("\t{0}".format(grp))
        print("================================================================================================")
        print("Rule :")
        if self.rule:
            self.rule.show()

class DirectoryData:
    """ Data collected from the Samba4 directory
    """

    def __init__(self, rules, config):
        """ The collected data
        :param samDB: Samba "connection"
        :type samDB: `SamDB`
        :param domain_dn: The domain dn configured in samba
        :type domain_dn: Samba domain DN
        :param rules: The extraction rules
        :type rules: `List` of `ExtractRule` instance
        :param rawUserData: The user data
        :type rawUserData: `List` of `DirectoryUser` instance
        :param groups: `List` of groups to create in samba
        :type groups: `List`
        :param sambaGroups: Founed groups in samba
        :type sambaGroups: `List`
        :param config: General configuration
        :type sourceAttr: `String`
        :param sourceAttr: Group extraction string source attribute
        """
        self.samDB = self.__openConn__()
        self.domain_dn = self.samDB.domain_dn()
        self.rules = rules
        self.config = config
        if 'attributSource' in config:
            self.sourceAttr = config['attributSource']
        else:
            self.sourceAttr = 'division'
        self.rawUserData = self.__getRawData__()
        self.groups = self.__getAllGroups__()
        self.sambaGroups = self.__getSambaGroups__()

    def __openConn__(self):
        """ Open the samba connection"""
        # Loading Parameters
        lp = LoadParm()

        # Loading Credentials
        creds = Credentials()
        creds.guess(lp)
        creds.set_username('admin')

        samDB = SamDB(session_info=system_session(), credentials=creds, lp=lp)
        return samDB

    def __getSambaGroups__(self):
        """ Get all groups from samba directory"""
        expression=("(objectClass=group)")
        grps = []
        res = self.samDB.search(self.domain_dn, scope=ldb.SCOPE_SUBTREE,
              expression=expression,
              attrs=["samaccountname", "grouptype"])

        for grp in res:
            grps.append(grp.get('samaccountname', idx=0))

        return grps

    def __getRawData__(self):
        """ Create a list of DirectoryUser instances with the user information
        like mail, division, rules ...
        """
        data = []
        expression=("(&(objectClass=user)(userAccountControl:%s:=%u))" %
                   (ldb.OID_COMPARATOR_AND, dsdb.UF_NORMAL_ACCOUNT))
        res = [el for el in self.samDB.search(self.domain_dn, scope=ldb.SCOPE_SUBTREE,
              expression=expression,
              attrs=[self.sourceAttr, "samaccountname", "mail"])
              if self.sourceAttr in el]
        foundrdiv = 0
        for elm in res:
#            print(elm.get('division',idx=0))
            if elm:
                rdivs = [rd for rd in elm.get(self.sourceAttr)]
                for rdiv in rdivs:
                    if rdiv:
                        foundrdiv += 1
                        name = elm.get('samaccountname', idx=0)
                        mail = elm.get('mail', idx=0)
                        if rdiv :
                            divs = extract_division_elements(rdiv)
                            divs = [normalize(div.lower()) for div in divs]

                            # Creating the user list with the rules selected with the
                            # separator
                            for rl in self.rules:
                                if rl:
                                    if rl.separator in ','.join(divs):
                                        if not mail:
                                           mail = ''
    #                                       print "name=",name,"\ndivs=",divs,"\nmail=",mail,"\nrdiv=",rdiv,"\nrl=",rl,"\n"
                                        data.append(DirectoryUser(name, divs, mail, rdiv, rl))
                    else:
                        continue
            else:
                continue

        if foundrdiv == 0:
            raise Exception("Bad attribute : {0}".format(self.sourceAttr))
        else:
            return data


    def __getAllGroups__(self):
        """ Create the full list of groups to create"""
        grps = []
        for user in self.rawUserData:
            grps = list(set(grps + user.groups))

        grps.sort()
        return grps

    def __renameGroup__(self, old_grp, new_grp):
        """Rename group
        """
        old_dn =  self.samDB.search(self.domain_dn, scope=ldb.SCOPE_SUBTREE, expression="(samaccountname={})".format(old_grp), attrs=[])[0].get('dn').get_linearized()
        new_dn = old_dn.replace(old_grp, new_grp)
        self.samDB.rename(old_dn, new_dn)
        ldif = """dn: {}
changetype: modify
replace: samaccountname
samaccountname: {}""".format(new_dn, new_grp)
        self.samDB.modify_ldif(ldif)

    def __cleanGroups__(self, cachefile, pgroups):
            try:
                oPgroups = pickle.load(open(cachefile, "rb"))
            except Exception as e:
                logger.error("Error loading cache !!!")
                logger.error("No action will be performed !!!")
                logger.error(e)

                print("|__ Error loading cache !!!")
                print("  |__ No action will be performed !!!")
                print("  |__ {0}".format(e))
                return False

            toRemoveFromGrp = {}
            groupsToRemove = list(set(oPgroups.keys()) - set(pgroups.keys()))
            for group in pgroups:
                if group in oPgroups.keys() :
                    res = list(set(oPgroups[group]) - set(pgroups[group]))
                    if res:
                        toRemoveFromGrp[group] = res

            # Clean Groups
            for grp in toRemoveFromGrp:
                try:
                    if not silent:
                        print("\nRemoving users from group {0}".format(grp))
                    logger.info("Removing users from group {0}".format(grp))
                    for usr in toRemoveFromGrp[grp]:
                        if not silent:
                            print("   - {0}".format(usr))
                        logger.info("Removed {0}".format(usr))
                    self.samDB.add_remove_group_members(grp, toRemoveFromGrp[grp],
                                                        add_members_operation=False)
                except Exception as e:
                    if not silent:
                        print("|__ Error removing users from group {0}".format(grp))
                        print("   |__ {0}".format(e))
                    logger.error("Error removing users from group {0}".format(grp))
                    logger.error(e)

            if groupsToRemove:
                if not silent:
                    print("\n")
                for grp in groupsToRemove:
                    if not silent:
                        print("Removing group {0}".format(grp))
                    logger.info("Removing group {0}".format(grp))
                    try:
                        self.__renameGroup__(grp, 'SAUVEGARDE.' + grp)
                    except Exception as e:
                        if not silent:
                            print("|__ Error removing group {0}".format(grp))
                            print("   |__ {0}".format(e))
                        logger.error("Error removing group {0}".format(grp))
                        logger.error(e)
                        continue

    def createGroups(self, silent):
        """ Create all groups in Samba Directory """
        res = True
        failed = []
        print self.sambaGroups
        for grp,dest in self.groups:
            try:
                if grp in self.sambaGroups:
                    if not silent:
                        print("Group {0} allready exists".format(grp))
                    logger.info("Group {0} allready exists".format(grp))
                else:
                    if 'SAUVEGARDE.' + grp in self.sambaGroups:
                        if not silent:
                            print("Renaming group : {0}".format(grp))
                        logger.info("Renaming group : {0}".format(grp))
                        self.__renameGroup__('SAUVEGARDE.' + grp, grp)
                    else:
                        if not silent:
                            print("Creating group : {0}".format(grp))
                        logger.info("Creating group : {0}".format(grp))
                        self.samDB.newgroup(grp, groupou=dest)
            except Exception as e:
                if not silent:
                    print("|___ Error Creating group {0}".format(grp))
                    print("    |___ {0}".format(e))
                logger.error("Error Creating group {0}".format(grp))
                logger.error(e)
                failed.append(grp)

        if len(failed) != 0:
            if not silent:
                print("\nGroup creation failed for :")
            for elm in failed:
                if not silent:
                    print("   - {0}".format(elm))
                logger.error("Group creation failed : [{0}]".format(elm))
            return False
        else:
            return True

    def affectUsers(self, silent):
        """ Put users in groups according to the affectRules """
        pgroups = {}
        failed = []
        cachefile = None
        oPgroups = None

        for grp,dest in self.groups:
            pgroups[grp] = []

        for user in self.rawUserData:
            for group,dest in user.groups:
                if user.samaccountname not in pgroups[group]:
                    pgroups[group].append(user.samaccountname)

        if 'cachefile' in self.config.keys():
            cachefile = self.config['cachefile']
        else:
            cachefile = '/var/lib/eole/extractcache.p'

        if os.path.exists(cachefile):
            self.__cleanGroups__(cachefile, pgroups)

        pickle.dump(pgroups, open(cachefile, "wb"))

        for key in pgroups:
            try:
                if not silent:
                    print("\nAdding users to group {0}".format(key))
                for usr in pgroups[key]:
                    if not silent:
                        print("   + {0}".format(usr))
                    logger.info("Adding {0} user a to {1} group".format(usr, key))
                self.samDB.add_remove_group_members(key,pgroups[key], add_members_operation=True)
            except Exception as e:
                if not silent:
                    print("|___ Error adding users to group {0}".format(key))
                    print("    |___ {0}".format(e))
                logger.error("Error adding users to group {0}".format(key))
                failed.append(key)

        if len(failed) != 0:
            if not silent:
                print("\nJoining users failed for groups:")
                for elm in failed:
                    print("      - {0}".format(elm))
            return False
        else:
            return True


    def show(self):
        if self.rawUserData:
            for dt in self.rawUserData:
                dt.show()

parser = argparse.ArgumentParser(description='Process some integers.')
parser.add_argument('-c', '--config', help=u'Configuration file', metavar='CONFIG_FILE')
parser.add_argument('-s', '--silent', help=u'Silent mode (no output)', action='store_true')
args = parser.parse_args()


if args.config:
    conf = Configuration(args.config)
    if 'logfile' in conf.general:
        logfile = conf.general['logfile']
    else:
        logfile = "/tmp/manageGroups.log"

    logger = logging.getLogger('manageGroups')
    hdlr = logging.FileHandler(logfile)
    formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
    hdlr.setFormatter(formatter)
    logger.addHandler(hdlr)
    logger.setLevel(logging.INFO)

    if args.silent:
        silent = True
    else:
        silent = False
    conf.load()

    try:
        data = DirectoryData(conf.rules, conf.general)
        data.createGroups(silent)
        data.affectUsers(silent)
    except Exception as e:
        print("General Error !")
        print("\__ {0}".format(e))
        logger.error("General Error {0}".format(e))

else:
    parser.print_help()

