# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour afficher les partitions montées et leur utilisation
"""

from twisted.internet.utils import getProcessOutput

from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.data import HTMLData, TableData
from zephir.monitor.agentmanager.util import percent

# limites à partir des quelles on créé une alerte
SOFT_LIMIT = 90
HARD_LIMIT = 96

def stat_format(x):
    return "%.1f" % x

def perc2img(perc):
    p = int(perc[0:-1])
    if p >= HARD_LIMIT :
        color = 'red'
    elif p >= SOFT_LIMIT :
        color = 'orange'
    else:
        color = 'green'

    return """<img src="/static/img/%spix.gif" height="16" width="%s">
<img src="/static/img/whitepix.gif" height="16" width="%s">""" % (color,str(p),str(100-p))


class DiskSpace(Agent):

    def __init__(self, name,
                 **params):
        Agent.__init__(self, name, **params)
        self.table = TableData([
            ('name', 'Montage', {'align':'right'}, None),
            ('device', 'Partition', {'align':'left'}, None),
            ('type', 'Type', {'align':'right'}, None),
            ('inodes', 'Inodes', {'align':'right'}, None),
            ('perc', 'Utilisation', {'align':'right'}, None),
            ('used', 'Utilisé (Mo)', {'align':'right'}, None),
            ('free', 'Libre (Mo)', {'align':'right'}, None),
            ('size', 'Taille (Mo)', {'align':'right'}, None),
            ('graph', 'Graphe', {'align':'left'}, None) ])
        self.data = [self.table]

    def measure(self):
        cmd = getProcessOutput('/bin/mount',
                               env = {'LC_ALL': 'C'})
        cmd.addCallback(self.measure_mounts)
        cmd.addErrback(self.measure_error)
        return cmd

    def measure_mounts(self, output):
        # recherche des montages de type 'bind'
        mount_binds = []
        for mnt_line in output.split('\n'):
            try:
                options = mnt_line.split('(')[1]
                if 'bind' in options:
                    dev_name = mnt_line.split()[0]
                    mount_binds.append(dev_name)
            except:
                # pas d'options affichées ?
                pass
        cmd = getProcessOutput('df',
                               args = ['-iP'],
                               env = {'LC_ALL': 'C'})
        cmd.addCallback(self.measure_inodes, mount_binds)
        cmd.addErrback(self.measure_error)
        return cmd

    def measure_inodes(self, mnt, mount_binds):
        # tableau du pourcentage d'inodes utilisés
        # récupération des lignes
        lmnt = mnt.splitlines()
        # traitement mnt
        inodes = {}
        for lmn in lmnt:
            try:
                inodes[lmn.split()[0]] = lmn.split()[4]
                # systèmes de fichiers sans inodes (#1829)
                if inodes[lmn.split()[0]] == '-':
                    inodes[lmn.split()[0]] = '0%'
            except:
                pass
        cmd = getProcessOutput('df',
                               args = ['-kP','--print-type','--exclude-type=tmpfs','--exclude-type=usbfs'],
                               env = {'LC_ALL': 'C'})
        cmd.addCallback(self.measure_process, mount_binds, inodes)
        cmd.addErrback(self.measure_error)
        return cmd

    def measure_process(self, mnt, mount_binds, inodes):
        # récupération des lignes
        lmnt = mnt.splitlines()
        # traitement mnt
        lmnt = lmnt[1:]
        liste=[]
        for lmn in lmnt :
            try:
                l = {}
                l['name'] = lmn.split()[6] # montage
                l['device'] = lmn.split()[0] # partition
                l['type'] = lmn.split()[1] # jointure sur le type
                l['perc'] = lmn.split()[5] # utilisation (%)
                l['free'] = int(lmn.split()[4])/1024 # dispo
                l['used'] = int(lmn.split()[3])/1024 # utilisé
                l['size'] = int(lmn.split()[2])/1024 # taille
                l['graph'] = perc2img(l['perc'])
                l['inodes'] = inodes[l['device']]
            except:
                pass
            if l['device'] not in mount_binds:
                liste.append(l)
                self.measure_data[l['name']] = (int(l['perc'][:-1]), int(l['size']), int(l['used']), int(l['free']), l['type'])

        return {'statistics': liste}

    def measure_error(self, res):
        """ la commande df ne retourne rien ?"""
        return {'statistics': [{'name':'---', 'device':'---', 'type':'---',
                                'inodes':'---', 'perc':'---', 'free':'---',
                                'used':'---', 'size':'---', 'graph':'---'}]}

    def check_status(self):
        err = 0
        warn = 0
        err_nodes = 0
        warn_nodes = 0
        if self.last_measure is not None:
            if self.last_measure.value['statistics'][0]['name'] == '---':
                return status.Unknown()
            # Alerte si une partition est remplie à plus de "LIMIT" (%)
            for device in self.last_measure.value['statistics']:
                # on ignore les cdroms pour les alertes
                if not device['type'] in ['iso9660', 'supermount', 'cifs']:
                    if int(device['perc'][:-1]) >= HARD_LIMIT:
                        err += 1
                    if int(device['inodes'][:-1]) >= HARD_LIMIT:
                        err_nodes += 1
                    if int(device['perc'][:-1]) >= SOFT_LIMIT:
                        warn += 1
                    if int(device['inodes'][:-1]) >= SOFT_LIMIT:
                        warn_nodes += 1
        if err == 1:
            return status.Error("%d partition remplie à plus de %d %%"%(err,HARD_LIMIT))
        if err > 1:
            return status.Error("%d partitions remplies à plus de %d %%"%(err,HARD_LIMIT))
        if err_nodes == 1:
            return status.Error("%d partition utilise plus de %d %% des Inodes"%(err_nodes,HARD_LIMIT))
        if err_nodes > 1:
            return status.Error("%d partitions utilisent plus de %d %% des Inodes"%(err_nodes,HARD_LIMIT))
        if warn == 1:
            return status.Warn("%d partition remplie à plus de %d %%"%(warn,SOFT_LIMIT))
        if warn > 1:
            return status.Warn("%d partitions remplies à plus de %d %%"%(warn,SOFT_LIMIT))
        if warn_nodes == 1:
            return status.Warn("%d partition utilise plus de %d %% des Inodes"%(warn_nodes,SOFT_LIMIT))
        if warn_nodes > 1:
            return status.Warn("%d partitions utilisent plus de %d %% des Inodes"%(warn_nodes,SOFT_LIMIT))
        return status.OK()

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['statistics']

