# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agents zephir affichant les Informations Système
"""

from zephir.monitor.agentmanager.agent import MultiRRDAgent
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.data import TableData, HTMLData
from zephir.monitor.agentmanager.util import percent

from twisted.internet import defer
from twisted.internet.utils import getProcessOutput
import re

SECONDS_PER_DAY = 3600*24

# column titles in /proc/net/dev
(IBYTES, IPACKETS, IERRS,
 IDROP, IFIFO, IFRAME, ICOMPRESSED, IMULTICAST,
 OBYTES, OPACKETS, OERRS,
 ODROP, OFIFO, OCOLLS, OCARRIER, OCOMPRESSED,
 ) = range(0,16)

IPADDRESS_RE = re.compile('(?<=addr:)\d\d?\d?.\d\d?\d?.\d\d?\d?.\d\d?\d?')
ERROR_RATE_ALERT_THRESHOLD = 10 # in percent

def _mega(val):
    """ transfo de la valeur passée (string en octets) en Mo
    """
    return (int(val)/1024/1024)

def _stat_format(x):
    return "%.1f" % x

class NetStat(MultiRRDAgent):
    """
    Bilan de l'etat des cartes réseau
    présentation en tableau
    + graphe pour chaque carte
    """

    def __init__(self, name, **params):
        self.status = status.OK()
        MultiRRDAgent.__init__(self, name, **params)
        self.last_measure = None
        self.table = TableData([
            ('name', 'Nom', {'align':'right'}, None),
            ('address', 'Adresse', {'align':'left'}, None),
            ('input KB', 'Entrée (Ko)', {'align':'right'}, _stat_format),
            ('input err%', '(% err)', {'align':'right'}, _stat_format),
            ('output KB', 'Sortie (Ko)', {'align':'right'}, _stat_format),
            ('output err%', '(% err)', {'align':'right'}, _stat_format) ])
        title1 = HTMLData("<h3>Interfaces réseau<h3>")
        title2 = HTMLData("<h3>Statistiques réseau (Entrées/Sorties)<h3>")
        self.data.extend([title1, self.table, title2])

    def init_data(self, archive_dir):
        """on initialise les archives rrd, et on définit
        la liste des données"""
        MultiRRDAgent.init_data(self,archive_dir)

    def measure(self):
        self.status = status.OK()
        ifconfig = getProcessOutput('/sbin/ifconfig',
                                    env = {'LC_ALL': 'C'})
        catproc = getProcessOutput('/bin/cat',
                                   args = ['/proc/net/dev'],
                                   env = {'LC_ALL': 'C'})
        cmds = defer.DeferredList([ifconfig, catproc])
        cmds.addCallback(self.measure_process)
        return cmds

    def measure_process(self, cmds_results):
        [(ifconfig_success, ifconfig),
         (catproc_success, catproc)] = cmds_results
        if not (ifconfig_success and catproc_success):
            self.status = status.Error("Erreur à la lecture des informations des interfaces réseau")
            return None
        # get address of each interface
        addresses = {}
        blocks = ifconfig.strip().split('\n\n')
        for b in blocks:
            name_line, addr_line = b.splitlines()[0:2]
            name = name_line.split()[0].replace('.','_')
            # en mode conteneur, on ne fait pas de statistiques sur les cartes virtuelles
            if (not name.strip().startswith('ve')) and name.strip() != 'br0':
                addr_match = IPADDRESS_RE.search(addr_line)
                #FIXME set agent status
                if addr_match is not None:
                    addr = addr_match.group()
                    addresses[name] = addr

        statistics = []
        dico={}
        # get interfaces statistics from /proc
        lines = catproc.splitlines()[2:] # drop header lines
        for l in lines:
            if_name, stats = l.split(':')
            # en mode conteneur, on ne fait pas de statistiques sur les cartes virtuelles
            if (not if_name.strip().startswith('ve')) and name.strip() != 'br0':
                if_name = if_name.strip().replace('.','_')
                if addresses.has_key(if_name):
                    if_addr = addresses[if_name]
                else:
                    # pas d'infos sur une des interfaces
                    if_addr = "non configurée"
                stats = stats.split()
                inkb = int(float(stats[IBYTES])/1024.0)
                outkb = int(float(stats[OBYTES])/1024.0)
                iner = percent(stats[IERRS], stats[IPACKETS])
                outer = percent(stats[OERRS], stats[OPACKETS])
                if_stats = {'name': if_name,
                            'address': if_addr,
                            'input KB': inkb,
                            'output KB': outkb,
                            'input err%': iner,
                            'output err%': outer }
                statistics.append(if_stats)
                dico['in_%s' % if_name] = inkb
                dico['iner_%s' % if_name] = iner
                dico['out_%s' % if_name] = outkb
                dico['outer_%s' % if_name] = outer
                self.measure_data.update(dico)
        # données du tableau
        dico['statistics'] = statistics
        return dico

    def save_measure(self, measure):
        if measure.value is not None:
            MultiRRDAgent.save_measure(self, measure)
            measure.value = {'statistics':measure.value['statistics']}
        self.last_measure = measure

    def write_data(self):
        MultiRRDAgent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['statistics']

    def check_status(self):
        return self.status
