# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour le test des tunnels
"""
from os import system
from os.path import isfile
from IPy import IP
from creole import client
from twisted.internet.utils import getProcessOutput
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.agent import Agent, RRDAgent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img, log


def parse_ipsec_statusall(zephir_client_noaction, response):
    """parses ipsec statusall info
    response format :
        Connections:
        amontestha-sphynxtestha1:  192.168.0.6...192.168.0.16, dpddelay=120s
        amontestha-sphynxtestha1:   local:  [C=fr, O=gouv, OU=education, OU=ac-dijon, CN=0210066H-15] uses public key authentication
        amontestha-sphynxtestha1:   remote: [C=fr, O=gouv, OU=education, OU=ac-dijon, CN=AGRIATES-DIJON-10] uses any authentication
        dmz-reseau172:   child:  10.121.11.0/24 === 172.16.0.0/12 , dpdaction=clear
        admin-reseau_eth1:   child:  10.21.11.0/24 === 172.30.107.0/25 , dpdaction=clear
        admin-reseau10:   child:  10.21.11.0/24 === 10.0.0.0/8 , dpdaction=clear
        admin-reseau172:   child:  10.21.11.0/24 === 172.16.0.0/12 , dpdaction=clear
        Security Associations:
        amontestha-sphynxtestha1[2]: ESTABLISHED 15 minutes ago, 192.168.0.6[C=fr, O=gouv, OU=education, OU=ac-dijon, CN=0210066H-15]...192.168.0.16[C=fr, O=gouv, OU=education, OU=ac-dijon, CN=AGRIATES-DIJON-10]
        amontestha-sphynxtestha1[2]: IKE SPIs: 7c61e6943c503455_i* 670a092581afa023_r, public key reauthentication in 36 minutes
        amontestha-sphynxtestha1[2]: IKE proposal: AES_CBC_128/HMAC_SHA2_256_128/PRF_HMAC_SHA2_256/MODP_2048
        admin-reseau_eth1{2}:  INSTALLED, TUNNEL, ESP SPIs: c5abf3db_i c1abb8d6_o
        admin-reseau_eth1{2}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 15 minutes
        admin-reseau_eth1{2}:   10.21.11.0/24 === 172.30.107.0/25
        admin-reseau10{3}:  INSTALLED, TUNNEL, ESP SPIs: c8e253f3_i c9bbec32_o
        admin-reseau10{3}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 15 minutes
        admin-reseau10{3}:   10.21.11.0/24 === 10.0.0.0/8
        dmz-reseau172{1}:  INSTALLED, TUNNEL, ESP SPIs: c64a5816_i c6b2622d_o
        dmz-reseau172{1}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 13 minutes
        dmz-reseau172{1}:   10.121.11.0/24 === 172.16.0.0/12
        admin-reseau172{4}:  INSTALLED, TUNNEL, ESP SPIs: c1b2d485_i c5be1b6d_o
        admin-reseau172{4}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 16 minutes
        admin-reseau172{4}:   10.21.11.0/24 === 172.16.0.0/12
    """

    idle_threads = 0
    try:
        if "statusall" in response:
            raise Exception("Frozen ipsec")
        if response == "":
            raise Exception("Empty response")
        response = response.split('\n')
        for line in response:
            if "worker threads: " in line:
                idle_threads = int(line.split('worker threads: ')[1].split(' ')[0])
                break
        # On mémorise les connexions définies
        startindex_connections_list = response.index('Connections:')
        endindex_connections_list = 0
        for line in response:
            if 'Security Associations' in line:
                break
            endindex_connections_list += 1
        tunnels = []
        total = 0
        peer_name = None
        for line in response[startindex_connections_list+1:endindex_connections_list]:
            data = line.split(':')
            if 'child' not in line:
                # Il s'agit d'un peer
                tmp_name = data[0].lstrip()
                if tmp_name != peer_name:
                    # Nouveau peer: on extrait le nom et les IP src et dst
                    peer_name = tmp_name
                    subnets = data[1].split('...')
                    src = subnets[0].strip()
                    dst = subnets[1].split(',')[0].strip()
                    dst = dst.split(' ')[0].strip()
                elif 'local' in data[1]:
                    # On extrait le certif source
                    src = src + data[2].split('uses')[0].strip()
                elif 'remote' in data[1]:
                    # On extrait le certif dest et on ajoute à la liste
                    # Le child_name est vide si c'est un peer
                    dst = dst + data[2].split('uses')[0].strip()
                    tunnels.append({'peer_name':peer_name, 'child_name':'',
                                    'src':src, 'dst':dst, 'status':''})
            else:
                # Il s'agit d'un child
                child_name = data[0].lstrip()
                subnets = data[2].split(' === ')
                src = subnets[0].strip()
                if src == '':
                    src = 'Réseau(x) source et/ou destination non renseigné(s) !!!'
                dst = subnets[1].split(',')[0].strip()
                dst = dst.split(' ')[0].strip()
                if dst == '':
                    dst = 'Réseau(x) source et/ou destination non renseigné(s) !!!'
                tunnels.append({'peer_name':peer_name, 'child_name':child_name,
                                'src':src, 'dst':dst, 'status':''})
                total += 1

        # On mémorise les connexions actives
        active_tunnels = {}
        peer_name = None
        child_name = None
        for line in response[endindex_connections_list+1:]:
            if 'none' in line:
                active_tunnels['none'] = {'peer_name':peer_name,
                                          'child_name':child_name,
                                          'src':'', 'dst':'', 'status':''}
            else:
                data = line.split(':')
                if '[' in data[0]:
                    # Il s'agit d'un peer
                    tmp_name = data[0].split('[')[0].lstrip()
                    if tmp_name != peer_name:
                        # Nouveau peer: on extrait le nom, IP et certif src et dest et le statut
                        peer_name = tmp_name
                        if 'ESTABLISHED' in data[1]:
                            subnets = data[1].split('...')
                            src = subnets[0].split('ago,')[1].strip()
                            dst = subnets[1].strip()
                            status = data[1].strip().split(' ')[0]
                        elif 'CONNECTING' in data[1]:
                            src = ''
                            dst = ''
                            status = 'CONNECTING'
                        elif 'DELETING' in data[1]:
                            src = ''
                            dst = ''
                            status = 'DELETING'
                        else:
                            src = ''
                            dst = ''
                            status = 'UNKNOWN'
                        if status != 'CONNECTING' and status != 'DELETING':
                            active_tunnels[peer_name] = {'peer_name':peer_name, 'child_name':'', 'src':src, 'dst':dst, 'status':status}
                    else:
                        if status == 'CONNECTING':
                            if '0000000000000000_r' in data[2]:
                                src = ''
                                dst = ''
                                status = 'NO_REMOTE_KEY'
                                active_tunnels[peer_name] = {'peer_name':peer_name, 'child_name':'', 'src':src, 'dst':dst, 'status':status}
                                log.msg("""No remote key for "{0}" connection""".format(peer_name))
                        elif status == 'DELETING':
                            if 'Tasks queued' in data[1]:
                                log.msg("""DELETING SA still in task queued for "{0}" connection""".format(peer_name))
                        elif 'Tasks queued' in data[1]:
                            if len(data[2].split(' ')) > 10:
                                log.msg("""Too much tasks queued for "{0}" connection""".format(peer_name))
                elif '{' in data[0]:
                    # Il s'agit d'un child
                    tmp_name = data[0].split('{')[0].lstrip()
                    if tmp_name != child_name:
                        # Nouveau child: on extrait le nom et le statut
                        child_name = tmp_name
                        status = data[1].split(',')[0].strip()
                    elif "===" in data[1]:
                        # Dans les lignes suivantes, on extrait les réseaux src et dst
                        subnets = data[1].split('===')
                        src = subnets[0].strip()
                        dst = subnets[1].strip()
                        active_tunnels[peer_name+child_name] = {'peer_name':peer_name, 'child_name':child_name, 'src':src, 'dst':dst, 'status':status}

        # On regarde l'état des tunnels
        up = down = 0
        if active_tunnels.has_key('none'):
            # Aucune connexion active
            for tunnel in tunnels:
                tunnel['status'] = 'none'
                down += 1
        else:
            for tunnel in tunnels:
                # on renseigne le status de la connexion
                if tunnel['child_name'] == '':
                    try:
                        if active_tunnels.has_key(tunnel['peer_name']):
                            if active_tunnels[tunnel['peer_name']]['status'] == 'ESTABLISHED':
                                tunnel['status'] = 'On'
                            elif active_tunnels[tunnel['peer_name']]['status'] == 'NO_REMOTE_KEY':
                                tunnel['dst'] += " injoignable !!!"
                                tunnel['status'] = 'Off'
                            else:
                                tunnel['status'] = 'Off'
                        else:
                            tunnel['status'] = 'Off'
                    except:
                        tunnel['status'] = 'Off'
                # on renseigne le status de chaque tunnel
                else:
                    try:
                        if active_tunnels.has_key(tunnel['peer_name']+tunnel['child_name']):
                            if active_tunnels[tunnel['peer_name']+tunnel['child_name']]['status'] == 'INSTALLED':
                                tunnel['status'] = 'On'
                                up += 1
                            else:
                                tunnel['status'] = 'Off'
                                down += 1
                        else:
                            tunnel['status'] = 'Off'
                    except:
                        tunnel['status'] = 'Off'
                        down += 1

        # Si on est sur Amon, on vérifie les test-rvp
        ip_list = []
        try:
            test_rvp = open('/usr/share/eole/test-rvp','r').read().split('\n')
            for line in test_rvp:
                if 'fping' in line:
                    ip_list.append(line.split('fping')[1].strip())
        except:
            pass
        try:
            test_rvp = open('/usr/share/eole/test-rvp_more_ip','r').read().split('\n')
            for line in test_rvp:
                if 'fping' in line:
                    ip_list.append(line.split('fping')[1].strip())
        except:
            pass
        for tunnel in tunnels:
            if tunnel['child_name'] != '':
                ip_in_tunnel = ip_err = 0
                try:
                    subnet = IP(tunnel['dst'])
                    for ip in ip_list:
                        if IP(ip) in subnet:
                            ip_in_tunnel +=1
                            cmd = "fping -r 1 -t 2000 "+str(ip)+"  >/dev/null 2>&1"
                            errcode = system(cmd)
                            if errcode != 0:
                                tunnel['dst'] += " -- "+ip+" injoignable"
                                if tunnel['status'] == 'On':
                                    tunnel['status'] = 'Off'
                                    down +=1
                                    up -=1
                            else:
                                tunnel['dst'] += " -- "+ip+" joignable"
                    if tunnel['status'] == 'Off':
                        log.msg("tunnel "+str(tunnel['src'])+" vers "+str(tunnel['dst']+ " en erreur"))
                except:
                    pass

    except Exception, msg:
        tunnels = []
        total = up = down = idle_threads = -1
        log.msg(msg)

    return tunnels, total, up, down, idle_threads

class RvpAmon(Agent):

    def __init__(self, name,
                 **params):
        Agent.__init__(self, name, **params)
        self.status = status.Unknown()
        self.table = TableData([
            ('peer_name', "Connexion", {'align':'left'}, None),
            ('child_name', "Tunnel", {'align':'left'}, None),
            ('status', "Etat", {'align':'center'}, status_to_img),
            ('src', "Source", {'align':'left'}, None),
            ('dst', "Destination", {'align':'left'}, None),
            ])
        self.data = [self.table]
        self.measure_data = {}

    def frozen_ipsec_callback(self, response):
        if response != '':
            cmd = 'echo'
        else:
            cmd = "/usr/sbin/ipsec"
        update = getProcessOutput(cmd, ['statusall'])
        return update.addCallbacks(self.callback_tunnels, self.errback_tunnels)

    def frozen_ipsec_errback(self):
        log.msg("It doesn't work")

    def measure(self):
        self.status = status.OK()
        cmd = "pgrep"
        res = getProcessOutput(cmd, ['stroke'])
        return res.addCallbacks(self.frozen_ipsec_callback, self.frozen_ipsec_errback)

    def callback_tunnels(self,response):
        tunnels, total, up, down, idle_threads = parse_ipsec_statusall('oui', response)
        tunnels_status_content = []
        tunnels_status_content.append(str(tunnels))
        tunnels_status_content.append(str(total))
        tunnels_status_content.append(str(up))
        tunnels_status_content.append(str(down))
        tunnels_status = open('/tmp/tunnels_status.txt', "w")
        tunnels_status.write('\n'.join(tunnels_status_content))
        tunnels_status.close()
        # On force l'action en cas d'erreur
        if down > 0 or total != up + down:
            self.last_status = status.OK()
            self.status = status.Error()
        elif total == -1 and down == -1 and up == -1 and idle_threads == -1:
            log.msg("Can't parse 'ipsec statusall' response")
            self.last_status = status.OK()
            self.status = status.Error()
        for data in tunnels:
            self.measure_data[data['peer_name']+data['child_name']] = (data['src'], data['dst'], data['status'])
        self.measure_data['ok'] = up
        self.measure_data['bad'] = down
        return {'statistics':tunnels}

    def errback_tunnels(self, err):
        self.status = status.Error("Erreur d'execution : ipsec status")
        return {'statistics':None}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['statistics']

    def check_status(self):
        return self.status

class RvpSphynx(RRDAgent):

    def __init__(self, name, **params):
        RRDAgent.__init__(self, name, **params)
        self.status = status.Unknown()
        self.measure_data = {}
        self.pourcentok = 100
        conf_eole = client.CreoleClient()
        self.activer_haute_dispo = conf_eole.get_creole('activer_haute_dispo', 'non')
        self.localnode = conf_eole.get_creole('nom_machine')
        if self.activer_haute_dispo == 'maitre':
            self.remotenode = conf_eole.get_creole('nom_machine_esclave')
        elif self.activer_haute_dispo == 'esclave':
            self.remotenode = conf_eole.get_creole('nom_machine_maitre')
        else:
            self.remotenode = None

# test frozen
    def frozen_ipsec_callback(self, response):
        if response != '':
            cmd = 'echo'
        else:
            cmd = "/usr/sbin/ipsec"
        update = getProcessOutput(cmd, ['statusall'])
        return update.addCallbacks(self.callback_tunnels, self.errback_tunnels)

    def frozen_ipsec_errback(self):
        log.msg("It doesn't work")

    def measure(self):
        self.status = status.OK()
        cmd = "pgrep"
        res = getProcessOutput(cmd, ['stroke'])
        return res.addCallbacks(self.frozen_ipsec_callback, self.frozen_ipsec_errback)

    def callback_tunnels(self,response):
        tunnels, total, up, down, idle_threads = parse_ipsec_statusall('oui', response)
        tunnels_status_content = []
        tunnels_status_content.append(str(tunnels))
        tunnels_status_content.append(str(total))
        tunnels_status_content.append(str(up))
        tunnels_status_content.append(str(down))
        tunnels_status = open('/tmp/tunnels_status.txt', "w")
        tunnels_status.write('\n'.join(tunnels_status_content))
        tunnels_status.close()
        self.measure_data['ok'] = up
        self.measure_data['bad'] = down
        if self.activer_haute_dispo == 'non':
            if total > 0:
                self.pourcentok = (100*up)/total
            elif total == -1:
                self.pourcentok = 0
            else:
                self.pourcentok = 100
        else:
            # Si haute dispo activée, on teste sur quel nœud ipsec est activé
            cmd = "crm resource show VIPCluster 2>>/dev/null| grep -sqE \"VIPCluster is running on: {0} $\"".format(self.localnode)
            errcode = system(cmd)
            # Si le nœud est actif, on renvoi les stats
            if errcode == 0:
                if total > 0:
                    self.pourcentok = (100*up)/total
                elif total == -1:
                    self.pourcentok = 0
                else:
                    self.pourcentok = 100
            else:
                cmd = "crm resource show VIPCluster 2>>/dev/null| grep -sqE \"VIPCluster is running on: {0} $\"".format(self.remotenode)
                errcode = system(cmd)
                if errcode == 0:
                    # Si IPsec tourne sur le nœud distant, on dit que tout est bon
                    self.pourcentok = 100
                else:
                    self.pourcentok = 0
        return {'ok':up, 'bad':down, 'total':total,
                'pourcentok':self.pourcentok}

    def errback_tunnels(self, err):
        self.status = status.Error("Erreur d'execution : ipsec statusall")
        res = None
        self.pourcentok = 0
        return res

    def check_status(self):
        if self.pourcentok < 75:
            return status.Error("plus de 25% de tunnels en erreur")
        if self.pourcentok < 95:
            return status.Warn("plus de 5% de tunnels en erreur")
        return status.OK()

class strongSwan_threads(RRDAgent):

    def __init__(self, name, **params):
        RRDAgent.__init__(self, name, **params)
        self.status = status.Unknown()
        self.measure_data = {}
        conf_eole = client.CreoleClient()
        self.activer_haute_dispo = conf_eole.get_creole('activer_haute_dispo', 'non')
        self.localnode = conf_eole.get_creole('nom_machine')
        self.max_threads = conf_eole.get_creole('sw_threads')
        self.running_threads = 0

# test frozen
    def frozen_ipsec_callback(self, response):
        if response != '':
            cmd = 'echo'
        else:
            cmd = "/usr/sbin/ipsec"
        update = getProcessOutput(cmd, ['statusall'])
        return update.addCallbacks(self.callback_tunnels, self.errback_tunnels)

    def frozen_ipsec_errback(self):
       log.msg("It doesn't work")

    def measure(self):
        self.status = status.OK()
        cmd = "pgrep"
        res = getProcessOutput(cmd, ['stroke'])
        return res.addCallbacks(self.frozen_ipsec_callback, self.frozen_ipsec_errback)

    def callback_tunnels(self,response):
        tunnels, total, up, down, idle_threads = parse_ipsec_statusall('oui', response)
        if self.activer_haute_dispo == 'non':
            self.running_threads = self.max_threads - idle_threads
        else:
            # Si haute dispo activée, on teste sur quel nœud ipsec est activé
            cmd = "crm resource show VIPCluster 2>>/dev/null| grep -sqE \"VIPCluster is running on: {0} $\"".format(self.localnode)
            errcode = system(cmd)
            # Si le nœud est actif, on renvoi les stats
            if errcode == 0:
                self.running_threads = self.max_threads - idle_threads
            else:
                self.running_threads = 0
        return {'max_threads':self.max_threads,'running_threads':self.running_threads}

    def errback_tunnels(self, err):
        self.status = status.Error("Erreur d'execution : ipsec statusall")
        res = None
        return res

    def check_status(self):
        if (self.max_threads - self.running_threads) < 2:
            return status.Error("Moins de 2 threads strongSwan disponibles")
        if (self.max_threads - self.running_threads) < 5:
            return status.Warn("Moins de 5 threads strongSwan disponibles")
        return status.OK()
