# -*- coding: utf-8 -*-

import pwd
import re
from os import makedirs, readlink, unlink, system, stat
from os.path import (abspath, dirname, join,
 isfile, isdir, islink, ismount, exists, realpath)
from configobj import ConfigObj
from pickle import dump, load
from time import sleep, time
from datetime import datetime, timedelta

import ConfigParser
from pyeole2.service import service_out
from pyeole2.process import system_out, tcpcheck
from pyeole2.schedule import load_schedule
from creole2.eolelock import check_lock
from creole2.parsedico import parse_dico

from creole2 import config
import locale
locale.setlocale(locale.LC_ALL,'')

BACULA_CONF = "/etc/eole/bacula.conf"

BCONSOLE_CONF = '/etc/bacula/bconsole.conf'

BACULA_SUPPORT = '/var/lib/eole/config/baculasupport.conf'
BACULA_MAIL = '/var/lib/eole/config/baculamail.conf'
BACULA_JOB = '/var/lib/eole/config/baculajobs.conf'
BACULA_RAPPORT = '/var/lib/eole/reports/resultat-bacula'
BACULA_CONFIG_TMPL = '/var/lib/creole/baculasupport.conf'
BACULA_CONFIG_MAILALLTMPL = '/var/lib/creole/baculamailall.conf'
BACULA_CONFIG_MAILONERRORTMPL = '/var/lib/creole/baculamailonerror.conf'
BACULA_CONFIG_FILE = '/etc/bacula/baculasupport.conf'
BACULA_CONFIG_JOB = '/etc/bacula/baculaschedule.conf'
BACULA_CONFIG_MAILALL = '/etc/bacula/baculamailall.conf'
BACULA_CONFIG_MAILONERROR = '/etc/bacula/baculamailonerror.conf'
BACULA_CONFIG_JOBSCHED = '/etc/bacula/baculaschedulepost.conf'
BACULA_CONFIG_DIR_XML = '/usr/share/eole/bacula/dicos/'
MOUNT_POINT = '/mnt/sauvegardes'
TEST_MOUNT_FILE = join(MOUNT_POINT, 'eole_test.txt')
FILE_LOG_BACKUP = '/var/log/rsyslog/local/bacula-dir/bacula-dir.err.log'

BACULA_RAPPORT_OK = "1"
BACULA_RAPPORT_ERR = "-1"
BACULA_RAPPORT_UNKNOWN = "0"

dico = None

def load_dico():
    global dico
    if dico == None:
        dico = parse_dico()

class Disabled(Exception):
    pass

def bacula_active():
    load_dico()
    return dico['activer_bacula'] == 'oui'

def bacula_active_sd():
    load_dico()
    if bacula_active():
        return 'oui' == dico['activer_bacula_sd']
    return False

def bacula_active_dir():
    load_dico()
    if bacula_active():
        return 'oui' == dico['activer_bacula_dir']
    return False

def load_bacula_conf(filename):
    cfgparser = ConfigParser.ConfigParser()
    dic = {}
    cfgparser.read(filename)
    for section in cfgparser.sections():
        dic[section] = eval(cfgparser.get(section, 'val'))[0]
    return dic

def reload_conf(test='both'):
    if test not in ['both', 'sd', 'dir']:
        raise Exception('ERREUR : test doit être "both", "sd" ou "dir')
    if check_lock('eolesauvegarde'):
        raise Exception('ERREUR : une sauvegarde est en cours : le rechargement de la configuration impossible')
    #both + dir
    if test != 'sd':
        cmd = ['/usr/sbin/bacula-dir', '-t']
        ret, stdout, stderr = system_out(cmd)
        if ret != 0:
            raise Exception('ERREUR : erreur au test de la configuration : {0}, {1}'.format(stdout, stderr))
        run_bacula_cmd('reload')
    #both + sd
    if test != 'dir':
        cmd = ['/usr/sbin/bacula-sd', '-t']
        ret, stdout, stderr = system_out(cmd)
        if ret != 0:
            raise Exception('ERREUR : erreur au test de la configuration : {0}, {1}'.format(stdout, stderr))
        service_out('bacula-sd', 'restart')

def _save_bacula_conf(filename, dic):
    fh = file(filename, 'w')
    cfgparser = ConfigParser.ConfigParser()
    for section, val in dic.items():
        cfgparser.add_section(section)
        if type(val) == list:
            cfgparser.set(section, 'val', val)
        else:
            cfgparser.set(section, 'val', [val])
        cfgparser.set(section, 'valprec', [])
        cfgparser.set(section, 'valdefault', [])
    cfgparser.write(fh)
    fh.close()

def system_out_catched(cmd):
    """
    Retourne une exception avec le message d'erreur en cas
    de sortie non nulle.

    :param cmd: commande à exécuter
    :type cmd: str
    """

    code, result, stderr = system_out(cmd)
    if code != 0:
        raise OSError(code, stderr)
    else:
        return result

def parse_mounts():
    """
    Retourne un dictionnaire des points de montage.

    Les entrées sont de la forme :
    point de montage : {nœud,
                        type de système de fichiers,
                        options de montage}
    """

    mount_output_re = r'(?P<node>.*?) (on )?(?P<mount_point>.*?) (type )?(?P<fs_type>.*?) \((?P<options>.*)\).*'
    mounts_list = [re.search(mount_output_re, mount).groupdict()
            for mount in system_out_catched(['/bin/mount']).split('\n')
            if mount != '']
    mounts = {}
    for mount in mounts_list:
        mounts[mount['mount_point']] = {'node': mount['node'],
                    'fs_type': mount['fs_type'],
                    'options': mount['options'].replace(')','').split(',')}
    return mounts

def test_mount_point(dic):
    """
    Retourne l'état du point de montage parmi l'un des trois états suivants :
    1) point de montage libre --> montage
    2) point de montage occupé par le bon périphérique avec les bons paramètres --> rien à faire
    3) point de montage occupé par le mauvais périphérique ou le bon périphérique avec les mauvais paramètres --> démontage puis montage

    Il y a deux paramètres à vérifier :
    - l'identité du périphérique monté (adresse, contenu)
    - les droits associés à ce périphérique

    :param dic: paramètres de bacula saisis dans l'ead ou avec baculaconfig -s
    :type dic: dictionnaire
    """

    # drapeaux
    mount_point_busy = False
    right_node_mounted = False
    right_rights = False
    # test de l'identité du périphérique monté (drapeaux mount_point_busy et
    # right_node_mounted)
    mounts = parse_mounts()
    if MOUNT_POINT in mounts:
        mount_point_busy = True
        if dic['support'] == 'usb' and \
           mounts[MOUNT_POINT]['node'] == realpath(dic['usb_path']):
            right_node_mounted = True
        elif dic['support'] == 'smb' and \
           dic['smb_machine'] in mounts[MOUNT_POINT]['node']:
            right_node_mounted = True
        elif dic['support'] == 'manual':
            right_node_mounted = True
        else:
            right_node_mounted = False

    # test d'écriture si le montage est le bon (drapeau right_rights)
    if right_node_mounted == True: # implique mount_point_busy == True
        try:
            system_out_catched(['su', '-', 'bacula', '-c', "touch {0}".format(TEST_MOUNT_FILE), '-s', '/bin/bash'])
            system_out_catched(['su', '-', 'bacula', '-c', "rm -f {0}".format(TEST_MOUNT_FILE), '-s', '/bin/bash'])
            right_rights = True
        except OSError as e:
            if isfile(TEST_MOUNT_FILE):
                system_out_catched('rm -f {0}'.format(TEST_MOUNT_FILE))

    return {'is_busy':mount_point_busy,
            'is_right_node': right_node_mounted,
            'has_rights': right_rights}


def mount_status_to_str(status):
    """
    Retourne le statut du support de sauvegarde pour affichage.
    :param status: résultat de la commande test_mount_point.
    :type status: dict
    """
    messages = {'is_right_node': 'point de montage',
                'has_rights': 'permissions',
                'is_busy': 'montage',
                'manual': 'Pas de montage en mode manuel'}

    return '\n'.join([messages[st] + " : OK" for st in status.keys() if status[st] == True] +
            [messages[st] + " : Erreur" for st in status.keys() if not status[st] == True])

def parse_custom_mount_file(conf):
    """
    Retourne les commandes de montage présentes dans le fichier conf et
    les paramètres si possible.

    Les commandes de montage sont filtrées pour extraire les paramètres
    de montage et enlever les options qui empêchent le fonctionnement
    de test_mount_point (-n qui ne créé pas d'entrée dans /etc/mtab).

    :param conf: chemin du fichier
    :type conf: str
    """

    conf = ConfigObj(conf)
    return conf

def get_mount_command(dic, uid):
    """
    Renvoie la commande pour monter le support en donnant la priorité
    au fichier BACULA_CONF si il existe.


    :param dic: paramètres de bacula
    :type dic: dictionnaire
    """

    bacula_config = parse_custom_mount_file(BACULA_CONF)

    DISTANT_LOGIN_MOUNT = bacula_config.get('DISTANT_LOGIN_MOUNT', '/bin/mount -t smbfs -o username={0},password={1},ip={2},uid={3},noexec,nosuid,nodev //{4}/{5} {6}')
    DISTANT_MOUNT = bacula_config.get('DISTANT_MOUNT', '/bin/mount -t smbfs -o password={0},ip={1},uid={2},noexec,nosuid,nodev //{3}/{4} {5}')
    if dic['support'] == 'usb':
        #si usb_path est un lien symbolique, suivre le lien (#2447)
        usb_path = realpath(dic['usb_path'])
        USB_MOUNT = bacula_config.get('USB_MOUNT', None)
        if USB_MOUNT is None:
            #if USB_MOUNT not set:
            #if volume if in VFAT, add uid option
            code, vtype, stderr = system_out(['/sbin/blkid', '-sTYPE', '-ovalue', usb_path])
            if code != 0:
                if stderr:
                    raise Exception('Erreur avec blkid : {0}'.format(stderr))
                else:
                    raise Exception('ERREUR : périphérique {0} non reconnu'.format(usb_path))
            if vtype.rstrip() in ['vfat', 'ntfs']:
                USB_MOUNT = '/bin/mount {0} {1} -o noexec,nosuid,nodev,uid={2},umask=0077'
            else:
                USB_MOUNT = '/bin/mount {0} {1} -o noexec,nosuid,nodev'
                # chown_mount_point = True

        cmd = USB_MOUNT.format(usb_path, MOUNT_POINT, uid).split()
    elif dic['support'] == 'smb':
        ret = tcpcheck(dic['smb_ip'], '139', 2)
        if not ret:
            ret = tcpcheck(dic['smb_ip'], '445', 2)
            if not ret:
                raise Exception("La machine {0} n'est pas accessible sur les ports 139 et 445".format(dic['smb_ip']))
        if dic['smb_login'] != None:
            cmd = DISTANT_LOGIN_MOUNT.format(dic['smb_login'],
                    dic['smb_password'], dic['smb_ip'], uid,
                    dic['smb_machine'], dic['smb_partage'], MOUNT_POINT).split()
        else:
            cmd = DISTANT_MOUNT.format(dic['smb_password'], dic['smb_ip'], uid,
                    dic['smb_machine'], dic['smb_partage'], MOUNT_POINT).split()
    return cmd

def mount_bacula_support(retry=40, chown=False):
    """
    Monte le support configuré via l'ead ou baculaconfig.py -s
    sur le point de montage MOUNT_POINT.
    Renvoie Vrai si le montage est effectif, Faux sinon.

    :param retry: nombre d'essais pour le montage usb
    :type retry: entier
    """
    attempt = retry
    is_mounted = False
    # créé le répertoire MOUNT_POINT si il n'existe pas
    if not isdir(MOUNT_POINT):
        makedirs(MOUNT_POINT)

    #quitte si manual ou pas configuré
    dic = load_bacula_support()
    if dic['support'] == 'manual':
        return True, {'manual': 'Pas de montage pour la configuration manuelle'}

    if dic['support'] == 'none':
        raise Exception("ERREUR : bacula n'est pas configuré")

    # configure l'uid à utiliser pour le montage
    try:
        uid = pwd.getpwnam('bacula').pw_uid
    except:
        uid = 102

    # démonte et monte seulement si le
    # point de montage est occupé par un autre périphérique
    mount_point_status = test_mount_point(dic)
    if mount_point_status['is_busy'] == True:
        if mount_point_status['is_right_node'] == True:
            if mount_point_status['has_rights'] == True:
                is_mounted = True
            elif chown == True:
                try:
                    system_out_catched(['chown', 'bacula:tape', MOUNT_POINT])
                except Exception as e:
                    print str(e)
                else:
                    is_mounted = True
            else:
                umount_bacula_support()
        else:
            umount_bacula_support()
    if is_mounted == False:
        if attempt > 0:
            # récupère la commande à utiliser pour monter
            try:
                cmd = get_mount_command(dic, uid)
                system_out_catched(cmd)
            except Exception as error:
                #print "Problème de montage ({0} essais restants)\n{1}".format(attempt-1, error)
                umount_bacula_support()
                sleep(3)
            finally:
                return mount_bacula_support(retry=attempt-1, chown=chown)
    return is_mounted, mount_point_status


def umount_bacula_support(volume='', error=False):
    dic = load_bacula_support()
    if isfile(volume) == True:
        volume = volume.split('-')[-2]
    if dic['support'] in ['manual', 'none'] or volume in ['inc', 'full', 'diff', '/var/lib/bacula/']:
        return True
    while ismount(MOUNT_POINT):
        cmd = ['/bin/umount', MOUNT_POINT]
        ret, stdout, stderr = system_out(cmd)
        if ret != 0 and error:
            raise Exception('ERREUR : démontage impossible : {0}, {1}'.format(stdout, stderr))
    return True

def device_is_mount(device):
    if not exists(device):
        raise Exception("ERREUR : le périphérique {0} n'existe pas".format(device))
    cmd = ['grep', '-q',  '^{0} '.format(device), '/etc/mtab']
    ret = system_out(cmd)
    return ret[0] == 0


def get_queries():
    """
    Return dictionnary of queries (conventionnal name
    and order of appearance in /etc/bacula/scripts/query.sql)
    """
    query_header = re.compile(r'^:')
    eole_query = re.compile(r'\(EOLE:(?P<query>\w+)\)')
    with open('/etc/bacula/scripts/query.sql', 'r') as query_file:
        queries_list = [query for query in query_file.readlines()
                        if query_header.match(query)]
    queries = {}
    for i in range(len(queries_list)):
        query = eole_query.search(queries_list[i])
        if query:
            queries[query.group('query')] = i + 1
    return queries


def run_bacula_cmd(stdin):
    cmd = ['bconsole', '-c', BCONSOLE_CONF]
    return system_out(cmd, stdin=stdin)


def bacula_query(query, *args):
    """
    Return bacula query command result
    :param query: query id as specified in /etc/bacula/scripts/query.sql
    :type query: str
    :param args: list of parameters needed for query
    :type args: list
    """
    queries = get_queries()
    try:
        query_num = queries[query]
    except KeyError:
        raise Exception('La liste des fonctions disponibles est la suivante : \n{0}'.format('\n'.join(queries.keys())))
    results_list = []
    params = '\n'.join(args) + '\nquit'
    status, res, err = run_bacula_cmd('query\n{0}\n{1}\n'.format(query_num, params))
    bacula_query_re = re.compile(r'\+(-+\+)+')
    try:
        res = res.split(bacula_query_re.search(res).group())
        header = [title.strip() for title in res[1].split('|') if title != '\n']
        results = [[j.strip() for j in i.split('|') if j != ''] for i in res[2].split('\n') if i != '']
        for result in results:
            result_dict = {}
            for value, key in zip(result, header):
                result_dict[key] = value
            results_list.append(result_dict)
    except AttributeError:
        raise Exception('Résultat de la commande bconsole imprévu')
    return results_list


def _empty_file(filename):
    if isfile(filename):
        unlink(filename)
    fh = file(filename,'w')
    fh.write('')
    fh.close()

def run_backup(level='Full', when=''):
    if level not in ['Full', 'Incremental', 'Differential']:
        raise Exception('level inconnu')
    if when != '':
        if re.match(r'[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}:[0-9]{2}', when) == None:
            raise Exception('horaire mal renseigné')
        else:
            when = ' When=\"{0}\"'.format(when)
    ret, stdout, stderr = run_bacula_cmd("run JobSchedulePre Level=Full{0}\n\n".format(when))
    if ret != 0:
        raise Exception('ERREUR : de sauvegarde {0} : {1}, {2}'.format(level, stdout, stderr))
    ret, stdout, stderr = run_bacula_cmd("run JobSauvegarde Level={0}{1}\n\n".format(level, when))
    if ret != 0:
        raise Exception('ERREUR : de sauvegarde {0} : {1}, {2}'.format(level, stdout, stderr))
    #run catalog backup every time
    ret, stdout, stderr = run_bacula_cmd("run BackupCatalog level=Full{0}\n\n".format(when))
    if ret != 0:
        raise Exception('ERREUR : de sauvegarde du catalogue {0} : {1}, {2}'.format(level, stdout, stderr))
    return "Sauvegarde {0} lancée\nVous pouvez suivre son évolution dans le fichier {1}".format(level, FILE_LOG_BACKUP)

def instance_bacula(tmpl, dest, eol):
    if not isfile(tmpl):
        raise Exception('Fichier template {0} inexistant'.format(tmpl))
    from creole2.creolecat import instance
    xml_dir = []
    xml_dir.extend(config.eoledirs)
    xml_dir.append(BACULA_CONFIG_DIR_XML)
    xml_dir = ','.join(xml_dir)
    eol = [config.configeol, eol]
    instance(tmpl, xml_dir, dest, eol)

# -------------------------------------------------
# -------------- Bacula support -------------------
# -------------------------------------------------
def load_bacula_support():
    if not bacula_active_sd():
        raise Disabled('Bacula sd désactivé')
    if not isfile(BACULA_SUPPORT):
        return {'support': 'none'}
    return load_bacula_conf(filename=BACULA_SUPPORT)

def save_bacula_support_usb(usb_path):
    if not bacula_active_sd():
        raise Disabled('Bacula sd désactivé')
    if usb_path == None:
        raise Exception('Chemin usb inconnu')
    dic = {'usb_path': usb_path, 'support': 'usb'}
    _save_bacula_conf(filename=BACULA_SUPPORT, dic=dic)
    apply_bacula_support()

def save_bacula_support_smb(smb_machine, smb_ip,
        smb_partage, smb_login=None, smb_password=None):
    if not bacula_active_sd():
        raise Disabled('Bacula sd désactivé')

    if smb_machine == None or smb_ip  == None or smb_partage == None:
        raise Exception('Nom machine, IP ou nom du partage inconnu')
    dic = {'support': 'smb'}
    if smb_login != None and smb_password == None:
        raise Exception('Login spécifié mais password absent')
    if smb_login == None and smb_password != None:
        raise Exception('Password spécifié mais login absent')

    dic['smb_machine'] = smb_machine
    dic['smb_ip'] = smb_ip
    dic['smb_partage'] = smb_partage
    dic['smb_login'] = smb_login
    dic['smb_password'] = smb_password
    _save_bacula_conf(filename=BACULA_SUPPORT, dic=dic)
    apply_bacula_support()

def save_bacula_support_manual():
    if not bacula_active_sd():
        raise Disabled('Bacula sd désactivé')
    dic = {'support': 'manual'}
    _save_bacula_conf(filename=BACULA_SUPPORT, dic=dic)
    apply_bacula_support()

def apply_bacula_support(load=True):
    if not bacula_active_sd():
        if isfile(BACULA_CONFIG_FILE):
            unlink(BACULA_CONFIG_FILE)
        raise Disabled('Bacula sd désactivé')
    if load_bacula_support()['support'] == 'none':
        print "bacula n'est pas configuré"
        _empty_file(BACULA_CONFIG_FILE)
        #cree un fichier eol minimal
        dic = {'support': 'none'}
        _save_bacula_conf(filename=BACULA_SUPPORT, dic=dic)
    instance_bacula(BACULA_CONFIG_TMPL, BACULA_CONFIG_FILE, BACULA_SUPPORT)
    if load:
        reload_conf(test='sd')

def test_bacula_support(force=False, chown=False):
    """
    :param force: drapeau pour forcer l'exécution du test
    :type force: booléen
    """

    if force or bacula_active_sd():
        status, dic = mount_bacula_support(retry=2, chown=chown)
        if status == True:
            umount_bacula_support()
        comment = mount_status_to_str(dic)
    else:
        status = True
        comment = 'Support non testé'
    return status, comment


# -------------------------------------------------
# ----------------- Bacula mail -------------------
# -------------------------------------------------
def load_bacula_mail():
    if not bacula_active_dir():
        raise Disabled('Bacula dir désactivé')
    if not isfile(BACULA_MAIL):
        return False
    return load_bacula_conf(filename=BACULA_MAIL)

def save_bacula_mail(mail_ok=None, mail_error=None):
    if not bacula_active_dir():
        raise Disabled('Bacula dir désactivé')
    dic = {}
    if mail_ok != None:
        dic['mail_ok'] = mail_ok
    if mail_error != None:
        dic['mail_error'] = mail_error
    _save_bacula_conf(filename=BACULA_MAIL, dic=dic)
    apply_bacula_mail()

def apply_bacula_mail(load=True):
    if not bacula_active_dir():
        if isfile(BACULA_CONFIG_MAILALL):
            unlink(BACULA_CONFIG_MAILALL)
        if isfile(BACULA_CONFIG_MAILONERROR):
            unlink(BACULA_CONFIG_MAILONERROR)
        raise Disabled('Bacula dir désactivé')
    if load_bacula_mail() == False:
        print "les mails bacula ne sont pas configurés"
        _empty_file(BACULA_CONFIG_MAILALL)
        _empty_file(BACULA_CONFIG_MAILONERROR)
        return
    instance_bacula(BACULA_CONFIG_MAILALLTMPL, BACULA_CONFIG_MAILALL,
                    BACULA_MAIL)
    instance_bacula(BACULA_CONFIG_MAILONERRORTMPL, BACULA_CONFIG_MAILONERROR,
                    BACULA_MAIL)
    if load:
        reload_conf(test='dir')

# -------------------------------------------------
# ------------------ Bacula job -------------------
# -------------------------------------------------

DAY_TO_STRING = {1: 'dimanche au lundi', 2: 'lundi au mardi',
        3: 'mardi au mercredi', 4: 'mercredi au jeudi', 5: 'jeudi au vendredi',
        6: 'vendredi au samedi', 7: 'samedi au dimanche'}

DAY_TO_BACULA = {1: 'mon', 2: 'tue', 3: 'wed', 4: 'thu', 5: 'fri',
                 6: 'sat', 7: 'sun'}
LEVEL_TO_FR = {'Incremental':'Incrémentale', 'Full':'Totale',
               'Differential': 'Différentielle'}

def load_bacula_jobs(check_active=True):
    """
    check_active: don't load dicos to check if bacula-dir is activate
    """
    if check_active and not bacula_active_dir():
        raise Disabled('Bacula dir désactivé')
    try:
        dicosched = load(file(BACULA_JOB, 'r'))
    except:
        dicosched = []
    return dicosched

def save_bacula_job(dic):
    if not bacula_active_dir():
        raise Disabled('Bacula dir désactivé')
    dump(dic, file(BACULA_JOB, 'w'))

def add_job(job, level, day, hour, end_day=None):
    """
    job: weekly, daily or monthly
    level: Full/Incremental/Differential
    day: day in week
    hour: hour of job
    end_day: used only for daily job
    """
    if not bacula_active_dir():
        raise Disabled('Bacula dir désactivé')
    def raise_if_conflict(day, index):
        raise Exception('Sauvegarde de la nuit du {0} entre en conflit avec la sauvegarde {1}'.format(DAY_TO_STRING[int(day)], index))

    if level not in ['Full', 'Incremental', 'Differential']:
        raise Exception('level inconnu')

    if job not in ['weekly', 'daily', 'monthly']:
        raise Exception ('job inconnu')
    try:
        day = int(day)
    except:
        raise Exception("Le jour n'est pas un entier ({0})".format(day))
    if not 0 < day < 8:
        raise Exception("Le jour {0} n'est pas correct n'est pas entre 1 et 7".format(day))
    if job == 'daily':
        if end_day == None:
            raise Exception('end_day doit être renseigné')
        try:
            end_day = int(end_day)
        except:
            raise Exception("Le jour de fin n'est pas un entier ({0})".format(end_day))
        if not 0 < end_day < 8:
            raise Exception("Le jour de fin {0} n'est pas correct n'est pas entre 1 et 7".format(end_day))
        if day == end_day:
            raise Exception('Jour de début et de fin identique, veuiller faire une sauvegarde hebdomadaire')
        if day > end_day:
            raise Exception('Le jour de fin est inférieur au jour de début')
    elif end_day != None:
        raise Exception('end_day ne concerne que les job daily')
    try:
        hour = int(hour)
    except:
        raise Exception("L'heure n'est pas un entier ({0})".format(hour))
    if not 0 <= hour < 24:
        raise Exception("L'heure {0} n'est pas correct entre 0 et 23".format(hour))
    jobs = load_bacula_jobs()

    for tjob in jobs:
        index = jobs.index(tjob)+1
        #test si le jour du job est déjà utilisé pour le même type de job
        if day == tjob['day'] and job == tjob['job']:
            raise_if_conflict(day, index)
        #si le nouveau job est de type daily, il ne faut pas de monthly ou
        #de weekly le même jour
        if job == 'daily':
            if tjob['job'] == 'weekly' and day == tjob['day']:
                raise_if_conflict(day, index)
            #si le job existant est du type daily, vérifier la plage et pas
            #seulement le jour
            if tjob['job'] == 'daily':
                if tjob['day'] < day <= tjob['end_day']:
                    raise_if_conflict(day, index)
        #si le job existant est de type daily, il faut verifié que le nouveau
        #job monthly ou weekly ne soit pas définit le même jour
        elif tjob['job'] == 'daily' and job == 'weekly':
            if tjob['day'] <= day <= tjob['end_day']:
                raise_if_conflict(day, index)

    retjob = {'job': job,
              'level': level,
              'day': day,
              'hour': hour}
    if end_day != None:
        retjob['end_day'] = end_day
    jobs.append(retjob)
    save_bacula_job(jobs)
    apply_bacula_jobs()

def del_job(jobnumber):
    if not bacula_active_dir():
        raise Disabled('Bacula dir désactivé')
    try:
        jobindex = int(jobnumber) - 1
    except ValueError:
        raise Exception("jobnumber n'est pas un nombre")
    jobs = load_bacula_jobs()
    try:
        if jobindex < 0:
            raise IndexError
        jobs.pop(jobindex)
    except IndexError:
        raise Exception("Le job {1} n'existe pas :\n{0}".format(pprint_bacula_jobs(), jobnumber))
    save_bacula_job(jobs)
    apply_bacula_jobs()

def apply_bacula_jobs(load=False):
    if not bacula_active_dir():
        if isfile(BACULA_CONFIG_JOB):
            unlink(BACULA_CONFIG_JOB)
        if isfile(BACULA_CONFIG_JOBSCHED):
            unlink(BACULA_CONFIG_JOBSCHED)
        raise Disabled('Bacula dir désactivé')
    def write_job(job, day, monthly=False):
        level = job['level']
        hour = job['hour']
        day = int(day)
        #weeknumber must be 1st for monthly jobs
        #weeknumber must be 2nd-5th if monthly job set same day
        if monthly:
            weeknumber = '1st '
        else:
            if str(day) in monthjobs:
                weeknumber = '2nd-5th '
            else:
                weeknumber = ''
        #if 12 <= day < 24: day = yesterday
        #if 0 <= day < 12: day = today

        schehour = schedule['hour']
        if int(hour) >= 12:
            if day == 1:
                newday = 7
            else:
                newday = day-1
        else:
            newday = day
            if schehour < hour:
                schehour = hour

        tday = DAY_TO_BACULA[newday]
        scheday = DAY_TO_BACULA[day]
        schemin = str(schedule['minute'])
        #time must be 4:01 not 4:1
        if len(schemin) == 1:
            schemin = '0{0}'.format(schemin)
        fhjob.write('Run = {0} {1}{2} at {3}:00\n'.format(level, weeknumber, tday, hour))
        fhjobpost.write('Run = Full {0}{1} at {2}:{3}\n'.format(weeknumber, scheday, schehour, schemin))
    jobs = load_bacula_jobs()
    if jobs == []:
        print "les programmations bacula ne sont pas configurées"
        _empty_file(BACULA_CONFIG_JOB)
        _empty_file(BACULA_CONFIG_JOBSCHED)
        return None
    fhjob = open(BACULA_CONFIG_JOB, 'w')
    fhjobpost = open(BACULA_CONFIG_JOBSCHED, 'w')

    #get all monthly job
    monthjobs = []
    for job in jobs:
        if job['job'] == 'monthly':
            monthjobs.append(str(job['day']))

    #load schedule
    schedule = load_schedule()
    for job in jobs:
        if job['job'] == 'daily':
            for i in range(int(job['day']), int(job['end_day'])+1):
                write_job(job, i)
        elif job['job'] == 'weekly':
            write_job(job, job['day'])
        elif job['job'] == 'monthly':
            write_job(job, job['day'], True)
        else:
            raise Exception('Type de job inconnu')
    fhjob.close()
    fhjobpost.close()

    if load:
        reload_conf(test='dir')

def display_bacula_jobs():
    if not bacula_active_dir():
        raise Disabled('Bacula dir désactivé')
    jobs = load_bacula_jobs()
    if jobs == []:
        return None
    disp = []
    for save in jobs:
        jobnumber = jobs.index(save)
        day = save['day']
        level = LEVEL_TO_FR[save['level']]
        hour = save['hour']
        job = save['job']
        message = "{0} : Sauvegarde {1} ".format(jobnumber+1, level.lower())
        if job == 'daily':
            end_day = save['end_day']
            message += "de la nuit du {0} à la nuit du {1} ".format(DAY_TO_STRING[int(day)], DAY_TO_STRING[int(end_day)])
        elif job == 'weekly':
            message += "dans la nuit du {0} ".format(DAY_TO_STRING[int(day)])
        else:
            message += "dans la première nuit du mois du {0} ".format(DAY_TO_STRING[int(day)])
        message += "à {0}:00".format(hour)
        disp.append((jobnumber, message))
    return disp

def pprint_bacula_jobs():
    jobs = display_bacula_jobs()
    if jobs != None:
        pprinted_bacula_jobs = "\n".join(["\t{0}".format(job) for num, job in display_bacula_jobs()])
    else:
        pprinted_bacula_jobs = "Aucun job programmé."
    return pprinted_bacula_jobs

def is_running_jobs():
    if not bacula_active():
        raise Exception("Bacula n'est pas actif")
    code, out, err = run_bacula_cmd('status dir')
    try:
        running_lines = out.split('\n====\n')[1]
    except:
        print "Erreur de status"
        print out, err
        return True

    if not 'No Jobs running.' in running_lines and \
        not 'Pas de job en cours.' in running_lines:
        print "Jobs en cours : Bacula est occupé"
        print "\n".join(running_lines.split('\n')[5:])
        return True

    return False


def _date_from_epoch(epoch):
    return datetime.fromtimestamp(epoch).strftime("%A %d %B %Y à %H:%M")

def _is_more_than_one_day(epoch):
    return datetime.now() > datetime.fromtimestamp(epoch) + timedelta(1)

def _bacula_rapport(job_type, text, status):
    date = time()
    try:
        dic = load(file(BACULA_RAPPORT, 'r'))
    except EOFError, IOError:
        dic = {}
    dic[job_type] = {'text' : text,
        'date': date,
        'status': str(status)}
    dump(dic, file(BACULA_RAPPORT, 'w'))

def bacula_rapport_in_progress(job_type):
    _bacula_rapport(job_type, "Sauvegarde en cours depuis le {0}.", BACULA_RAPPORT_UNKNOWN)

def bacula_rapport_ok(job_type):
    _bacula_rapport(job_type, "Sauvegarde terminée le {0}.", BACULA_RAPPORT_OK)

def bacula_rapport_err(job_type):
    _bacula_rapport(job_type, "Sauvegarde échouée le {0}.", BACULA_RAPPORT_ERR)

def bacula_rapport_load(report='sauvegarde'):
    if not isfile(BACULA_RAPPORT) or stat(BACULA_RAPPORT).st_size == 0:
        return (BACULA_RAPPORT_UNKNOWN, 'Aucune sauvegarde')
    else:
        try:
            info = load(file(BACULA_RAPPORT, 'r'))[report]
        except:
            return (BACULA_RAPPORT_UNKNOWN, "Impossible de lire le rapport de sauvegarde")

        status = info['status']
        text = info['text'].format(_date_from_epoch(info['date']))
        if status == BACULA_RAPPORT_UNKNOWN:
            #verifie si l'etat de sauvegarde n'a pas plus de 24 heures
            if _is_more_than_one_day(info['date']):
                status = BACULA_RAPPORT_ERR
                text = "Sauvegarde démarrée depuis plus d'une journée (date de démarrage : %s)"% _date_from_epoch(info['date'])
        elif status not in [BACULA_RAPPORT_UNKNOWN, BACULA_RAPPORT_OK, BACULA_RAPPORT_ERR]:
            status = BACULA_RAPPORT_UNKNOWN
            text = 'Status inconnu : %s' % info['status']
        return (status, text)
