# -*- coding: utf-8 -*-

import sys
from pyeole2.process import system_code, system_out
from pyeole2.service import service_out
from pyeole2.bacula import BACULA_CONF, BACULA_SUPPORT, BACULA_JOB, \
        BACULA_MAIL, MOUNT_POINT, mount_bacula_support
from pyeole2.ansiprint import print_red, print_orange, print_blue
from creole2.config import configeol
from os import system, unlink, stat, listdir, chown
from creole2.parsedico import parse_dico
from os.path import isfile, dirname, basename, normpath, join as joinpath
from tempfile import mkstemp
from shutil import move
import re

catalog_label = '-catalog-'
catalog_pool_name_tmpl = '{0}' + catalog_label
catalog_db_file = '/var/lib/bacula/bacula.db'
catalog_sql_file = '/var/lib/bacula/bacula.sql'

# Bacula may speek french (or not) (#8644)
cd_regex = re.compile(r'cd (?P<cd>.*?)\n(?P<invalide>Invalid path given.\n)?cwd is: (?P<cwd>.*?)\n')
cd_regex_fr = re.compile(r'cd (?P<cd>.*?)\n(?P<invalide>Invalid path given.\n)?Le répertoire courant est : (?P<cwd>.*?)\n')

dico = None

def load_dico():
    global dico
    if dico == None:
        dico = parse_dico()

def bacula_active():
    load_dico()
    return dico['activer_bacula'] == 'oui' and dico['activer_bacula_dir'] == 'oui'

def bacula_create_restore_log_file():
    filename = '/var/log/bacula/restore.txt'
    fh = open(filename, 'w')
    fh.write('')
    fh.close()
    print """La restauration est lancée en tâche de fond.
Vous pouvez suivre son évolution dans le fichier {0}""".format(filename)

def bacula_restore_one_file(filename, jobid):
    bconsole_command("file={0} done\n{1}\n".format(filename, jobid), stdout_param="@output /dev/null\n")
    bacula_create_restore_log_file()

def bacula_ls_one_folder(foldername, jobid, test_mode=False):
    foldername = normpath(foldername)
    if foldername[0] != "/":
        print_orange("Un chemin complet est requis")
        if test_mode == True:
            return False
        else:
            foldername = "/" + foldername
            print_orange("Essai avec {0}".format(foldername))
    stdout = bconsole_command("\ncd {0}\nls\ndone\n".format(foldername)).split("$ ")[1:]
    ls_result = stdout[1].split('\n')[1:]
    cd_result = re.search(cd_regex, stdout[0])
    if cd_result is None:
        cd_result = re.search(cd_regex_fr, stdout[0])
        if cd_result is None:
            print_red("Réponse de Bacula non reconnue")
            print '"""\n{0}"""'.format(stdout[0])
            sys.exit(1)
    if cd_result.groupdict()['invalide'] == None:
        if test_mode == True:
            return True
        else:
            print '\n'.join(ls_result)
    else:
        print_red("Le repertoire {0} n'existe pas.".format(foldername))
        if test_mode == True:
            return False
        else:
            foldername = dirname(foldername)
            print "liste du contenu du repertoire parent {0} :".format(foldername)
        bacula_ls_one_folder(foldername, jobid)

def bacula_restore_one_folder(foldername, jobid):
    foldername = normpath(foldername)
    if not bacula_ls_one_folder(foldername, jobid, test_mode=True) == True:
        sys.exit(0)
    else:
        parent = dirname(foldername)
        dirn = basename(foldername)
        bconsole_command("\ncd {0}\nmark {1}\ndone\n{2}\n".format(parent, dirn,
            jobid), stdout_param="@output /dev/null\n")
        bacula_create_restore_log_file()

def bacula_restore_all_files(jobid):
    bconsole_command('\ndone\n{0}\n'.format(jobid), prefix='select current all yes', stdout_param="@output /dev/null\n")
    bacula_create_restore_log_file()

def bacula_delete_catalog():
    if isfile(catalog_db_file):
        service_out('bacula-director', 'stop')
        unlink(catalog_db_file)

def bacula_restore_dump(filename=catalog_sql_file):
    if isfile(catalog_db_file):
        print "Le catalogue existe déjà"
        sys.exit(1)
    system('/usr/bin/sqlite3 {0} < {1}'.format(catalog_db_file, filename))
    system_code(['chown', 'bacula:root', catalog_db_file])
    service_out('bacula-director', 'start')

def bacula_search(filenames):
    if type(filenames) != list or filenames == []:
        raise Exception('Il faut une liste de fichier pour bacula_search')
    cmd = '\n@output\nfind ' + ' '.join(filenames)
    stdout = bconsole_command(cmd).split('$')[1].split('\n')[2:]
    stdout = [f for f in stdout if f != '']
    if len(stdout) > 0:
        print '\n'.join(stdout)
    else:
        print_orange('Aucun resultat')

def bconsole_command(cmd, prefix='select current yes', stdout_param=""):
    if not bacula_active():
        raise Exception("Bacula n'est pas actif, impossible de restaurer")
    bacula_fd = dico['bacula_fd_name']
    code, stdout, stderr = system_out(["bconsole", "-c", "/etc/bacula/bconsole.conf"], stdin="{3}restore FileSet=FileSetSauvegarde Client={0} {1} {2}\n".format(bacula_fd, prefix, cmd, stdout_param))
    if code != 0:
        raise Exception('Erreur bconsole')
    return stdout

def get_volume_name(bacula_dir_name):
    try:
        mount_bacula_support()
    except:
        pass
    volume_names = [v for v in listdir(MOUNT_POINT) if catalog_label in v]
    if len(volume_names) > 0:
        filtered_volume_names = [v for v in volume_names if catalog_pool_name_tmpl.format(bacula_dir_name) in v]
        if len(filtered_volume_names) > 0:
            return max([(stat(joinpath(MOUNT_POINT, i)).st_mtime, i)
                for i in filtered_volume_names])[1]
        else:
            print_red("échec de la restauration")
            print "\n".join(["Les noms de directeur possibles sont :"] + ["\t" + v.split(catalog_label)[0] for v in volume_names])
            sys.exit(1)
    else:
        print_red("échec de la restauration")
        print "Aucun volume respectant le modèle de nommage dans le répertoire {0}".format(MOUNT_POINT)
        sys.exit(1)

def extract_bacula_files(bacula_dir_name=None):
    if bacula_dir_name == None:
        load_dico()
        bacula_dir_name = dico['bacula_dir_name']
    system_code(['/bin/sed', '-i',
            's@  Archive Device = /nonexistant/path/to/file/archive/dir@  Archive Device = {0}@g'.format(MOUNT_POINT),
            '/etc/bacula/bacula-sd.conf'])
    volume_name = get_volume_name(bacula_dir_name)
    extract_file(catalog_sql_file, bacula_dir_name, volume_name=volume_name)
    extract_file('/var/lib/bacula/{0}-JobDefsSauvegarde.bsr'.format(bacula_dir_name), bacula_dir_name, volume_name=volume_name)
    system_code(['chown', 'bacula:bacula', '/var/lib/bacula/{0}-JobDefsSauvegarde.bsr'.format(bacula_dir_name)])
    extract_file('/var/lib/bacula/{0}-JobDefsCatalog.bsr'.format(bacula_dir_name), bacula_dir_name, error_if_not_found=False, volume_name=volume_name)
    system_code(['chown', 'bacula:bacula', '/var/lib/bacula/{0}-JobDefsCatalog.bsr'.format(bacula_dir_name)])
    extract_file(BACULA_SUPPORT, bacula_dir_name, error_if_not_found=False, volume_name=volume_name)
    extract_file(BACULA_MAIL, bacula_dir_name, error_if_not_found=False, volume_name=volume_name)
    extract_file(BACULA_JOB, bacula_dir_name, error_if_not_found=False, volume_name=volume_name)
    extract_file(BACULA_CONF, bacula_dir_name, error_if_not_found=False, volume_name=volume_name)
    extract_file(configeol, bacula_dir_name, dest_file='/root/zephir-restore.eol', volume_name=volume_name)

def extract_file(filename, bacula_dir_name, dest_file=None, error_if_not_found=True, volume_name=None):
    """Extract file directly in a volume identified by file name
    (bacula tools use base path name from configuration file)"""
    tmp_file = mkstemp()[1]
    include_file = open(tmp_file, 'w')
    include_file.write(filename)
    include_file.close()
    extract_dir = '/tmp'
    if volume_name is None:
        volume_name = get_volume_name(bacula_dir_name)
    code, stdout, stderr = system_out(['/usr/sbin/bls', '-V', volume_name, 'FileStorage', '-i', tmp_file])
    if code == 1:
        unlink(tmp_file)
        raise Exception('Erreur lors de bls : {0},{1}'.format(stdout, stderr))
    if stdout.strip().split('\n')[-1] == "0 files found.":
        print "Pas de fichier {0} dans le volume {1}".format(filename, volume_name)
        if error_if_not_found:
            raise Exception("Impossible de lister le catalogue {0}".format(volume_name))
    else:
        code, stdout, stderr = system_out(['/usr/sbin/bextract', '-V', volume_name, 'FileStorage', extract_dir, '-i', tmp_file])
        if code == 1:
            unlink(tmp_file)
            print "Impossible de restaurer {0} du catalogue {1}".format(filename, volume_name)
            raise Exception(stderr)
        if dest_file == None:
            dest_file = filename
        move("{0}{1}".format(extract_dir, filename), dest_file)
    unlink(tmp_file)

def exit_if_running_jobs():
    if not bacula_active():
        raise Exception("Bacula n'est pas actif, impossible de restaurer")
    code, out, err = system_out(["/usr/bin/bconsole"], stdin='status dir')
    try:
        running_lines = out.split('\n====\n')[1]
    except:
        print "Erreur de status"
        print out, err
        sys.exit(1)

    if not 'No Jobs running.' in running_lines and not 'Pas de job en cours.' in running_lines:
        print "Impossible de restaurer, jobs en cours"
        print "\n".join(running_lines.split('\n')[5:])
        sys.exit(1)

