#-*-coding:utf-8-*-
"""

    Common eole log library

"""
import logging

EOLE_FORMAT = "%(asctime)s - %(name)s - %(levelname)s - '%(funcName)s' %(message)s"
DEFAULT_LOG_FILE = "/var/log/eole/default.log"
DEFAULT_LOG_LEVEL = 'ERROR'

class DebugFilter(logging.Filter):
    def filter(self, rec):
        if rec.levelno <= logging.DEBUG:
            return logging.Filter.filter(self, rec)

class InfoFilter(logging.Filter):
    def filter(self, rec):
        if logging.INFO >= rec.levelno > logging.DEBUG:
            return logging.Filter.filter(self, rec)

class DebugInfoFilter(logging.Filter):
    def filter(self, rec):
        if logging.INFO >= rec.levelno:
            return logging.Filter.filter(self, rec)

class ErrorFilter(logging.Filter):
    def filter(self, rec):
        if rec.levelno >= logging.WARNING:
            return logging.Filter.filter(self, rec)

def get_formatter(logfmt, datefmt):
    if datefmt:
        return logging.Formatter(logfmt, datefmt)
    return logging.Formatter(logfmt)

def get_log_level(level=DEFAULT_LOG_LEVEL):
    """
        permet de renvoyer un objet loglevel avec un comportement standard
        :param level: string in 
                      ['CRITICAL', 'ERROR', 'WARNING', 'WARN', 'INFO', 'DEBUG']
        :return: logging.loglevel
    """
    level = level.upper()
    if level in ['CRITICAL', 'ERROR', 'WARNING', 'WARN', 'INFO', 'DEBUG']:
        return getattr(logging, level)
    return logging.INFO

def make_file_handler(logfile, loglevel, logfilter, logformat, datefmt):
    """

        créee un handler de type FileHandler
        :param logfile: le fichier de log
        :param loglevel: (string) le niveau de sévérité ['CRITICAL', 'ERROR', 'WARNING', 'INFO', 'DEBUG']
        :param logfilter: class filtre à appliquer au handler
        :param logformat: (string) format du log
        :return: le handler

    """
    handler = logging.FileHandler(logfile)
    handler.setFormatter(get_formatter(logformat, datefmt))
    handler.setLevel(get_log_level(loglevel))
    if logfilter:
        handler.addFilter(logfilter)
    return handler

def make_logger(packagename, logfile=DEFAULT_LOG_FILE, errfile=None, debugfile=None, loglevel=DEFAULT_LOG_LEVEL, splitlog=False, logfmt=EOLE_FORMAT, datefmt=None):
    """

        point d'entrée principal du module pour la création du loggeur
        :Return: le logger

        Initialise un logger pour le package packagename
        :param packagename: (string) nom du package appelant ex: creole.eosfunc
        :param logfile: (string) fichier de log du package
        :param errfile: (string) optionnel, fichier de log des erreurs (CRITICAL & ERROR)
        :param debugfile: (string) optionnel, fichier de log du debug (DEBUG) 
        :param loglevel: (string) niveau demandé
        :param splitlog: (boolean) optionnel, séparer les niveau de log suivant les fichiers ou concaténer les niveaux inférieurs
        :param logfmt: (string) optionnel, format du log (http://docs.python.org/library/logging.html#formatter-objects)

    """
    logger = logging.getLogger(packagename)
    logfilter = None
    # si déjà existant, le renvoie
    if logger.handlers:
        logger.setLevel(get_log_level(loglevel))
        return logger
    if errfile:
        if splitlog:
            logfilter = ErrorFilter()
        logger.addHandler(make_file_handler(errfile, 'WARNING', logfilter, logfmt, datefmt))
    if debugfile:
        if splitlog:
            logfilter = DebugFilter()
        # Si on a un fichier pour le DEBUG, on log à part
        logger.addHandler(make_file_handler(debugfile, 'DEBUG', logfilter, logfmt, datefmt))
        if splitlog:
            logfilter = InfoFilter()
        logger.addHandler(make_file_handler(logfile, 'INFO', logfilter, logfmt, datefmt))
    else:
        if splitlog:
            logfilter = DebugInfoFilter()
        # Sinon on log DEBUG+INFO dans le même fichier
        logger.addHandler(make_file_handler(logfile, 'DEBUG', logfilter, logfmt, datefmt))
    logger.setLevel(get_log_level(loglevel))
    return logger

