# -*- coding: utf-8 -*-

import sys, random
from os import chdir, listdir, readlink, symlink, unlink
from os.path import join, isfile, isdir, islink, basename
from pickle import dump, load
from pyeole2.ansiprint import print_orange
from pyeole2.process import system_out

## configuration
SCHEDULE_FILE = '/var/lib/eole/config/schedule.conf'
SCHEDULE_DIR = '/usr/share/eole/schedule'

#BACULA_FILE = '/var/lib/eole/config/baculajobs.conf'

CRON_FILE='/etc/cron.d/schedule'

DAY_TO_STRING = {1: 'lundi', 2: 'mardi', 3: 'mercredi', 4: 'jeudi', 5: 'vendredi', 6: 'samedi', 7: 'dimanche'}

def _list_all_available_file(mod):
    """
    list all available cron files

    mod: pre or post
    """
    directory = join(SCHEDULE_DIR, mod)
    if not isdir(directory):
        raise Exception("no directory {0}".format(directory))

    return listdir(directory)

def _list_all_enable_file(mod):
    """
    list all enable cron files
    """
    #list all links in separate cron directories
    dest_file = []
    for tday in ['daily', 'weekly', 'monthly', 'once']:
        tdirectory = join(SCHEDULE_DIR, tday, mod)
        if not isdir(tdirectory):
            raise Exception('no directory {0}'.format(tdirectory))
        else:
            for tdest_file in listdir(tdirectory):
                if tdest_file.endswith('~'):
                    continue
                if islink(join(tdirectory, tdest_file)):
                    tdest_link = basename(readlink(join(tdirectory, tdest_file)))
                dest_file.append((tday, tdest_file))
    return dest_file

def _compare_orig_dest(available, enable):
    """
    remove enable cron files in available cron files

    available: available cron file
    enable: enable cron file
    """
    for tday, tdest in enable:
        if tday != 'once' and tdest in available:
            available.remove(tdest)

def list_schedule(mod, day=None):
    """
    list available and enable schedule file

    mod: pre or post

    return (available, enable)
    """
    orig_file = _list_all_available_file(mod)
    dest_file = _list_all_enable_file(mod)
    if day != 'once':
        _compare_orig_dest(orig_file, dest_file)
    return (orig_file, dest_file)

def description(mod, script, day):
    if mod not in ['pre', 'post']:
        raise Exception('Not a valid mod')
    if day not in ['daily', 'weekly', 'monthly']:
        raise Exception('Not a valid day')
    script_file = join(SCHEDULE_DIR, day, mod, script)
    code, stdout, stderr = system_out(['grep', "^DESC=", script_file])
    if code != 0:
        return script
    try:
        exec(stdout)
        return DESC
    except:
        return script

def add_del_schedule(mod, script, day, action):
    """
    mod: pre or post
    script: name of schedule script
    day: daily, weekly, monthly or once
    action: add or del
    """
    if mod not in ['pre', 'post']:
        raise Exception("unknown option {0}".format(mod))

    orig_file, dest_file = list_schedule(mod, day)

    if day not in ['daily', 'weekly', 'monthly', 'once']:
        raise Exception('unknown day {0}'.format(day))

    if action not in ['add', 'del']:
        raise Exception('unknown action {0}'.format(action))

    orig = join(SCHEDULE_DIR, mod, script)
    ddir = join(SCHEDULE_DIR, day, mod)
    dest = join(ddir, script)
    if action == 'add':
        if script not in orig_file:
            for tday, dest in dest_file:
                if script in dest:
                    raise Exception("{0} already a {1} script".format(script, tday))
            raise Exception("unknown script {0}".format(script))

        symlink(orig, dest)
    elif action == 'del':
        is_link=False
        for tdest_file in listdir(ddir):
            if islink(join(ddir, tdest_file)):
                tdest_link = basename(readlink(join(ddir, tdest_file)))
                if tdest_file == script:
                    is_link=True
                    break
        if not is_link:
            if isfile(join(ddir, script)):
                raise Exception("{0} is not a link in {1}, cannot del it".format(script, ddir))
            else:
                raise Exception("{0} is not a file in {1}".format(script, ddir))
        unlink(dest)

def add_post_schedule(script, day):
    """
    script: name of schedule script
    day: daily, weekly, monthly or once
    """
    add_del_schedule(mod='post', script=script, day=day, action='add')

def add_pre_schedule(script, day):
    """
    script: name of schedule script
    day: daily, weekly, monthly or once
    """
    add_del_schedule(mod='pre', script=script, day=day, action='add')

def del_post_schedule(script, day):
    """
    script: name of schedule script
    day: daily, weekly, monthly or once
    """
    add_del_schedule(mod='post', script=script, day=day, action='del')

def del_pre_schedule(script, day):
    """
    script: name of schedule script
    day: daily, weekly, monthly or once
    """
    add_del_schedule(mod='pre', script=script, day=day, action='del')

def hour_display(number):
    if number < 10:
        return '0{0}'.format(number)
    return str(number)

def prog_schedule(force=False):
    """
    Programmation automatique d'une mise à jour hebdomadaire

    force: True: remove SCHEDULE_FILE if exists
           False: write_schedule if exists
    """
    if isfile(SCHEDULE_FILE):
        if not force:
            #just rewrite crontab file
            write_schedule()
            return True
    dic = {'hour': random.randint(1, 5)}
    dic['minute'] = random.randint(0, 59)

    dic['weekday'] = random.randint(1, 7)

    #init monthday with same value as weekday
    #monthday and weekday must not have same value
    dic['monthday'] = dic['weekday']
    while dic['monthday'] == dic['weekday']:
        dic['monthday'] = random.randint(1, 7)

    schedule_file = file(SCHEDULE_FILE, 'w')
    dump(dic, schedule_file)
    schedule_file.close()
    write_schedule()
    list_schedules_pre = _list_all_enable_file('pre')
    list_schedules_post = (_list_all_enable_file('post'))
    print_orange("Tâches planifiées EOLE :")
    print_orange("* Les tâches journalières se feront tous les jours à {0}:{1} (hors sauvegarde)".format(hour_display(dic['hour']), hour_display(dic['minute'])))
    for day, script in list_schedules_pre:
        if day == 'daily':
            print("  * {0}".format(description('pre', script, 'daily')))
    for day, script in list_schedules_post:
        if day == 'daily':
            print("  * {0}".format(description('post', script, 'daily')))
    print_orange("* Les tâches hebdomadaires se feront le {0} à {1}:{2} (hors sauvegarde)".format(DAY_TO_STRING[dic['weekday']], hour_display(dic['hour']), hour_display(dic['minute'])))
    for day, script in list_schedules_pre:
        if day == 'weekly':
            print("  * {0}".format(description('pre', script, 'weekly')))
    for day, script in list_schedules_post:
        if day == 'weekly':
            print("  * {0}".format(description('post', script, 'weekly')))
    print_orange("* Les tâches mensuelles se feront le premier {0} du mois à {1}:{2} (hors sauvegarde)".format(DAY_TO_STRING[dic['monthday']], hour_display(dic['hour']), hour_display(dic['minute'])))
    for day, script in list_schedules_pre:
        if day == 'monthly':
            print("  * {0}".format(description('pre', script, 'monthly')))
    for day, script in list_schedules_post:
        if day == 'monthly':
            print("  * {0}".format(description('post', script, 'monthly')))
    return True

def load_schedule():
    if not isfile(SCHEDULE_FILE):
        raise Exception("ERROR: {0} not exist".format(SCHEDULE_FILE))

    schedule_file = file(SCHEDULE_FILE, 'r')
    dic = load(schedule_file)
    schedule_file.close()
    return dic

def write_schedule():
    dic = load_schedule()
    cron = file(CRON_FILE, 'w')
    cron.write("""#EOLE don't touch this file
{0} {1} * * * root python {2}/schedule cron >> /var/log/schedule.log 2>&1
""".format(dic['minute'], dic['hour'], SCHEDULE_DIR))
    cron.close()

