# -*- coding: UTF-8 -*-
from time import sleep
from shutil import move

from glob import glob
from os import unlink
from os.path import join, isfile
from pyeole2.process import system_out, system_code
from creole2.config import init_services_file, VIRTMASTER, containers_file
try:
    import json
except:
    import simplejson as json

def _init_service(service):
    """
    charge le fichier init_services_file et retourne les valeurs correspond au
    service
    """
    if isfile(init_services_file):
        dico_init = json.load(file(init_services_file))
    else:
        dico_init = {}
    return dico_init.get(service, ('service', False, []))

def get_container_path(container, path=''):
    """
    renvoie le chemin d'un conteneur
    ou le chemin d'un fichier dans un conteneur
    """
    dico_container = {}
    execfile(containers_file, {}, dico_container)
    cpath = dico_container.get('container_path_%s' % container, '')
    return cpath+path

def get_containers_group(containers):
    """
    renvoie les groupes associés à des conteneurs
    """
    groups = set()
    dico_container = {}
    execfile(containers_file, {}, dico_container)
    for container in containers:
        groups.add(dico_container['container_name_%s' % container])
    return list(groups)

def _gen_service_cmd(service, action):
    """
    génération de la commande en fonction des services
    """
    cmds = []
    init, pty, containers = _init_service(service)
    if init == 'upstart':
        cmds.append(['/sbin/restart-wrapper', service, action])
    elif action == 'restart':
        if init == 'initd':
            cmds.append(['/etc/init.d/%s' % service, 'stop'])
            cmds.append(['sleep', '1'])
            cmds.append(['/etc/init.d/%s' % service, 'restart'])
        else:
            cmds.append(['/usr/sbin/service', service, "stop"])
            cmds.append(['sleep', '1'])
            cmds.append(['/usr/sbin/service', service, "start"])
    elif init == 'initd':
        cmds.append(['/etc/init.d/%s' % service, action])
    elif init == 'apache':
        return None, None
    else:
        cmds.append(['/usr/sbin/service', service, action])
    return cmds, pty

def service_code(service, action, container=VIRTMASTER):
    """
    action sur un service :
    - affichage des messages (stdout/stderr) "à l'écran"
    - retourne le *dernier* code de retour sous la forme d'un entier
    @service : nom du service
    @action : action sur le service (stop, start, restart)
    @container : conteneur du service
    """
    cmds, use_pty = _gen_service_cmd(service, action)
    if cmds != None:
        for cmd in cmds:
            if cmd[0] == 'sleep':
                ret = sleep(int(cmd[1]))
            else:
                env = None
                pty = False
                if use_pty:
                    terminfo_dir = get_container_path(container, '/etc/terminfo/')
                    try:
                        #gestion correct des [ OK ]
                        import sys, fcntl, termios, struct
                        data = fcntl.ioctl(sys.stdout.fileno(), termios.TIOCGWINSZ, '1234')
                        cols = struct.unpack('hh', data)[1]
                        ti_file = '/tmp/eole.ti'
                        fh = open(ti_file, 'w')
                        fh.write('eole,\n\txenl,\n\thpa=\E[%i%p1%dG,\n\tsetaf=\E[3%p1%dm,\n\tcols#{0},\n\top=\E[39;49m,\n'.format(cols))
                        fh.close()
                        ticcmd = ['/usr/bin/tic', '-o', terminfo_dir, ti_file]
                        system_code(ticcmd)
                        env = {'TERM': 'eole'}
                        pty = True
                        unlink(ti_file)
                    except:
                        pty = False
                        env = None
                ret = system_code(cmd, container=container, env=env, pty=pty)
        return ret
    else:
        return 0

def service_code_no_container(service, action):
    """
    action sur un service sans connaitre le nom du conteneur :
    - affichage des messages (stdout/stderr) "à l'écran"
    - retourne le *dernier* code de retour sous la forme d'un entier
    @service : nom du service
    @action : action sur le service (stop, start, restart)
    """
    containers = _init_service(service)[2]
    if containers == []:
        raise Exception('{0} est un service non activé'.format(service))
    if len(containers) > 1:
        containers = get_containers_group(containers)
    tret = 0
    for container in containers:
        ret = service_code(service, action, container)
        if ret != 0:
            tret = ret
    return tret

def service_out(service, action, container=VIRTMASTER):
    """
    action sur un service :
    - retourne un tuple contenant (pour la *dernière* commande) :
        - le code de retour sous la forme d'un entier
        - stdout
        - stderr
    @service : nom du service
    @action : action sur le service (stop, start, restart)
    @container : conteneur du service
    """
    cmds, pty = _gen_service_cmd(service, action)
    if cmds != None:
        for cmd in cmds:
            if cmd[0] == 'sleep':
                ret = sleep(int(cmd[1]))
            else:
                #no env for system_out
                ret = system_out(cmd, container=container)
        return ret
    else:
        return 0, '', ''

def update_rcd(service, action, container=VIRTMASTER, startlevel='', stoplevel=''):
    """
    activation/désactivation des services au démarrage
    """
    if action not in ['set', 'remove']:
        raise Exception ('update_rcd action must be set or remove, not %s' % action)

    init, pty, containers = _init_service(service)
    if init == 'upstart':
        if startlevel != '' or stoplevel != '':
            raise Exception('pas de support de startlevel ou stoplevel pour les services gérés par upstart')
        #copie de /etc/init/service.conf en /etc/init/service.conf.noexec et inversement
        path = get_container_path(container)
        active = join('/', path, 'etc/init', "%s.conf" % service)
        desactive = join('/', path, 'etc/init', "%s.noexec" % service)
        if action == 'set':
            if not isfile(active):
                if isfile(desactive):
                    move(desactive, active)
                else:
                    raise Exception("unknown service %s" % service)
        else:
            if isfile(active):
                if isfile(desactive):
                    # should not happen (#4481)
                    unlink(desactive)
                move(active, desactive)
    elif init == 'apache':
        if startlevel != '' or stoplevel != '':
            raise Exception('pas de support de startlevel ou stoplevel pour les services gérés par upstart')

        path = get_container_path(container)
        active = join('/', path, 'etc/apache2/sites-enabled', service)
        if action == 'set':
            if not isfile(active):
                system_out(['/usr/sbin/a2ensite', service], container=container)
        else:
            if isfile(active):
                system_out(['/usr/sbin/a2dissite', service], container=container)
    else:
        if action == 'set':
            need_update = False
            cmd = ['/usr/sbin/update-rc.d', service, 'defaults']
            active = glob(get_container_path(container, "/etc/rc*.d/S??%s" % service))
            if active == []:
                # nouveau service à activer
                need_update = True
            elif startlevel != '':
                if startlevel not in active[0]:
                    # mise à jour du startlevel d'un service (#1998)
                    update_rcd(service, 'remove', container=container)
                    need_update = True
                    cmd.append(startlevel)
                    if stoplevel != '':
                        cmd.append(stoplevel)
            if need_update:
                # ajout nécessaire
                system_out(cmd, container=container)
        elif glob(get_container_path(container, "/etc/rc*.d/S??%s" % service)):
            # suppression nécessaire
            cmd = ['/usr/sbin/update-rc.d', '-f', service, 'remove']
            system_out(cmd, container=container)
