/*
 * @(#) video-capture.c - modified from vidcat.c
 *
 * Copyright (C) 1998 Rasca, Berlin
 * EMail: thron@gmx.de
 * Modifications (C) 2001, Nick Andrew <nick@nick-andrew.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "video_device.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif
#ifdef HAVE_LIBPNG
#include <png.h>
#endif
#ifdef HAVE_LIBJPEG
#include <jpeglib.h>
#endif

#define VERSION		"0.2 (looped video capture added by <nick@nick-andrew.net>)"

#define DEF_WIDTH	320
#define DEF_HEIGHT	240
// default colour depth (changing this will break everything)
#define DEF_DEPTH	3

#define FMT_UNKNOWN	0
#define FMT_PPM		1
#define FMT_PNG		2
#define FMT_JPEG	3
#define FMT_DEFAULT	2

#define IN_TV			0
#define IN_COMPOSITE1	1
#define IN_COMPOSITE2	2
#define IN_SVIDEO		3
#define IN_DEFAULT		1

#define NORM_PAL		0
#define NORM_NTSC		1
#define NORM_SECAM		2
#define NORM_DEFAULT	0

#define QUAL_DEFAULT	80

/*
 */
void
usage (char *pname)
{
	fprintf (stderr,
	"video-capture, Version %s\n"
	"Usage: %s <options>\n"
	" -s NxN                      define size of the output image (default:"
		" %dx%d)\n"
	" -f {ppm|jpeg|png}           output format of the image\n"
	" -i {tv|comp1|comp2|s-video} which input channel to use\n"
	" -q <quality>                only for jpeg: quality setting (1-100,"
		" default: %d)\n"
	" -d <device>                 video device (default: $VIDEO_DEV)\n"
	" -l <loops>                  loop to pick up <loops> frames\n"
	" -b                          make a raw PPM instead of an ASCII one\n"
	" -v                          Verbose mode (log ioctls issued)\n"
	"Example: vidcat | xsetbg stdin\n",
		VERSION, basename(pname), DEF_WIDTH, DEF_HEIGHT, QUAL_DEFAULT);
	exit (1);
}

/*
 */
void
put_image_jpeg (char *image, int width, int height, int quality, int frame)
{
#ifdef HAVE_LIBJPEG
	int y, x, line_width;
	JSAMPROW row_ptr[1];
	struct jpeg_compress_struct cjpeg;
	struct jpeg_error_mgr jerr;
	char *line;

	line = malloc (width * 3);
	if (!line)
		return;
	cjpeg.err = jpeg_std_error(&jerr);
	jpeg_create_compress (&cjpeg);
	cjpeg.image_width = width;
	cjpeg.image_height= height;
	cjpeg.input_components = 3;
	cjpeg.in_color_space = JCS_RGB;
	jpeg_set_defaults (&cjpeg);

	jpeg_set_quality (&cjpeg, quality, TRUE);
	cjpeg.dct_method = JDCT_FASTEST;
	jpeg_stdio_dest (&cjpeg, stdout);

	jpeg_start_compress (&cjpeg, TRUE);

	row_ptr[0] = line;
	line_width = width * 3;
	for ( y = 0; y < height; y++) {
	for (x = 0; x < line_width; x+=3) {
			line[x]   = image[x+2];
			line[x+1] = image[x+1];
			line[x+2] = image[x];
		}
		jpeg_write_scanlines (&cjpeg, row_ptr, 1);
		image += line_width;
	}
	jpeg_finish_compress (&cjpeg);
	jpeg_destroy_compress (&cjpeg);
	free (line);
#else
	fprintf(stderr, "libjpeg not available - cannot write jpeg!\n");
#endif
}

/*
 * write png image to stdout
 */
void
put_image_png (char *image, int width, int height, int frame)
{
#ifdef HAVE_LIBPNG
	int y;
	char *p;
	png_infop info_ptr;
	png_structp png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
						NULL, NULL, NULL);
	if (!png_ptr)
		return;
	info_ptr = png_create_info_struct (png_ptr);
	if (!info_ptr)
		return;

	png_init_io (png_ptr, stdout);
	png_set_IHDR (png_ptr, info_ptr, width, height,
					8, PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
					PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
	png_set_bgr (png_ptr);
	png_write_info (png_ptr, info_ptr);
	p = image;
	for (y = 0; y < height; y++) {
		png_write_row (png_ptr, p);
		p+=width*3;
	}
	png_write_end (png_ptr, info_ptr);
#endif
}

/*
 * write ppm image to stdout
 */
#define WRITE_MODE	"w"

int
put_image_ppm (char *image, int width, int height, int binary, int frame)
{
	int x, y, ls=0;
	unsigned char *p = (unsigned char *)image;
	FILE	*out_fp = stdout;
	int	rc = 0;

	if (frame >= 0) {
		/* Output to a file numbered by the frame number */
		char	fn[32];

		sprintf(fn, "frame%d.ppm", frame);

		out_fp = fopen(fn, WRITE_MODE);
		if (!out_fp) {
			fprintf(stderr, "Unable to open %s for write: ", fn);
			perror("");
			return -1;
		}
	}

	if (!binary) {
	fprintf(out_fp, "P3\n%d %d\n%d\n", width, height, 255);
	for (x = 0; x < width; x++) {
		for (y = 0; y < height; y++) {
			fprintf(out_fp, "%03d %03d %03d  ", p[2], p[1], p[0]);
			p += 3;
			if (ls++ > 4) {
				fprintf(out_fp, "\n");
				ls = 0;
			}
		}
	}
	fprintf(out_fp, "\n");
	} else {
		unsigned char	buff[3 * width * height];
		unsigned char	*bp = buff;

		fprintf(out_fp, "P6\n%d %d\n%d\n", width, height, 255);
		for (x = 0; x < width * height; x++) {
			*bp++ = p[2];
			*bp++ = p[1];
			*bp++ = p[0];
			p += 3;
		}
		fwrite(buff, width * height, 3, out_fp);
	}

	if (ferror(out_fp)) {
		rc = -1;
	}

	if (frame >= 0) {
		fclose(out_fp);
	} else {
		fflush(out_fp);
	}

	return rc;
}

/*
 * main()
 */
int
main (int argc, char *argv[])
{
	int width = DEF_WIDTH, height = DEF_HEIGHT, size, c;
	int	dev;		// fd of open video device
	video_device_t	*vd;	// the instance data
	char *image;
	char *device = getenv("VIDEO_DEV");
	int max_try = 5;	/* we try 5 seconds/times to open the device */
	int quality = QUAL_DEFAULT;	/* default jpeg quality setting */
	int input = IN_DEFAULT;
	int norm  = NORM_DEFAULT;
	int loop = 1;		/* once around the loop only */
	int frame = 0;		/* the frame number we are capturing */
	int binary = 0;
	int no_save = 0;
	int debug = 0;
#ifdef HAVE_LIBJPEG
	int format = FMT_JPEG;
#else
#ifdef HAVE_LIBPNG
	int format = FMT_PNG;
#else
	int format = FMT_PPM;
#endif
#endif

	while ((c = getopt (argc, argv, "bs:f:q:i:d:l:tv")) != EOF) {
		switch (c) {
			case 'b':
				binary = 1;
				break;
			case 'd':
				device = optarg;
				break;
			case 's':
				sscanf (optarg, "%dx%d", &width, &height);
				break;
			case 'f':
				if (strcasecmp ("ppm", optarg) == 0)
					format = FMT_PPM;
				else if (strcasecmp ("png", optarg) == 0)
					format = FMT_PNG;
				else if (strcasecmp ("jpeg", optarg) == 0)
					format = FMT_JPEG;
				else
					format = FMT_UNKNOWN;
				break;
			case 'q':
				sscanf (optarg, "%d", &quality);
				break;
			case 'i':
				if (strcasecmp ("tv", optarg) == 0) {
					input = IN_TV;
				} else if (strcasecmp ("comp1", optarg) == 0) {
					input = IN_COMPOSITE1;
				} else if (strcasecmp ("comp2", optarg) ==0) {
					input = IN_COMPOSITE2;
				} else if (strcasecmp ("s-video", optarg) == 0) {
					input = IN_SVIDEO;
				}
				break;
			case 'l':
				loop = -1;
				sscanf(optarg, "%d", &loop);
				fprintf(stderr, "Looping %d times.\n", loop);
				break;
			case 't':
				no_save = 1;
				break;
			case 'v':
				debug = 1;
				break;
			default:
				usage (argv[0]);
				break;
		}
	}

	if (format != FMT_PPM && format != FMT_PNG && format != FMT_JPEG) {
		fprintf(stderr, "Unknown format (%d)\n", format);
		exit(8);
	}

	if (device == (char *) 0) {
		fprintf(stderr, "No device set - use -d <device> or set $VIDEO_DEV\n");
		exit(8);
	}

	/* open the video4linux device */
	while (max_try) {
		dev = open (device, O_RDWR);
		if (dev == -1) {
			if (!--max_try) {
				fprintf (stderr, "Can't open device %s\n", device);
				exit (0);
			}
			sleep (1);
		} else {
			break;
		}
	}

	vd = vd_setup(width, height, DEF_DEPTH, dev);

	if (debug) {
		vd_debug(vd, 1, stderr);
	}

	if (!vd_setup_capture_mode(vd)) {
		fprintf(stderr, "Unable to setup capture mode.\n");
		exit(8);
	}

	if (!vd_setup_video_source(vd, input, norm)) {
		// unavailable channel selected
		fprintf(stderr, "Unable to setup video source!\n");
		exit(8);
	}

	frame = (loop == 1) ? -1 : 0;

	do {
//		fprintf(stderr, "Getting frame %d\n", frame);

		image = (char *) vd_get_image(vd);
		if (!image) {
			fprintf (stderr, "Error: Can't get image\n");
			exit(8);
		}

		if (!no_save) {
		/* now save it */
		switch (format) {
			case FMT_PPM:
				put_image_ppm (image, width, height, binary, frame);
				break;
			case FMT_PNG:
				put_image_png (image, width, height, frame);
				break;
			case FMT_JPEG:
				put_image_jpeg (image, width, height, quality, frame);
				break;
			default:
				fprintf(stderr, "No known output format %d!\n", format);
				/*NOTREACHED*/
				break;
		}}

		// Now tell the driver we're done with it
		vd_image_done(vd);

		frame++;

	} while (loop < 0 || --loop > 0);

	vd_close(vd);
	return 0;
}
