// Copyright 2015, VIXL authors
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of ARM Limited nor the names of its contributors may be
//     used to endorse or promote products derived from this software without
//     specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

extern "C" {
#include <stdint.h>
}

#include <cassert>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iomanip>
#include <iostream>

#include "utils-vixl.h"
#include "aarch32/constants-aarch32.h"
#include "aarch32/disasm-aarch32.h"
#include "aarch32/instructions-aarch32.h"
#include "aarch32/operands-aarch32.h"

namespace vixl {
namespace aarch32 {

using internal::Int64;
using internal::Uint32;

class T32CodeAddressIncrementer {
  uint32_t* code_address_;
  uint32_t increment_;

 public:
  T32CodeAddressIncrementer(uint32_t instr, uint32_t* code_address)
      : code_address_(code_address),
        increment_(Disassembler::Is16BitEncoding(instr) ? 2 : 4) {}
  ~T32CodeAddressIncrementer() { *code_address_ += increment_; }
};

class A32CodeAddressIncrementer {
  uint32_t* code_address_;

 public:
  explicit A32CodeAddressIncrementer(uint32_t* code_address)
      : code_address_(code_address) {}
  ~A32CodeAddressIncrementer() { *code_address_ += 4; }
};

class DecodeNeon {
  int lane_;
  SpacingType spacing_;
  bool valid_;

 public:
  DecodeNeon(int lane, SpacingType spacing)
      : lane_(lane), spacing_(spacing), valid_(true) {}
  DecodeNeon() : lane_(0), spacing_(kSingle), valid_(false) {}
  int GetLane() const { return lane_; }
  SpacingType GetSpacing() const { return spacing_; }
  bool IsValid() const { return valid_; }
};

class DecodeNeonAndAlign : public DecodeNeon {
 public:
  Alignment align_;
  DecodeNeonAndAlign(int lanes, SpacingType spacing, Alignment align)
      : DecodeNeon(lanes, spacing), align_(align) {}
  DecodeNeonAndAlign() : align_(kBadAlignment) {}
  Alignment GetAlign() const { return align_; }
};

// Start of generated code.
DataTypeValue Dt_L_imm6_1_Decode(uint32_t value, uint32_t type_value) {
  if ((value & 0xf) == 0x1) {
    switch (type_value) {
      case 0x0:
        return S8;
      case 0x1:
        return U8;
    }
  } else if ((value & 0xe) == 0x2) {
    switch (type_value) {
      case 0x0:
        return S16;
      case 0x1:
        return U16;
    }
  } else if ((value & 0xc) == 0x4) {
    switch (type_value) {
      case 0x0:
        return S32;
      case 0x1:
        return U32;
    }
  } else if ((value & 0x8) == 0x8) {
    switch (type_value) {
      case 0x0:
        return S64;
      case 0x1:
        return U64;
    }
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_L_imm6_2_Decode(uint32_t value, uint32_t type_value) {
  if ((value & 0xf) == 0x1) {
    if (type_value == 0x1) return S8;
  } else if ((value & 0xe) == 0x2) {
    if (type_value == 0x1) return S16;
  } else if ((value & 0xc) == 0x4) {
    if (type_value == 0x1) return S32;
  } else if ((value & 0x8) == 0x8) {
    if (type_value == 0x1) return S64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_L_imm6_3_Decode(uint32_t value) {
  if ((value & 0xf) == 0x1) {
    return I8;
  } else if ((value & 0xe) == 0x2) {
    return I16;
  } else if ((value & 0xc) == 0x4) {
    return I32;
  } else if ((value & 0x8) == 0x8) {
    return I64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_L_imm6_4_Decode(uint32_t value) {
  if ((value & 0xf) == 0x1) {
    return Untyped8;
  } else if ((value & 0xe) == 0x2) {
    return Untyped16;
  } else if ((value & 0xc) == 0x4) {
    return Untyped32;
  } else if ((value & 0x8) == 0x8) {
    return Untyped64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_imm6_1_Decode(uint32_t value, uint32_t type_value) {
  if ((value & 0x7) == 0x1) {
    switch (type_value) {
      case 0x0:
        return S16;
      case 0x1:
        return U16;
    }
  } else if ((value & 0x6) == 0x2) {
    switch (type_value) {
      case 0x0:
        return S32;
      case 0x1:
        return U32;
    }
  } else if ((value & 0x4) == 0x4) {
    switch (type_value) {
      case 0x0:
        return S64;
      case 0x1:
        return U64;
    }
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_imm6_2_Decode(uint32_t value, uint32_t type_value) {
  if ((value & 0x7) == 0x1) {
    if (type_value == 0x1) return S16;
  } else if ((value & 0x6) == 0x2) {
    if (type_value == 0x1) return S32;
  } else if ((value & 0x4) == 0x4) {
    if (type_value == 0x1) return S64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_imm6_3_Decode(uint32_t value) {
  if ((value & 0x7) == 0x1) {
    return I16;
  } else if ((value & 0x6) == 0x2) {
    return I32;
  } else if ((value & 0x4) == 0x4) {
    return I64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_imm6_4_Decode(uint32_t value, uint32_t type_value) {
  if ((value & 0x7) == 0x1) {
    switch (type_value) {
      case 0x0:
        return S8;
      case 0x1:
        return U8;
    }
  } else if ((value & 0x6) == 0x2) {
    switch (type_value) {
      case 0x0:
        return S16;
      case 0x1:
        return U16;
    }
  } else if ((value & 0x4) == 0x4) {
    switch (type_value) {
      case 0x0:
        return S32;
      case 0x1:
        return U32;
    }
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_U_size_1_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S8;
    case 0x1:
      return S16;
    case 0x2:
      return S32;
    case 0x4:
      return U8;
    case 0x5:
      return U16;
    case 0x6:
      return U32;
    case 0x8:
      return P8;
    case 0xa:
      return P64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_size_1_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return I8;
    case 0x1:
      return I16;
    case 0x2:
      return I32;
    case 0x4:
      return P8;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_size_2_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S8;
    case 0x1:
      return S16;
    case 0x2:
      return S32;
    case 0x4:
      return U8;
    case 0x5:
      return U16;
    case 0x6:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_size_3_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S16;
    case 0x1:
      return S32;
    case 0x2:
      return S64;
    case 0x4:
      return U16;
    case 0x5:
      return U32;
    case 0x6:
      return U64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_U_imm3H_1_Decode(uint32_t value) {
  switch (value) {
    case 0x1:
      return S8;
    case 0x2:
      return S16;
    case 0x4:
      return S32;
    case 0x9:
      return U8;
    case 0xa:
      return U16;
    case 0xc:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_U_opc1_opc2_1_Decode(uint32_t value, unsigned* lane) {
  if ((value & 0x18) == 0x8) {
    *lane = value & 7;
    return S8;
  }
  if ((value & 0x19) == 0x1) {
    *lane = (value >> 1) & 3;
    return S16;
  }
  if ((value & 0x18) == 0x18) {
    *lane = value & 7;
    return U8;
  }
  if ((value & 0x19) == 0x11) {
    *lane = (value >> 1) & 3;
    return U16;
  }
  if ((value & 0x1b) == 0x0) {
    *lane = (value >> 2) & 1;
    return Untyped32;
  }
  *lane = ~0U;
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_opc1_opc2_1_Decode(uint32_t value, unsigned* lane) {
  if ((value & 0x8) == 0x8) {
    *lane = value & 7;
    return Untyped8;
  }
  if ((value & 0x9) == 0x1) {
    *lane = (value >> 1) & 3;
    return Untyped16;
  }
  if ((value & 0xb) == 0x0) {
    *lane = (value >> 2) & 1;
    return Untyped32;
  }
  *lane = ~0U;
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_imm4_1_Decode(uint32_t value, unsigned* lane) {
  if ((value & 0x1) == 0x1) {
    *lane = (value >> 1) & 7;
    return Untyped8;
  }
  if ((value & 0x3) == 0x2) {
    *lane = (value >> 2) & 3;
    return Untyped16;
  }
  if ((value & 0x7) == 0x4) {
    *lane = (value >> 3) & 1;
    return Untyped32;
  }
  *lane = ~0U;
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_B_E_1_Decode(uint32_t value) {
  switch (value) {
    case 0x2:
      return Untyped8;
    case 0x1:
      return Untyped16;
    case 0x0:
      return Untyped32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_1_Decode1(uint32_t value) {
  switch (value) {
    case 0x0:
      return F32;
    case 0x1:
      return F32;
    case 0x2:
      return S32;
    case 0x3:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_1_Decode2(uint32_t value) {
  switch (value) {
    case 0x0:
      return S32;
    case 0x1:
      return U32;
    case 0x2:
      return F32;
    case 0x3:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_2_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return U32;
    case 0x1:
      return S32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_3_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S32;
    case 0x1:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_U_sx_1_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S16;
    case 0x1:
      return S32;
    case 0x2:
      return U16;
    case 0x3:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_U_1_Decode1(uint32_t value) {
  switch (value) {
    case 0x0:
      return F32;
    case 0x1:
      return F32;
    case 0x2:
      return S32;
    case 0x3:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_op_U_1_Decode2(uint32_t value) {
  switch (value) {
    case 0x0:
      return S32;
    case 0x1:
      return U32;
    case 0x2:
      return F32;
    case 0x3:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_sz_1_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_F_size_1_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S8;
    case 0x1:
      return S16;
    case 0x2:
      return S32;
    case 0x6:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_F_size_2_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return I8;
    case 0x1:
      return I16;
    case 0x2:
      return I32;
    case 0x6:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_F_size_3_Decode(uint32_t value) {
  switch (value) {
    case 0x1:
      return I16;
    case 0x2:
      return I32;
    case 0x6:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_F_size_4_Decode(uint32_t value) {
  switch (value) {
    case 0x2:
      return U32;
    case 0x6:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_U_size_1_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S8;
    case 0x1:
      return S16;
    case 0x2:
      return S32;
    case 0x4:
      return U8;
    case 0x5:
      return U16;
    case 0x6:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_U_size_2_Decode(uint32_t value) {
  switch (value) {
    case 0x1:
      return S16;
    case 0x2:
      return S32;
    case 0x5:
      return U16;
    case 0x6:
      return U32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_U_size_3_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S8;
    case 0x1:
      return S16;
    case 0x2:
      return S32;
    case 0x3:
      return S64;
    case 0x4:
      return U8;
    case 0x5:
      return U16;
    case 0x6:
      return U32;
    case 0x7:
      return U64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_1_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return Untyped8;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_2_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return I8;
    case 0x1:
      return I16;
    case 0x2:
      return I32;
    case 0x3:
      return I64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_3_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return I16;
    case 0x1:
      return I32;
    case 0x2:
      return I64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_4_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return I8;
    case 0x1:
      return I16;
    case 0x2:
      return I32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_5_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S8;
    case 0x1:
      return S16;
    case 0x2:
      return S32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_6_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return Untyped8;
    case 0x1:
      return Untyped16;
    case 0x2:
      return Untyped32;
    case 0x3:
      return Untyped64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_7_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return Untyped8;
    case 0x1:
      return Untyped16;
    case 0x2:
      return Untyped32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_8_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return Untyped8;
    case 0x1:
      return Untyped16;
    case 0x2:
      return Untyped32;
    case 0x3:
      return Untyped32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_9_Decode(uint32_t value, uint32_t type_value) {
  switch (value) {
    case 0x1:
      switch (type_value) {
        case 0x0:
          return I16;
      }
      break;
    case 0x2:
      switch (type_value) {
        case 0x0:
          return I32;
        case 0x1:
          return F32;
      }
      break;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_10_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return I8;
    case 0x1:
      return I16;
    case 0x2:
      return I32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_11_Decode(uint32_t value, uint32_t type_value) {
  switch (value) {
    case 0x1:
      switch (type_value) {
        case 0x0:
          return S16;
        case 0x1:
          return U16;
      }
      break;
    case 0x2:
      switch (type_value) {
        case 0x0:
          return S32;
        case 0x1:
          return U32;
      }
      break;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_12_Decode(uint32_t value, uint32_t type_value) {
  switch (value) {
    case 0x0:
      switch (type_value) {
        case 0x0:
          return S8;
        case 0x1:
          return U8;
      }
      break;
    case 0x1:
      switch (type_value) {
        case 0x0:
          return S16;
        case 0x1:
          return U16;
      }
      break;
    case 0x2:
      switch (type_value) {
        case 0x0:
          return S32;
        case 0x1:
          return U32;
      }
      break;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_13_Decode(uint32_t value) {
  switch (value) {
    case 0x1:
      return S16;
    case 0x2:
      return S32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_14_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return S16;
    case 0x1:
      return S32;
    case 0x2:
      return S64;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_15_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return Untyped8;
    case 0x1:
      return Untyped16;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_16_Decode(uint32_t value) {
  switch (value) {
    case 0x2:
      return F32;
  }
  return kDataTypeValueInvalid;
}

DataTypeValue Dt_size_17_Decode(uint32_t value) {
  switch (value) {
    case 0x0:
      return I8;
    case 0x1:
      return I16;
    case 0x2:
      return I32;
  }
  return kDataTypeValueInvalid;
}

DecodeNeon Index_1_Decode(uint32_t value, DataType dt) {
  switch (dt.GetValue()) {
    case Untyped8: {
      int lane = (value >> 1) & 0x7;
      if ((value & 1) != 0) break;
      SpacingType spacing = kSingle;
      return DecodeNeon(lane, spacing);
    }
    case Untyped16: {
      int lane = (value >> 2) & 0x3;
      if ((value & 1) != 0) break;
      SpacingType spacing = ((value & 3) == 2) ? kDouble : kSingle;
      return DecodeNeon(lane, spacing);
    }
    case Untyped32: {
      int lane = (value >> 3) & 0x1;
      if ((value & 3) != 0) break;
      SpacingType spacing = ((value & 7) == 4) ? kDouble : kSingle;
      return DecodeNeon(lane, spacing);
    }
    default:
      break;
  }
  return DecodeNeon();
}

DecodeNeonAndAlign Align_index_align_1_Decode(uint32_t value, DataType dt) {
  switch (dt.GetValue()) {
    case Untyped8: {
      AlignmentType align;
      if ((value & 1) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 1) & 0x7;
      SpacingType spacing = kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    case Untyped16: {
      AlignmentType align;
      if ((value & 3) == 1) {
        align = k16BitAlign;
      } else if ((value & 3) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 2) & 0x3;
      SpacingType spacing = kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    case Untyped32: {
      AlignmentType align;
      if ((value & 7) == 3) {
        align = k32BitAlign;
      } else if ((value & 7) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 3) & 0x1;
      SpacingType spacing = kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    default:
      break;
  }
  return DecodeNeonAndAlign();
}

DecodeNeonAndAlign Align_index_align_2_Decode(uint32_t value, DataType dt) {
  switch (dt.GetValue()) {
    case Untyped8: {
      AlignmentType align;
      if ((value & 1) == 1) {
        align = k16BitAlign;
      } else if ((value & 1) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 1) & 0x7;
      SpacingType spacing = kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    case Untyped16: {
      AlignmentType align;
      if ((value & 1) == 1) {
        align = k32BitAlign;
      } else if ((value & 1) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 2) & 0x3;
      SpacingType spacing = ((value & 2) == 2) ? kDouble : kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    case Untyped32: {
      AlignmentType align;
      if ((value & 3) == 1) {
        align = k64BitAlign;
      } else if ((value & 3) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 3) & 0x1;
      SpacingType spacing = ((value & 4) == 4) ? kDouble : kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    default:
      break;
  }
  return DecodeNeonAndAlign();
}

DecodeNeonAndAlign Align_index_align_3_Decode(uint32_t value, DataType dt) {
  switch (dt.GetValue()) {
    case Untyped8: {
      AlignmentType align;
      if ((value & 1) == 1) {
        align = k32BitAlign;
      } else if ((value & 1) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 1) & 0x7;
      SpacingType spacing = kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    case Untyped16: {
      AlignmentType align;
      if ((value & 1) == 1) {
        align = k64BitAlign;
      } else if ((value & 1) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 2) & 0x3;
      SpacingType spacing = ((value & 2) == 2) ? kDouble : kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    case Untyped32: {
      AlignmentType align;
      if ((value & 3) == 1) {
        align = k64BitAlign;
      } else if ((value & 3) == 2) {
        align = k128BitAlign;
      } else if ((value & 3) == 0) {
        align = kNoAlignment;
      } else {
        break;
      }
      int lane = (value >> 3) & 0x1;
      SpacingType spacing = ((value & 4) == 4) ? kDouble : kSingle;
      return DecodeNeonAndAlign(lane, spacing, align);
    }
    default:
      break;
  }
  return DecodeNeonAndAlign();
}

Alignment Align_a_1_Decode(uint32_t value, DataType dt) {
  switch (value) {
    case 0:
      return kNoAlignment;
    case 1:
      if (dt.Is(Untyped16)) return k16BitAlign;
      if (dt.Is(Untyped32)) return k32BitAlign;
      break;
    default:
      break;
  }
  return kBadAlignment;
}

Alignment Align_a_2_Decode(uint32_t value, DataType dt) {
  switch (value) {
    case 0:
      return kNoAlignment;
    case 1:
      if (dt.Is(Untyped8)) return k16BitAlign;
      if (dt.Is(Untyped16)) return k32BitAlign;
      if (dt.Is(Untyped32)) return k64BitAlign;
      break;
    default:
      break;
  }
  return kBadAlignment;
}

Alignment Align_a_3_Decode(uint32_t value, DataType dt, uint32_t size) {
  switch (value) {
    case 0:
      if (size != 3) return kNoAlignment;
      break;
    case 1:
      if (dt.Is(Untyped8)) return k32BitAlign;
      if (dt.Is(Untyped16)) return k64BitAlign;
      if (size == 2) return k64BitAlign;
      if (size == 3) return k128BitAlign;
      break;
    default:
      break;
  }
  return kBadAlignment;
}

Alignment Align_align_1_Decode(uint32_t value) {
  switch (value) {
    case 0:
      return kNoAlignment;
    case 1:
      return k64BitAlign;
    case 2:
      return k128BitAlign;
    case 3:
      return k256BitAlign;
    default:
      break;
  }
  return kBadAlignment;
}

Alignment Align_align_2_Decode(uint32_t value) {
  switch (value) {
    case 0:
      return kNoAlignment;
    case 1:
      return k64BitAlign;
    case 2:
      return k128BitAlign;
    case 3:
      return k256BitAlign;
    default:
      break;
  }
  return kBadAlignment;
}

Alignment Align_align_3_Decode(uint32_t value) {
  switch (value) {
    case 0:
      return kNoAlignment;
    case 1:
      return k64BitAlign;
    default:
      break;
  }
  return kBadAlignment;
}

Alignment Align_align_4_Decode(uint32_t value) {
  switch (value) {
    case 0:
      return kNoAlignment;
    case 1:
      return k64BitAlign;
    case 2:
      return k128BitAlign;
    case 3:
      return k256BitAlign;
    default:
      break;
  }
  return kBadAlignment;
}

Alignment Align_align_5_Decode(uint32_t value) {
  switch (value) {
    case 0:
      return kNoAlignment;
    case 1:
      return k64BitAlign;
    case 2:
      return k128BitAlign;
    case 3:
      return k256BitAlign;
    default:
      break;
  }
  return kBadAlignment;
}


void Disassembler::adc(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kAdc, kArithmetic);
  os() << ToCString(kAdc) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::adcs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kAdcs, kArithmetic);
  os() << ToCString(kAdcs) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::add(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kAdd, kArithmetic);
  os() << ToCString(kAdd) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::add(Condition cond, Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kAdd, kArithmetic);
  os() << ToCString(kAdd) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << operand;
}

void Disassembler::adds(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kAdds, kArithmetic);
  os() << ToCString(kAdds) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::adds(Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kAdds, kArithmetic);
  os() << ToCString(kAdds) << " " << rd << ", " << operand;
}

void Disassembler::addw(Condition cond,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kAddw, kArithmetic);
  os() << ToCString(kAddw) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::adr(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Location* location) {
  os().SetCurrentInstruction(kAdr, kAddress);
  os() << ToCString(kAdr) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", "
       << PrintLabel(kAnyLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::and_(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kAnd, kBitwise);
  os() << ToCString(kAnd) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::ands(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kAnds, kBitwise);
  os() << ToCString(kAnds) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::asr(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rm,
                       const Operand& operand) {
  os().SetCurrentInstruction(kAsr, kShift);
  os() << ToCString(kAsr) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::asrs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rm,
                        const Operand& operand) {
  os().SetCurrentInstruction(kAsrs, kShift);
  os() << ToCString(kAsrs) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::b(Condition cond, EncodingSize size, Location* location) {
  os().SetCurrentInstruction(kB, kAddress | kBranch);
  os() << ToCString(kB) << ConditionPrinter(it_block_, cond) << size << " "
       << PrintLabel(kCodeLocation, location, GetCodeAddress());
}

void Disassembler::bfc(Condition cond,
                       Register rd,
                       uint32_t lsb,
                       uint32_t width) {
  os().SetCurrentInstruction(kBfc, kShift);
  os() << ToCString(kBfc) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << ImmediatePrinter(lsb) << ", " << ImmediatePrinter(width);
}

void Disassembler::bfi(
    Condition cond, Register rd, Register rn, uint32_t lsb, uint32_t width) {
  os().SetCurrentInstruction(kBfi, kShift);
  os() << ToCString(kBfi) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << ImmediatePrinter(lsb) << ", "
       << ImmediatePrinter(width);
}

void Disassembler::bic(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kBic, kBitwise);
  os() << ToCString(kBic) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::bics(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kBics, kBitwise);
  os() << ToCString(kBics) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::bkpt(Condition cond, uint32_t imm) {
  os().SetCurrentInstruction(kBkpt, kSystem);
  os() << ToCString(kBkpt) << ConditionPrinter(it_block_, cond) << " "
       << RawImmediatePrinter(imm);
}

void Disassembler::bl(Condition cond, Location* location) {
  os().SetCurrentInstruction(kBl, kAddress | kBranch);
  os() << ToCString(kBl) << ConditionPrinter(it_block_, cond) << " "
       << PrintLabel(kCodeLocation, location, GetCodeAddress());
}

void Disassembler::blx(Condition cond, Location* location) {
  os().SetCurrentInstruction(kBlx, kAddress | kBranch);
  os() << ToCString(kBlx) << ConditionPrinter(it_block_, cond) << " "
       << PrintLabel(kCodeLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::blx(Condition cond, Register rm) {
  os().SetCurrentInstruction(kBlx, kAddress | kBranch);
  os() << ToCString(kBlx) << ConditionPrinter(it_block_, cond) << " " << rm;
}

void Disassembler::bx(Condition cond, Register rm) {
  os().SetCurrentInstruction(kBx, kAddress | kBranch);
  os() << ToCString(kBx) << ConditionPrinter(it_block_, cond) << " " << rm;
}

void Disassembler::bxj(Condition cond, Register rm) {
  os().SetCurrentInstruction(kBxj, kAddress | kBranch);
  os() << ToCString(kBxj) << ConditionPrinter(it_block_, cond) << " " << rm;
}

void Disassembler::cbnz(Register rn, Location* location) {
  os().SetCurrentInstruction(kCbnz, kAddress | kBranch);
  os() << ToCString(kCbnz) << " " << rn << ", "
       << PrintLabel(kCodeLocation, location, GetCodeAddress());
}

void Disassembler::cbz(Register rn, Location* location) {
  os().SetCurrentInstruction(kCbz, kAddress | kBranch);
  os() << ToCString(kCbz) << " " << rn << ", "
       << PrintLabel(kCodeLocation, location, GetCodeAddress());
}

void Disassembler::clrex(Condition cond) {
  os().SetCurrentInstruction(kClrex, kNoAttribute);
  os() << ToCString(kClrex) << ConditionPrinter(it_block_, cond);
}

void Disassembler::clz(Condition cond, Register rd, Register rm) {
  os().SetCurrentInstruction(kClz, kNoAttribute);
  os() << ToCString(kClz) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rm;
}

void Disassembler::cmn(Condition cond,
                       EncodingSize size,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kCmn, kArithmetic);
  os() << ToCString(kCmn) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << ", " << operand;
}

void Disassembler::cmp(Condition cond,
                       EncodingSize size,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kCmp, kArithmetic);
  os() << ToCString(kCmp) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << ", " << operand;
}

void Disassembler::crc32b(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kCrc32b, kNoAttribute);
  os() << ToCString(kCrc32b) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm;
}

void Disassembler::crc32cb(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kCrc32cb, kNoAttribute);
  os() << ToCString(kCrc32cb) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm;
}

void Disassembler::crc32ch(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kCrc32ch, kNoAttribute);
  os() << ToCString(kCrc32ch) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm;
}

void Disassembler::crc32cw(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kCrc32cw, kNoAttribute);
  os() << ToCString(kCrc32cw) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm;
}

void Disassembler::crc32h(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kCrc32h, kNoAttribute);
  os() << ToCString(kCrc32h) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm;
}

void Disassembler::crc32w(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kCrc32w, kNoAttribute);
  os() << ToCString(kCrc32w) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm;
}

void Disassembler::dmb(Condition cond, MemoryBarrier option) {
  os().SetCurrentInstruction(kDmb, kNoAttribute);
  os() << ToCString(kDmb) << ConditionPrinter(it_block_, cond) << " " << option;
}

void Disassembler::dsb(Condition cond, MemoryBarrier option) {
  os().SetCurrentInstruction(kDsb, kNoAttribute);
  os() << ToCString(kDsb) << ConditionPrinter(it_block_, cond) << " " << option;
}

void Disassembler::eor(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kEor, kBitwise);
  os() << ToCString(kEor) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::eors(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kEors, kBitwise);
  os() << ToCString(kEors) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::fldmdbx(Condition cond,
                           Register rn,
                           WriteBack write_back,
                           DRegisterList dreglist) {
  os().SetCurrentInstruction(kFldmdbx,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kFldmdbx) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << dreglist;
}

void Disassembler::fldmiax(Condition cond,
                           Register rn,
                           WriteBack write_back,
                           DRegisterList dreglist) {
  os().SetCurrentInstruction(kFldmiax,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kFldmiax) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << dreglist;
}

void Disassembler::fstmdbx(Condition cond,
                           Register rn,
                           WriteBack write_back,
                           DRegisterList dreglist) {
  os().SetCurrentInstruction(kFstmdbx,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kFstmdbx) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << dreglist;
}

void Disassembler::fstmiax(Condition cond,
                           Register rn,
                           WriteBack write_back,
                           DRegisterList dreglist) {
  os().SetCurrentInstruction(kFstmiax,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kFstmiax) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << dreglist;
}

void Disassembler::hlt(Condition cond, uint32_t imm) {
  os().SetCurrentInstruction(kHlt, kSystem);
  os() << ToCString(kHlt) << ConditionPrinter(it_block_, cond) << " "
       << RawImmediatePrinter(imm);
}

void Disassembler::hvc(Condition cond, uint32_t imm) {
  os().SetCurrentInstruction(kHvc, kSystem);
  os() << ToCString(kHvc) << ConditionPrinter(it_block_, cond) << " "
       << RawImmediatePrinter(imm);
}

void Disassembler::isb(Condition cond, MemoryBarrier option) {
  os().SetCurrentInstruction(kIsb, kNoAttribute);
  os() << ToCString(kIsb) << ConditionPrinter(it_block_, cond) << " " << option;
}

void Disassembler::it(Condition cond, uint16_t mask) {
  os().SetCurrentInstruction(kIt, kNoAttribute);
  os() << ToCString(kIt);
  int count;
  if ((mask & 0x1) != 0) {
    count = 3;
  } else if ((mask & 0x2) != 0) {
    count = 2;
  } else if ((mask & 0x4) != 0) {
    count = 1;
  } else {
    count = 0;
  }
  uint16_t tmp = 0x8;
  uint16_t ref = (cond.GetCondition() & 0x1) << 3;
  while (count-- > 0) {
    os() << (((mask & tmp) == ref) ? "t" : "e");
    tmp >>= 1;
    ref >>= 1;
  }
  if (cond.Is(al)) {
    os() << " al";
  } else {
    os() << " " << cond;
  }
}

void Disassembler::lda(Condition cond, Register rt, const MemOperand& operand) {
  os().SetCurrentInstruction(kLda, kAddress | kLoadStore);
  os() << ToCString(kLda) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadWordLocation, operand);
}

void Disassembler::ldab(Condition cond,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kLdab, kAddress | kLoadStore);
  os() << ToCString(kLdab) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadByteLocation, operand);
}

void Disassembler::ldaex(Condition cond,
                         Register rt,
                         const MemOperand& operand) {
  os().SetCurrentInstruction(kLdaex, kAddress | kLoadStore);
  os() << ToCString(kLdaex) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadWordLocation, operand);
}

void Disassembler::ldaexb(Condition cond,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kLdaexb, kAddress | kLoadStore);
  os() << ToCString(kLdaexb) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadByteLocation, operand);
}

void Disassembler::ldaexd(Condition cond,
                          Register rt,
                          Register rt2,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kLdaexd, kAddress | kLoadStore);
  os() << ToCString(kLdaexd) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rt2 << ", "
       << PrintMemOperand(kLoadDoubleWordLocation, operand);
}

void Disassembler::ldaexh(Condition cond,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kLdaexh, kAddress | kLoadStore);
  os() << ToCString(kLdaexh) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadHalfWordLocation, operand);
}

void Disassembler::ldah(Condition cond,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kLdah, kAddress | kLoadStore);
  os() << ToCString(kLdah) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadHalfWordLocation, operand);
}

void Disassembler::ldm(Condition cond,
                       EncodingSize size,
                       Register rn,
                       WriteBack write_back,
                       RegisterList registers) {
  os().SetCurrentInstruction(kLdm, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdm) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << write_back << ", " << registers;
}

void Disassembler::ldmda(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kLdmda, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdmda) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::ldmdb(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kLdmdb, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdmdb) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::ldmea(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kLdmea, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdmea) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::ldmed(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kLdmed, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdmed) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::ldmfa(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kLdmfa, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdmfa) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::ldmfd(Condition cond,
                         EncodingSize size,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kLdmfd, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdmfd) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << write_back << ", " << registers;
}

void Disassembler::ldmib(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kLdmib, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kLdmib) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::ldr(Condition cond,
                       EncodingSize size,
                       Register rt,
                       const MemOperand& operand) {
  os().SetCurrentInstruction(kLdr, kAddress | kLoadStore);
  os() << ToCString(kLdr) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kLoadWordLocation, operand);
}

void Disassembler::ldr(Condition cond,
                       EncodingSize size,
                       Register rt,
                       Location* location) {
  os().SetCurrentInstruction(kLdr, kAddress | kLoadStore);
  os() << ToCString(kLdr) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", "
       << PrintLabel(kLoadWordLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::ldrb(Condition cond,
                        EncodingSize size,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrb, kAddress | kLoadStore);
  os() << ToCString(kLdrb) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kLoadByteLocation, operand);
}

void Disassembler::ldrb(Condition cond, Register rt, Location* location) {
  os().SetCurrentInstruction(kLdrb, kAddress | kLoadStore);
  os() << ToCString(kLdrb) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", "
       << PrintLabel(kLoadByteLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::ldrd(Condition cond,
                        Register rt,
                        Register rt2,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrd, kAddress | kLoadStore);
  os() << ToCString(kLdrd) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rt2 << ", "
       << PrintMemOperand(kLoadDoubleWordLocation, operand);
}

void Disassembler::ldrd(Condition cond,
                        Register rt,
                        Register rt2,
                        Location* location) {
  os().SetCurrentInstruction(kLdrd, kAddress | kLoadStore);
  os() << ToCString(kLdrd) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rt2 << ", "
       << PrintLabel(kLoadDoubleWordLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::ldrex(Condition cond,
                         Register rt,
                         const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrex, kAddress | kLoadStore);
  os() << ToCString(kLdrex) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadWordLocation, operand);
}

void Disassembler::ldrexb(Condition cond,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrexb, kAddress | kLoadStore);
  os() << ToCString(kLdrexb) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadByteLocation, operand);
}

void Disassembler::ldrexd(Condition cond,
                          Register rt,
                          Register rt2,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrexd, kAddress | kLoadStore);
  os() << ToCString(kLdrexd) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rt2 << ", "
       << PrintMemOperand(kLoadDoubleWordLocation, operand);
}

void Disassembler::ldrexh(Condition cond,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrexh, kAddress | kLoadStore);
  os() << ToCString(kLdrexh) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kLoadHalfWordLocation, operand);
}

void Disassembler::ldrh(Condition cond,
                        EncodingSize size,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrh, kAddress | kLoadStore);
  os() << ToCString(kLdrh) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kLoadHalfWordLocation, operand);
}

void Disassembler::ldrh(Condition cond, Register rt, Location* location) {
  os().SetCurrentInstruction(kLdrh, kAddress | kLoadStore);
  os() << ToCString(kLdrh) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", "
       << PrintLabel(kLoadHalfWordLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::ldrsb(Condition cond,
                         EncodingSize size,
                         Register rt,
                         const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrsb, kAddress | kLoadStore);
  os() << ToCString(kLdrsb) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kLoadSignedByteLocation, operand);
}

void Disassembler::ldrsb(Condition cond, Register rt, Location* location) {
  os().SetCurrentInstruction(kLdrsb, kAddress | kLoadStore);
  os() << ToCString(kLdrsb) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", "
       << PrintLabel(kLoadSignedByteLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::ldrsh(Condition cond,
                         EncodingSize size,
                         Register rt,
                         const MemOperand& operand) {
  os().SetCurrentInstruction(kLdrsh, kAddress | kLoadStore);
  os() << ToCString(kLdrsh) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kLoadSignedHalfWordLocation, operand);
}

void Disassembler::ldrsh(Condition cond, Register rt, Location* location) {
  os().SetCurrentInstruction(kLdrsh, kAddress | kLoadStore);
  os() << ToCString(kLdrsh) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintLabel(kLoadSignedHalfWordLocation,
                             location,
                             GetCodeAddress() & ~3);
}

void Disassembler::lsl(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rm,
                       const Operand& operand) {
  os().SetCurrentInstruction(kLsl, kShift);
  os() << ToCString(kLsl) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::lsls(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rm,
                        const Operand& operand) {
  os().SetCurrentInstruction(kLsls, kShift);
  os() << ToCString(kLsls) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::lsr(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rm,
                       const Operand& operand) {
  os().SetCurrentInstruction(kLsr, kShift);
  os() << ToCString(kLsr) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::lsrs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rm,
                        const Operand& operand) {
  os().SetCurrentInstruction(kLsrs, kShift);
  os() << ToCString(kLsrs) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::mla(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kMla, kArithmetic);
  os() << ToCString(kMla) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::mlas(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kMlas, kArithmetic);
  os() << ToCString(kMlas) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::mls(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kMls, kArithmetic);
  os() << ToCString(kMls) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::mov(Condition cond,
                       EncodingSize size,
                       Register rd,
                       const Operand& operand) {
  os().SetCurrentInstruction(kMov, kNoAttribute);
  os() << ToCString(kMov) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << operand;
}

void Disassembler::movs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        const Operand& operand) {
  os().SetCurrentInstruction(kMovs, kNoAttribute);
  os() << ToCString(kMovs) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << operand;
}

void Disassembler::movt(Condition cond, Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kMovt, kNoAttribute);
  os() << ToCString(kMovt) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << operand;
}

void Disassembler::movw(Condition cond, Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kMovw, kNoAttribute);
  os() << ToCString(kMovw) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << operand;
}

void Disassembler::mrs(Condition cond, Register rd, SpecialRegister spec_reg) {
  os().SetCurrentInstruction(kMrs, kNoAttribute);
  os() << ToCString(kMrs) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << spec_reg;
}

void Disassembler::msr(Condition cond,
                       MaskedSpecialRegister spec_reg,
                       const Operand& operand) {
  os().SetCurrentInstruction(kMsr, kNoAttribute);
  os() << ToCString(kMsr) << ConditionPrinter(it_block_, cond) << " "
       << spec_reg << ", " << operand;
}

void Disassembler::mul(
    Condition cond, EncodingSize size, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kMul, kArithmetic);
  os() << ToCString(kMul) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::muls(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kMuls, kArithmetic);
  os() << ToCString(kMuls) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm;
}

void Disassembler::mvn(Condition cond,
                       EncodingSize size,
                       Register rd,
                       const Operand& operand) {
  os().SetCurrentInstruction(kMvn, kNoAttribute);
  os() << ToCString(kMvn) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << operand;
}

void Disassembler::mvns(Condition cond,
                        EncodingSize size,
                        Register rd,
                        const Operand& operand) {
  os().SetCurrentInstruction(kMvns, kNoAttribute);
  os() << ToCString(kMvns) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << operand;
}

void Disassembler::nop(Condition cond, EncodingSize size) {
  os().SetCurrentInstruction(kNop, kNoAttribute);
  os() << ToCString(kNop) << ConditionPrinter(it_block_, cond) << size;
}

void Disassembler::orn(Condition cond,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kOrn, kBitwise);
  os() << ToCString(kOrn) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::orns(Condition cond,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kOrns, kBitwise);
  os() << ToCString(kOrns) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::orr(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kOrr, kBitwise);
  os() << ToCString(kOrr) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::orrs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kOrrs, kBitwise);
  os() << ToCString(kOrrs) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::pkhbt(Condition cond,
                         Register rd,
                         Register rn,
                         const Operand& operand) {
  os().SetCurrentInstruction(kPkhbt, kNoAttribute);
  os() << ToCString(kPkhbt) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::pkhtb(Condition cond,
                         Register rd,
                         Register rn,
                         const Operand& operand) {
  os().SetCurrentInstruction(kPkhtb, kNoAttribute);
  os() << ToCString(kPkhtb) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::pld(Condition cond, Location* location) {
  os().SetCurrentInstruction(kPld, kAddress);
  os() << ToCString(kPld) << ConditionPrinter(it_block_, cond) << " "
       << PrintLabel(kDataLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::pld(Condition cond, const MemOperand& operand) {
  os().SetCurrentInstruction(kPld, kAddress);
  os() << ToCString(kPld) << ConditionPrinter(it_block_, cond) << " "
       << PrintMemOperand(kDataLocation, operand);
}

void Disassembler::pldw(Condition cond, const MemOperand& operand) {
  os().SetCurrentInstruction(kPldw, kAddress);
  os() << ToCString(kPldw) << ConditionPrinter(it_block_, cond) << " "
       << PrintMemOperand(kDataLocation, operand);
}

void Disassembler::pli(Condition cond, const MemOperand& operand) {
  os().SetCurrentInstruction(kPli, kAddress);
  os() << ToCString(kPli) << ConditionPrinter(it_block_, cond) << " "
       << PrintMemOperand(kCodeLocation, operand);
}

void Disassembler::pli(Condition cond, Location* location) {
  os().SetCurrentInstruction(kPli, kAddress);
  os() << ToCString(kPli) << ConditionPrinter(it_block_, cond) << " "
       << PrintLabel(kCodeLocation, location, GetCodeAddress() & ~3);
}

void Disassembler::pop(Condition cond,
                       EncodingSize size,
                       RegisterList registers) {
  os().SetCurrentInstruction(kPop, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kPop) << ConditionPrinter(it_block_, cond) << size << " "
       << registers;
}

void Disassembler::pop(Condition cond, EncodingSize size, Register rt) {
  os().SetCurrentInstruction(kPop, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kPop) << ConditionPrinter(it_block_, cond) << size << " "
       << "{" << rt << "}";
}

void Disassembler::push(Condition cond,
                        EncodingSize size,
                        RegisterList registers) {
  os().SetCurrentInstruction(kPush, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kPush) << ConditionPrinter(it_block_, cond) << size << " "
       << registers;
}

void Disassembler::push(Condition cond, EncodingSize size, Register rt) {
  os().SetCurrentInstruction(kPush, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kPush) << ConditionPrinter(it_block_, cond) << size << " "
       << "{" << rt << "}";
}

void Disassembler::qadd(Condition cond, Register rd, Register rm, Register rn) {
  os().SetCurrentInstruction(kQadd, kArithmetic);
  os() << ToCString(kQadd) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::qadd16(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kQadd16, kArithmetic);
  os() << ToCString(kQadd16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::qadd8(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kQadd8, kArithmetic);
  os() << ToCString(kQadd8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::qasx(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kQasx, kArithmetic);
  os() << ToCString(kQasx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::qdadd(Condition cond,
                         Register rd,
                         Register rm,
                         Register rn) {
  os().SetCurrentInstruction(kQdadd, kArithmetic);
  os() << ToCString(kQdadd) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::qdsub(Condition cond,
                         Register rd,
                         Register rm,
                         Register rn) {
  os().SetCurrentInstruction(kQdsub, kArithmetic);
  os() << ToCString(kQdsub) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::qsax(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kQsax, kArithmetic);
  os() << ToCString(kQsax) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::qsub(Condition cond, Register rd, Register rm, Register rn) {
  os().SetCurrentInstruction(kQsub, kArithmetic);
  os() << ToCString(kQsub) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::qsub16(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kQsub16, kArithmetic);
  os() << ToCString(kQsub16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::qsub8(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kQsub8, kArithmetic);
  os() << ToCString(kQsub8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::rbit(Condition cond, Register rd, Register rm) {
  os().SetCurrentInstruction(kRbit, kNoAttribute);
  os() << ToCString(kRbit) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rm;
}

void Disassembler::rev(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rm) {
  os().SetCurrentInstruction(kRev, kNoAttribute);
  os() << ToCString(kRev) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << rm;
}

void Disassembler::rev16(Condition cond,
                         EncodingSize size,
                         Register rd,
                         Register rm) {
  os().SetCurrentInstruction(kRev16, kNoAttribute);
  os() << ToCString(kRev16) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << rm;
}

void Disassembler::revsh(Condition cond,
                         EncodingSize size,
                         Register rd,
                         Register rm) {
  os().SetCurrentInstruction(kRevsh, kNoAttribute);
  os() << ToCString(kRevsh) << ConditionPrinter(it_block_, cond) << size << " "
       << rd << ", " << rm;
}

void Disassembler::ror(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rm,
                       const Operand& operand) {
  os().SetCurrentInstruction(kRor, kShift);
  os() << ToCString(kRor) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::rors(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rm,
                        const Operand& operand) {
  os().SetCurrentInstruction(kRors, kShift);
  os() << ToCString(kRors) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::rrx(Condition cond, Register rd, Register rm) {
  os().SetCurrentInstruction(kRrx, kShift);
  os() << ToCString(kRrx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm;
}

void Disassembler::rrxs(Condition cond, Register rd, Register rm) {
  os().SetCurrentInstruction(kRrxs, kShift);
  os() << ToCString(kRrxs) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm;
}

void Disassembler::rsb(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kRsb, kArithmetic);
  os() << ToCString(kRsb) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::rsbs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kRsbs, kArithmetic);
  os() << ToCString(kRsbs) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::rsc(Condition cond,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kRsc, kArithmetic);
  os() << ToCString(kRsc) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::rscs(Condition cond,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kRscs, kArithmetic);
  os() << ToCString(kRscs) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::sadd16(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSadd16, kArithmetic);
  os() << ToCString(kSadd16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::sadd8(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kSadd8, kArithmetic);
  os() << ToCString(kSadd8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::sasx(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kSasx, kArithmetic);
  os() << ToCString(kSasx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::sbc(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kSbc, kArithmetic);
  os() << ToCString(kSbc) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::sbcs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kSbcs, kArithmetic);
  os() << ToCString(kSbcs) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::sbfx(
    Condition cond, Register rd, Register rn, uint32_t lsb, uint32_t width) {
  os().SetCurrentInstruction(kSbfx, kShift);
  os() << ToCString(kSbfx) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << ImmediatePrinter(lsb) << ", "
       << ImmediatePrinter(width);
}

void Disassembler::sdiv(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kSdiv, kArithmetic);
  os() << ToCString(kSdiv) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::sel(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kSel, kNoAttribute);
  os() << ToCString(kSel) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::shadd16(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kShadd16, kArithmetic);
  os() << ToCString(kShadd16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::shadd8(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kShadd8, kArithmetic);
  os() << ToCString(kShadd8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::shasx(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kShasx, kArithmetic);
  os() << ToCString(kShasx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::shsax(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kShsax, kArithmetic);
  os() << ToCString(kShsax) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::shsub16(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kShsub16, kArithmetic);
  os() << ToCString(kShsub16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::shsub8(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kShsub8, kArithmetic);
  os() << ToCString(kShsub8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smlabb(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlabb, kArithmetic);
  os() << ToCString(kSmlabb) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlabt(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlabt, kArithmetic);
  os() << ToCString(kSmlabt) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlad(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlad, kArithmetic);
  os() << ToCString(kSmlad) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smladx(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmladx, kArithmetic);
  os() << ToCString(kSmladx) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlal(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlal, kArithmetic);
  os() << ToCString(kSmlal) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlalbb(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlalbb, kArithmetic);
  os() << ToCString(kSmlalbb) << ConditionPrinter(it_block_, cond) << " "
       << rdlo << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlalbt(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlalbt, kArithmetic);
  os() << ToCString(kSmlalbt) << ConditionPrinter(it_block_, cond) << " "
       << rdlo << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlald(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlald, kArithmetic);
  os() << ToCString(kSmlald) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlaldx(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlaldx, kArithmetic);
  os() << ToCString(kSmlaldx) << ConditionPrinter(it_block_, cond) << " "
       << rdlo << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlals(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlals, kArithmetic);
  os() << ToCString(kSmlals) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlaltb(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlaltb, kArithmetic);
  os() << ToCString(kSmlaltb) << ConditionPrinter(it_block_, cond) << " "
       << rdlo << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlaltt(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlaltt, kArithmetic);
  os() << ToCString(kSmlaltt) << ConditionPrinter(it_block_, cond) << " "
       << rdlo << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlatb(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlatb, kArithmetic);
  os() << ToCString(kSmlatb) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlatt(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlatt, kArithmetic);
  os() << ToCString(kSmlatt) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlawb(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlawb, kArithmetic);
  os() << ToCString(kSmlawb) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlawt(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlawt, kArithmetic);
  os() << ToCString(kSmlawt) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlsd(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlsd, kArithmetic);
  os() << ToCString(kSmlsd) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlsdx(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmlsdx, kArithmetic);
  os() << ToCString(kSmlsdx) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smlsld(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlsld, kArithmetic);
  os() << ToCString(kSmlsld) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smlsldx(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmlsldx, kArithmetic);
  os() << ToCString(kSmlsldx) << ConditionPrinter(it_block_, cond) << " "
       << rdlo << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smmla(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmmla, kArithmetic);
  os() << ToCString(kSmmla) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smmlar(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmmlar, kArithmetic);
  os() << ToCString(kSmmlar) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smmls(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmmls, kArithmetic);
  os() << ToCString(kSmmls) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smmlsr(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kSmmlsr, kArithmetic);
  os() << ToCString(kSmmlsr) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::smmul(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kSmmul, kArithmetic);
  os() << ToCString(kSmmul) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smmulr(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmmulr, kArithmetic);
  os() << ToCString(kSmmulr) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smuad(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kSmuad, kArithmetic);
  os() << ToCString(kSmuad) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smuadx(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmuadx, kArithmetic);
  os() << ToCString(kSmuadx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smulbb(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmulbb, kArithmetic);
  os() << ToCString(kSmulbb) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smulbt(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmulbt, kArithmetic);
  os() << ToCString(kSmulbt) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smull(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmull, kArithmetic);
  os() << ToCString(kSmull) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smulls(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kSmulls, kArithmetic);
  os() << ToCString(kSmulls) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::smultb(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmultb, kArithmetic);
  os() << ToCString(kSmultb) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smultt(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmultt, kArithmetic);
  os() << ToCString(kSmultt) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smulwb(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmulwb, kArithmetic);
  os() << ToCString(kSmulwb) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smulwt(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmulwt, kArithmetic);
  os() << ToCString(kSmulwt) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smusd(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kSmusd, kArithmetic);
  os() << ToCString(kSmusd) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::smusdx(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSmusdx, kArithmetic);
  os() << ToCString(kSmusdx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::ssat(Condition cond,
                        Register rd,
                        uint32_t imm,
                        const Operand& operand) {
  os().SetCurrentInstruction(kSsat, kArithmetic);
  os() << ToCString(kSsat) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << ImmediatePrinter(imm) << ", " << operand;
}

void Disassembler::ssat16(Condition cond,
                          Register rd,
                          uint32_t imm,
                          Register rn) {
  os().SetCurrentInstruction(kSsat16, kArithmetic);
  os() << ToCString(kSsat16) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << ImmediatePrinter(imm) << ", " << rn;
}

void Disassembler::ssax(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kSsax, kArithmetic);
  os() << ToCString(kSsax) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::ssub16(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kSsub16, kArithmetic);
  os() << ToCString(kSsub16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::ssub8(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kSsub8, kArithmetic);
  os() << ToCString(kSsub8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::stl(Condition cond, Register rt, const MemOperand& operand) {
  os().SetCurrentInstruction(kStl, kAddress | kLoadStore);
  os() << ToCString(kStl) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kStoreWordLocation, operand);
}

void Disassembler::stlb(Condition cond,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kStlb, kAddress | kLoadStore);
  os() << ToCString(kStlb) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kStoreByteLocation, operand);
}

void Disassembler::stlex(Condition cond,
                         Register rd,
                         Register rt,
                         const MemOperand& operand) {
  os().SetCurrentInstruction(kStlex, kAddress | kLoadStore);
  os() << ToCString(kStlex) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", " << PrintMemOperand(kStoreWordLocation, operand);
}

void Disassembler::stlexb(Condition cond,
                          Register rd,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kStlexb, kAddress | kLoadStore);
  os() << ToCString(kStlexb) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", " << PrintMemOperand(kStoreByteLocation, operand);
}

void Disassembler::stlexd(Condition cond,
                          Register rd,
                          Register rt,
                          Register rt2,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kStlexd, kAddress | kLoadStore);
  os() << ToCString(kStlexd) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", " << rt2 << ", "
       << PrintMemOperand(kStoreDoubleWordLocation, operand);
}

void Disassembler::stlexh(Condition cond,
                          Register rd,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kStlexh, kAddress | kLoadStore);
  os() << ToCString(kStlexh) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", "
       << PrintMemOperand(kStoreHalfWordLocation, operand);
}

void Disassembler::stlh(Condition cond,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kStlh, kAddress | kLoadStore);
  os() << ToCString(kStlh) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << PrintMemOperand(kStoreHalfWordLocation, operand);
}

void Disassembler::stm(Condition cond,
                       EncodingSize size,
                       Register rn,
                       WriteBack write_back,
                       RegisterList registers) {
  os().SetCurrentInstruction(kStm, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStm) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << write_back << ", " << registers;
}

void Disassembler::stmda(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kStmda, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStmda) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::stmdb(Condition cond,
                         EncodingSize size,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kStmdb, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStmdb) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << write_back << ", " << registers;
}

void Disassembler::stmea(Condition cond,
                         EncodingSize size,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kStmea, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStmea) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << write_back << ", " << registers;
}

void Disassembler::stmed(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kStmed, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStmed) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::stmfa(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kStmfa, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStmfa) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::stmfd(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kStmfd, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStmfd) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::stmib(Condition cond,
                         Register rn,
                         WriteBack write_back,
                         RegisterList registers) {
  os().SetCurrentInstruction(kStmib, kLoadStore | kLoadStoreMultiple);
  os() << ToCString(kStmib) << ConditionPrinter(it_block_, cond) << " " << rn
       << write_back << ", " << registers;
}

void Disassembler::str(Condition cond,
                       EncodingSize size,
                       Register rt,
                       const MemOperand& operand) {
  os().SetCurrentInstruction(kStr, kAddress | kLoadStore);
  os() << ToCString(kStr) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kStoreWordLocation, operand);
}

void Disassembler::strb(Condition cond,
                        EncodingSize size,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kStrb, kAddress | kLoadStore);
  os() << ToCString(kStrb) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kStoreByteLocation, operand);
}

void Disassembler::strd(Condition cond,
                        Register rt,
                        Register rt2,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kStrd, kAddress | kLoadStore);
  os() << ToCString(kStrd) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rt2 << ", "
       << PrintMemOperand(kStoreDoubleWordLocation, operand);
}

void Disassembler::strex(Condition cond,
                         Register rd,
                         Register rt,
                         const MemOperand& operand) {
  os().SetCurrentInstruction(kStrex, kAddress | kLoadStore);
  os() << ToCString(kStrex) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", " << PrintMemOperand(kStoreWordLocation, operand);
}

void Disassembler::strexb(Condition cond,
                          Register rd,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kStrexb, kAddress | kLoadStore);
  os() << ToCString(kStrexb) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", " << PrintMemOperand(kStoreByteLocation, operand);
}

void Disassembler::strexd(Condition cond,
                          Register rd,
                          Register rt,
                          Register rt2,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kStrexd, kAddress | kLoadStore);
  os() << ToCString(kStrexd) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", " << rt2 << ", "
       << PrintMemOperand(kStoreDoubleWordLocation, operand);
}

void Disassembler::strexh(Condition cond,
                          Register rd,
                          Register rt,
                          const MemOperand& operand) {
  os().SetCurrentInstruction(kStrexh, kAddress | kLoadStore);
  os() << ToCString(kStrexh) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rt << ", "
       << PrintMemOperand(kStoreHalfWordLocation, operand);
}

void Disassembler::strh(Condition cond,
                        EncodingSize size,
                        Register rt,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kStrh, kAddress | kLoadStore);
  os() << ToCString(kStrh) << ConditionPrinter(it_block_, cond) << size << " "
       << rt << ", " << PrintMemOperand(kStoreHalfWordLocation, operand);
}

void Disassembler::sub(Condition cond,
                       EncodingSize size,
                       Register rd,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kSub, kArithmetic);
  os() << ToCString(kSub) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::sub(Condition cond, Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kSub, kArithmetic);
  os() << ToCString(kSub) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << operand;
}

void Disassembler::subs(Condition cond,
                        EncodingSize size,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kSubs, kArithmetic);
  os() << ToCString(kSubs) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::subs(Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kSubs, kArithmetic);
  os() << ToCString(kSubs) << " " << rd << ", " << operand;
}

void Disassembler::subw(Condition cond,
                        Register rd,
                        Register rn,
                        const Operand& operand) {
  os().SetCurrentInstruction(kSubw, kArithmetic);
  os() << ToCString(kSubw) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::svc(Condition cond, uint32_t imm) {
  os().SetCurrentInstruction(kSvc, kSystem);
  os() << ToCString(kSvc) << ConditionPrinter(it_block_, cond) << " "
       << RawImmediatePrinter(imm);
}

void Disassembler::sxtab(Condition cond,
                         Register rd,
                         Register rn,
                         const Operand& operand) {
  os().SetCurrentInstruction(kSxtab, kArithmetic);
  os() << ToCString(kSxtab) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::sxtab16(Condition cond,
                           Register rd,
                           Register rn,
                           const Operand& operand) {
  os().SetCurrentInstruction(kSxtab16, kArithmetic);
  os() << ToCString(kSxtab16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::sxtah(Condition cond,
                         Register rd,
                         Register rn,
                         const Operand& operand) {
  os().SetCurrentInstruction(kSxtah, kArithmetic);
  os() << ToCString(kSxtah) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::sxtb(Condition cond,
                        EncodingSize size,
                        Register rd,
                        const Operand& operand) {
  os().SetCurrentInstruction(kSxtb, kArithmetic);
  os() << ToCString(kSxtb) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(operand.GetBaseRegister()) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << operand;
}

void Disassembler::sxtb16(Condition cond, Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kSxtb16, kArithmetic);
  os() << ToCString(kSxtb16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(operand.GetBaseRegister()) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << operand;
}

void Disassembler::sxth(Condition cond,
                        EncodingSize size,
                        Register rd,
                        const Operand& operand) {
  os().SetCurrentInstruction(kSxth, kArithmetic);
  os() << ToCString(kSxth) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(operand.GetBaseRegister()) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << operand;
}

void Disassembler::tbb(Condition cond, Register rn, Register rm) {
  os().SetCurrentInstruction(kTbb, kBranch);
  os() << ToCString(kTbb) << ConditionPrinter(it_block_, cond) << " "
       << MemOperand(rn, rm);
}

void Disassembler::tbh(Condition cond, Register rn, Register rm) {
  os().SetCurrentInstruction(kTbh, kBranch);
  os() << ToCString(kTbh) << ConditionPrinter(it_block_, cond) << " "
       << MemOperand(rn, plus, rm, LSL, 1);
}

void Disassembler::teq(Condition cond, Register rn, const Operand& operand) {
  os().SetCurrentInstruction(kTeq, kBitwise);
  os() << ToCString(kTeq) << ConditionPrinter(it_block_, cond) << " " << rn
       << ", " << operand;
}

void Disassembler::tst(Condition cond,
                       EncodingSize size,
                       Register rn,
                       const Operand& operand) {
  os().SetCurrentInstruction(kTst, kBitwise);
  os() << ToCString(kTst) << ConditionPrinter(it_block_, cond) << size << " "
       << rn << ", " << operand;
}

void Disassembler::uadd16(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kUadd16, kArithmetic);
  os() << ToCString(kUadd16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uadd8(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kUadd8, kArithmetic);
  os() << ToCString(kUadd8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uasx(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kUasx, kArithmetic);
  os() << ToCString(kUasx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::ubfx(
    Condition cond, Register rd, Register rn, uint32_t lsb, uint32_t width) {
  os().SetCurrentInstruction(kUbfx, kShift);
  os() << ToCString(kUbfx) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << ImmediatePrinter(lsb) << ", "
       << ImmediatePrinter(width);
}

void Disassembler::udf(Condition cond, EncodingSize size, uint32_t imm) {
  os().SetCurrentInstruction(kUdf, kNoAttribute);
  os() << ToCString(kUdf) << ConditionPrinter(it_block_, cond) << size << " "
       << RawImmediatePrinter(imm);
}

void Disassembler::udiv(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kUdiv, kArithmetic);
  os() << ToCString(kUdiv) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uhadd16(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kUhadd16, kArithmetic);
  os() << ToCString(kUhadd16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uhadd8(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kUhadd8, kArithmetic);
  os() << ToCString(kUhadd8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uhasx(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kUhasx, kArithmetic);
  os() << ToCString(kUhasx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uhsax(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kUhsax, kArithmetic);
  os() << ToCString(kUhsax) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uhsub16(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kUhsub16, kArithmetic);
  os() << ToCString(kUhsub16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uhsub8(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kUhsub8, kArithmetic);
  os() << ToCString(kUhsub8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::umaal(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kUmaal, kArithmetic);
  os() << ToCString(kUmaal) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::umlal(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kUmlal, kArithmetic);
  os() << ToCString(kUmlal) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::umlals(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kUmlals, kArithmetic);
  os() << ToCString(kUmlals) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::umull(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kUmull, kArithmetic);
  os() << ToCString(kUmull) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::umulls(
    Condition cond, Register rdlo, Register rdhi, Register rn, Register rm) {
  os().SetCurrentInstruction(kUmulls, kArithmetic);
  os() << ToCString(kUmulls) << ConditionPrinter(it_block_, cond) << " " << rdlo
       << ", " << rdhi << ", " << rn << ", " << rm;
}

void Disassembler::uqadd16(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kUqadd16, kArithmetic);
  os() << ToCString(kUqadd16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uqadd8(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kUqadd8, kArithmetic);
  os() << ToCString(kUqadd8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uqasx(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kUqasx, kArithmetic);
  os() << ToCString(kUqasx) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uqsax(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kUqsax, kArithmetic);
  os() << ToCString(kUqsax) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uqsub16(Condition cond,
                           Register rd,
                           Register rn,
                           Register rm) {
  os().SetCurrentInstruction(kUqsub16, kArithmetic);
  os() << ToCString(kUqsub16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uqsub8(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kUqsub8, kArithmetic);
  os() << ToCString(kUqsub8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::usad8(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kUsad8, kArithmetic);
  os() << ToCString(kUsad8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::usada8(
    Condition cond, Register rd, Register rn, Register rm, Register ra) {
  os().SetCurrentInstruction(kUsada8, kArithmetic);
  os() << ToCString(kUsada8) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << rn << ", " << rm << ", " << ra;
}

void Disassembler::usat(Condition cond,
                        Register rd,
                        uint32_t imm,
                        const Operand& operand) {
  os().SetCurrentInstruction(kUsat, kArithmetic);
  os() << ToCString(kUsat) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << ImmediatePrinter(imm) << ", " << operand;
}

void Disassembler::usat16(Condition cond,
                          Register rd,
                          uint32_t imm,
                          Register rn) {
  os().SetCurrentInstruction(kUsat16, kArithmetic);
  os() << ToCString(kUsat16) << ConditionPrinter(it_block_, cond) << " " << rd
       << ", " << ImmediatePrinter(imm) << ", " << rn;
}

void Disassembler::usax(Condition cond, Register rd, Register rn, Register rm) {
  os().SetCurrentInstruction(kUsax, kArithmetic);
  os() << ToCString(kUsax) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::usub16(Condition cond,
                          Register rd,
                          Register rn,
                          Register rm) {
  os().SetCurrentInstruction(kUsub16, kArithmetic);
  os() << ToCString(kUsub16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::usub8(Condition cond,
                         Register rd,
                         Register rn,
                         Register rm) {
  os().SetCurrentInstruction(kUsub8, kArithmetic);
  os() << ToCString(kUsub8) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::uxtab(Condition cond,
                         Register rd,
                         Register rn,
                         const Operand& operand) {
  os().SetCurrentInstruction(kUxtab, kArithmetic);
  os() << ToCString(kUxtab) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::uxtab16(Condition cond,
                           Register rd,
                           Register rn,
                           const Operand& operand) {
  os().SetCurrentInstruction(kUxtab16, kArithmetic);
  os() << ToCString(kUxtab16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::uxtah(Condition cond,
                         Register rd,
                         Register rn,
                         const Operand& operand) {
  os().SetCurrentInstruction(kUxtah, kArithmetic);
  os() << ToCString(kUxtah) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::uxtb(Condition cond,
                        EncodingSize size,
                        Register rd,
                        const Operand& operand) {
  os().SetCurrentInstruction(kUxtb, kArithmetic);
  os() << ToCString(kUxtb) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(operand.GetBaseRegister()) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << operand;
}

void Disassembler::uxtb16(Condition cond, Register rd, const Operand& operand) {
  os().SetCurrentInstruction(kUxtb16, kArithmetic);
  os() << ToCString(kUxtb16) << ConditionPrinter(it_block_, cond);
  os() << " ";
  if (!rd.Is(operand.GetBaseRegister()) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << operand;
}

void Disassembler::uxth(Condition cond,
                        EncodingSize size,
                        Register rd,
                        const Operand& operand) {
  os().SetCurrentInstruction(kUxth, kArithmetic);
  os() << ToCString(kUxth) << ConditionPrinter(it_block_, cond) << size;
  os() << " ";
  if (!rd.Is(operand.GetBaseRegister()) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << operand;
}

void Disassembler::vaba(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVaba, kFpNeon);
  os() << ToCString(kVaba) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vaba(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVaba, kFpNeon);
  os() << ToCString(kVaba) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vabal(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVabal, kFpNeon);
  os() << ToCString(kVabal) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vabd(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVabd, kFpNeon);
  os() << ToCString(kVabd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vabd(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVabd, kFpNeon);
  os() << ToCString(kVabd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vabdl(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVabdl, kFpNeon);
  os() << ToCString(kVabdl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vabs(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVabs, kFpNeon);
  os() << ToCString(kVabs) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vabs(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVabs, kFpNeon);
  os() << ToCString(kVabs) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vabs(Condition cond,
                        DataType dt,
                        SRegister rd,
                        SRegister rm) {
  os().SetCurrentInstruction(kVabs, kFpNeon);
  os() << ToCString(kVabs) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vacge(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVacge, kFpNeon);
  os() << ToCString(kVacge) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vacge(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVacge, kFpNeon);
  os() << ToCString(kVacge) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vacgt(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVacgt, kFpNeon);
  os() << ToCString(kVacgt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vacgt(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVacgt, kFpNeon);
  os() << ToCString(kVacgt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vacle(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVacle, kFpNeon);
  os() << ToCString(kVacle) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vacle(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVacle, kFpNeon);
  os() << ToCString(kVacle) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vaclt(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVaclt, kFpNeon);
  os() << ToCString(kVaclt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vaclt(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVaclt, kFpNeon);
  os() << ToCString(kVaclt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vadd(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVadd, kFpNeon);
  os() << ToCString(kVadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vadd(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVadd, kFpNeon);
  os() << ToCString(kVadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vadd(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVadd, kFpNeon);
  os() << ToCString(kVadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vaddhn(
    Condition cond, DataType dt, DRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVaddhn, kFpNeon);
  os() << ToCString(kVaddhn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vaddl(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVaddl, kFpNeon);
  os() << ToCString(kVaddl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vaddw(
    Condition cond, DataType dt, QRegister rd, QRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVaddw, kFpNeon);
  os() << ToCString(kVaddw) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vand(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rn,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVand, kFpNeon);
  os() << ToCString(kVand) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vand(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rn,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVand, kFpNeon);
  os() << ToCString(kVand) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vbic(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rn,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVbic, kFpNeon);
  os() << ToCString(kVbic) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vbic(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rn,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVbic, kFpNeon);
  os() << ToCString(kVbic) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vbif(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVbif, kFpNeon);
  os() << ToCString(kVbif) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vbif(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVbif, kFpNeon);
  os() << ToCString(kVbif) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vbit(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVbit, kFpNeon);
  os() << ToCString(kVbit) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vbit(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVbit, kFpNeon);
  os() << ToCString(kVbit) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vbsl(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVbsl, kFpNeon);
  os() << ToCString(kVbsl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vbsl(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVbsl, kFpNeon);
  os() << ToCString(kVbsl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vceq(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVceq, kFpNeon);
  os() << ToCString(kVceq) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vceq(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVceq, kFpNeon);
  os() << ToCString(kVceq) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vceq(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVceq, kFpNeon);
  os() << ToCString(kVceq) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vceq(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVceq, kFpNeon);
  os() << ToCString(kVceq) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vcge(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVcge, kFpNeon);
  os() << ToCString(kVcge) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vcge(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVcge, kFpNeon);
  os() << ToCString(kVcge) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vcge(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVcge, kFpNeon);
  os() << ToCString(kVcge) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vcge(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVcge, kFpNeon);
  os() << ToCString(kVcge) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vcgt(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVcgt, kFpNeon);
  os() << ToCString(kVcgt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vcgt(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVcgt, kFpNeon);
  os() << ToCString(kVcgt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vcgt(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVcgt, kFpNeon);
  os() << ToCString(kVcgt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vcgt(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVcgt, kFpNeon);
  os() << ToCString(kVcgt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vcle(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVcle, kFpNeon);
  os() << ToCString(kVcle) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vcle(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVcle, kFpNeon);
  os() << ToCString(kVcle) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vcle(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVcle, kFpNeon);
  os() << ToCString(kVcle) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vcle(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVcle, kFpNeon);
  os() << ToCString(kVcle) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vcls(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVcls, kFpNeon);
  os() << ToCString(kVcls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vcls(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVcls, kFpNeon);
  os() << ToCString(kVcls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vclt(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVclt, kFpNeon);
  os() << ToCString(kVclt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vclt(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVclt, kFpNeon);
  os() << ToCString(kVclt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vclt(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVclt, kFpNeon);
  os() << ToCString(kVclt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vclt(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVclt, kFpNeon);
  os() << ToCString(kVclt) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vclz(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVclz, kFpNeon);
  os() << ToCString(kVclz) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vclz(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVclz, kFpNeon);
  os() << ToCString(kVclz) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vcmp(Condition cond,
                        DataType dt,
                        SRegister rd,
                        const SOperand& operand) {
  os().SetCurrentInstruction(kVcmp, kFpNeon);
  os() << ToCString(kVcmp) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vcmp(Condition cond,
                        DataType dt,
                        DRegister rd,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVcmp, kFpNeon);
  os() << ToCString(kVcmp) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vcmpe(Condition cond,
                         DataType dt,
                         SRegister rd,
                         const SOperand& operand) {
  os().SetCurrentInstruction(kVcmpe, kFpNeon);
  os() << ToCString(kVcmpe) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vcmpe(Condition cond,
                         DataType dt,
                         DRegister rd,
                         const DOperand& operand) {
  os().SetCurrentInstruction(kVcmpe, kFpNeon);
  os() << ToCString(kVcmpe) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vcnt(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVcnt, kFpNeon);
  os() << ToCString(kVcnt) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vcnt(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVcnt, kFpNeon);
  os() << ToCString(kVcnt) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vcvt(
    Condition cond, DataType dt1, DataType dt2, DRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvt(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvt(Condition cond,
                        DataType dt1,
                        DataType dt2,
                        DRegister rd,
                        DRegister rm,
                        int32_t fbits) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm << ", " << SignedImmediatePrinter(fbits);
}

void Disassembler::vcvt(Condition cond,
                        DataType dt1,
                        DataType dt2,
                        QRegister rd,
                        QRegister rm,
                        int32_t fbits) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm << ", " << SignedImmediatePrinter(fbits);
}

void Disassembler::vcvt(Condition cond,
                        DataType dt1,
                        DataType dt2,
                        SRegister rd,
                        SRegister rm,
                        int32_t fbits) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm << ", " << SignedImmediatePrinter(fbits);
}

void Disassembler::vcvt(
    Condition cond, DataType dt1, DataType dt2, DRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvt(
    Condition cond, DataType dt1, DataType dt2, QRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvt(
    Condition cond, DataType dt1, DataType dt2, DRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvt(
    Condition cond, DataType dt1, DataType dt2, QRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvt(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVcvt, kFpNeon);
  os() << ToCString(kVcvt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvta(DataType dt1,
                         DataType dt2,
                         DRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvta, kFpNeon);
  os() << ToCString(kVcvta) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvta(DataType dt1,
                         DataType dt2,
                         QRegister rd,
                         QRegister rm) {
  os().SetCurrentInstruction(kVcvta, kFpNeon);
  os() << ToCString(kVcvta) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvta(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         SRegister rm) {
  os().SetCurrentInstruction(kVcvta, kFpNeon);
  os() << ToCString(kVcvta) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvta(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvta, kFpNeon);
  os() << ToCString(kVcvta) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtb(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVcvtb, kFpNeon);
  os() << ToCString(kVcvtb) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvtb(
    Condition cond, DataType dt1, DataType dt2, DRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVcvtb, kFpNeon);
  os() << ToCString(kVcvtb) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvtb(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVcvtb, kFpNeon);
  os() << ToCString(kVcvtb) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvtm(DataType dt1,
                         DataType dt2,
                         DRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvtm, kFpNeon);
  os() << ToCString(kVcvtm) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtm(DataType dt1,
                         DataType dt2,
                         QRegister rd,
                         QRegister rm) {
  os().SetCurrentInstruction(kVcvtm, kFpNeon);
  os() << ToCString(kVcvtm) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtm(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         SRegister rm) {
  os().SetCurrentInstruction(kVcvtm, kFpNeon);
  os() << ToCString(kVcvtm) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtm(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvtm, kFpNeon);
  os() << ToCString(kVcvtm) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtn(DataType dt1,
                         DataType dt2,
                         DRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvtn, kFpNeon);
  os() << ToCString(kVcvtn) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtn(DataType dt1,
                         DataType dt2,
                         QRegister rd,
                         QRegister rm) {
  os().SetCurrentInstruction(kVcvtn, kFpNeon);
  os() << ToCString(kVcvtn) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtn(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         SRegister rm) {
  os().SetCurrentInstruction(kVcvtn, kFpNeon);
  os() << ToCString(kVcvtn) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtn(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvtn, kFpNeon);
  os() << ToCString(kVcvtn) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtp(DataType dt1,
                         DataType dt2,
                         DRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvtp, kFpNeon);
  os() << ToCString(kVcvtp) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtp(DataType dt1,
                         DataType dt2,
                         QRegister rd,
                         QRegister rm) {
  os().SetCurrentInstruction(kVcvtp, kFpNeon);
  os() << ToCString(kVcvtp) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtp(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         SRegister rm) {
  os().SetCurrentInstruction(kVcvtp, kFpNeon);
  os() << ToCString(kVcvtp) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtp(DataType dt1,
                         DataType dt2,
                         SRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVcvtp, kFpNeon);
  os() << ToCString(kVcvtp) << dt1 << dt2 << " " << rd << ", " << rm;
}

void Disassembler::vcvtr(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVcvtr, kFpNeon);
  os() << ToCString(kVcvtr) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvtr(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVcvtr, kFpNeon);
  os() << ToCString(kVcvtr) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvtt(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVcvtt, kFpNeon);
  os() << ToCString(kVcvtt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvtt(
    Condition cond, DataType dt1, DataType dt2, DRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVcvtt, kFpNeon);
  os() << ToCString(kVcvtt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vcvtt(
    Condition cond, DataType dt1, DataType dt2, SRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVcvtt, kFpNeon);
  os() << ToCString(kVcvtt) << ConditionPrinter(it_block_, cond) << dt1 << dt2
       << " " << rd << ", " << rm;
}

void Disassembler::vdiv(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVdiv, kFpNeon);
  os() << ToCString(kVdiv) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vdiv(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVdiv, kFpNeon);
  os() << ToCString(kVdiv) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vdup(Condition cond,
                        DataType dt,
                        QRegister rd,
                        Register rt) {
  os().SetCurrentInstruction(kVdup, kFpNeon);
  os() << ToCString(kVdup) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rt;
}

void Disassembler::vdup(Condition cond,
                        DataType dt,
                        DRegister rd,
                        Register rt) {
  os().SetCurrentInstruction(kVdup, kFpNeon);
  os() << ToCString(kVdup) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rt;
}

void Disassembler::vdup(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegisterLane rm) {
  os().SetCurrentInstruction(kVdup, kFpNeon);
  os() << ToCString(kVdup) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vdup(Condition cond,
                        DataType dt,
                        QRegister rd,
                        DRegisterLane rm) {
  os().SetCurrentInstruction(kVdup, kFpNeon);
  os() << ToCString(kVdup) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::veor(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVeor, kFpNeon);
  os() << ToCString(kVeor) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::veor(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVeor, kFpNeon);
  os() << ToCString(kVeor) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vext(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rn,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVext, kFpNeon);
  os() << ToCString(kVext) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm << ", " << operand;
}

void Disassembler::vext(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rn,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVext, kFpNeon);
  os() << ToCString(kVext) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm << ", " << operand;
}

void Disassembler::vfma(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVfma, kFpNeon);
  os() << ToCString(kVfma) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfma(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVfma, kFpNeon);
  os() << ToCString(kVfma) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfma(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVfma, kFpNeon);
  os() << ToCString(kVfma) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfms(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVfms, kFpNeon);
  os() << ToCString(kVfms) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfms(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVfms, kFpNeon);
  os() << ToCString(kVfms) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfms(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVfms, kFpNeon);
  os() << ToCString(kVfms) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfnma(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVfnma, kFpNeon);
  os() << ToCString(kVfnma) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfnma(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVfnma, kFpNeon);
  os() << ToCString(kVfnma) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfnms(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVfnms, kFpNeon);
  os() << ToCString(kVfnms) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vfnms(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVfnms, kFpNeon);
  os() << ToCString(kVfnms) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vhadd(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVhadd, kFpNeon);
  os() << ToCString(kVhadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vhadd(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVhadd, kFpNeon);
  os() << ToCString(kVhadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vhsub(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVhsub, kFpNeon);
  os() << ToCString(kVhsub) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vhsub(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVhsub, kFpNeon);
  os() << ToCString(kVhsub) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vld1(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVld1, kFpNeon);
  os() << ToCString(kVld1) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVld1Location, operand);
}

void Disassembler::vld2(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVld2, kFpNeon);
  os() << ToCString(kVld2) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVld2Location, operand);
}

void Disassembler::vld3(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVld3, kFpNeon);
  os() << ToCString(kVld3) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVld3Location, operand);
}

void Disassembler::vld3(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kVld3, kFpNeon);
  os() << ToCString(kVld3) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintMemOperand(kVld3Location, operand);
}

void Disassembler::vld4(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVld4, kFpNeon);
  os() << ToCString(kVld4) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVld4Location, operand);
}

void Disassembler::vldm(Condition cond,
                        DataType dt,
                        Register rn,
                        WriteBack write_back,
                        DRegisterList dreglist) {
  os().SetCurrentInstruction(kVldm, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVldm) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << dreglist;
}

void Disassembler::vldm(Condition cond,
                        DataType dt,
                        Register rn,
                        WriteBack write_back,
                        SRegisterList sreglist) {
  os().SetCurrentInstruction(kVldm, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVldm) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << sreglist;
}

void Disassembler::vldmdb(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          DRegisterList dreglist) {
  os().SetCurrentInstruction(kVldmdb,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVldmdb) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << dreglist;
}

void Disassembler::vldmdb(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          SRegisterList sreglist) {
  os().SetCurrentInstruction(kVldmdb,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVldmdb) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << sreglist;
}

void Disassembler::vldmia(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          DRegisterList dreglist) {
  os().SetCurrentInstruction(kVldmia,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVldmia) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << dreglist;
}

void Disassembler::vldmia(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          SRegisterList sreglist) {
  os().SetCurrentInstruction(kVldmia,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVldmia) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << sreglist;
}

void Disassembler::vldr(Condition cond,
                        DataType dt,
                        DRegister rd,
                        Location* location) {
  os().SetCurrentInstruction(kVldr, kFpNeon);
  os() << ToCString(kVldr) << ConditionPrinter(it_block_, cond)
       << DtPrinter(dt, Untyped64) << " " << rd << ", "
       << PrintLabel(kLoadDoublePrecisionLocation,
                     location,
                     GetCodeAddress() & ~3);
}

void Disassembler::vldr(Condition cond,
                        DataType dt,
                        DRegister rd,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kVldr, kFpNeon);
  os() << ToCString(kVldr) << ConditionPrinter(it_block_, cond)
       << DtPrinter(dt, Untyped64) << " " << rd << ", "
       << PrintMemOperand(kLoadDoublePrecisionLocation, operand);
}

void Disassembler::vldr(Condition cond,
                        DataType dt,
                        SRegister rd,
                        Location* location) {
  os().SetCurrentInstruction(kVldr, kFpNeon);
  os() << ToCString(kVldr) << ConditionPrinter(it_block_, cond)
       << DtPrinter(dt, Untyped32) << " " << rd << ", "
       << PrintLabel(kLoadSinglePrecisionLocation,
                     location,
                     GetCodeAddress() & ~3);
}

void Disassembler::vldr(Condition cond,
                        DataType dt,
                        SRegister rd,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kVldr, kFpNeon);
  os() << ToCString(kVldr) << ConditionPrinter(it_block_, cond)
       << DtPrinter(dt, Untyped32) << " " << rd << ", "
       << PrintMemOperand(kLoadSinglePrecisionLocation, operand);
}

void Disassembler::vmax(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmax, kFpNeon);
  os() << ToCString(kVmax) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vmax(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVmax, kFpNeon);
  os() << ToCString(kVmax) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vmaxnm(DataType dt,
                          DRegister rd,
                          DRegister rn,
                          DRegister rm) {
  os().SetCurrentInstruction(kVmaxnm, kFpNeon);
  os() << ToCString(kVmaxnm) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmaxnm(DataType dt,
                          QRegister rd,
                          QRegister rn,
                          QRegister rm) {
  os().SetCurrentInstruction(kVmaxnm, kFpNeon);
  os() << ToCString(kVmaxnm) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmaxnm(DataType dt,
                          SRegister rd,
                          SRegister rn,
                          SRegister rm) {
  os().SetCurrentInstruction(kVmaxnm, kFpNeon);
  os() << ToCString(kVmaxnm) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmin(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmin, kFpNeon);
  os() << ToCString(kVmin) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vmin(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVmin, kFpNeon);
  os() << ToCString(kVmin) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vminnm(DataType dt,
                          DRegister rd,
                          DRegister rn,
                          DRegister rm) {
  os().SetCurrentInstruction(kVminnm, kFpNeon);
  os() << ToCString(kVminnm) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vminnm(DataType dt,
                          QRegister rd,
                          QRegister rn,
                          QRegister rm) {
  os().SetCurrentInstruction(kVminnm, kFpNeon);
  os() << ToCString(kVminnm) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vminnm(DataType dt,
                          SRegister rd,
                          SRegister rn,
                          SRegister rm) {
  os().SetCurrentInstruction(kVminnm, kFpNeon);
  os() << ToCString(kVminnm) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmla(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVmla, kFpNeon);
  os() << ToCString(kVmla) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmla(
    Condition cond, DataType dt, QRegister rd, QRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVmla, kFpNeon);
  os() << ToCString(kVmla) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmla(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmla, kFpNeon);
  os() << ToCString(kVmla) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmla(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVmla, kFpNeon);
  os() << ToCString(kVmla) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmla(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVmla, kFpNeon);
  os() << ToCString(kVmla) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmlal(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVmlal, kFpNeon);
  os() << ToCString(kVmlal) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmlal(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmlal, kFpNeon);
  os() << ToCString(kVmlal) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmls(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVmls, kFpNeon);
  os() << ToCString(kVmls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmls(
    Condition cond, DataType dt, QRegister rd, QRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVmls, kFpNeon);
  os() << ToCString(kVmls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmls(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmls, kFpNeon);
  os() << ToCString(kVmls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmls(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVmls, kFpNeon);
  os() << ToCString(kVmls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmls(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVmls, kFpNeon);
  os() << ToCString(kVmls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmlsl(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVmlsl, kFpNeon);
  os() << ToCString(kVmlsl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmlsl(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmlsl, kFpNeon);
  os() << ToCString(kVmlsl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmov(Condition cond, Register rt, SRegister rn) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rn;
}

void Disassembler::vmov(Condition cond, SRegister rn, Register rt) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << " " << rn
       << ", " << rt;
}

void Disassembler::vmov(Condition cond,
                        Register rt,
                        Register rt2,
                        DRegister rm) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rt2 << ", " << rm;
}

void Disassembler::vmov(Condition cond,
                        DRegister rm,
                        Register rt,
                        Register rt2) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << " " << rm
       << ", " << rt << ", " << rt2;
}

void Disassembler::vmov(
    Condition cond, Register rt, Register rt2, SRegister rm, SRegister rm1) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << rt2 << ", " << rm << ", " << rm1;
}

void Disassembler::vmov(
    Condition cond, SRegister rm, SRegister rm1, Register rt, Register rt2) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << " " << rm
       << ", " << rm1 << ", " << rt << ", " << rt2;
}

void Disassembler::vmov(Condition cond,
                        DataType dt,
                        DRegisterLane rd,
                        Register rt) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rt;
}

void Disassembler::vmov(Condition cond,
                        DataType dt,
                        DRegister rd,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vmov(Condition cond,
                        DataType dt,
                        QRegister rd,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vmov(Condition cond,
                        DataType dt,
                        SRegister rd,
                        const SOperand& operand) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vmov(Condition cond,
                        DataType dt,
                        Register rt,
                        DRegisterLane rn) {
  os().SetCurrentInstruction(kVmov, kFpNeon);
  os() << ToCString(kVmov) << ConditionPrinter(it_block_, cond) << dt << " "
       << rt << ", " << rn;
}

void Disassembler::vmovl(Condition cond,
                         DataType dt,
                         QRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVmovl, kFpNeon);
  os() << ToCString(kVmovl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vmovn(Condition cond,
                         DataType dt,
                         DRegister rd,
                         QRegister rm) {
  os().SetCurrentInstruction(kVmovn, kFpNeon);
  os() << ToCString(kVmovn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vmrs(Condition cond,
                        RegisterOrAPSR_nzcv rt,
                        SpecialFPRegister spec_reg) {
  os().SetCurrentInstruction(kVmrs, kFpNeon);
  os() << ToCString(kVmrs) << ConditionPrinter(it_block_, cond) << " " << rt
       << ", " << spec_reg;
}

void Disassembler::vmsr(Condition cond,
                        SpecialFPRegister spec_reg,
                        Register rt) {
  os().SetCurrentInstruction(kVmsr, kFpNeon);
  os() << ToCString(kVmsr) << ConditionPrinter(it_block_, cond) << " "
       << spec_reg << ", " << rt;
}

void Disassembler::vmul(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rn,
                        DRegister dm,
                        unsigned index) {
  os().SetCurrentInstruction(kVmul, kFpNeon);
  os() << ToCString(kVmul) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << IndexedRegisterPrinter(dm, index);
}

void Disassembler::vmul(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rn,
                        DRegister dm,
                        unsigned index) {
  os().SetCurrentInstruction(kVmul, kFpNeon);
  os() << ToCString(kVmul) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << IndexedRegisterPrinter(dm, index);
}

void Disassembler::vmul(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmul, kFpNeon);
  os() << ToCString(kVmul) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vmul(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVmul, kFpNeon);
  os() << ToCString(kVmul) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vmul(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVmul, kFpNeon);
  os() << ToCString(kVmul) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vmull(Condition cond,
                         DataType dt,
                         QRegister rd,
                         DRegister rn,
                         DRegister dm,
                         unsigned index) {
  os().SetCurrentInstruction(kVmull, kFpNeon);
  os() << ToCString(kVmull) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << IndexedRegisterPrinter(dm, index);
}

void Disassembler::vmull(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVmull, kFpNeon);
  os() << ToCString(kVmull) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vmvn(Condition cond,
                        DataType dt,
                        DRegister rd,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVmvn, kFpNeon);
  os() << ToCString(kVmvn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vmvn(Condition cond,
                        DataType dt,
                        QRegister rd,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVmvn, kFpNeon);
  os() << ToCString(kVmvn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << operand;
}

void Disassembler::vneg(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVneg, kFpNeon);
  os() << ToCString(kVneg) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vneg(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVneg, kFpNeon);
  os() << ToCString(kVneg) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vneg(Condition cond,
                        DataType dt,
                        SRegister rd,
                        SRegister rm) {
  os().SetCurrentInstruction(kVneg, kFpNeon);
  os() << ToCString(kVneg) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vnmla(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVnmla, kFpNeon);
  os() << ToCString(kVnmla) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vnmla(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVnmla, kFpNeon);
  os() << ToCString(kVnmla) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vnmls(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVnmls, kFpNeon);
  os() << ToCString(kVnmls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vnmls(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVnmls, kFpNeon);
  os() << ToCString(kVnmls) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vnmul(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVnmul, kFpNeon);
  os() << ToCString(kVnmul) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vnmul(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVnmul, kFpNeon);
  os() << ToCString(kVnmul) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vorn(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rn,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVorn, kFpNeon);
  os() << ToCString(kVorn) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vorn(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rn,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVorn, kFpNeon);
  os() << ToCString(kVorn) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vorr(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rn,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVorr, kFpNeon);
  os() << ToCString(kVorr) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vorr(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rn,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVorr, kFpNeon);
  os() << ToCString(kVorr) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << operand;
}

void Disassembler::vpadal(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVpadal, kFpNeon);
  os() << ToCString(kVpadal) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vpadal(Condition cond,
                          DataType dt,
                          QRegister rd,
                          QRegister rm) {
  os().SetCurrentInstruction(kVpadal, kFpNeon);
  os() << ToCString(kVpadal) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vpadd(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVpadd, kFpNeon);
  os() << ToCString(kVpadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vpaddl(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVpaddl, kFpNeon);
  os() << ToCString(kVpaddl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vpaddl(Condition cond,
                          DataType dt,
                          QRegister rd,
                          QRegister rm) {
  os().SetCurrentInstruction(kVpaddl, kFpNeon);
  os() << ToCString(kVpaddl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vpmax(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVpmax, kFpNeon);
  os() << ToCString(kVpmax) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vpmin(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVpmin, kFpNeon);
  os() << ToCString(kVpmin) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vpop(Condition cond, DataType dt, DRegisterList dreglist) {
  os().SetCurrentInstruction(kVpop, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVpop) << ConditionPrinter(it_block_, cond) << dt << " "
       << dreglist;
}

void Disassembler::vpop(Condition cond, DataType dt, SRegisterList sreglist) {
  os().SetCurrentInstruction(kVpop, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVpop) << ConditionPrinter(it_block_, cond) << dt << " "
       << sreglist;
}

void Disassembler::vpush(Condition cond, DataType dt, DRegisterList dreglist) {
  os().SetCurrentInstruction(kVpush, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVpush) << ConditionPrinter(it_block_, cond) << dt << " "
       << dreglist;
}

void Disassembler::vpush(Condition cond, DataType dt, SRegisterList sreglist) {
  os().SetCurrentInstruction(kVpush, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVpush) << ConditionPrinter(it_block_, cond) << dt << " "
       << sreglist;
}

void Disassembler::vqabs(Condition cond,
                         DataType dt,
                         DRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVqabs, kFpNeon);
  os() << ToCString(kVqabs) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vqabs(Condition cond,
                         DataType dt,
                         QRegister rd,
                         QRegister rm) {
  os().SetCurrentInstruction(kVqabs, kFpNeon);
  os() << ToCString(kVqabs) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vqadd(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVqadd, kFpNeon);
  os() << ToCString(kVqadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqadd(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVqadd, kFpNeon);
  os() << ToCString(kVqadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqdmlal(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVqdmlal, kFpNeon);
  os() << ToCString(kVqdmlal) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vqdmlal(Condition cond,
                           DataType dt,
                           QRegister rd,
                           DRegister rn,
                           DRegister dm,
                           unsigned index) {
  os().SetCurrentInstruction(kVqdmlal, kFpNeon);
  os() << ToCString(kVqdmlal) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << IndexedRegisterPrinter(dm, index);
}

void Disassembler::vqdmlsl(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVqdmlsl, kFpNeon);
  os() << ToCString(kVqdmlsl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vqdmlsl(Condition cond,
                           DataType dt,
                           QRegister rd,
                           DRegister rn,
                           DRegister dm,
                           unsigned index) {
  os().SetCurrentInstruction(kVqdmlsl, kFpNeon);
  os() << ToCString(kVqdmlsl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << IndexedRegisterPrinter(dm, index);
}

void Disassembler::vqdmulh(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVqdmulh, kFpNeon);
  os() << ToCString(kVqdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqdmulh(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVqdmulh, kFpNeon);
  os() << ToCString(kVqdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqdmulh(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVqdmulh, kFpNeon);
  os() << ToCString(kVqdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqdmulh(
    Condition cond, DataType dt, QRegister rd, QRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVqdmulh, kFpNeon);
  os() << ToCString(kVqdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqdmull(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVqdmull, kFpNeon);
  os() << ToCString(kVqdmull) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vqdmull(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVqdmull, kFpNeon);
  os() << ToCString(kVqdmull) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vqmovn(Condition cond,
                          DataType dt,
                          DRegister rd,
                          QRegister rm) {
  os().SetCurrentInstruction(kVqmovn, kFpNeon);
  os() << ToCString(kVqmovn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vqmovun(Condition cond,
                           DataType dt,
                           DRegister rd,
                           QRegister rm) {
  os().SetCurrentInstruction(kVqmovun, kFpNeon);
  os() << ToCString(kVqmovun) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vqneg(Condition cond,
                         DataType dt,
                         DRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVqneg, kFpNeon);
  os() << ToCString(kVqneg) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vqneg(Condition cond,
                         DataType dt,
                         QRegister rd,
                         QRegister rm) {
  os().SetCurrentInstruction(kVqneg, kFpNeon);
  os() << ToCString(kVqneg) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vqrdmulh(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVqrdmulh, kFpNeon);
  os() << ToCString(kVqrdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqrdmulh(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVqrdmulh, kFpNeon);
  os() << ToCString(kVqrdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqrdmulh(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVqrdmulh, kFpNeon);
  os() << ToCString(kVqrdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqrdmulh(
    Condition cond, DataType dt, QRegister rd, QRegister rn, DRegisterLane rm) {
  os().SetCurrentInstruction(kVqrdmulh, kFpNeon);
  os() << ToCString(kVqrdmulh) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqrshl(
    Condition cond, DataType dt, DRegister rd, DRegister rm, DRegister rn) {
  os().SetCurrentInstruction(kVqrshl, kFpNeon);
  os() << ToCString(kVqrshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::vqrshl(
    Condition cond, DataType dt, QRegister rd, QRegister rm, QRegister rn) {
  os().SetCurrentInstruction(kVqrshl, kFpNeon);
  os() << ToCString(kVqrshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::vqrshrn(Condition cond,
                           DataType dt,
                           DRegister rd,
                           QRegister rm,
                           const QOperand& operand) {
  os().SetCurrentInstruction(kVqrshrn, kFpNeon);
  os() << ToCString(kVqrshrn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm << ", " << operand;
}

void Disassembler::vqrshrun(Condition cond,
                            DataType dt,
                            DRegister rd,
                            QRegister rm,
                            const QOperand& operand) {
  os().SetCurrentInstruction(kVqrshrun, kFpNeon);
  os() << ToCString(kVqrshrun) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm << ", " << operand;
}

void Disassembler::vqshl(Condition cond,
                         DataType dt,
                         DRegister rd,
                         DRegister rm,
                         const DOperand& operand) {
  os().SetCurrentInstruction(kVqshl, kFpNeon);
  os() << ToCString(kVqshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vqshl(Condition cond,
                         DataType dt,
                         QRegister rd,
                         QRegister rm,
                         const QOperand& operand) {
  os().SetCurrentInstruction(kVqshl, kFpNeon);
  os() << ToCString(kVqshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vqshlu(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm,
                          const DOperand& operand) {
  os().SetCurrentInstruction(kVqshlu, kFpNeon);
  os() << ToCString(kVqshlu) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vqshlu(Condition cond,
                          DataType dt,
                          QRegister rd,
                          QRegister rm,
                          const QOperand& operand) {
  os().SetCurrentInstruction(kVqshlu, kFpNeon);
  os() << ToCString(kVqshlu) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vqshrn(Condition cond,
                          DataType dt,
                          DRegister rd,
                          QRegister rm,
                          const QOperand& operand) {
  os().SetCurrentInstruction(kVqshrn, kFpNeon);
  os() << ToCString(kVqshrn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm << ", " << operand;
}

void Disassembler::vqshrun(Condition cond,
                           DataType dt,
                           DRegister rd,
                           QRegister rm,
                           const QOperand& operand) {
  os().SetCurrentInstruction(kVqshrun, kFpNeon);
  os() << ToCString(kVqshrun) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm << ", " << operand;
}

void Disassembler::vqsub(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVqsub, kFpNeon);
  os() << ToCString(kVqsub) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vqsub(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVqsub, kFpNeon);
  os() << ToCString(kVqsub) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vraddhn(
    Condition cond, DataType dt, DRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVraddhn, kFpNeon);
  os() << ToCString(kVraddhn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vrecpe(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVrecpe, kFpNeon);
  os() << ToCString(kVrecpe) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrecpe(Condition cond,
                          DataType dt,
                          QRegister rd,
                          QRegister rm) {
  os().SetCurrentInstruction(kVrecpe, kFpNeon);
  os() << ToCString(kVrecpe) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrecps(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVrecps, kFpNeon);
  os() << ToCString(kVrecps) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vrecps(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVrecps, kFpNeon);
  os() << ToCString(kVrecps) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vrev16(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVrev16, kFpNeon);
  os() << ToCString(kVrev16) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrev16(Condition cond,
                          DataType dt,
                          QRegister rd,
                          QRegister rm) {
  os().SetCurrentInstruction(kVrev16, kFpNeon);
  os() << ToCString(kVrev16) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrev32(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVrev32, kFpNeon);
  os() << ToCString(kVrev32) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrev32(Condition cond,
                          DataType dt,
                          QRegister rd,
                          QRegister rm) {
  os().SetCurrentInstruction(kVrev32, kFpNeon);
  os() << ToCString(kVrev32) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrev64(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVrev64, kFpNeon);
  os() << ToCString(kVrev64) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrev64(Condition cond,
                          DataType dt,
                          QRegister rd,
                          QRegister rm) {
  os().SetCurrentInstruction(kVrev64, kFpNeon);
  os() << ToCString(kVrev64) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrhadd(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVrhadd, kFpNeon);
  os() << ToCString(kVrhadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vrhadd(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVrhadd, kFpNeon);
  os() << ToCString(kVrhadd) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vrinta(DataType dt, DRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVrinta, kFpNeon);
  os() << ToCString(kVrinta) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrinta(DataType dt, QRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVrinta, kFpNeon);
  os() << ToCString(kVrinta) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrinta(DataType dt, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVrinta, kFpNeon);
  os() << ToCString(kVrinta) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintm(DataType dt, DRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVrintm, kFpNeon);
  os() << ToCString(kVrintm) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintm(DataType dt, QRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVrintm, kFpNeon);
  os() << ToCString(kVrintm) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintm(DataType dt, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVrintm, kFpNeon);
  os() << ToCString(kVrintm) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintn(DataType dt, DRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVrintn, kFpNeon);
  os() << ToCString(kVrintn) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintn(DataType dt, QRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVrintn, kFpNeon);
  os() << ToCString(kVrintn) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintn(DataType dt, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVrintn, kFpNeon);
  os() << ToCString(kVrintn) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintp(DataType dt, DRegister rd, DRegister rm) {
  os().SetCurrentInstruction(kVrintp, kFpNeon);
  os() << ToCString(kVrintp) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintp(DataType dt, QRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVrintp, kFpNeon);
  os() << ToCString(kVrintp) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintp(DataType dt, SRegister rd, SRegister rm) {
  os().SetCurrentInstruction(kVrintp, kFpNeon);
  os() << ToCString(kVrintp) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintr(Condition cond,
                          DataType dt,
                          SRegister rd,
                          SRegister rm) {
  os().SetCurrentInstruction(kVrintr, kFpNeon);
  os() << ToCString(kVrintr) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrintr(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVrintr, kFpNeon);
  os() << ToCString(kVrintr) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrintx(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVrintx, kFpNeon);
  os() << ToCString(kVrintx) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrintx(DataType dt, QRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVrintx, kFpNeon);
  os() << ToCString(kVrintx) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintx(Condition cond,
                          DataType dt,
                          SRegister rd,
                          SRegister rm) {
  os().SetCurrentInstruction(kVrintx, kFpNeon);
  os() << ToCString(kVrintx) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrintz(Condition cond,
                          DataType dt,
                          DRegister rd,
                          DRegister rm) {
  os().SetCurrentInstruction(kVrintz, kFpNeon);
  os() << ToCString(kVrintz) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrintz(DataType dt, QRegister rd, QRegister rm) {
  os().SetCurrentInstruction(kVrintz, kFpNeon);
  os() << ToCString(kVrintz) << dt << " " << rd << ", " << rm;
}

void Disassembler::vrintz(Condition cond,
                          DataType dt,
                          SRegister rd,
                          SRegister rm) {
  os().SetCurrentInstruction(kVrintz, kFpNeon);
  os() << ToCString(kVrintz) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrshl(
    Condition cond, DataType dt, DRegister rd, DRegister rm, DRegister rn) {
  os().SetCurrentInstruction(kVrshl, kFpNeon);
  os() << ToCString(kVrshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::vrshl(
    Condition cond, DataType dt, QRegister rd, QRegister rm, QRegister rn) {
  os().SetCurrentInstruction(kVrshl, kFpNeon);
  os() << ToCString(kVrshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << rn;
}

void Disassembler::vrshr(Condition cond,
                         DataType dt,
                         DRegister rd,
                         DRegister rm,
                         const DOperand& operand) {
  os().SetCurrentInstruction(kVrshr, kFpNeon);
  os() << ToCString(kVrshr) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vrshr(Condition cond,
                         DataType dt,
                         QRegister rd,
                         QRegister rm,
                         const QOperand& operand) {
  os().SetCurrentInstruction(kVrshr, kFpNeon);
  os() << ToCString(kVrshr) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vrshrn(Condition cond,
                          DataType dt,
                          DRegister rd,
                          QRegister rm,
                          const QOperand& operand) {
  os().SetCurrentInstruction(kVrshrn, kFpNeon);
  os() << ToCString(kVrshrn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm << ", " << operand;
}

void Disassembler::vrsqrte(Condition cond,
                           DataType dt,
                           DRegister rd,
                           DRegister rm) {
  os().SetCurrentInstruction(kVrsqrte, kFpNeon);
  os() << ToCString(kVrsqrte) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrsqrte(Condition cond,
                           DataType dt,
                           QRegister rd,
                           QRegister rm) {
  os().SetCurrentInstruction(kVrsqrte, kFpNeon);
  os() << ToCString(kVrsqrte) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vrsqrts(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVrsqrts, kFpNeon);
  os() << ToCString(kVrsqrts) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vrsqrts(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVrsqrts, kFpNeon);
  os() << ToCString(kVrsqrts) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vrsra(Condition cond,
                         DataType dt,
                         DRegister rd,
                         DRegister rm,
                         const DOperand& operand) {
  os().SetCurrentInstruction(kVrsra, kFpNeon);
  os() << ToCString(kVrsra) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vrsra(Condition cond,
                         DataType dt,
                         QRegister rd,
                         QRegister rm,
                         const QOperand& operand) {
  os().SetCurrentInstruction(kVrsra, kFpNeon);
  os() << ToCString(kVrsra) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vrsubhn(
    Condition cond, DataType dt, DRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVrsubhn, kFpNeon);
  os() << ToCString(kVrsubhn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vseleq(DataType dt,
                          DRegister rd,
                          DRegister rn,
                          DRegister rm) {
  os().SetCurrentInstruction(kVseleq, kFpNeon);
  os() << ToCString(kVseleq) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vseleq(DataType dt,
                          SRegister rd,
                          SRegister rn,
                          SRegister rm) {
  os().SetCurrentInstruction(kVseleq, kFpNeon);
  os() << ToCString(kVseleq) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vselge(DataType dt,
                          DRegister rd,
                          DRegister rn,
                          DRegister rm) {
  os().SetCurrentInstruction(kVselge, kFpNeon);
  os() << ToCString(kVselge) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vselge(DataType dt,
                          SRegister rd,
                          SRegister rn,
                          SRegister rm) {
  os().SetCurrentInstruction(kVselge, kFpNeon);
  os() << ToCString(kVselge) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vselgt(DataType dt,
                          DRegister rd,
                          DRegister rn,
                          DRegister rm) {
  os().SetCurrentInstruction(kVselgt, kFpNeon);
  os() << ToCString(kVselgt) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vselgt(DataType dt,
                          SRegister rd,
                          SRegister rn,
                          SRegister rm) {
  os().SetCurrentInstruction(kVselgt, kFpNeon);
  os() << ToCString(kVselgt) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vselvs(DataType dt,
                          DRegister rd,
                          DRegister rn,
                          DRegister rm) {
  os().SetCurrentInstruction(kVselvs, kFpNeon);
  os() << ToCString(kVselvs) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vselvs(DataType dt,
                          SRegister rd,
                          SRegister rn,
                          SRegister rm) {
  os().SetCurrentInstruction(kVselvs, kFpNeon);
  os() << ToCString(kVselvs) << dt << " " << rd << ", " << rn << ", " << rm;
}

void Disassembler::vshl(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVshl, kFpNeon);
  os() << ToCString(kVshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vshl(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVshl, kFpNeon);
  os() << ToCString(kVshl) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vshll(Condition cond,
                         DataType dt,
                         QRegister rd,
                         DRegister rm,
                         const DOperand& operand) {
  os().SetCurrentInstruction(kVshll, kFpNeon);
  os() << ToCString(kVshll) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm << ", " << operand;
}

void Disassembler::vshr(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVshr, kFpNeon);
  os() << ToCString(kVshr) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vshr(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVshr, kFpNeon);
  os() << ToCString(kVshr) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vshrn(Condition cond,
                         DataType dt,
                         DRegister rd,
                         QRegister rm,
                         const QOperand& operand) {
  os().SetCurrentInstruction(kVshrn, kFpNeon);
  os() << ToCString(kVshrn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm << ", " << operand;
}

void Disassembler::vsli(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVsli, kFpNeon);
  os() << ToCString(kVsli) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vsli(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVsli, kFpNeon);
  os() << ToCString(kVsli) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vsqrt(Condition cond,
                         DataType dt,
                         SRegister rd,
                         SRegister rm) {
  os().SetCurrentInstruction(kVsqrt, kFpNeon);
  os() << ToCString(kVsqrt) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vsqrt(Condition cond,
                         DataType dt,
                         DRegister rd,
                         DRegister rm) {
  os().SetCurrentInstruction(kVsqrt, kFpNeon);
  os() << ToCString(kVsqrt) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vsra(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVsra, kFpNeon);
  os() << ToCString(kVsra) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vsra(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVsra, kFpNeon);
  os() << ToCString(kVsra) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vsri(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm,
                        const DOperand& operand) {
  os().SetCurrentInstruction(kVsri, kFpNeon);
  os() << ToCString(kVsri) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vsri(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm,
                        const QOperand& operand) {
  os().SetCurrentInstruction(kVsri, kFpNeon);
  os() << ToCString(kVsri) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rm) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rm << ", " << operand;
}

void Disassembler::vst1(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVst1, kFpNeon);
  os() << ToCString(kVst1) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVst1Location, operand);
}

void Disassembler::vst2(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVst2, kFpNeon);
  os() << ToCString(kVst2) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVst2Location, operand);
}

void Disassembler::vst3(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVst3, kFpNeon);
  os() << ToCString(kVst3) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVst3Location, operand);
}

void Disassembler::vst3(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kVst3, kFpNeon);
  os() << ToCString(kVst3) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintMemOperand(kVst3Location, operand);
}

void Disassembler::vst4(Condition cond,
                        DataType dt,
                        const NeonRegisterList& nreglist,
                        const AlignedMemOperand& operand) {
  os().SetCurrentInstruction(kVst4, kFpNeon);
  os() << ToCString(kVst4) << ConditionPrinter(it_block_, cond) << dt << " "
       << nreglist << ", " << PrintAlignedMemOperand(kVst4Location, operand);
}

void Disassembler::vstm(Condition cond,
                        DataType dt,
                        Register rn,
                        WriteBack write_back,
                        DRegisterList dreglist) {
  os().SetCurrentInstruction(kVstm, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVstm) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << dreglist;
}

void Disassembler::vstm(Condition cond,
                        DataType dt,
                        Register rn,
                        WriteBack write_back,
                        SRegisterList sreglist) {
  os().SetCurrentInstruction(kVstm, kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVstm) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << sreglist;
}

void Disassembler::vstmdb(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          DRegisterList dreglist) {
  os().SetCurrentInstruction(kVstmdb,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVstmdb) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << dreglist;
}

void Disassembler::vstmdb(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          SRegisterList sreglist) {
  os().SetCurrentInstruction(kVstmdb,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVstmdb) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << sreglist;
}

void Disassembler::vstmia(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          DRegisterList dreglist) {
  os().SetCurrentInstruction(kVstmia,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVstmia) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << dreglist;
}

void Disassembler::vstmia(Condition cond,
                          DataType dt,
                          Register rn,
                          WriteBack write_back,
                          SRegisterList sreglist) {
  os().SetCurrentInstruction(kVstmia,
                             kLoadStore | kLoadStoreMultiple | kFpNeon);
  os() << ToCString(kVstmia) << ConditionPrinter(it_block_, cond) << dt << " "
       << rn << write_back << ", " << sreglist;
}

void Disassembler::vstr(Condition cond,
                        DataType dt,
                        DRegister rd,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kVstr, kFpNeon);
  os() << ToCString(kVstr) << ConditionPrinter(it_block_, cond)
       << DtPrinter(dt, Untyped64) << " " << rd << ", "
       << PrintMemOperand(kStoreDoublePrecisionLocation, operand);
}

void Disassembler::vstr(Condition cond,
                        DataType dt,
                        SRegister rd,
                        const MemOperand& operand) {
  os().SetCurrentInstruction(kVstr, kFpNeon);
  os() << ToCString(kVstr) << ConditionPrinter(it_block_, cond)
       << DtPrinter(dt, Untyped32) << " " << rd << ", "
       << PrintMemOperand(kStoreSinglePrecisionLocation, operand);
}

void Disassembler::vsub(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVsub, kFpNeon);
  os() << ToCString(kVsub) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vsub(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVsub, kFpNeon);
  os() << ToCString(kVsub) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vsub(
    Condition cond, DataType dt, SRegister rd, SRegister rn, SRegister rm) {
  os().SetCurrentInstruction(kVsub, kFpNeon);
  os() << ToCString(kVsub) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vsubhn(
    Condition cond, DataType dt, DRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVsubhn, kFpNeon);
  os() << ToCString(kVsubhn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vsubl(
    Condition cond, DataType dt, QRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVsubl, kFpNeon);
  os() << ToCString(kVsubl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rn << ", " << rm;
}

void Disassembler::vsubw(
    Condition cond, DataType dt, QRegister rd, QRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVsubw, kFpNeon);
  os() << ToCString(kVsubw) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vswp(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVswp, kFpNeon);
  os() << ToCString(kVswp) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vswp(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVswp, kFpNeon);
  os() << ToCString(kVswp) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vtbl(Condition cond,
                        DataType dt,
                        DRegister rd,
                        const NeonRegisterList& nreglist,
                        DRegister rm) {
  os().SetCurrentInstruction(kVtbl, kFpNeon);
  os() << ToCString(kVtbl) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << nreglist << ", " << rm;
}

void Disassembler::vtbx(Condition cond,
                        DataType dt,
                        DRegister rd,
                        const NeonRegisterList& nreglist,
                        DRegister rm) {
  os().SetCurrentInstruction(kVtbx, kFpNeon);
  os() << ToCString(kVtbx) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << nreglist << ", " << rm;
}

void Disassembler::vtrn(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVtrn, kFpNeon);
  os() << ToCString(kVtrn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vtrn(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVtrn, kFpNeon);
  os() << ToCString(kVtrn) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vtst(
    Condition cond, DataType dt, DRegister rd, DRegister rn, DRegister rm) {
  os().SetCurrentInstruction(kVtst, kFpNeon);
  os() << ToCString(kVtst) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vtst(
    Condition cond, DataType dt, QRegister rd, QRegister rn, QRegister rm) {
  os().SetCurrentInstruction(kVtst, kFpNeon);
  os() << ToCString(kVtst) << ConditionPrinter(it_block_, cond) << dt;
  os() << " ";
  if (!rd.Is(rn) || !use_short_hand_form_) {
    os() << rd << ", ";
  }
  os() << rn << ", " << rm;
}

void Disassembler::vuzp(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVuzp, kFpNeon);
  os() << ToCString(kVuzp) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vuzp(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVuzp, kFpNeon);
  os() << ToCString(kVuzp) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vzip(Condition cond,
                        DataType dt,
                        DRegister rd,
                        DRegister rm) {
  os().SetCurrentInstruction(kVzip, kFpNeon);
  os() << ToCString(kVzip) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::vzip(Condition cond,
                        DataType dt,
                        QRegister rd,
                        QRegister rm) {
  os().SetCurrentInstruction(kVzip, kFpNeon);
  os() << ToCString(kVzip) << ConditionPrinter(it_block_, cond) << dt << " "
       << rd << ", " << rm;
}

void Disassembler::yield(Condition cond, EncodingSize size) {
  os().SetCurrentInstruction(kYield, kNoAttribute);
  os() << ToCString(kYield) << ConditionPrinter(it_block_, cond) << size;
}

int Disassembler::T32Size(uint32_t instr) {
  if ((instr & 0xe0000000) == 0xe0000000) {
    switch (instr & 0x08000000) {
      case 0x00000000:
        if ((instr & 0x10000000) == 0x10000000) return 4;
        return 2;
      case 0x08000000:
        return 4;
      default:
        return 2;
    }
  }
  return 2;
}

void Disassembler::DecodeT32(uint32_t instr) {
  T32CodeAddressIncrementer incrementer(instr, &code_address_);
  ITBlockScope it_scope(&it_block_);

  switch (instr & 0xe0000000) {
    case 0x00000000: {
      // 0x00000000
      switch (instr & 0x18000000) {
        case 0x18000000: {
          // 0x18000000
          switch (instr & 0x06000000) {
            case 0x00000000: {
              // 0x18000000
              unsigned rd = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              unsigned rm = (instr >> 22) & 0x7;
              if (InITBlock()) {
                // ADD<c>{<q>} <Rd>, <Rn>, <Rm> ; T1
                add(CurrentCond(),
                    Narrow,
                    Register(rd),
                    Register(rn),
                    Register(rm));
              } else {
                VIXL_ASSERT(OutsideITBlock());
                // ADDS{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                adds(Condition::None(),
                     Narrow,
                     Register(rd),
                     Register(rn),
                     Register(rm));
              }
              break;
            }
            case 0x02000000: {
              // 0x1a000000
              unsigned rd = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              unsigned rm = (instr >> 22) & 0x7;
              if (InITBlock()) {
                // SUB<c>{<q>} <Rd>, <Rn>, <Rm> ; T1
                sub(CurrentCond(),
                    Narrow,
                    Register(rd),
                    Register(rn),
                    Register(rm));
              } else {
                VIXL_ASSERT(OutsideITBlock());
                // SUBS{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                subs(Condition::None(),
                     Narrow,
                     Register(rd),
                     Register(rn),
                     Register(rm));
              }
              break;
            }
            case 0x04000000: {
              // 0x1c000000
              unsigned rd = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              uint32_t imm = (instr >> 22) & 0x7;
              if (InITBlock()) {
                // ADD<c>{<q>} <Rd>, <Rn>, #<imm3> ; T1
                add(CurrentCond(), Narrow, Register(rd), Register(rn), imm);
              } else {
                VIXL_ASSERT(OutsideITBlock());
                // ADDS{<q>} <Rd>, <Rn>, #<imm3> ; T1
                adds(Condition::None(),
                     Narrow,
                     Register(rd),
                     Register(rn),
                     imm);
              }
              break;
            }
            case 0x06000000: {
              // 0x1e000000
              unsigned rd = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              uint32_t imm = (instr >> 22) & 0x7;
              if (InITBlock()) {
                // SUB<c>{<q>} <Rd>, <Rn>, #<imm3> ; T1
                sub(CurrentCond(), Narrow, Register(rd), Register(rn), imm);
              } else {
                VIXL_ASSERT(OutsideITBlock());
                // SUBS{<q>} <Rd>, <Rn>, #<imm3> ; T1
                subs(Condition::None(),
                     Narrow,
                     Register(rd),
                     Register(rn),
                     imm);
              }
              break;
            }
          }
          break;
        }
        default: {
          if (((instr & 0x18000000) == 0x18000000)) {
            UnallocatedT32(instr);
            return;
          }
          if (((Uint32((instr >> 27)) & Uint32(0x3)) == Uint32(0x2)) &&
              InITBlock()) {
            unsigned rd = (instr >> 16) & 0x7;
            unsigned rm = (instr >> 19) & 0x7;
            uint32_t amount = (instr >> 22) & 0x1f;
            if (amount == 0) amount = 32;
            // ASR<c>{<q>} {<Rd>}, <Rm>, #<imm> ; T2
            asr(CurrentCond(), Narrow, Register(rd), Register(rm), amount);
            return;
          }
          if (((Uint32((instr >> 27)) & Uint32(0x3)) == Uint32(0x2)) &&
              !InITBlock()) {
            unsigned rd = (instr >> 16) & 0x7;
            unsigned rm = (instr >> 19) & 0x7;
            uint32_t amount = (instr >> 22) & 0x1f;
            if (amount == 0) amount = 32;
            // ASRS{<q>} {<Rd>}, <Rm>, #<imm> ; T2
            asrs(Condition::None(), Narrow, Register(rd), Register(rm), amount);
            return;
          }
          if (((Uint32((instr >> 27)) & Uint32(0x3)) == Uint32(0x0)) &&
              ((instr & 0x07c00000) != 0x00000000) && InITBlock()) {
            unsigned rd = (instr >> 16) & 0x7;
            unsigned rm = (instr >> 19) & 0x7;
            uint32_t amount = (instr >> 22) & 0x1f;
            // LSL<c>{<q>} {<Rd>}, <Rm>, #<imm> ; T2
            lsl(CurrentCond(), Narrow, Register(rd), Register(rm), amount);
            return;
          }
          if (((Uint32((instr >> 27)) & Uint32(0x3)) == Uint32(0x0)) &&
              ((instr & 0x07c00000) != 0x00000000) && !InITBlock()) {
            unsigned rd = (instr >> 16) & 0x7;
            unsigned rm = (instr >> 19) & 0x7;
            uint32_t amount = (instr >> 22) & 0x1f;
            // LSLS{<q>} {<Rd>}, <Rm>, #<imm> ; T2
            lsls(Condition::None(), Narrow, Register(rd), Register(rm), amount);
            return;
          }
          if (((Uint32((instr >> 27)) & Uint32(0x3)) == Uint32(0x1)) &&
              InITBlock()) {
            unsigned rd = (instr >> 16) & 0x7;
            unsigned rm = (instr >> 19) & 0x7;
            uint32_t amount = (instr >> 22) & 0x1f;
            if (amount == 0) amount = 32;
            // LSR<c>{<q>} {<Rd>}, <Rm>, #<imm> ; T2
            lsr(CurrentCond(), Narrow, Register(rd), Register(rm), amount);
            return;
          }
          if (((Uint32((instr >> 27)) & Uint32(0x3)) == Uint32(0x1)) &&
              !InITBlock()) {
            unsigned rd = (instr >> 16) & 0x7;
            unsigned rm = (instr >> 19) & 0x7;
            uint32_t amount = (instr >> 22) & 0x1f;
            if (amount == 0) amount = 32;
            // LSRS{<q>} {<Rd>}, <Rm>, #<imm> ; T2
            lsrs(Condition::None(), Narrow, Register(rd), Register(rm), amount);
            return;
          }
          unsigned rd = (instr >> 16) & 0x7;
          unsigned rm = (instr >> 19) & 0x7;
          ImmediateShiftOperand shift_operand((instr >> 27) & 0x3,
                                              (instr >> 22) & 0x1f);
          if (InITBlock()) {
            // MOV<c>{<q>} <Rd>, <Rm> {, <shift> #<amount> } ; T2
            mov(CurrentCond(),
                Narrow,
                Register(rd),
                Operand(Register(rm),
                        shift_operand.GetType(),
                        shift_operand.GetAmount()));
          } else {
            VIXL_ASSERT(OutsideITBlock());
            // MOVS{<q>} <Rd>, <Rm> {, <shift> #<amount> } ; T2
            movs(Condition::None(),
                 Narrow,
                 Register(rd),
                 Operand(Register(rm),
                         shift_operand.GetType(),
                         shift_operand.GetAmount()));
          }
          break;
        }
      }
      break;
    }
    case 0x20000000: {
      // 0x20000000
      switch (instr & 0x18000000) {
        case 0x00000000: {
          // 0x20000000
          unsigned rd = (instr >> 24) & 0x7;
          uint32_t imm = (instr >> 16) & 0xff;
          if (InITBlock()) {
            // MOV<c>{<q>} <Rd>, #<imm8> ; T1
            mov(CurrentCond(), Narrow, Register(rd), imm);
          } else {
            VIXL_ASSERT(OutsideITBlock());
            // MOVS{<q>} <Rd>, #<imm8> ; T1
            movs(Condition::None(), Narrow, Register(rd), imm);
          }
          break;
        }
        case 0x08000000: {
          // 0x28000000
          unsigned rn = (instr >> 24) & 0x7;
          uint32_t imm = (instr >> 16) & 0xff;
          // CMP{<c>}{<q>} <Rn>, #<imm8> ; T1
          cmp(CurrentCond(), Narrow, Register(rn), imm);
          break;
        }
        case 0x10000000: {
          // 0x30000000
          unsigned rd = (instr >> 24) & 0x7;
          uint32_t imm = (instr >> 16) & 0xff;
          if (InITBlock() && ((imm <= 7))) {
            // ADD<c>{<q>} <Rdn>, #<imm8> ; T2
            add(CurrentCond(), Register(rd), imm);
          } else if (InITBlock() && ((imm > 7))) {
            // ADD<c>{<q>} {<Rdn>}, <Rdn>, #<imm8> ; T2
            add(CurrentCond(), Narrow, Register(rd), Register(rd), imm);
          } else if (OutsideITBlock() && ((imm <= 7))) {
            // ADDS{<q>} <Rdn>, #<imm8> ; T2
            adds(Register(rd), imm);
          } else if (OutsideITBlock() && ((imm > 7))) {
            // ADDS{<q>} {<Rdn>}, <Rdn>, #<imm8> ; T2
            adds(Condition::None(), Narrow, Register(rd), Register(rd), imm);
          } else {
            UnallocatedT32(instr);
          }
          break;
        }
        case 0x18000000: {
          // 0x38000000
          unsigned rd = (instr >> 24) & 0x7;
          uint32_t imm = (instr >> 16) & 0xff;
          if (InITBlock() && ((imm <= 7))) {
            // SUB<c>{<q>} <Rdn>, #<imm8> ; T2
            sub(CurrentCond(), Register(rd), imm);
          } else if (InITBlock() && ((imm > 7))) {
            // SUB<c>{<q>} {<Rdn>}, <Rdn>, #<imm8> ; T2
            sub(CurrentCond(), Narrow, Register(rd), Register(rd), imm);
          } else if (OutsideITBlock() && ((imm <= 7))) {
            // SUBS{<q>} <Rdn>, #<imm8> ; T2
            subs(Register(rd), imm);
          } else if (OutsideITBlock() && ((imm > 7))) {
            // SUBS{<q>} {<Rdn>}, <Rdn>, #<imm8> ; T2
            subs(Condition::None(), Narrow, Register(rd), Register(rd), imm);
          } else {
            UnallocatedT32(instr);
          }
          break;
        }
      }
      break;
    }
    case 0x40000000: {
      // 0x40000000
      switch (instr & 0x18000000) {
        case 0x00000000: {
          // 0x40000000
          switch (instr & 0x07000000) {
            case 0x00000000: {
              // 0x40000000
              switch (instr & 0x00c00000) {
                case 0x00000000: {
                  // 0x40000000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // AND<c>{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    and_(CurrentCond(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rm));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // ANDS{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    ands(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rm));
                  }
                  break;
                }
                case 0x00400000: {
                  // 0x40400000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // EOR<c>{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    eor(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rm));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // EORS{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    eors(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rm));
                  }
                  break;
                }
                case 0x00800000: {
                  // 0x40800000
                  if (InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // LSL<c>{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    lsl(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rs));
                    return;
                  }
                  if (!InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // LSLS{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    lsls(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rs));
                    return;
                  }
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 16) & 0x7;
                  unsigned rs = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // MOV<c>{<q>} <Rdm>, <Rdm>, LSL <Rs> ; T1
                    mov(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Operand(Register(rm), LSL, Register(rs)));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // MOVS{<q>} <Rdm>, <Rdm>, LSL <Rs> ; T1
                    movs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Operand(Register(rm), LSL, Register(rs)));
                  }
                  break;
                }
                case 0x00c00000: {
                  // 0x40c00000
                  if (InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // LSR<c>{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    lsr(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rs));
                    return;
                  }
                  if (!InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // LSRS{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    lsrs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rs));
                    return;
                  }
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 16) & 0x7;
                  unsigned rs = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // MOV<c>{<q>} <Rdm>, <Rdm>, LSR <Rs> ; T1
                    mov(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Operand(Register(rm), LSR, Register(rs)));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // MOVS{<q>} <Rdm>, <Rdm>, LSR <Rs> ; T1
                    movs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Operand(Register(rm), LSR, Register(rs)));
                  }
                  break;
                }
              }
              break;
            }
            case 0x01000000: {
              // 0x41000000
              switch (instr & 0x00c00000) {
                case 0x00000000: {
                  // 0x41000000
                  if (InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // ASR<c>{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    asr(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rs));
                    return;
                  }
                  if (!InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // ASRS{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    asrs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rs));
                    return;
                  }
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 16) & 0x7;
                  unsigned rs = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // MOV<c>{<q>} <Rdm>, <Rdm>, ASR <Rs> ; T1
                    mov(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Operand(Register(rm), ASR, Register(rs)));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // MOVS{<q>} <Rdm>, <Rdm>, ASR <Rs> ; T1
                    movs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Operand(Register(rm), ASR, Register(rs)));
                  }
                  break;
                }
                case 0x00400000: {
                  // 0x41400000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // ADC<c>{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    adc(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rm));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // ADCS{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    adcs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rm));
                  }
                  break;
                }
                case 0x00800000: {
                  // 0x41800000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // SBC<c>{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    sbc(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rm));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // SBCS{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    sbcs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rm));
                  }
                  break;
                }
                case 0x00c00000: {
                  // 0x41c00000
                  if (InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // ROR<c>{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    ror(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rs));
                    return;
                  }
                  if (!InITBlock()) {
                    unsigned rd = (instr >> 16) & 0x7;
                    unsigned rs = (instr >> 19) & 0x7;
                    // RORS{<q>} {<Rdm>}, <Rdm>, <Rs> ; T1
                    rors(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rs));
                    return;
                  }
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 16) & 0x7;
                  unsigned rs = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // MOV<c>{<q>} <Rdm>, <Rdm>, ROR <Rs> ; T1
                    mov(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Operand(Register(rm), ROR, Register(rs)));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // MOVS{<q>} <Rdm>, <Rdm>, ROR <Rs> ; T1
                    movs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Operand(Register(rm), ROR, Register(rs)));
                  }
                  break;
                }
              }
              break;
            }
            case 0x02000000: {
              // 0x42000000
              switch (instr & 0x00c00000) {
                case 0x00000000: {
                  // 0x42000000
                  unsigned rn = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  // TST{<c>}{<q>} <Rn>, <Rm> ; T1
                  tst(CurrentCond(), Narrow, Register(rn), Register(rm));
                  break;
                }
                case 0x00400000: {
                  // 0x42400000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rn = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // RSB<c>{<q>} {<Rd>}, <Rn>, #0 ; T1
                    rsb(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rn),
                        UINT32_C(0));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // RSBS{<q>} {<Rd>}, <Rn>, #0 ; T1
                    rsbs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rn),
                         UINT32_C(0));
                  }
                  break;
                }
                case 0x00800000: {
                  // 0x42800000
                  unsigned rn = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  // CMP{<c>}{<q>} <Rn>, <Rm> ; T1
                  cmp(CurrentCond(), Narrow, Register(rn), Register(rm));
                  break;
                }
                case 0x00c00000: {
                  // 0x42c00000
                  unsigned rn = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  // CMN{<c>}{<q>} <Rn>, <Rm> ; T1
                  cmn(CurrentCond(), Narrow, Register(rn), Register(rm));
                  break;
                }
              }
              break;
            }
            case 0x03000000: {
              // 0x43000000
              switch (instr & 0x00c00000) {
                case 0x00000000: {
                  // 0x43000000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // ORR<c>{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    orr(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rm));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // ORRS{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    orrs(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rm));
                  }
                  break;
                }
                case 0x00400000: {
                  // 0x43400000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rn = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // MUL<c>{<q>} <Rdm>, <Rn>, {<Rdm>} ; T1
                    mul(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rn),
                        Register(rd));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // MULS{<q>} <Rdm>, <Rn>, {<Rdm>} ; T1
                    muls(Condition::None(),
                         Register(rd),
                         Register(rn),
                         Register(rd));
                  }
                  break;
                }
                case 0x00800000: {
                  // 0x43800000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // BIC<c>{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    bic(CurrentCond(),
                        Narrow,
                        Register(rd),
                        Register(rd),
                        Register(rm));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // BICS{<q>} {<Rdn>}, <Rdn>, <Rm> ; T1
                    bics(Condition::None(),
                         Narrow,
                         Register(rd),
                         Register(rd),
                         Register(rm));
                  }
                  break;
                }
                case 0x00c00000: {
                  // 0x43c00000
                  unsigned rd = (instr >> 16) & 0x7;
                  unsigned rm = (instr >> 19) & 0x7;
                  if (InITBlock()) {
                    // MVN<c>{<q>} <Rd>, <Rm> ; T1
                    mvn(CurrentCond(), Narrow, Register(rd), Register(rm));
                  } else {
                    VIXL_ASSERT(OutsideITBlock());
                    // MVNS{<q>} <Rd>, <Rm> ; T1
                    mvns(Condition::None(), Narrow, Register(rd), Register(rm));
                  }
                  break;
                }
              }
              break;
            }
            case 0x04000000: {
              // 0x44000000
              switch (instr & 0x00780000) {
                case 0x00680000: {
                  // 0x44680000
                  unsigned rd = ((instr >> 16) & 0x7) | ((instr >> 20) & 0x8);
                  // ADD{<c>}{<q>} {<Rdm>}, SP, <Rdm> ; T1
                  add(CurrentCond(), Narrow, Register(rd), sp, Register(rd));
                  break;
                }
                default: {
                  switch (instr & 0x00870000) {
                    case 0x00850000: {
                      // 0x44850000
                      if (((instr & 0x780000) == 0x680000)) {
                        UnallocatedT32(instr);
                        return;
                      }
                      unsigned rm = (instr >> 19) & 0xf;
                      // ADD{<c>}{<q>} {SP}, SP, <Rm> ; T2
                      add(CurrentCond(), Narrow, sp, sp, Register(rm));
                      break;
                    }
                    default: {
                      if (((instr & 0x780000) == 0x680000) ||
                          ((instr & 0x870000) == 0x850000)) {
                        UnallocatedT32(instr);
                        return;
                      }
                      unsigned rd =
                          ((instr >> 16) & 0x7) | ((instr >> 20) & 0x8);
                      unsigned rm = (instr >> 19) & 0xf;
                      if (InITBlock()) {
                        // ADD<c>{<q>} <Rdn>, <Rm> ; T2
                        add(CurrentCond(), Register(rd), Register(rm));
                      } else {
                        // ADD{<c>}{<q>} {<Rdn>}, <Rdn>, <Rm> ; T2
                        add(CurrentCond(),
                            Narrow,
                            Register(rd),
                            Register(rd),
                            Register(rm));
                      }
                      break;
                    }
                  }
                  break;
                }
              }
              break;
            }
            case 0x05000000: {
              // 0x45000000
              unsigned rn = ((instr >> 16) & 0x7) | ((instr >> 20) & 0x8);
              unsigned rm = (instr >> 19) & 0xf;
              // CMP{<c>}{<q>} <Rn>, <Rm> ; T2
              cmp(CurrentCond(), Narrow, Register(rn), Register(rm));
              break;
            }
            case 0x06000000: {
              // 0x46000000
              unsigned rd = ((instr >> 16) & 0x7) | ((instr >> 20) & 0x8);
              unsigned rm = (instr >> 19) & 0xf;
              // MOV{<c>}{<q>} <Rd>, <Rm> ; T1
              mov(CurrentCond(), Narrow, Register(rd), Register(rm));
              break;
            }
            case 0x07000000: {
              // 0x47000000
              switch (instr & 0x00800000) {
                case 0x00000000: {
                  // 0x47000000
                  unsigned rm = (instr >> 19) & 0xf;
                  // BX{<c>}{<q>} <Rm> ; T1
                  bx(CurrentCond(), Register(rm));
                  if (((instr & 0xff870000) != 0x47000000)) {
                    UnpredictableT32(instr);
                  }
                  break;
                }
                case 0x00800000: {
                  // 0x47800000
                  unsigned rm = (instr >> 19) & 0xf;
                  // BLX{<c>}{<q>} <Rm> ; T1
                  blx(CurrentCond(), Register(rm));
                  if (((instr & 0xff870000) != 0x47800000)) {
                    UnpredictableT32(instr);
                  }
                  break;
                }
              }
              break;
            }
          }
          break;
        }
        case 0x08000000: {
          // 0x48000000
          unsigned rt = (instr >> 24) & 0x7;
          int32_t imm = ((instr >> 16) & 0xff) << 2;
          Location location(imm, kT32PcDelta);
          // LDR{<c>}{<q>} <Rt>, <label> ; T1
          ldr(CurrentCond(), Narrow, Register(rt), &location);
          break;
        }
        case 0x10000000: {
          // 0x50000000
          switch (instr & 0x06000000) {
            case 0x00000000: {
              // 0x50000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // STR{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              str(CurrentCond(),
                  Narrow,
                  Register(rt),
                  MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
            case 0x02000000: {
              // 0x52000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // STRH{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              strh(CurrentCond(),
                   Narrow,
                   Register(rt),
                   MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
            case 0x04000000: {
              // 0x54000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // STRB{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              strb(CurrentCond(),
                   Narrow,
                   Register(rt),
                   MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
            case 0x06000000: {
              // 0x56000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // LDRSB{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              ldrsb(CurrentCond(),
                    Narrow,
                    Register(rt),
                    MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
          }
          break;
        }
        case 0x18000000: {
          // 0x58000000
          switch (instr & 0x06000000) {
            case 0x00000000: {
              // 0x58000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // LDR{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              ldr(CurrentCond(),
                  Narrow,
                  Register(rt),
                  MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
            case 0x02000000: {
              // 0x5a000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // LDRH{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              ldrh(CurrentCond(),
                   Narrow,
                   Register(rt),
                   MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
            case 0x04000000: {
              // 0x5c000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // LDRB{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              ldrb(CurrentCond(),
                   Narrow,
                   Register(rt),
                   MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
            case 0x06000000: {
              // 0x5e000000
              unsigned rt = (instr >> 16) & 0x7;
              unsigned rn = (instr >> 19) & 0x7;
              Sign sign(plus);
              unsigned rm = (instr >> 22) & 0x7;
              AddrMode addrmode = Offset;
              // LDRSH{<c>}{<q>} <Rt>, [<Rn>, #{+}<Rm>] ; T1
              ldrsh(CurrentCond(),
                    Narrow,
                    Register(rt),
                    MemOperand(Register(rn), sign, Register(rm), addrmode));
              break;
            }
          }
          break;
        }
      }
      break;
    }
    case 0x60000000: {
      // 0x60000000
      switch (instr & 0x18000000) {
        case 0x00000000: {
          // 0x60000000
          unsigned rt = (instr >> 16) & 0x7;
          unsigned rn = (instr >> 19) & 0x7;
          int32_t offset = ((instr >> 22) & 0x1f) << 2;
          // STR{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1
          str(CurrentCond(),
              Narrow,
              Register(rt),
              MemOperand(Register(rn), plus, offset, Offset));
          break;
        }
        case 0x08000000: {
          // 0x68000000
          unsigned rt = (instr >> 16) & 0x7;
          unsigned rn = (instr >> 19) & 0x7;
          int32_t offset = ((instr >> 22) & 0x1f) << 2;
          // LDR{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1
          ldr(CurrentCond(),
              Narrow,
              Register(rt),
              MemOperand(Register(rn), plus, offset, Offset));
          break;
        }
        case 0x10000000: {
          // 0x70000000
          unsigned rt = (instr >> 16) & 0x7;
          unsigned rn = (instr >> 19) & 0x7;
          int32_t offset = (instr >> 22) & 0x1f;
          // STRB{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1
          strb(CurrentCond(),
               Narrow,
               Register(rt),
               MemOperand(Register(rn), plus, offset, Offset));
          break;
        }
        case 0x18000000: {
          // 0x78000000
          unsigned rt = (instr >> 16) & 0x7;
          unsigned rn = (instr >> 19) & 0x7;
          int32_t offset = (instr >> 22) & 0x1f;
          // LDRB{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1
          ldrb(CurrentCond(),
               Narrow,
               Register(rt),
               MemOperand(Register(rn), plus, offset, Offset));
          break;
        }
      }
      break;
    }
    case 0x80000000: {
      // 0x80000000
      switch (instr & 0x18000000) {
        case 0x00000000: {
          // 0x80000000
          unsigned rt = (instr >> 16) & 0x7;
          unsigned rn = (instr >> 19) & 0x7;
          int32_t offset = ((instr >> 22) & 0x1f) << 1;
          // STRH{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1
          strh(CurrentCond(),
               Narrow,
               Register(rt),
               MemOperand(Register(rn), plus, offset, Offset));
          break;
        }
        case 0x08000000: {
          // 0x88000000
          unsigned rt = (instr >> 16) & 0x7;
          unsigned rn = (instr >> 19) & 0x7;
          int32_t offset = ((instr >> 22) & 0x1f) << 1;
          // LDRH{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1
          ldrh(CurrentCond(),
               Narrow,
               Register(rt),
               MemOperand(Register(rn), plus, offset, Offset));
          break;
        }
        case 0x10000000: {
          // 0x90000000
          unsigned rt = (instr >> 24) & 0x7;
          int32_t offset = ((instr >> 16) & 0xff) << 2;
          // STR{<c>}{<q>} <Rt>, [SP{, #{+}<imm>}] ; T2
          str(CurrentCond(),
              Narrow,
              Register(rt),
              MemOperand(sp, plus, offset, Offset));
          break;
        }
        case 0x18000000: {
          // 0x98000000
          unsigned rt = (instr >> 24) & 0x7;
          int32_t offset = ((instr >> 16) & 0xff) << 2;
          // LDR{<c>}{<q>} <Rt>, [SP{, #{+}<imm>}] ; T2
          ldr(CurrentCond(),
              Narrow,
              Register(rt),
              MemOperand(sp, plus, offset, Offset));
          break;
        }
      }
      break;
    }
    case 0xa0000000: {
      // 0xa0000000
      switch (instr & 0x18000000) {
        case 0x00000000: {
          // 0xa0000000
          unsigned rd = (instr >> 24) & 0x7;
          int32_t imm = ((instr >> 16) & 0xff) << 2;
          Location location(imm, kT32PcDelta);
          // ADR{<c>}{<q>} <Rd>, <label> ; T1
          adr(CurrentCond(), Narrow, Register(rd), &location);
          break;
        }
        case 0x08000000: {
          // 0xa8000000
          unsigned rd = (instr >> 24) & 0x7;
          uint32_t imm = ((instr >> 16) & 0xff) << 2;
          // ADD{<c>}{<q>} <Rd>, SP, #<imm8> ; T1
          add(CurrentCond(), Narrow, Register(rd), sp, imm);
          break;
        }
        case 0x10000000: {
          // 0xb0000000
          switch (instr & 0x04000000) {
            case 0x00000000: {
              // 0xb0000000
              switch (instr & 0x01000000) {
                case 0x00000000: {
                  // 0xb0000000
                  switch (instr & 0x02800000) {
                    case 0x00000000: {
                      // 0xb0000000
                      uint32_t imm = ((instr >> 16) & 0x7f) << 2;
                      // ADD{<c>}{<q>} {SP}, SP, #<imm7> ; T2
                      add(CurrentCond(), Narrow, sp, sp, imm);
                      break;
                    }
                    case 0x00800000: {
                      // 0xb0800000
                      uint32_t imm = ((instr >> 16) & 0x7f) << 2;
                      // SUB{<c>}{<q>} {SP}, SP, #<imm7> ; T1
                      sub(CurrentCond(), Narrow, sp, sp, imm);
                      break;
                    }
                    case 0x02000000: {
                      // 0xb2000000
                      switch (instr & 0x00400000) {
                        case 0x00000000: {
                          // 0xb2000000
                          unsigned rd = (instr >> 16) & 0x7;
                          unsigned rm = (instr >> 19) & 0x7;
                          // SXTH{<c>}{<q>} {<Rd>}, <Rm> ; T1
                          sxth(CurrentCond(),
                               Narrow,
                               Register(rd),
                               Register(rm));
                          break;
                        }
                        case 0x00400000: {
                          // 0xb2400000
                          unsigned rd = (instr >> 16) & 0x7;
                          unsigned rm = (instr >> 19) & 0x7;
                          // SXTB{<c>}{<q>} {<Rd>}, <Rm> ; T1
                          sxtb(CurrentCond(),
                               Narrow,
                               Register(rd),
                               Register(rm));
                          break;
                        }
                      }
                      break;
                    }
                    case 0x02800000: {
                      // 0xb2800000
                      switch (instr & 0x00400000) {
                        case 0x00000000: {
                          // 0xb2800000
                          unsigned rd = (instr >> 16) & 0x7;
                          unsigned rm = (instr >> 19) & 0x7;
                          // UXTH{<c>}{<q>} {<Rd>}, <Rm> ; T1
                          uxth(CurrentCond(),
                               Narrow,
                               Register(rd),
                               Register(rm));
                          break;
                        }
                        case 0x00400000: {
                          // 0xb2c00000
                          unsigned rd = (instr >> 16) & 0x7;
                          unsigned rm = (instr >> 19) & 0x7;
                          // UXTB{<c>}{<q>} {<Rd>}, <Rm> ; T1
                          uxtb(CurrentCond(),
                               Narrow,
                               Register(rd),
                               Register(rm));
                          break;
                        }
                      }
                      break;
                    }
                  }
                  break;
                }
                case 0x01000000: {
                  // 0xb1000000
                  unsigned rn = (instr >> 16) & 0x7;
                  int32_t imm =
                      (((instr >> 19) & 0x1f) | ((instr >> 20) & 0x20)) << 1;
                  Location location(imm, kT32PcDelta);
                  // CBZ{<q>} <Rn>, <label> ; T1
                  cbz(Register(rn), &location);
                  break;
                }
              }
              break;
            }
            case 0x04000000: {
              // 0xb4000000
              switch (instr & 0x02000000) {
                case 0x00000000: {
                  // 0xb4000000
                  RegisterList registers((((instr >> 24) & 0x1) << kLRRegNum) |
                                         ((instr >> 16) & 0xff));
                  // PUSH{<c>}{<q>} <registers> ; T1
                  push(CurrentCond(), Narrow, registers);
                  break;
                }
                case 0x02000000: {
                  // 0xb6000000
                  switch (instr & 0x01e00000) {
                    case 0x00400000: {
                      // 0xb6400000
                      UnimplementedT32_16("SETEND", instr);
                      break;
                    }
                    case 0x00600000: {
                      // 0xb6600000
                      switch (instr & 0x00100000) {
                        case 0x00000000: {
                          // 0xb6600000
                          UnimplementedT32_16("CPSIE", instr);
                          break;
                        }
                        case 0x00100000: {
                          // 0xb6700000
                          UnimplementedT32_16("CPSID", instr);
                          break;
                        }
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
              }
              break;
            }
          }
          break;
        }
        case 0x18000000: {
          // 0xb8000000
          switch (instr & 0x04000000) {
            case 0x00000000: {
              // 0xb8000000
              switch (instr & 0x01000000) {
                case 0x00000000: {
                  // 0xb8000000
                  switch (instr & 0x02c00000) {
                    case 0x02000000: {
                      // 0xba000000
                      unsigned rd = (instr >> 16) & 0x7;
                      unsigned rm = (instr >> 19) & 0x7;
                      // REV{<c>}{<q>} <Rd>, <Rm> ; T1
                      rev(CurrentCond(), Narrow, Register(rd), Register(rm));
                      break;
                    }
                    case 0x02400000: {
                      // 0xba400000
                      unsigned rd = (instr >> 16) & 0x7;
                      unsigned rm = (instr >> 19) & 0x7;
                      // REV16{<c>}{<q>} <Rd>, <Rm> ; T1
                      rev16(CurrentCond(), Narrow, Register(rd), Register(rm));
                      break;
                    }
                    case 0x02800000: {
                      // 0xba800000
                      uint32_t imm = (instr >> 16) & 0x3f;
                      // HLT{<q>} {#}<imm> ; T1
                      hlt(Condition::None(), imm);
                      break;
                    }
                    case 0x02c00000: {
                      // 0xbac00000
                      unsigned rd = (instr >> 16) & 0x7;
                      unsigned rm = (instr >> 19) & 0x7;
                      // REVSH{<c>}{<q>} <Rd>, <Rm> ; T1
                      revsh(CurrentCond(), Narrow, Register(rd), Register(rm));
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x01000000: {
                  // 0xb9000000
                  unsigned rn = (instr >> 16) & 0x7;
                  int32_t imm =
                      (((instr >> 19) & 0x1f) | ((instr >> 20) & 0x20)) << 1;
                  Location location(imm, kT32PcDelta);
                  // CBNZ{<q>} <Rn>, <label> ; T1
                  cbnz(Register(rn), &location);
                  break;
                }
              }
              break;
            }
            case 0x04000000: {
              // 0xbc000000
              switch (instr & 0x02000000) {
                case 0x00000000: {
                  // 0xbc000000
                  RegisterList registers((((instr >> 24) & 0x1) << kPCRegNum) |
                                         ((instr >> 16) & 0xff));
                  // POP{<c>}{<q>} <registers> ; T1
                  pop(CurrentCond(), Narrow, registers);
                  break;
                }
                case 0x02000000: {
                  // 0xbe000000
                  switch (instr & 0x01000000) {
                    case 0x00000000: {
                      // 0xbe000000
                      uint32_t imm = (instr >> 16) & 0xff;
                      // BKPT{<q>} {#}<imm> ; T1
                      bkpt(Condition::None(), imm);
                      break;
                    }
                    case 0x01000000: {
                      // 0xbf000000
                      switch (instr & 0x000f0000) {
                        case 0x00000000: {
                          // 0xbf000000
                          switch (instr & 0x00f00000) {
                            case 0x00000000: {
                              // 0xbf000000
                              // NOP{<c>}{<q>} ; T1
                              nop(CurrentCond(), Narrow);
                              break;
                            }
                            case 0x00100000: {
                              // 0xbf100000
                              // YIELD{<c>}{<q>} ; T1
                              yield(CurrentCond(), Narrow);
                              break;
                            }
                            case 0x00200000: {
                              // 0xbf200000
                              UnimplementedT32_16("WFE", instr);
                              break;
                            }
                            case 0x00300000: {
                              // 0xbf300000
                              UnimplementedT32_16("WFI", instr);
                              break;
                            }
                            case 0x00400000: {
                              // 0xbf400000
                              UnimplementedT32_16("SEV", instr);
                              break;
                            }
                            case 0x00500000: {
                              // 0xbf500000
                              UnimplementedT32_16("SEVL", instr);
                              break;
                            }
                            default:
                              UnallocatedT32(instr);
                              break;
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0x0)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned first_cond = (instr >> 20) & 0xf;
                          unsigned mask = (instr >> 16) & 0xf;
                          bool was_in_it_block = InITBlock();
                          SetIT(Condition(first_cond), mask);
                          it(Condition(first_cond), mask);
                          if (was_in_it_block || (first_cond == 15) ||
                              ((first_cond == al) &&
                               (BitCount(Uint32(mask)) != 1))) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                  }
                  break;
                }
              }
              break;
            }
          }
          break;
        }
      }
      break;
    }
    case 0xc0000000: {
      // 0xc0000000
      switch (instr & 0x10000000) {
        case 0x00000000: {
          // 0xc0000000
          switch (instr & 0x08000000) {
            case 0x00000000: {
              // 0xc0000000
              unsigned rn = (instr >> 24) & 0x7;
              RegisterList registers(((instr >> 16) & 0xff));
              // STM{<c>}{<q>} <Rn>!, <registers> ; T1
              stm(CurrentCond(),
                  Narrow,
                  Register(rn),
                  WriteBack(WRITE_BACK),
                  registers);
              break;
            }
            case 0x08000000: {
              // 0xc8000000
              unsigned rn = (instr >> 24) & 0x7;
              RegisterList registers(((instr >> 16) & 0xff));
              // LDM{<c>}{<q>} <Rn>{!}, <registers> ; T1
              ldm(CurrentCond(),
                  Narrow,
                  Register(rn),
                  WriteBack(!registers.Includes(Register(rn))),
                  registers);
              break;
            }
          }
          break;
        }
        case 0x10000000: {
          // 0xd0000000
          switch (instr & 0x0e000000) {
            case 0x0e000000: {
              // 0xde000000
              switch (instr & 0x01000000) {
                case 0x00000000: {
                  // 0xde000000
                  uint32_t imm = (instr >> 16) & 0xff;
                  // UDF{<c>}{<q>} {#}<imm> ; T1
                  udf(CurrentCond(), Narrow, imm);
                  break;
                }
                case 0x01000000: {
                  // 0xdf000000
                  uint32_t imm = (instr >> 16) & 0xff;
                  // SVC{<c>}{<q>} {#}<imm> ; T1
                  svc(CurrentCond(), imm);
                  break;
                }
              }
              break;
            }
            default: {
              if (((instr & 0xe000000) == 0xe000000)) {
                UnallocatedT32(instr);
                return;
              }
              Condition condition((instr >> 24) & 0xf);
              int32_t imm = SignExtend<int32_t>(((instr >> 16) & 0xff) << 1, 9);
              Location location(imm, kT32PcDelta);
              // B<c>{<q>} <label> ; T1
              b(condition, Narrow, &location);
              if (InITBlock()) {
                UnpredictableT32(instr);
              }
              break;
            }
          }
          break;
        }
      }
      break;
    }
    case 0xe0000000: {
      // 0xe0000000
      switch (instr & 0x08000000) {
        case 0x00000000: {
          // 0xe0000000
          switch (instr & 0x10000000) {
            case 0x00000000: {
              // 0xe0000000
              int32_t imm =
                  SignExtend<int32_t>(((instr >> 16) & 0x7ff) << 1, 12);
              Location location(imm, kT32PcDelta);
              // B{<c>}{<q>} <label> ; T2
              b(CurrentCond(), Narrow, &location);
              if (!OutsideITBlockOrLast()) {
                UnpredictableT32(instr);
              }
              break;
            }
            case 0x10000000: {
              // 0xf0000000
              switch (instr & 0x00008000) {
                case 0x00000000: {
                  // 0xf0000000
                  switch (instr & 0x03f00000) {
                    case 0x00000000: {
                      // 0xf0000000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // AND{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      and_(CurrentCond(),
                           Best,
                           Register(rd),
                           Register(rn),
                           imm);
                      break;
                    }
                    case 0x00100000: {
                      // 0xf0100000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xf0100f00
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // TST{<c>}{<q>} <Rn>, #<const> ; T1
                          tst(CurrentCond(), Best, Register(rn), imm);
                          break;
                        }
                        default: {
                          if (((instr & 0xf00) == 0xf00)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // ANDS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                          ands(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               imm);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00200000: {
                      // 0xf0200000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // BIC{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      bic(CurrentCond(), Best, Register(rd), Register(rn), imm);
                      break;
                    }
                    case 0x00300000: {
                      // 0xf0300000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // BICS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      bics(CurrentCond(),
                           Best,
                           Register(rd),
                           Register(rn),
                           imm);
                      break;
                    }
                    case 0x00400000: {
                      // 0xf0400000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xf04f0000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              ((rd < kNumberOfT32LowRegisters) &&
                               (imm <= 255))) {
                            // MOV<c>.W <Rd>, #<const> ; T2
                            mov(CurrentCond(), Wide, Register(rd), imm);
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // MOV{<c>}{<q>} <Rd>, #<const> ; T2
                            mov(CurrentCond(), Best, Register(rd), imm);
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // ORR{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                          orr(CurrentCond(),
                              Best,
                              Register(rd),
                              Register(rn),
                              imm);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00500000: {
                      // 0xf0500000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xf05f0000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          if (OutsideITBlock() &&
                              (instr & 0x00100000) == 0x00100000 &&
                              ((rd < kNumberOfT32LowRegisters) &&
                               (imm <= 255))) {
                            // MOVS.W <Rd>, #<const> ; T2
                            movs(Condition::None(), Wide, Register(rd), imm);
                          } else if ((instr & 0x00100000) == 0x00100000) {
                            // MOVS{<c>}{<q>} <Rd>, #<const> ; T2
                            movs(CurrentCond(), Best, Register(rd), imm);
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // ORRS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                          orrs(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               imm);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00600000: {
                      // 0xf0600000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xf06f0000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // MVN{<c>}{<q>} <Rd>, #<const> ; T1
                          mvn(CurrentCond(), Best, Register(rd), imm);
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // ORN{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                          orn(CurrentCond(), Register(rd), Register(rn), imm);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00700000: {
                      // 0xf0700000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xf07f0000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // MVNS{<c>}{<q>} <Rd>, #<const> ; T1
                          mvns(CurrentCond(), Best, Register(rd), imm);
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // ORNS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                          orns(CurrentCond(), Register(rd), Register(rn), imm);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00800000: {
                      // 0xf0800000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // EOR{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      eor(CurrentCond(), Best, Register(rd), Register(rn), imm);
                      break;
                    }
                    case 0x00900000: {
                      // 0xf0900000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xf0900f00
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // TEQ{<c>}{<q>} <Rn>, #<const> ; T1
                          teq(CurrentCond(), Register(rn), imm);
                          break;
                        }
                        default: {
                          if (((instr & 0xf00) == 0xf00)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // EORS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                          eors(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               imm);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01000000: {
                      // 0xf1000000
                      switch (instr & 0x000f0000) {
                        case 0x000d0000: {
                          // 0xf10d0000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          if ((instr & 0x00100000) == 0x00000000 &&
                              (((rd < kNumberOfT32LowRegisters) &&
                                ((imm <= 1020) && ((imm & 3) == 0))) ||
                               ((rd == sp.GetCode()) &&
                                ((imm <= 508) && ((imm & 3) == 0))))) {
                            // ADD{<c>}.W {<Rd>}, SP, #<const> ; T3
                            add(CurrentCond(), Wide, Register(rd), sp, imm);
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // ADD{<c>}{<q>} {<Rd>}, SP, #<const> ; T3
                            add(CurrentCond(), Best, Register(rd), sp, imm);
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0xd0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              (((rd < kNumberOfT32LowRegisters) &&
                                (rn < kNumberOfT32LowRegisters) &&
                                (imm <= 7)) ||
                               ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                                (imm <= 255)))) {
                            // ADD<c>.W {<Rd>}, <Rn>, #<const> ; T3
                            add(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rn),
                                imm);
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // ADD{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T3
                            add(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                imm);
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01100000: {
                      // 0xf1100000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xf1100f00
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          // CMN{<c>}{<q>} <Rn>, #<const> ; T1
                          cmn(CurrentCond(), Best, Register(rn), imm);
                          break;
                        }
                        default: {
                          switch (instr & 0x000f0000) {
                            case 0x000d0000: {
                              // 0xf11d0000
                              if (((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              uint32_t imm = ImmediateT32::Decode(
                                  (instr & 0xff) | ((instr >> 4) & 0x700) |
                                  ((instr >> 15) & 0x800));
                              // ADDS{<c>}{<q>} {<Rd>}, SP, #<const> ; T3
                              adds(CurrentCond(), Best, Register(rd), sp, imm);
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xd0000) ||
                                  ((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              uint32_t imm = ImmediateT32::Decode(
                                  (instr & 0xff) | ((instr >> 4) & 0x700) |
                                  ((instr >> 15) & 0x800));
                              if (OutsideITBlock() &&
                                  (instr & 0x00100000) == 0x00100000 &&
                                  (((rd < kNumberOfT32LowRegisters) &&
                                    (rn < kNumberOfT32LowRegisters) &&
                                    (imm <= 7)) ||
                                   ((rd == rn) &&
                                    (rd < kNumberOfT32LowRegisters) &&
                                    (imm <= 255)))) {
                                // ADDS.W {<Rd>}, <Rn>, #<const> ; T3
                                adds(Condition::None(),
                                     Wide,
                                     Register(rd),
                                     Register(rn),
                                     imm);
                              } else if ((instr & 0x00100000) == 0x00100000) {
                                // ADDS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T3
                                adds(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Register(rn),
                                     imm);
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01400000: {
                      // 0xf1400000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // ADC{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      adc(CurrentCond(), Best, Register(rd), Register(rn), imm);
                      break;
                    }
                    case 0x01500000: {
                      // 0xf1500000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // ADCS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      adcs(CurrentCond(),
                           Best,
                           Register(rd),
                           Register(rn),
                           imm);
                      break;
                    }
                    case 0x01600000: {
                      // 0xf1600000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // SBC{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      sbc(CurrentCond(), Best, Register(rd), Register(rn), imm);
                      break;
                    }
                    case 0x01700000: {
                      // 0xf1700000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      // SBCS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T1
                      sbcs(CurrentCond(),
                           Best,
                           Register(rd),
                           Register(rn),
                           imm);
                      break;
                    }
                    case 0x01a00000: {
                      // 0xf1a00000
                      switch (instr & 0x000f0000) {
                        case 0x000d0000: {
                          // 0xf1ad0000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          if ((instr & 0x00100000) == 0x00000000 &&
                              ((rd == sp.GetCode()) &&
                               ((imm <= 508) && ((imm & 3) == 0)))) {
                            // SUB{<c>}.W {<Rd>}, SP, #<const> ; T2
                            sub(CurrentCond(), Wide, Register(rd), sp, imm);
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // SUB{<c>}{<q>} {<Rd>}, SP, #<const> ; T2
                            sub(CurrentCond(), Best, Register(rd), sp, imm);
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0xd0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              (((rd < kNumberOfT32LowRegisters) &&
                                (rn < kNumberOfT32LowRegisters) &&
                                (imm <= 7)) ||
                               ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                                (imm <= 255)))) {
                            // SUB<c>.W {<Rd>}, <Rn>, #<const> ; T3
                            sub(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rn),
                                imm);
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // SUB{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T3
                            sub(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                imm);
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01b00000: {
                      // 0xf1b00000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xf1b00f00
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = ImmediateT32::Decode(
                              (instr & 0xff) | ((instr >> 4) & 0x700) |
                              ((instr >> 15) & 0x800));
                          if ((rn < kNumberOfT32LowRegisters) && (imm <= 255)) {
                            // CMP{<c>}.W <Rn>, #<const> ; T2
                            cmp(CurrentCond(), Wide, Register(rn), imm);
                          } else {
                            // CMP{<c>}{<q>} <Rn>, #<const> ; T2
                            cmp(CurrentCond(), Best, Register(rn), imm);
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000f0000) {
                            case 0x000d0000: {
                              // 0xf1bd0000
                              if (((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              uint32_t imm = ImmediateT32::Decode(
                                  (instr & 0xff) | ((instr >> 4) & 0x700) |
                                  ((instr >> 15) & 0x800));
                              // SUBS{<c>}{<q>} {<Rd>}, SP, #<const> ; T2
                              subs(CurrentCond(), Best, Register(rd), sp, imm);
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xd0000) ||
                                  ((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              uint32_t imm = ImmediateT32::Decode(
                                  (instr & 0xff) | ((instr >> 4) & 0x700) |
                                  ((instr >> 15) & 0x800));
                              if (OutsideITBlock() &&
                                  (instr & 0x00100000) == 0x00100000 &&
                                  (((rd < kNumberOfT32LowRegisters) &&
                                    (rn < kNumberOfT32LowRegisters) &&
                                    (imm <= 7)) ||
                                   ((rd == rn) &&
                                    (rd < kNumberOfT32LowRegisters) &&
                                    (imm <= 255)))) {
                                // SUBS.W {<Rd>}, <Rn>, #<const> ; T3
                                subs(Condition::None(),
                                     Wide,
                                     Register(rd),
                                     Register(rn),
                                     imm);
                              } else if ((instr & 0x00100000) == 0x00100000) {
                                // SUBS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T3
                                subs(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Register(rn),
                                     imm);
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01c00000: {
                      // 0xf1c00000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      if (InITBlock() && (instr & 0x00100000) == 0x00000000 &&
                          (imm == 0) &&
                          ((rd < kNumberOfT32LowRegisters) &&
                           (rn < kNumberOfT32LowRegisters) && (imm == 0))) {
                        // RSB<c>.W {<Rd>}, <Rn>, #0 ; T2
                        rsb(CurrentCond(),
                            Wide,
                            Register(rd),
                            Register(rn),
                            UINT32_C(0));
                      } else if ((instr & 0x00100000) == 0x00000000) {
                        // RSB{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T2
                        rsb(CurrentCond(),
                            Best,
                            Register(rd),
                            Register(rn),
                            imm);
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x01d00000: {
                      // 0xf1d00000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t imm = ImmediateT32::Decode(
                          (instr & 0xff) | ((instr >> 4) & 0x700) |
                          ((instr >> 15) & 0x800));
                      if (OutsideITBlock() &&
                          (instr & 0x00100000) == 0x00100000 && (imm == 0) &&
                          ((rd < kNumberOfT32LowRegisters) &&
                           (rn < kNumberOfT32LowRegisters) && (imm == 0))) {
                        // RSBS.W {<Rd>}, <Rn>, #0 ; T2
                        rsbs(Condition::None(),
                             Wide,
                             Register(rd),
                             Register(rn),
                             UINT32_C(0));
                      } else if ((instr & 0x00100000) == 0x00100000) {
                        // RSBS{<c>}{<q>} {<Rd>}, <Rn>, #<const> ; T2
                        rsbs(CurrentCond(),
                             Best,
                             Register(rd),
                             Register(rn),
                             imm);
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x02000000: {
                      // 0xf2000000
                      switch (instr & 0x000d0000) {
                        case 0x000d0000: {
                          // 0xf20d0000
                          switch (instr & 0x00020000) {
                            case 0x00000000: {
                              // 0xf20d0000
                              unsigned rd = (instr >> 8) & 0xf;
                              uint32_t imm = (instr & 0xff) |
                                             ((instr >> 4) & 0x700) |
                                             ((instr >> 15) & 0x800);
                              if (((rd >= kNumberOfT32LowRegisters) ||
                                   ((imm > 1020) || ((imm & 3) != 0))) &&
                                  ((rd != sp.GetCode()) ||
                                   ((imm > 508) || ((imm & 3) != 0))) &&
                                  (!ImmediateT32::IsImmediateT32(imm))) {
                                // ADD{<c>}{<q>} {<Rd>}, SP, #<imm12> ; T4
                                add(CurrentCond(), Best, Register(rd), sp, imm);
                              } else {
                                // ADDW{<c>}{<q>} {<Rd>}, SP, #<imm12> ; T4
                                addw(CurrentCond(), Register(rd), sp, imm);
                              }
                              break;
                            }
                            case 0x00020000: {
                              // 0xf20f0000
                              unsigned rd = (instr >> 8) & 0xf;
                              int32_t imm = (instr & 0xff) |
                                            ((instr >> 4) & 0x700) |
                                            ((instr >> 15) & 0x800);
                              Location location(imm, kT32PcDelta);
                              if ((imm >= 0) && (imm <= 4095) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (imm >= 0) && (imm <= 1020) &&
                                   ((imm & 3) == 0))) {
                                // ADR{<c>}.W <Rd>, <label> ; T3
                                adr(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    &location);
                              } else if ((imm >= 0) && (imm <= 4095)) {
                                // ADR{<c>}{<q>} <Rd>, <label> ; T3
                                adr(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    &location);
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xd0000) == 0xd0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = (instr & 0xff) |
                                         ((instr >> 4) & 0x700) |
                                         ((instr >> 15) & 0x800);
                          if ((InITBlock() ||
                               (rd >= kNumberOfT32LowRegisters) ||
                               (rn >= kNumberOfT32LowRegisters) || (imm > 7)) &&
                              (InITBlock() || (rd != rn) ||
                               (rd >= kNumberOfT32LowRegisters) ||
                               (imm > 255)) &&
                              (!ImmediateT32::IsImmediateT32(imm))) {
                            // ADD{<c>}{<q>} {<Rd>}, <Rn>, #<imm12> ; T4
                            add(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                imm);
                          } else {
                            // ADDW{<c>}{<q>} {<Rd>}, <Rn>, #<imm12> ; T4
                            addw(CurrentCond(),
                                 Register(rd),
                                 Register(rn),
                                 imm);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x02400000: {
                      // 0xf2400000
                      unsigned rd = (instr >> 8) & 0xf;
                      uint32_t imm = (instr & 0xff) | ((instr >> 4) & 0x700) |
                                     ((instr >> 15) & 0x800) |
                                     ((instr >> 4) & 0xf000);
                      if ((InITBlock() || (rd >= kNumberOfT32LowRegisters) ||
                           (imm > 255)) &&
                          (!ImmediateT32::IsImmediateT32(imm))) {
                        // MOV{<c>}{<q>} <Rd>, #<imm16> ; T3
                        mov(CurrentCond(), Best, Register(rd), imm);
                      } else {
                        // MOVW{<c>}{<q>} <Rd>, #<imm16> ; T3
                        movw(CurrentCond(), Register(rd), imm);
                      }
                      break;
                    }
                    case 0x02a00000: {
                      // 0xf2a00000
                      switch (instr & 0x000d0000) {
                        case 0x000d0000: {
                          // 0xf2ad0000
                          switch (instr & 0x00020000) {
                            case 0x00000000: {
                              // 0xf2ad0000
                              unsigned rd = (instr >> 8) & 0xf;
                              uint32_t imm = (instr & 0xff) |
                                             ((instr >> 4) & 0x700) |
                                             ((instr >> 15) & 0x800);
                              if (((rd != sp.GetCode()) ||
                                   ((imm > 508) || ((imm & 3) != 0))) &&
                                  (!ImmediateT32::IsImmediateT32(imm))) {
                                // SUB{<c>}{<q>} {<Rd>}, SP, #<imm12> ; T3
                                sub(CurrentCond(), Best, Register(rd), sp, imm);
                              } else {
                                // SUBW{<c>}{<q>} {<Rd>}, SP, #<imm12> ; T3
                                subw(CurrentCond(), Register(rd), sp, imm);
                              }
                              break;
                            }
                            case 0x00020000: {
                              // 0xf2af0000
                              if (((((Uint32((instr >> 26)) & Uint32(0x1))
                                     << 11) |
                                    ((Uint32((instr >> 12)) & Uint32(0x7))
                                     << 8) |
                                    (Uint32(instr) & Uint32(0xff))) ==
                                   Uint32(0x0))) {
                                unsigned rd = (instr >> 8) & 0xf;
                                uint32_t imm = (instr & 0xff) |
                                               ((instr >> 4) & 0x700) |
                                               ((instr >> 15) & 0x800);
                                // SUB{<c>}{<q>} <Rd>, PC, #<imm12> ; T2
                                sub(CurrentCond(), Best, Register(rd), pc, imm);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              int32_t imm = (instr & 0xff) |
                                            ((instr >> 4) & 0x700) |
                                            ((instr >> 15) & 0x800);
                              Location location(-imm, kT32PcDelta);
                              // ADR{<c>}{<q>} <Rd>, <label> ; T2
                              adr(CurrentCond(), Best, Register(rd), &location);
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xd0000) == 0xd0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t imm = (instr & 0xff) |
                                         ((instr >> 4) & 0x700) |
                                         ((instr >> 15) & 0x800);
                          if ((InITBlock() ||
                               (rd >= kNumberOfT32LowRegisters) ||
                               (rn >= kNumberOfT32LowRegisters) || (imm > 7)) &&
                              (InITBlock() || (rd != rn) ||
                               (rd >= kNumberOfT32LowRegisters) ||
                               (imm > 255)) &&
                              (!ImmediateT32::IsImmediateT32(imm))) {
                            // SUB{<c>}{<q>} {<Rd>}, <Rn>, #<imm12> ; T4
                            sub(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                imm);
                          } else {
                            // SUBW{<c>}{<q>} {<Rd>}, <Rn>, #<imm12> ; T4
                            subw(CurrentCond(),
                                 Register(rd),
                                 Register(rn),
                                 imm);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x02c00000: {
                      // 0xf2c00000
                      unsigned rd = (instr >> 8) & 0xf;
                      uint32_t imm = (instr & 0xff) | ((instr >> 4) & 0x700) |
                                     ((instr >> 15) & 0x800) |
                                     ((instr >> 4) & 0xf000);
                      // MOVT{<c>}{<q>} <Rd>, #<imm16> ; T1
                      movt(CurrentCond(), Register(rd), imm);
                      break;
                    }
                    case 0x03000000: {
                      // 0xf3000000
                      unsigned rd = (instr >> 8) & 0xf;
                      uint32_t imm = (instr & 0x1f) + 1;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t amount =
                          ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                      // SSAT{<c>}{<q>} <Rd>, #<imm>, <Rn> {, LSL #<amount> } ; T1 NOLINT(whitespace/line_length)
                      ssat(CurrentCond(),
                           Register(rd),
                           imm,
                           Operand(Register(rn), LSL, amount));
                      if (((instr & 0xfff08020) != 0xf3000000)) {
                        UnpredictableT32(instr);
                      }
                      break;
                    }
                    case 0x03200000: {
                      // 0xf3200000
                      switch (instr & 0x000070c0) {
                        case 0x00000000: {
                          // 0xf3200000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = (instr & 0xf) + 1;
                          unsigned rn = (instr >> 16) & 0xf;
                          // SSAT16{<c>}{<q>} <Rd>, #<imm>, <Rn> ; T1
                          ssat16(CurrentCond(),
                                 Register(rd),
                                 imm,
                                 Register(rn));
                          if (((instr & 0xfff0f0f0) != 0xf3200000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70c0) == 0x0)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = (instr & 0x1f) + 1;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t amount =
                              ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                          // SSAT{<c>}{<q>} <Rd>, #<imm>, <Rn>, ASR #<amount> ; T1 NOLINT(whitespace/line_length)
                          ssat(CurrentCond(),
                               Register(rd),
                               imm,
                               Operand(Register(rn), ASR, amount));
                          if (((instr & 0xfff08020) != 0xf3200000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x03400000: {
                      // 0xf3400000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t lsb =
                          ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                      uint32_t widthm1 = instr & 0x1f;
                      uint32_t width = widthm1 + 1;
                      // SBFX{<c>}{<q>} <Rd>, <Rn>, #<lsb>, #<width> ; T1
                      sbfx(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           lsb,
                           width);
                      if (((instr & 0xfff08020) != 0xf3400000)) {
                        UnpredictableT32(instr);
                      }
                      break;
                    }
                    case 0x03600000: {
                      // 0xf3600000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xf36f0000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t lsb =
                              ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                          uint32_t msb = instr & 0x1f;
                          uint32_t width = msb - lsb + 1;
                          // BFC{<c>}{<q>} <Rd>, #<lsb>, #<width> ; T1
                          bfc(CurrentCond(), Register(rd), lsb, width);
                          if (((instr & 0xffff8020) != 0xf36f0000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t lsb =
                              ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                          uint32_t msb = instr & 0x1f;
                          uint32_t width = msb - lsb + 1;
                          // BFI{<c>}{<q>} <Rd>, <Rn>, #<lsb>, #<width> ; T1
                          bfi(CurrentCond(),
                              Register(rd),
                              Register(rn),
                              lsb,
                              width);
                          if (((instr & 0xfff08020) != 0xf3600000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x03800000: {
                      // 0xf3800000
                      unsigned rd = (instr >> 8) & 0xf;
                      uint32_t imm = instr & 0x1f;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t amount =
                          ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                      // USAT{<c>}{<q>} <Rd>, #<imm>, <Rn> {, LSL #<amount> } ; T1 NOLINT(whitespace/line_length)
                      usat(CurrentCond(),
                           Register(rd),
                           imm,
                           Operand(Register(rn), LSL, amount));
                      if (((instr & 0xfff08020) != 0xf3800000)) {
                        UnpredictableT32(instr);
                      }
                      break;
                    }
                    case 0x03a00000: {
                      // 0xf3a00000
                      switch (instr & 0x000070c0) {
                        case 0x00000000: {
                          // 0xf3a00000
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = instr & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          // USAT16{<c>}{<q>} <Rd>, #<imm>, <Rn> ; T1
                          usat16(CurrentCond(),
                                 Register(rd),
                                 imm,
                                 Register(rn));
                          if (((instr & 0xfff0f0f0) != 0xf3a00000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70c0) == 0x0)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          uint32_t imm = instr & 0x1f;
                          unsigned rn = (instr >> 16) & 0xf;
                          uint32_t amount =
                              ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                          // USAT{<c>}{<q>} <Rd>, #<imm>, <Rn>, ASR #<amount> ; T1 NOLINT(whitespace/line_length)
                          usat(CurrentCond(),
                               Register(rd),
                               imm,
                               Operand(Register(rn), ASR, amount));
                          if (((instr & 0xfff08020) != 0xf3a00000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x03c00000: {
                      // 0xf3c00000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      uint32_t lsb =
                          ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                      uint32_t widthm1 = instr & 0x1f;
                      uint32_t width = widthm1 + 1;
                      // UBFX{<c>}{<q>} <Rd>, <Rn>, #<lsb>, #<width> ; T1
                      ubfx(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           lsb,
                           width);
                      if (((instr & 0xfff08020) != 0xf3c00000)) {
                        UnpredictableT32(instr);
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x00008000: {
                  // 0xf0008000
                  switch (instr & 0x00005000) {
                    case 0x00000000: {
                      // 0xf0008000
                      switch (instr & 0x03800000) {
                        case 0x03800000: {
                          // 0xf3808000
                          switch (instr & 0x04600000) {
                            case 0x00000000: {
                              // 0xf3808000
                              switch (instr & 0x00000020) {
                                case 0x00000000: {
                                  // 0xf3808000
                                  unsigned spec_reg = ((instr >> 8) & 0xf) |
                                                      ((instr >> 16) & 0x10);
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // MSR{<c>}{<q>} <spec_reg>, <Rn> ; T1
                                  msr(CurrentCond(),
                                      MaskedSpecialRegister(spec_reg),
                                      Register(rn));
                                  if (((instr & 0xffe0f0ff) != 0xf3808000)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000020: {
                                  // 0xf3808020
                                  UnimplementedT32_32("MSR", instr);
                                  break;
                                }
                              }
                              break;
                            }
                            case 0x00200000: {
                              // 0xf3a08000
                              switch (instr & 0x00100000) {
                                case 0x00000000: {
                                  // 0xf3a08000
                                  switch (instr & 0x00000700) {
                                    case 0x00000000: {
                                      // 0xf3a08000
                                      switch (instr & 0x000000f0) {
                                        case 0x00000000: {
                                          // 0xf3a08000
                                          switch (instr & 0x0000000f) {
                                            case 0x00000000: {
                                              // 0xf3a08000
                                              // NOP{<c>}.W ; T2
                                              nop(CurrentCond(), Wide);
                                              if (((instr & 0xffffffff) !=
                                                   0xf3af8000)) {
                                                UnpredictableT32(instr);
                                              }
                                              break;
                                            }
                                            case 0x00000001: {
                                              // 0xf3a08001
                                              // YIELD{<c>}.W ; T2
                                              yield(CurrentCond(), Wide);
                                              if (((instr & 0xffffffff) !=
                                                   0xf3af8001)) {
                                                UnpredictableT32(instr);
                                              }
                                              break;
                                            }
                                            case 0x00000002: {
                                              // 0xf3a08002
                                              UnimplementedT32_32("WFE", instr);
                                              break;
                                            }
                                            case 0x00000003: {
                                              // 0xf3a08003
                                              UnimplementedT32_32("WFI", instr);
                                              break;
                                            }
                                            case 0x00000004: {
                                              // 0xf3a08004
                                              UnimplementedT32_32("SEV", instr);
                                              break;
                                            }
                                            case 0x00000005: {
                                              // 0xf3a08005
                                              UnimplementedT32_32("SEVL",
                                                                  instr);
                                              break;
                                            }
                                            default:
                                              UnallocatedT32(instr);
                                              break;
                                          }
                                          break;
                                        }
                                        case 0x000000f0: {
                                          // 0xf3a080f0
                                          UnimplementedT32_32("DBG", instr);
                                          break;
                                        }
                                        default:
                                          UnallocatedT32(instr);
                                          break;
                                      }
                                      break;
                                    }
                                    case 0x00000100: {
                                      // 0xf3a08100
                                      if ((instr & 0x000000e0) == 0x00000000) {
                                        UnimplementedT32_32("CPS", instr);
                                      } else {
                                        UnallocatedT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000400: {
                                      // 0xf3a08400
                                      if ((instr & 0x0000001f) == 0x00000000) {
                                        UnimplementedT32_32("CPSIE", instr);
                                      } else {
                                        UnallocatedT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000500: {
                                      // 0xf3a08500
                                      UnimplementedT32_32("CPSIE", instr);
                                      break;
                                    }
                                    case 0x00000600: {
                                      // 0xf3a08600
                                      if ((instr & 0x0000001f) == 0x00000000) {
                                        UnimplementedT32_32("CPSID", instr);
                                      } else {
                                        UnallocatedT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000700: {
                                      // 0xf3a08700
                                      UnimplementedT32_32("CPSID", instr);
                                      break;
                                    }
                                    default:
                                      UnallocatedT32(instr);
                                      break;
                                  }
                                  break;
                                }
                                case 0x00100000: {
                                  // 0xf3b08000
                                  switch (instr & 0x000000f0) {
                                    case 0x00000020: {
                                      // 0xf3b08020
                                      // CLREX{<c>}{<q>} ; T1
                                      clrex(CurrentCond());
                                      if (((instr & 0xffffffff) !=
                                           0xf3bf8f2f)) {
                                        UnpredictableT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000040: {
                                      // 0xf3b08040
                                      MemoryBarrier option(instr & 0xf);
                                      // DSB{<c>}{<q>} {<option>} ; T1
                                      dsb(CurrentCond(), option);
                                      if (((instr & 0xfffffff0) !=
                                           0xf3bf8f40)) {
                                        UnpredictableT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000050: {
                                      // 0xf3b08050
                                      MemoryBarrier option(instr & 0xf);
                                      // DMB{<c>}{<q>} {<option>} ; T1
                                      dmb(CurrentCond(), option);
                                      if (((instr & 0xfffffff0) !=
                                           0xf3bf8f50)) {
                                        UnpredictableT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000060: {
                                      // 0xf3b08060
                                      MemoryBarrier option(instr & 0xf);
                                      // ISB{<c>}{<q>} {<option>} ; T1
                                      isb(CurrentCond(), option);
                                      if (((instr & 0xfffffff0) !=
                                           0xf3bf8f60)) {
                                        UnpredictableT32(instr);
                                      }
                                      break;
                                    }
                                    default:
                                      UnallocatedT32(instr);
                                      break;
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                            case 0x00400000: {
                              // 0xf3c08000
                              switch (instr & 0x00100000) {
                                case 0x00000000: {
                                  // 0xf3c08000
                                  unsigned rm = (instr >> 16) & 0xf;
                                  // BXJ{<c>}{<q>} <Rm> ; T1
                                  bxj(CurrentCond(), Register(rm));
                                  if (((instr & 0xfff0ffff) != 0xf3c08f00)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00100000: {
                                  // 0xf3d08000
                                  switch (instr & 0x000000ff) {
                                    case 0x00000000: {
                                      // 0xf3d08000
                                      if ((instr & 0x000f0000) == 0x000e0000) {
                                        UnimplementedT32_32("ERET", instr);
                                      } else {
                                        UnallocatedT32(instr);
                                      }
                                      break;
                                    }
                                    default: {
                                      if (((instr & 0xff) == 0x0)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      uint32_t imm = instr & 0xff;
                                      // SUBS{<c>}{<q>} PC, LR, #<imm8> ; T5
                                      subs(CurrentCond(), Best, pc, lr, imm);
                                      if (((instr & 0xffffff00) !=
                                           0xf3de8f00)) {
                                        UnpredictableT32(instr);
                                      }
                                      break;
                                    }
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                            case 0x00600000: {
                              // 0xf3e08000
                              switch (instr & 0x00000020) {
                                case 0x00000000: {
                                  // 0xf3e08000
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned spec_reg = (instr >> 20) & 0x1;
                                  // MRS{<c>}{<q>} <Rd>, <spec_reg> ; T1
                                  mrs(CurrentCond(),
                                      Register(rd),
                                      SpecialRegister(spec_reg));
                                  if (((instr & 0xffeff0ff) != 0xf3ef8000)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000020: {
                                  // 0xf3e08020
                                  UnimplementedT32_32("MRS", instr);
                                  break;
                                }
                              }
                              break;
                            }
                            case 0x04000000: {
                              // 0xf7808000
                              switch (instr & 0x001f2fff) {
                                case 0x000f0001: {
                                  // 0xf78f8001
                                  UnimplementedT32_32("DCPS1", instr);
                                  break;
                                }
                                case 0x000f0002: {
                                  // 0xf78f8002
                                  UnimplementedT32_32("DCPS2", instr);
                                  break;
                                }
                                case 0x000f0003: {
                                  // 0xf78f8003
                                  UnimplementedT32_32("DCPS3", instr);
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                            case 0x04600000: {
                              // 0xf7e08000
                              switch (instr & 0x00102000) {
                                case 0x00000000: {
                                  // 0xf7e08000
                                  uint32_t imm =
                                      (instr & 0xfff) | ((instr >> 4) & 0xf000);
                                  // HVC{<q>} {#}<imm16> ; T1
                                  hvc(Condition::None(), imm);
                                  break;
                                }
                                case 0x00100000: {
                                  // 0xf7f08000
                                  UnimplementedT32_32("SMC", instr);
                                  break;
                                }
                                case 0x00102000: {
                                  // 0xf7f0a000
                                  uint32_t imm =
                                      (instr & 0xfff) | ((instr >> 4) & 0xf000);
                                  if ((imm <= 255)) {
                                    // UDF{<c>}.W {#}<imm> ; T2
                                    udf(CurrentCond(), Wide, imm);
                                  } else {
                                    // UDF{<c>}{<q>} {#}<imm> ; T2
                                    udf(CurrentCond(), Best, imm);
                                  }
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                            default:
                              UnallocatedT32(instr);
                              break;
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x3800000) == 0x3800000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          Condition condition((instr >> 22) & 0xf);
                          int32_t imm =
                              SignExtend<int32_t>(((instr & 0x7ff) |
                                                   ((instr >> 5) & 0x1f800) |
                                                   ((instr << 4) & 0x20000) |
                                                   ((instr << 7) & 0x40000) |
                                                   ((instr >> 7) & 0x80000))
                                                      << 1,
                                                  21);
                          Location location(imm, kT32PcDelta);
                          if ((imm >= -1048576) && (imm <= 1048574) &&
                              ((imm & 1) == 0) &&
                              ((imm >= -256) && (imm <= 254) &&
                               ((imm & 1) == 0))) {
                            // B<c>.W <label> ; T3
                            b(condition, Wide, &location);
                            if (InITBlock()) {
                              UnpredictableT32(instr);
                            }
                          } else if ((imm >= -1048576) && (imm <= 1048574) &&
                                     ((imm & 1) == 0)) {
                            // B<c>{<q>} <label> ; T3
                            b(condition, Best, &location);
                            if (InITBlock()) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00001000: {
                      // 0xf0009000
                      uint32_t encoded_imm =
                          (instr & 0x7ff) | ((instr >> 5) & 0x1ff800) |
                          ((instr << 10) & 0x200000) |
                          ((instr << 9) & 0x400000) | ((instr >> 3) & 0x800000);
                      uint32_t S = encoded_imm & (1 << 23);
                      encoded_imm ^= ((S >> 1) | (S >> 2)) ^ (3 << 21);
                      int32_t imm = SignExtend<int32_t>(encoded_imm << 1, 25);
                      Location location(imm, kT32PcDelta);
                      if ((imm >= -16777216) && (imm <= 16777214) &&
                          ((imm & 1) == 0) &&
                          ((imm >= -2048) && (imm <= 2046) &&
                           ((imm & 1) == 0))) {
                        // B{<c>}.W <label> ; T4
                        b(CurrentCond(), Wide, &location);
                      } else if ((imm >= -16777216) && (imm <= 16777214) &&
                                 ((imm & 1) == 0)) {
                        // B{<c>}{<q>} <label> ; T4
                        b(CurrentCond(), Best, &location);
                        if (!OutsideITBlockOrLast()) {
                          UnpredictableT32(instr);
                        }
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x00004000: {
                      // 0xf000c000
                      if ((instr & 0x00000001) == 0x00000000) {
                        uint32_t encoded_imm = ((instr >> 1) & 0x3ff) |
                                               ((instr >> 6) & 0xffc00) |
                                               ((instr << 9) & 0x100000) |
                                               ((instr << 8) & 0x200000) |
                                               ((instr >> 4) & 0x400000);
                        uint32_t S = encoded_imm & (1 << 22);
                        encoded_imm ^= ((S >> 1) | (S >> 2)) ^ (3 << 20);
                        int32_t imm = SignExtend<int32_t>(encoded_imm << 2, 25);
                        Location location(imm, kT32PcDelta);
                        // BLX{<c>}{<q>} <label> ; T2
                        blx(CurrentCond(), &location);
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x00005000: {
                      // 0xf000d000
                      uint32_t encoded_imm =
                          (instr & 0x7ff) | ((instr >> 5) & 0x1ff800) |
                          ((instr << 10) & 0x200000) |
                          ((instr << 9) & 0x400000) | ((instr >> 3) & 0x800000);
                      uint32_t S = encoded_imm & (1 << 23);
                      encoded_imm ^= ((S >> 1) | (S >> 2)) ^ (3 << 21);
                      int32_t imm = SignExtend<int32_t>(encoded_imm << 1, 25);
                      Location location(imm, kT32PcDelta);
                      // BL{<c>}{<q>} <label> ; T1
                      bl(CurrentCond(), &location);
                      break;
                    }
                  }
                  break;
                }
              }
              break;
            }
          }
          break;
        }
        case 0x08000000: {
          // 0xe8000000
          switch (instr & 0x06000000) {
            case 0x00000000: {
              // 0xe8000000
              switch (instr & 0x10100000) {
                case 0x00000000: {
                  // 0xe8000000
                  switch (instr & 0x01400000) {
                    case 0x00000000: {
                      // 0xe8000000
                      switch (instr & 0x00800000) {
                        case 0x00000000: {
                          // 0xe8000000
                          UnimplementedT32_32("SRSDB", instr);
                          break;
                        }
                        case 0x00800000: {
                          // 0xe8800000
                          unsigned rn = (instr >> 16) & 0xf;
                          WriteBack write_back((instr >> 21) & 0x1);
                          RegisterList registers(
                              (((instr >> 14) & 0x1) << kLRRegNum) |
                              (instr & 0x1fff));
                          if ((rn < kNumberOfT32LowRegisters) &&
                              write_back.DoesWriteBack() &&
                              ((registers.GetList() & ~0xff) == 0)) {
                            // STM{<c>}.W <Rn>{!}, <registers> ; T2
                            stm(CurrentCond(),
                                Wide,
                                Register(rn),
                                write_back,
                                registers);
                            if (((instr & 0xffd0a000) != 0xe8800000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            // STM{<c>}{<q>} <Rn>{!}, <registers> ; T2
                            stm(CurrentCond(),
                                Best,
                                Register(rn),
                                write_back,
                                registers);
                            if (((instr & 0xffd0a000) != 0xe8800000)) {
                              UnpredictableT32(instr);
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xe8400000
                      switch (instr & 0x00200000) {
                        case 0x00000000: {
                          // 0xe8400000
                          switch (instr & 0x00800000) {
                            case 0x00000000: {
                              // 0xe8400000
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              int32_t offset = (instr & 0xff) << 2;
                              // STREX{<c>}{<q>} <Rd>, <Rt>, [<Rn>{, #<imm>}] ; T1 NOLINT(whitespace/line_length)
                              strex(CurrentCond(),
                                    Register(rd),
                                    Register(rt),
                                    MemOperand(Register(rn),
                                               plus,
                                               offset,
                                               Offset));
                              break;
                            }
                            case 0x00800000: {
                              // 0xe8c00000
                              switch (instr & 0x000000f0) {
                                case 0x00000040: {
                                  // 0xe8c00040
                                  unsigned rd = instr & 0xf;
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STREXB{<c>}{<q>} <Rd>, <Rt>, [<Rn>] ; T1
                                  strexb(CurrentCond(),
                                         Register(rd),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00ff0) != 0xe8c00f40)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000050: {
                                  // 0xe8c00050
                                  unsigned rd = instr & 0xf;
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STREXH{<c>}{<q>} <Rd>, <Rt>, [<Rn>] ; T1
                                  strexh(CurrentCond(),
                                         Register(rd),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00ff0) != 0xe8c00f50)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000070: {
                                  // 0xe8c00070
                                  unsigned rd = instr & 0xf;
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rt2 = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STREXD{<c>}{<q>} <Rd>, <Rt>, <Rt2>, [<Rn>] ; T1 NOLINT(whitespace/line_length)
                                  strexd(CurrentCond(),
                                         Register(rd),
                                         Register(rt),
                                         Register(rt2),
                                         MemOperand(Register(rn), Offset));
                                  break;
                                }
                                case 0x00000080: {
                                  // 0xe8c00080
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STLB{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  stlb(CurrentCond(),
                                       Register(rt),
                                       MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8c00f8f)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000090: {
                                  // 0xe8c00090
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STLH{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  stlh(CurrentCond(),
                                       Register(rt),
                                       MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8c00f9f)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000a0: {
                                  // 0xe8c000a0
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STL{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  stl(CurrentCond(),
                                      Register(rt),
                                      MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8c00faf)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000c0: {
                                  // 0xe8c000c0
                                  unsigned rd = instr & 0xf;
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STLEXB{<c>}{<q>} <Rd>, <Rt>, [<Rn>] ; T1
                                  stlexb(CurrentCond(),
                                         Register(rd),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00ff0) != 0xe8c00fc0)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000d0: {
                                  // 0xe8c000d0
                                  unsigned rd = instr & 0xf;
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STLEXH{<c>}{<q>} <Rd>, <Rt>, [<Rn>] ; T1
                                  stlexh(CurrentCond(),
                                         Register(rd),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00ff0) != 0xe8c00fd0)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000e0: {
                                  // 0xe8c000e0
                                  unsigned rd = instr & 0xf;
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STLEX{<c>}{<q>} <Rd>, <Rt>, [<Rn>] ; T1
                                  stlex(CurrentCond(),
                                        Register(rd),
                                        Register(rt),
                                        MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00ff0) != 0xe8c00fe0)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000f0: {
                                  // 0xe8c000f0
                                  unsigned rd = instr & 0xf;
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rt2 = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // STLEXD{<c>}{<q>} <Rd>, <Rt>, <Rt2>, [<Rn>] ; T1 NOLINT(whitespace/line_length)
                                  stlexd(CurrentCond(),
                                         Register(rd),
                                         Register(rt),
                                         Register(rt2),
                                         MemOperand(Register(rn), Offset));
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00200000: {
                          // 0xe8600000
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rt2 = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 23) & 0x1) == 0) ? minus
                                                                 : plus);
                          int32_t offset = (instr & 0xff) << 2;
                          // STRD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>], #{+/-}<imm> ; T1 NOLINT(whitespace/line_length)
                          strd(CurrentCond(),
                               Register(rt),
                               Register(rt2),
                               MemOperand(Register(rn),
                                          sign,
                                          offset,
                                          PostIndex));
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01000000: {
                      // 0xe9000000
                      switch (instr & 0x00800000) {
                        case 0x00000000: {
                          // 0xe9000000
                          if (((Uint32((instr >> 21)) & Uint32(0x1)) ==
                               Uint32(0x1)) &&
                              ((Uint32((instr >> 16)) & Uint32(0xf)) ==
                               Uint32(0xd)) &&
                              (BitCount(((Uint32((instr >> 14)) & Uint32(0x1))
                                         << 13) |
                                        (Uint32(instr) & Uint32(0x1fff))) >
                               Int64(1))) {
                            RegisterList registers(
                                (((instr >> 14) & 0x1) << kLRRegNum) |
                                (instr & 0x1fff));
                            if (registers.IsR0toR7orLR()) {
                              // PUSH{<c>}.W <registers> ; T1
                              push(CurrentCond(), Wide, registers);
                              if (((instr & 0xffffa000) != 0xe92d0000)) {
                                UnpredictableT32(instr);
                              }
                            } else {
                              // PUSH{<c>}{<q>} <registers> ; T1
                              push(CurrentCond(), Best, registers);
                              if (((instr & 0xffffa000) != 0xe92d0000)) {
                                UnpredictableT32(instr);
                              }
                            }
                            return;
                          }
                          unsigned rn = (instr >> 16) & 0xf;
                          WriteBack write_back((instr >> 21) & 0x1);
                          RegisterList registers(
                              (((instr >> 14) & 0x1) << kLRRegNum) |
                              (instr & 0x1fff));
                          // STMDB{<c>}{<q>} <Rn>{!}, <registers> ; T1
                          stmdb(CurrentCond(),
                                Best,
                                Register(rn),
                                write_back,
                                registers);
                          if (((instr & 0xffd0a000) != 0xe9000000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        case 0x00800000: {
                          // 0xe9800000
                          UnimplementedT32_32("SRS{IA}", instr);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01400000: {
                      // 0xe9400000
                      switch (instr & 0x00200000) {
                        case 0x00000000: {
                          // 0xe9400000
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rt2 = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 23) & 0x1) == 0) ? minus
                                                                 : plus);
                          int32_t offset = (instr & 0xff) << 2;
                          // STRD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>{, #{+/-}<imm>}] ; T1 NOLINT(whitespace/line_length)
                          strd(CurrentCond(),
                               Register(rt),
                               Register(rt2),
                               MemOperand(Register(rn), sign, offset, Offset));
                          break;
                        }
                        case 0x00200000: {
                          // 0xe9600000
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rt2 = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 23) & 0x1) == 0) ? minus
                                                                 : plus);
                          int32_t offset = (instr & 0xff) << 2;
                          // STRD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>{, #{+/-}<imm>}]! ; T1 NOLINT(whitespace/line_length)
                          strd(CurrentCond(),
                               Register(rt),
                               Register(rt2),
                               MemOperand(Register(rn),
                                          sign,
                                          offset,
                                          PreIndex));
                          break;
                        }
                      }
                      break;
                    }
                  }
                  break;
                }
                case 0x00100000: {
                  // 0xe8100000
                  switch (instr & 0x00400000) {
                    case 0x00000000: {
                      // 0xe8100000
                      switch (instr & 0x01800000) {
                        case 0x00000000: {
                          // 0xe8100000
                          UnimplementedT32_32("RFEDB", instr);
                          break;
                        }
                        case 0x00800000: {
                          // 0xe8900000
                          if (((Uint32((instr >> 21)) & Uint32(0x1)) ==
                               Uint32(0x1)) &&
                              ((Uint32((instr >> 16)) & Uint32(0xf)) ==
                               Uint32(0xd)) &&
                              (BitCount(((Uint32((instr >> 15)) & Uint32(0x1))
                                         << 14) |
                                        ((Uint32((instr >> 14)) & Uint32(0x1))
                                         << 13) |
                                        (Uint32(instr) & Uint32(0x1fff))) >
                               Int64(1))) {
                            RegisterList registers(
                                (((instr >> 15) & 0x1) << kPCRegNum) |
                                (((instr >> 14) & 0x1) << kLRRegNum) |
                                (instr & 0x1fff));
                            if (registers.IsR0toR7orPC()) {
                              // POP{<c>}.W <registers> ; T2
                              pop(CurrentCond(), Wide, registers);
                              if (((instr & 0xffff2000) != 0xe8bd0000)) {
                                UnpredictableT32(instr);
                              }
                            } else {
                              // POP{<c>}{<q>} <registers> ; T2
                              pop(CurrentCond(), Best, registers);
                              if (((instr & 0xffff2000) != 0xe8bd0000)) {
                                UnpredictableT32(instr);
                              }
                            }
                            return;
                          }
                          unsigned rn = (instr >> 16) & 0xf;
                          WriteBack write_back((instr >> 21) & 0x1);
                          RegisterList registers(
                              (((instr >> 15) & 0x1) << kPCRegNum) |
                              (((instr >> 14) & 0x1) << kLRRegNum) |
                              (instr & 0x1fff));
                          if ((rn < kNumberOfT32LowRegisters) &&
                              (((registers.GetList() & (1 << rn)) == 0) ==
                               write_back.DoesWriteBack()) &&
                              ((registers.GetList() & ~0xff) == 0)) {
                            // LDM{<c>}.W <Rn>{!}, <registers> ; T2
                            ldm(CurrentCond(),
                                Wide,
                                Register(rn),
                                write_back,
                                registers);
                            if (((instr & 0xffd02000) != 0xe8900000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            // LDM{<c>}{<q>} <Rn>{!}, <registers> ; T2
                            ldm(CurrentCond(),
                                Best,
                                Register(rn),
                                write_back,
                                registers);
                            if (((instr & 0xffd02000) != 0xe8900000)) {
                              UnpredictableT32(instr);
                            }
                          }
                          break;
                        }
                        case 0x01000000: {
                          // 0xe9100000
                          unsigned rn = (instr >> 16) & 0xf;
                          WriteBack write_back((instr >> 21) & 0x1);
                          RegisterList registers(
                              (((instr >> 15) & 0x1) << kPCRegNum) |
                              (((instr >> 14) & 0x1) << kLRRegNum) |
                              (instr & 0x1fff));
                          // LDMDB{<c>}{<q>} <Rn>{!}, <registers> ; T1
                          ldmdb(CurrentCond(),
                                Register(rn),
                                write_back,
                                registers);
                          if (((instr & 0xffd02000) != 0xe9100000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        case 0x01800000: {
                          // 0xe9900000
                          UnimplementedT32_32("RFE{IA}", instr);
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xe8500000
                      switch (instr & 0x01200000) {
                        case 0x00000000: {
                          // 0xe8500000
                          switch (instr & 0x00800000) {
                            case 0x00000000: {
                              // 0xe8500000
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              int32_t offset = (instr & 0xff) << 2;
                              // LDREX{<c>}{<q>} <Rt>, [<Rn>{, #<imm>}] ; T1
                              ldrex(CurrentCond(),
                                    Register(rt),
                                    MemOperand(Register(rn),
                                               plus,
                                               offset,
                                               Offset));
                              if (((instr & 0xfff00f00) != 0xe8500f00)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            case 0x00800000: {
                              // 0xe8d00000
                              switch (instr & 0x000000f0) {
                                case 0x00000000: {
                                  // 0xe8d00000
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // TBB{<c>}{<q>} [<Rn>, <Rm>] ; T1
                                  tbb(CurrentCond(),
                                      Register(rn),
                                      Register(rm));
                                  if (((instr & 0xfff0fff0) != 0xe8d0f000) ||
                                      !OutsideITBlockOrLast()) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000010: {
                                  // 0xe8d00010
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // TBH{<c>}{<q>} [<Rn>, <Rm>, LSL #1] ; T1
                                  tbh(CurrentCond(),
                                      Register(rn),
                                      Register(rm));
                                  if (((instr & 0xfff0fff0) != 0xe8d0f010) ||
                                      !OutsideITBlockOrLast()) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000040: {
                                  // 0xe8d00040
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDREXB{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  ldrexb(CurrentCond(),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00f4f)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000050: {
                                  // 0xe8d00050
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDREXH{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  ldrexh(CurrentCond(),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00f5f)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000070: {
                                  // 0xe8d00070
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rt2 = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDREXD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>] ; T1
                                  ldrexd(CurrentCond(),
                                         Register(rt),
                                         Register(rt2),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff000ff) != 0xe8d0007f)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000080: {
                                  // 0xe8d00080
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDAB{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  ldab(CurrentCond(),
                                       Register(rt),
                                       MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00f8f)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x00000090: {
                                  // 0xe8d00090
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDAH{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  ldah(CurrentCond(),
                                       Register(rt),
                                       MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00f9f)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000a0: {
                                  // 0xe8d000a0
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDA{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  lda(CurrentCond(),
                                      Register(rt),
                                      MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00faf)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000c0: {
                                  // 0xe8d000c0
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDAEXB{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  ldaexb(CurrentCond(),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00fcf)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000d0: {
                                  // 0xe8d000d0
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDAEXH{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  ldaexh(CurrentCond(),
                                         Register(rt),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00fdf)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000e0: {
                                  // 0xe8d000e0
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDAEX{<c>}{<q>} <Rt>, [<Rn>] ; T1
                                  ldaex(CurrentCond(),
                                        Register(rt),
                                        MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff00fff) != 0xe8d00fef)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                case 0x000000f0: {
                                  // 0xe8d000f0
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rt2 = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // LDAEXD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>] ; T1
                                  ldaexd(CurrentCond(),
                                         Register(rt),
                                         Register(rt2),
                                         MemOperand(Register(rn), Offset));
                                  if (((instr & 0xfff000ff) != 0xe8d000ff)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00200000: {
                          // 0xe8700000
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xe87f0000
                              if (((instr & 0x1200000) == 0x0)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rt2 = (instr >> 8) & 0xf;
                              uint32_t U = (instr >> 23) & 0x1;
                              int32_t imm = instr & 0xff;
                              imm <<= 2;
                              if (U == 0) imm = -imm;
                              bool minus_zero = (imm == 0) && (U == 0);
                              Location location(imm, kT32PcDelta);
                              // LDRD{<c>}{<q>} <Rt>, <Rt2>, <label> ; T1
                              if (minus_zero) {
                                ldrd(CurrentCond(),
                                     Register(rt),
                                     Register(rt2),
                                     MemOperand(pc, minus, 0));
                              } else {
                                ldrd(CurrentCond(),
                                     Register(rt),
                                     Register(rt2),
                                     &location);
                              }
                              if (((instr & 0xff7f0000) != 0xe95f0000)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rt2 = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              Sign sign((((instr >> 23) & 0x1) == 0) ? minus
                                                                     : plus);
                              int32_t offset = (instr & 0xff) << 2;
                              // LDRD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>], #{+/-}<imm> ; T1 NOLINT(whitespace/line_length)
                              ldrd(CurrentCond(),
                                   Register(rt),
                                   Register(rt2),
                                   MemOperand(Register(rn),
                                              sign,
                                              offset,
                                              PostIndex));
                              break;
                            }
                          }
                          break;
                        }
                        case 0x01000000: {
                          // 0xe9500000
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xe95f0000
                              if (((instr & 0x1200000) == 0x0)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rt2 = (instr >> 8) & 0xf;
                              uint32_t U = (instr >> 23) & 0x1;
                              int32_t imm = instr & 0xff;
                              imm <<= 2;
                              if (U == 0) imm = -imm;
                              bool minus_zero = (imm == 0) && (U == 0);
                              Location location(imm, kT32PcDelta);
                              // LDRD{<c>}{<q>} <Rt>, <Rt2>, <label> ; T1
                              if (minus_zero) {
                                ldrd(CurrentCond(),
                                     Register(rt),
                                     Register(rt2),
                                     MemOperand(pc, minus, 0));
                              } else {
                                ldrd(CurrentCond(),
                                     Register(rt),
                                     Register(rt2),
                                     &location);
                              }
                              if (((instr & 0xff7f0000) != 0xe95f0000)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rt2 = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              Sign sign((((instr >> 23) & 0x1) == 0) ? minus
                                                                     : plus);
                              int32_t offset = (instr & 0xff) << 2;
                              // LDRD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>{, #{+/-}<imm>}] ; T1 NOLINT(whitespace/line_length)
                              ldrd(CurrentCond(),
                                   Register(rt),
                                   Register(rt2),
                                   MemOperand(Register(rn),
                                              sign,
                                              offset,
                                              Offset));
                              break;
                            }
                          }
                          break;
                        }
                        case 0x01200000: {
                          // 0xe9700000
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xe97f0000
                              if (((instr & 0x1200000) == 0x0)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rt2 = (instr >> 8) & 0xf;
                              uint32_t U = (instr >> 23) & 0x1;
                              int32_t imm = instr & 0xff;
                              imm <<= 2;
                              if (U == 0) imm = -imm;
                              bool minus_zero = (imm == 0) && (U == 0);
                              Location location(imm, kT32PcDelta);
                              // LDRD{<c>}{<q>} <Rt>, <Rt2>, <label> ; T1
                              if (minus_zero) {
                                ldrd(CurrentCond(),
                                     Register(rt),
                                     Register(rt2),
                                     MemOperand(pc, minus, 0));
                              } else {
                                ldrd(CurrentCond(),
                                     Register(rt),
                                     Register(rt2),
                                     &location);
                              }
                              if (((instr & 0xff7f0000) != 0xe95f0000)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rt2 = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              Sign sign((((instr >> 23) & 0x1) == 0) ? minus
                                                                     : plus);
                              int32_t offset = (instr & 0xff) << 2;
                              // LDRD{<c>}{<q>} <Rt>, <Rt2>, [<Rn>{, #{+/-}<imm>}]! ; T1 NOLINT(whitespace/line_length)
                              ldrd(CurrentCond(),
                                   Register(rt),
                                   Register(rt2),
                                   MemOperand(Register(rn),
                                              sign,
                                              offset,
                                              PreIndex));
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                  }
                  break;
                }
                case 0x10000000: {
                  // 0xf8000000
                  switch (instr & 0x01a00000) {
                    case 0x00000000: {
                      // 0xf8000000
                      switch (instr & 0x00400d00) {
                        case 0x00000000: {
                          // 0xf8000000
                          if ((instr & 0x000002c0) == 0x00000000) {
                            if (((instr & 0xf0000) == 0xf0000)) {
                              UnallocatedT32(instr);
                              return;
                            }
                            unsigned rt = (instr >> 12) & 0xf;
                            unsigned rn = (instr >> 16) & 0xf;
                            Sign sign(plus);
                            unsigned rm = instr & 0xf;
                            Shift shift = LSL;
                            uint32_t amount = (instr >> 4) & 0x3;
                            AddrMode addrmode = Offset;
                            if ((rt < kNumberOfT32LowRegisters) &&
                                (rn < kNumberOfT32LowRegisters) &&
                                (rm < kNumberOfT32LowRegisters) &&
                                shift.IsLSL() && (amount == 0) &&
                                sign.IsPlus()) {
                              // STRB{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2
                              strb(CurrentCond(),
                                   Wide,
                                   Register(rt),
                                   MemOperand(Register(rn),
                                              sign,
                                              Register(rm),
                                              addrmode));
                            } else {
                              // STRB{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                              strb(CurrentCond(),
                                   Best,
                                   Register(rt),
                                   MemOperand(Register(rn),
                                              sign,
                                              Register(rm),
                                              shift,
                                              amount,
                                              addrmode));
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                        case 0x00000900: {
                          // 0xf8000900
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 9) & 0x1) == 0) ? minus : plus);
                          int32_t offset = instr & 0xff;
                          // STRB{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_2> ; T3
                          strb(CurrentCond(),
                               Best,
                               Register(rt),
                               MemOperand(Register(rn),
                                          sign,
                                          offset,
                                          PostIndex));
                          break;
                        }
                        case 0x00000c00: {
                          // 0xf8000c00
                          switch (instr & 0x00000200) {
                            case 0x00000000: {
                              // 0xf8000c00
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              int32_t offset = instr & 0xff;
                              // STRB{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_2>}] ; T3
                              strb(CurrentCond(),
                                   Best,
                                   Register(rt),
                                   MemOperand(Register(rn),
                                              minus,
                                              offset,
                                              Offset));
                              break;
                            }
                            case 0x00000200: {
                              // 0xf8000e00
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              UnimplementedT32_32("STRBT", instr);
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000d00: {
                          // 0xf8000d00
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 9) & 0x1) == 0) ? minus : plus);
                          int32_t offset = instr & 0xff;
                          // STRB{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_2>}]! ; T3
                          strb(CurrentCond(),
                               Best,
                               Register(rt),
                               MemOperand(Register(rn),
                                          sign,
                                          offset,
                                          PreIndex));
                          break;
                        }
                        case 0x00400000: {
                          // 0xf8400000
                          if ((instr & 0x000002c0) == 0x00000000) {
                            if (((instr & 0xf0000) == 0xf0000)) {
                              UnallocatedT32(instr);
                              return;
                            }
                            unsigned rt = (instr >> 12) & 0xf;
                            unsigned rn = (instr >> 16) & 0xf;
                            Sign sign(plus);
                            unsigned rm = instr & 0xf;
                            Shift shift = LSL;
                            uint32_t amount = (instr >> 4) & 0x3;
                            AddrMode addrmode = Offset;
                            if ((rt < kNumberOfT32LowRegisters) &&
                                (rn < kNumberOfT32LowRegisters) &&
                                (rm < kNumberOfT32LowRegisters) &&
                                shift.IsLSL() && (amount == 0) &&
                                sign.IsPlus()) {
                              // STR{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2
                              str(CurrentCond(),
                                  Wide,
                                  Register(rt),
                                  MemOperand(Register(rn),
                                             sign,
                                             Register(rm),
                                             addrmode));
                            } else {
                              // STR{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                              str(CurrentCond(),
                                  Best,
                                  Register(rt),
                                  MemOperand(Register(rn),
                                             sign,
                                             Register(rm),
                                             shift,
                                             amount,
                                             addrmode));
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                        case 0x00400900: {
                          // 0xf8400900
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 9) & 0x1) == 0) ? minus : plus);
                          int32_t offset = instr & 0xff;
                          // STR{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_2> ; T4
                          str(CurrentCond(),
                              Best,
                              Register(rt),
                              MemOperand(Register(rn),
                                         sign,
                                         offset,
                                         PostIndex));
                          break;
                        }
                        case 0x00400c00: {
                          // 0xf8400c00
                          switch (instr & 0x00000200) {
                            case 0x00000000: {
                              // 0xf8400c00
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              int32_t offset = instr & 0xff;
                              // STR{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_2>}] ; T4
                              str(CurrentCond(),
                                  Best,
                                  Register(rt),
                                  MemOperand(Register(rn),
                                             minus,
                                             offset,
                                             Offset));
                              break;
                            }
                            case 0x00000200: {
                              // 0xf8400e00
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              UnimplementedT32_32("STRT", instr);
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00400d00: {
                          // 0xf8400d00
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          if (((Uint32((instr >> 16)) & Uint32(0xf)) ==
                               Uint32(0xd)) &&
                              ((Uint32((instr >> 9)) & Uint32(0x1)) ==
                               Uint32(0x0)) &&
                              ((Uint32(instr) & Uint32(0xff)) == Uint32(0x4))) {
                            unsigned rt = (instr >> 12) & 0xf;
                            if ((rt <= 7) || (rt == kLRRegNum)) {
                              // PUSH{<c>}.W <single_register_list> ; T4
                              push(CurrentCond(), Wide, Register(rt));
                            } else {
                              // PUSH{<c>}{<q>} <single_register_list> ; T4
                              push(CurrentCond(), Best, Register(rt));
                            }
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 9) & 0x1) == 0) ? minus : plus);
                          int32_t offset = instr & 0xff;
                          // STR{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_2>}]! ; T4
                          str(CurrentCond(),
                              Best,
                              Register(rt),
                              MemOperand(Register(rn), sign, offset, PreIndex));
                          break;
                        }
                        default:
                          UnallocatedT32(instr);
                          break;
                      }
                      break;
                    }
                    case 0x00200000: {
                      // 0xf8200000
                      switch (instr & 0x00400d00) {
                        case 0x00000000: {
                          // 0xf8200000
                          if ((instr & 0x000002c0) == 0x00000000) {
                            if (((instr & 0xf0000) == 0xf0000)) {
                              UnallocatedT32(instr);
                              return;
                            }
                            unsigned rt = (instr >> 12) & 0xf;
                            unsigned rn = (instr >> 16) & 0xf;
                            Sign sign(plus);
                            unsigned rm = instr & 0xf;
                            Shift shift = LSL;
                            uint32_t amount = (instr >> 4) & 0x3;
                            AddrMode addrmode = Offset;
                            if ((rt < kNumberOfT32LowRegisters) &&
                                (rn < kNumberOfT32LowRegisters) &&
                                (rm < kNumberOfT32LowRegisters) &&
                                shift.IsLSL() && (amount == 0) &&
                                sign.IsPlus()) {
                              // STRH{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2
                              strh(CurrentCond(),
                                   Wide,
                                   Register(rt),
                                   MemOperand(Register(rn),
                                              sign,
                                              Register(rm),
                                              addrmode));
                            } else {
                              // STRH{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                              strh(CurrentCond(),
                                   Best,
                                   Register(rt),
                                   MemOperand(Register(rn),
                                              sign,
                                              Register(rm),
                                              shift,
                                              amount,
                                              addrmode));
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                        case 0x00000900: {
                          // 0xf8200900
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 9) & 0x1) == 0) ? minus : plus);
                          int32_t offset = instr & 0xff;
                          // STRH{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_2> ; T3
                          strh(CurrentCond(),
                               Best,
                               Register(rt),
                               MemOperand(Register(rn),
                                          sign,
                                          offset,
                                          PostIndex));
                          break;
                        }
                        case 0x00000c00: {
                          // 0xf8200c00
                          switch (instr & 0x00000200) {
                            case 0x00000000: {
                              // 0xf8200c00
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              int32_t offset = instr & 0xff;
                              // STRH{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_2>}] ; T3
                              strh(CurrentCond(),
                                   Best,
                                   Register(rt),
                                   MemOperand(Register(rn),
                                              minus,
                                              offset,
                                              Offset));
                              break;
                            }
                            case 0x00000200: {
                              // 0xf8200e00
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              UnimplementedT32_32("STRHT", instr);
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000d00: {
                          // 0xf8200d00
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          Sign sign((((instr >> 9) & 0x1) == 0) ? minus : plus);
                          int32_t offset = instr & 0xff;
                          // STRH{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_2>}]! ; T3
                          strh(CurrentCond(),
                               Best,
                               Register(rt),
                               MemOperand(Register(rn),
                                          sign,
                                          offset,
                                          PreIndex));
                          break;
                        }
                        default:
                          UnallocatedT32(instr);
                          break;
                      }
                      break;
                    }
                    case 0x00800000: {
                      // 0xf8800000
                      switch (instr & 0x00400000) {
                        case 0x00000000: {
                          // 0xf8800000
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          int32_t offset = instr & 0xfff;
                          if ((rt < kNumberOfT32LowRegisters) &&
                              (rn < kNumberOfT32LowRegisters) &&
                              ((offset >= 0) && (offset <= 31))) {
                            // STRB{<c>}.W <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2
                            strb(CurrentCond(),
                                 Wide,
                                 Register(rt),
                                 MemOperand(Register(rn),
                                            plus,
                                            offset,
                                            Offset));
                          } else {
                            // STRB{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2
                            strb(CurrentCond(),
                                 Best,
                                 Register(rt),
                                 MemOperand(Register(rn),
                                            plus,
                                            offset,
                                            Offset));
                          }
                          break;
                        }
                        case 0x00400000: {
                          // 0xf8c00000
                          if (((instr & 0xf0000) == 0xf0000)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rt = (instr >> 12) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          int32_t offset = instr & 0xfff;
                          if (((rt < kNumberOfT32LowRegisters) &&
                               (rn < kNumberOfT32LowRegisters) &&
                               ((offset >= 0) && (offset <= 124) &&
                                ((offset & 3) == 0))) ||
                              ((rt < kNumberOfT32LowRegisters) &&
                               (rn == sp.GetCode()) &&
                               ((offset >= 0) && (offset <= 1020) &&
                                ((offset & 3) == 0)))) {
                            // STR{<c>}.W <Rt>, [<Rn>{, #{+}<imm_1>}] ; T3
                            str(CurrentCond(),
                                Wide,
                                Register(rt),
                                MemOperand(Register(rn), plus, offset, Offset));
                          } else {
                            // STR{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm_1>}] ; T3
                            str(CurrentCond(),
                                Best,
                                Register(rt),
                                MemOperand(Register(rn), plus, offset, Offset));
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00a00000: {
                      // 0xf8a00000
                      if ((instr & 0x00400000) == 0x00000000) {
                        if (((instr & 0xf0000) == 0xf0000)) {
                          UnallocatedT32(instr);
                          return;
                        }
                        unsigned rt = (instr >> 12) & 0xf;
                        unsigned rn = (instr >> 16) & 0xf;
                        int32_t offset = instr & 0xfff;
                        if ((rt < kNumberOfT32LowRegisters) &&
                            (rn < kNumberOfT32LowRegisters) &&
                            ((offset >= 0) && (offset <= 62) &&
                             ((offset & 1) == 0))) {
                          // STRH{<c>}.W <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2
                          strh(CurrentCond(),
                               Wide,
                               Register(rt),
                               MemOperand(Register(rn), plus, offset, Offset));
                        } else {
                          // STRH{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2
                          strh(CurrentCond(),
                               Best,
                               Register(rt),
                               MemOperand(Register(rn), plus, offset, Offset));
                        }
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x01000000: {
                      // 0xf9000000
                      switch (instr & 0x0000000d) {
                        case 0x0000000d: {
                          // 0xf900000d
                          switch (instr & 0x00000002) {
                            case 0x00000000: {
                              // 0xf900000d
                              switch (instr & 0x00000f00) {
                                case 0x00000000: {
                                  // 0xf900000d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000100: {
                                  // 0xf900010d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000200: {
                                  // 0xf900020d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000300: {
                                  // 0xf900030d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000400: {
                                  // 0xf900040d
                                  if (((instr & 0x20) == 0x20)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000500: {
                                  // 0xf900050d
                                  if (((instr & 0x20) == 0x20)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000600: {
                                  // 0xf900060d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000700: {
                                  // 0xf900070d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000800: {
                                  // 0xf900080d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000900: {
                                  // 0xf900090d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000a00: {
                                  // 0xf9000a0d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                            case 0x00000002: {
                              // 0xf900000f
                              switch (instr & 0x00000f00) {
                                case 0x00000000: {
                                  // 0xf900000d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000100: {
                                  // 0xf900010d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000200: {
                                  // 0xf900020d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000300: {
                                  // 0xf900030d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000400: {
                                  // 0xf900040d
                                  if (((instr & 0x20) == 0x20)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000500: {
                                  // 0xf900050d
                                  if (((instr & 0x20) == 0x20)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000600: {
                                  // 0xf900060d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000700: {
                                  // 0xf900070d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000800: {
                                  // 0xf900080d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000900: {
                                  // 0xf900090d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000a00: {
                                  // 0xf9000a0d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_5_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x00000f00) {
                            case 0x00000000: {
                              // 0xf9000000
                              if (((instr & 0xd) == 0xd)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_4_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x0:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                                case 0x1:
                                  length = 4;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst4(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000100: {
                              // 0xf9000100
                              if (((instr & 0xd) == 0xd)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_4_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x0:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                                case 0x1:
                                  length = 4;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst4(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000200: {
                              // 0xf9000200
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_5_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000300: {
                              // 0xf9000300
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe30) == 0x830)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_2_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x8:
                                  length = 2;
                                  spacing = kSingle;
                                  break;
                                case 0x9:
                                  length = 2;
                                  spacing = kDouble;
                                  break;
                                case 0x3:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst2(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000400: {
                              // 0xf9000400
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0x20) == 0x20)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_3_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x4:
                                  length = 3;
                                  spacing = kSingle;
                                  break;
                                case 0x5:
                                  length = 3;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst3(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000500: {
                              // 0xf9000500
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0x20) == 0x20)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_3_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x4:
                                  length = 3;
                                  spacing = kSingle;
                                  break;
                                case 0x5:
                                  length = 3;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst3(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000600: {
                              // 0xf9000600
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_5_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000700: {
                              // 0xf9000700
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_5_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000800: {
                              // 0xf9000800
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe30) == 0x830)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_2_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x8:
                                  length = 2;
                                  spacing = kSingle;
                                  break;
                                case 0x9:
                                  length = 2;
                                  spacing = kDouble;
                                  break;
                                case 0x3:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst2(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000900: {
                              // 0xf9000900
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe30) == 0x830)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_2_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x8:
                                  length = 2;
                                  spacing = kSingle;
                                  break;
                                case 0x9:
                                  length = 2;
                                  spacing = kDouble;
                                  break;
                                case 0x3:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst2(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000a00: {
                              // 0xf9000a00
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_5_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vst1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            default:
                              UnallocatedT32(instr);
                              break;
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01200000: {
                      // 0xf9200000
                      switch (instr & 0x0000000d) {
                        case 0x0000000d: {
                          // 0xf920000d
                          switch (instr & 0x00000002) {
                            case 0x00000000: {
                              // 0xf920000d
                              switch (instr & 0x00000f00) {
                                case 0x00000000: {
                                  // 0xf920000d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000100: {
                                  // 0xf920010d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000200: {
                                  // 0xf920020d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000300: {
                                  // 0xf920030d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000400: {
                                  // 0xf920040d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000500: {
                                  // 0xf920050d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000600: {
                                  // 0xf920060d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000700: {
                                  // 0xf920070d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000800: {
                                  // 0xf920080d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000900: {
                                  // 0xf920090d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                case 0x00000a00: {
                                  // 0xf9200a0d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         PostIndex));
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                            case 0x00000002: {
                              // 0xf920000f
                              switch (instr & 0x00000f00) {
                                case 0x00000000: {
                                  // 0xf920000d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000100: {
                                  // 0xf920010d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_4_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000200: {
                                  // 0xf920020d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000300: {
                                  // 0xf920030d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000400: {
                                  // 0xf920040d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000500: {
                                  // 0xf920050d
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_3_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x4:
                                      length = 3;
                                      spacing = kSingle;
                                      break;
                                    case 0x5:
                                      length = 3;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000600: {
                                  // 0xf920060d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000700: {
                                  // 0xf920070d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000800: {
                                  // 0xf920080d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000900: {
                                  // 0xf920090d
                                  if (((instr & 0xe30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_2_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x8:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x9:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                    case 0x3:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                case 0x00000a00: {
                                  // 0xf9200a0d
                                  if (((instr & 0xe20) == 0x620) ||
                                      ((instr & 0xf30) == 0xa30)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_6_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_align_1_Decode((instr >> 4) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 8) & 0xf) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x7:
                                      length = 1;
                                      break;
                                    case 0xa:
                                      length = 2;
                                      break;
                                    case 0x6:
                                      length = 3;
                                      break;
                                    case 0x2:
                                      length = 4;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kMultipleLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Offset));
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x00000f00) {
                            case 0x00000000: {
                              // 0xf9200000
                              if (((instr & 0xd) == 0xd)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_4_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x0:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                                case 0x1:
                                  length = 4;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld4(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000100: {
                              // 0xf9200100
                              if (((instr & 0xd) == 0xd)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_4_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x0:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                                case 0x1:
                                  length = 4;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld4(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000200: {
                              // 0xf9200200
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_1_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000300: {
                              // 0xf9200300
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe30) == 0x830)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_2_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x8:
                                  length = 2;
                                  spacing = kSingle;
                                  break;
                                case 0x9:
                                  length = 2;
                                  spacing = kDouble;
                                  break;
                                case 0x3:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld2(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000400: {
                              // 0xf9200400
                              if (((instr & 0xd) == 0xd)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_3_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x4:
                                  length = 3;
                                  spacing = kSingle;
                                  break;
                                case 0x5:
                                  length = 3;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld3(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000500: {
                              // 0xf9200500
                              if (((instr & 0xd) == 0xd)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_3_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x4:
                                  length = 3;
                                  spacing = kSingle;
                                  break;
                                case 0x5:
                                  length = 3;
                                  spacing = kDouble;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld3(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000600: {
                              // 0xf9200600
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_1_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000700: {
                              // 0xf9200700
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_1_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000800: {
                              // 0xf9200800
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe30) == 0x830)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_2_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x8:
                                  length = 2;
                                  spacing = kSingle;
                                  break;
                                case 0x9:
                                  length = 2;
                                  spacing = kDouble;
                                  break;
                                case 0x3:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld2(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000900: {
                              // 0xf9200900
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe30) == 0x830)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_7_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_2_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x8:
                                  length = 2;
                                  spacing = kSingle;
                                  break;
                                case 0x9:
                                  length = 2;
                                  spacing = kDouble;
                                  break;
                                case 0x3:
                                  length = 4;
                                  spacing = kSingle;
                                  break;
                              }
                              unsigned last =
                                  first +
                                  (length - 1) * (spacing == kSingle ? 1 : 2);
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld2(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            case 0x00000a00: {
                              // 0xf9200a00
                              if (((instr & 0xd) == 0xd) ||
                                  ((instr & 0xe20) == 0x620) ||
                                  ((instr & 0xf30) == 0xa30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              DataType dt =
                                  Dt_size_6_Decode((instr >> 6) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              Alignment align =
                                  Align_align_1_Decode((instr >> 4) & 0x3);
                              if (dt.Is(kDataTypeValueInvalid) ||
                                  align.Is(kBadAlignment)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned first = ExtractDRegister(instr, 22, 12);
                              unsigned length;
                              SpacingType spacing = kSingle;
                              switch ((instr >> 8) & 0xf) {
                                default:
                                  VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                case 0x7:
                                  length = 1;
                                  break;
                                case 0xa:
                                  length = 2;
                                  break;
                                case 0x6:
                                  length = 3;
                                  break;
                                case 0x2:
                                  length = 4;
                                  break;
                              }
                              unsigned last = first + length - 1;
                              TransferType transfer = kMultipleLanes;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                              vld1(CurrentCond(),
                                   dt,
                                   NeonRegisterList(DRegister(first),
                                                    DRegister(last),
                                                    spacing,
                                                    transfer),
                                   AlignedMemOperand(Register(rn),
                                                     align,
                                                     Register(rm),
                                                     PostIndex));
                              break;
                            }
                            default:
                              UnallocatedT32(instr);
                              break;
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01800000: {
                      // 0xf9800000
                      switch (instr & 0x00000300) {
                        case 0x00000000: {
                          // 0xf9800000
                          switch (instr & 0x00000c00) {
                            case 0x00000c00: {
                              // 0xf9800c00
                              UnallocatedT32(instr);
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf980000d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf980000d
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_1_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 1;
                                      unsigned last = first + length - 1;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vst1(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf980000f
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_1_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 1;
                                      unsigned last = first + length - 1;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vst1(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeonAndAlign decode_neon =
                                      Align_index_align_1_Decode((instr >> 4) &
                                                                     0xf,
                                                                 dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align = decode_neon.GetAlign();
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 1;
                                  unsigned last = first + length - 1;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VST1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vst1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000100: {
                          // 0xf9800100
                          switch (instr & 0x00000c00) {
                            case 0x00000c00: {
                              // 0xf9800d00
                              UnallocatedT32(instr);
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf980010d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf980010d
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_2_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 2;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vst2(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf980010f
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_2_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 2;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vst2(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeonAndAlign decode_neon =
                                      Align_index_align_2_Decode((instr >> 4) &
                                                                     0xf,
                                                                 dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align = decode_neon.GetAlign();
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 2;
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VST2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vst2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000200: {
                          // 0xf9800200
                          switch (instr & 0x00000c30) {
                            case 0x00000010: {
                              // 0xf9800210
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000030: {
                              // 0xf9800230
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000410: {
                              // 0xf9800610
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000430: {
                              // 0xf9800630
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000810: {
                              // 0xf9800a10
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000820: {
                              // 0xf9800a20
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000830: {
                              // 0xf9800a30
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000c00: {
                              // 0xf9800e00
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000c10: {
                              // 0xf9800e10
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000c20: {
                              // 0xf9800e20
                              UnallocatedT32(instr);
                              break;
                            }
                            case 0x00000c30: {
                              // 0xf9800e30
                              UnallocatedT32(instr);
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf980020d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf980020d
                                      if (((instr & 0xc00) == 0xc00) ||
                                          ((instr & 0x810) == 0x10) ||
                                          ((instr & 0xc30) == 0x810) ||
                                          ((instr & 0xc30) == 0x820) ||
                                          ((instr & 0xc30) == 0x830)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeon decode_neon =
                                          Index_1_Decode((instr >> 4) & 0xf,
                                                         dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 3;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST3{<c>}{<q>}.<dt> <list>, [<Rn>]! ; T1 NOLINT(whitespace/line_length)
                                      vst3(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           MemOperand(Register(rn), PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf980020f
                                      if (((instr & 0xc00) == 0xc00) ||
                                          ((instr & 0x810) == 0x10) ||
                                          ((instr & 0xc30) == 0x810) ||
                                          ((instr & 0xc30) == 0x820) ||
                                          ((instr & 0xc30) == 0x830)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeon decode_neon =
                                          Index_1_Decode((instr >> 4) & 0xf,
                                                         dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 3;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST3{<c>}{<q>}.<dt> <list>, [<Rn>] ; T1
                                      vst3(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           MemOperand(Register(rn), Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd) ||
                                      ((instr & 0x810) == 0x10) ||
                                      ((instr & 0xc30) == 0x810) ||
                                      ((instr & 0xc30) == 0x820) ||
                                      ((instr & 0xc30) == 0x830)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeon decode_neon =
                                      Index_1_Decode((instr >> 4) & 0xf, dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 3;
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  unsigned rn = (instr >> 16) & 0xf;
                                  Sign sign(plus);
                                  unsigned rm = instr & 0xf;
                                  // VST3{<c>}{<q>}.<dt> <list>, [<Rn>], #<Rm> ; T1 NOLINT(whitespace/line_length)
                                  vst3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       MemOperand(Register(rn),
                                                  sign,
                                                  Register(rm),
                                                  PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000300: {
                          // 0xf9800300
                          switch (instr & 0x00000c00) {
                            case 0x00000c00: {
                              // 0xf9800f00
                              UnallocatedT32(instr);
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf980030d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf980030d
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_3_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 4;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vst4(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf980030f
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_3_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 4;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vst4(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeonAndAlign decode_neon =
                                      Align_index_align_3_Decode((instr >> 4) &
                                                                     0xf,
                                                                 dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align = decode_neon.GetAlign();
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 4;
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VST4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vst4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x01a00000: {
                      // 0xf9a00000
                      switch (instr & 0x00000300) {
                        case 0x00000000: {
                          // 0xf9a00000
                          switch (instr & 0x00000c00) {
                            case 0x00000c00: {
                              // 0xf9a00c00
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf9a00c0d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf9a00c0d
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align =
                                          Align_a_1_Decode((instr >> 4) & 0x1,
                                                           dt);
                                      if (dt.Is(kDataTypeValueInvalid) ||
                                          align.Is(kBadAlignment)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length;
                                      SpacingType spacing = kSingle;
                                      switch ((instr >> 5) & 0x1) {
                                        default:
                                          VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                        case 0x0:
                                          length = 1;
                                          break;
                                        case 0x1:
                                          length = 2;
                                          break;
                                      }
                                      unsigned last = first + length - 1;
                                      TransferType transfer = kAllLanes;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vld1(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            transfer),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf9a00c0f
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align =
                                          Align_a_1_Decode((instr >> 4) & 0x1,
                                                           dt);
                                      if (dt.Is(kDataTypeValueInvalid) ||
                                          align.Is(kBadAlignment)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length;
                                      SpacingType spacing = kSingle;
                                      switch ((instr >> 5) & 0x1) {
                                        default:
                                          VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                        case 0x0:
                                          length = 1;
                                          break;
                                        case 0x1:
                                          length = 2;
                                          break;
                                      }
                                      unsigned last = first + length - 1;
                                      TransferType transfer = kAllLanes;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vld1(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            transfer),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_a_1_Decode((instr >> 4) & 0x1, dt);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing = kSingle;
                                  switch ((instr >> 5) & 0x1) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 1;
                                      break;
                                    case 0x1:
                                      length = 2;
                                      break;
                                  }
                                  unsigned last = first + length - 1;
                                  TransferType transfer = kAllLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf9a0000d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf9a0000d
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_1_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 1;
                                      unsigned last = first + length - 1;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vld1(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf9a0000f
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_1_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 1;
                                      unsigned last = first + length - 1;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vld1(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeonAndAlign decode_neon =
                                      Align_index_align_1_Decode((instr >> 4) &
                                                                     0xf,
                                                                 dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align = decode_neon.GetAlign();
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 1;
                                  unsigned last = first + length - 1;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VLD1{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vld1(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000100: {
                          // 0xf9a00100
                          switch (instr & 0x00000c00) {
                            case 0x00000c00: {
                              // 0xf9a00d00
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf9a00d0d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf9a00d0d
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align =
                                          Align_a_2_Decode((instr >> 4) & 0x1,
                                                           dt);
                                      if (dt.Is(kDataTypeValueInvalid) ||
                                          align.Is(kBadAlignment)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length;
                                      SpacingType spacing;
                                      switch ((instr >> 5) & 0x1) {
                                        default:
                                          VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                        case 0x0:
                                          length = 2;
                                          spacing = kSingle;
                                          break;
                                        case 0x1:
                                          length = 2;
                                          spacing = kDouble;
                                          break;
                                      }
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      TransferType transfer = kAllLanes;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vld2(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            transfer),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf9a00d0f
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align =
                                          Align_a_2_Decode((instr >> 4) & 0x1,
                                                           dt);
                                      if (dt.Is(kDataTypeValueInvalid) ||
                                          align.Is(kBadAlignment)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length;
                                      SpacingType spacing;
                                      switch ((instr >> 5) & 0x1) {
                                        default:
                                          VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                        case 0x0:
                                          length = 2;
                                          spacing = kSingle;
                                          break;
                                        case 0x1:
                                          length = 2;
                                          spacing = kDouble;
                                          break;
                                      }
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      TransferType transfer = kAllLanes;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vld2(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            transfer),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_a_2_Decode((instr >> 4) & 0x1, dt);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 5) & 0x1) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 2;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 2;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kAllLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf9a0010d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf9a0010d
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_2_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 2;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vld2(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf9a0010f
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_2_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 2;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vld2(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeonAndAlign decode_neon =
                                      Align_index_align_2_Decode((instr >> 4) &
                                                                     0xf,
                                                                 dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align = decode_neon.GetAlign();
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 2;
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VLD2{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vld2(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000200: {
                          // 0xf9a00200
                          switch (instr & 0x00000c00) {
                            case 0x00000c00: {
                              // 0xf9a00e00
                              switch (instr & 0x00000010) {
                                case 0x00000000: {
                                  // 0xf9a00e00
                                  switch (instr & 0x0000000d) {
                                    case 0x0000000d: {
                                      // 0xf9a00e0d
                                      switch (instr & 0x00000002) {
                                        case 0x00000000: {
                                          // 0xf9a00e0d
                                          DataType dt = Dt_size_7_Decode(
                                              (instr >> 6) & 0x3);
                                          if (dt.Is(kDataTypeValueInvalid)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned first =
                                              ExtractDRegister(instr, 22, 12);
                                          unsigned length;
                                          SpacingType spacing;
                                          switch ((instr >> 5) & 0x1) {
                                            default:
                                              VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                            case 0x0:
                                              length = 3;
                                              spacing = kSingle;
                                              break;
                                            case 0x1:
                                              length = 3;
                                              spacing = kDouble;
                                              break;
                                          }
                                          unsigned last =
                                              first +
                                              (length - 1) *
                                                  (spacing == kSingle ? 1 : 2);
                                          TransferType transfer = kAllLanes;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>]! ; T1 NOLINT(whitespace/line_length)
                                          vld3(CurrentCond(),
                                               dt,
                                               NeonRegisterList(DRegister(
                                                                    first),
                                                                DRegister(last),
                                                                spacing,
                                                                transfer),
                                               MemOperand(Register(rn),
                                                          PostIndex));
                                          break;
                                        }
                                        case 0x00000002: {
                                          // 0xf9a00e0f
                                          DataType dt = Dt_size_7_Decode(
                                              (instr >> 6) & 0x3);
                                          if (dt.Is(kDataTypeValueInvalid)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned first =
                                              ExtractDRegister(instr, 22, 12);
                                          unsigned length;
                                          SpacingType spacing;
                                          switch ((instr >> 5) & 0x1) {
                                            default:
                                              VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                            case 0x0:
                                              length = 3;
                                              spacing = kSingle;
                                              break;
                                            case 0x1:
                                              length = 3;
                                              spacing = kDouble;
                                              break;
                                          }
                                          unsigned last =
                                              first +
                                              (length - 1) *
                                                  (spacing == kSingle ? 1 : 2);
                                          TransferType transfer = kAllLanes;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>] ; T1 NOLINT(whitespace/line_length)
                                          vld3(CurrentCond(),
                                               dt,
                                               NeonRegisterList(DRegister(
                                                                    first),
                                                                DRegister(last),
                                                                spacing,
                                                                transfer),
                                               MemOperand(Register(rn),
                                                          Offset));
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    default: {
                                      if (((instr & 0xd) == 0xd)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length;
                                      SpacingType spacing;
                                      switch ((instr >> 5) & 0x1) {
                                        default:
                                          VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                        case 0x0:
                                          length = 3;
                                          spacing = kSingle;
                                          break;
                                        case 0x1:
                                          length = 3;
                                          spacing = kDouble;
                                          break;
                                      }
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      TransferType transfer = kAllLanes;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      Sign sign(plus);
                                      unsigned rm = instr & 0xf;
                                      // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>], #<Rm> ; T1 NOLINT(whitespace/line_length)
                                      vld3(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            transfer),
                                           MemOperand(Register(rn),
                                                      sign,
                                                      Register(rm),
                                                      PostIndex));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf9a0020d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf9a0020d
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeon decode_neon =
                                          Index_1_Decode((instr >> 4) & 0xf,
                                                         dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 3;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>]! ; T1 NOLINT(whitespace/line_length)
                                      vld3(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           MemOperand(Register(rn), PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf9a0020f
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeon decode_neon =
                                          Index_1_Decode((instr >> 4) & 0xf,
                                                         dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 3;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>] ; T1
                                      vld3(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           MemOperand(Register(rn), Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeon decode_neon =
                                      Index_1_Decode((instr >> 4) & 0xf, dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 3;
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  unsigned rn = (instr >> 16) & 0xf;
                                  Sign sign(plus);
                                  unsigned rm = instr & 0xf;
                                  // VLD3{<c>}{<q>}.<dt> <list>, [<Rn>], #<Rm> ; T1 NOLINT(whitespace/line_length)
                                  vld3(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       MemOperand(Register(rn),
                                                  sign,
                                                  Register(rm),
                                                  PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00000300: {
                          // 0xf9a00300
                          switch (instr & 0x00000c00) {
                            case 0x00000c00: {
                              // 0xf9a00f00
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf9a00f0d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf9a00f0d
                                      DataType dt =
                                          Dt_size_8_Decode((instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align =
                                          Align_a_3_Decode((instr >> 4) & 0x1,
                                                           dt,
                                                           (instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid) ||
                                          align.Is(kBadAlignment)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length;
                                      SpacingType spacing;
                                      switch ((instr >> 5) & 0x1) {
                                        default:
                                          VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                        case 0x0:
                                          length = 4;
                                          spacing = kSingle;
                                          break;
                                        case 0x1:
                                          length = 4;
                                          spacing = kDouble;
                                          break;
                                      }
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      TransferType transfer = kAllLanes;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vld4(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            transfer),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf9a00f0f
                                      DataType dt =
                                          Dt_size_8_Decode((instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align =
                                          Align_a_3_Decode((instr >> 4) & 0x1,
                                                           dt,
                                                           (instr >> 6) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid) ||
                                          align.Is(kBadAlignment)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length;
                                      SpacingType spacing;
                                      switch ((instr >> 5) & 0x1) {
                                        default:
                                          VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                        case 0x0:
                                          length = 4;
                                          spacing = kSingle;
                                          break;
                                        case 0x1:
                                          length = 4;
                                          spacing = kDouble;
                                          break;
                                      }
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      TransferType transfer = kAllLanes;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vld4(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            transfer),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_8_Decode((instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align =
                                      Align_a_3_Decode((instr >> 4) & 0x1,
                                                       dt,
                                                       (instr >> 6) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid) ||
                                      align.Is(kBadAlignment)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length;
                                  SpacingType spacing;
                                  switch ((instr >> 5) & 0x1) {
                                    default:
                                      VIXL_UNREACHABLE_OR_FALLTHROUGH();
                                    case 0x0:
                                      length = 4;
                                      spacing = kSingle;
                                      break;
                                    case 0x1:
                                      length = 4;
                                      spacing = kDouble;
                                      break;
                                  }
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  TransferType transfer = kAllLanes;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vld4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        transfer),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x0000000d) {
                                case 0x0000000d: {
                                  // 0xf9a0030d
                                  switch (instr & 0x00000002) {
                                    case 0x00000000: {
                                      // 0xf9a0030d
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_3_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 4;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}]! ; T1 NOLINT(whitespace/line_length)
                                      vld4(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             PostIndex));
                                      break;
                                    }
                                    case 0x00000002: {
                                      // 0xf9a0030f
                                      if (((instr & 0xc00) == 0xc00)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DataType dt =
                                          Dt_size_7_Decode((instr >> 10) & 0x3);
                                      if (dt.Is(kDataTypeValueInvalid)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      DecodeNeonAndAlign decode_neon =
                                          Align_index_align_3_Decode((instr >>
                                                                      4) &
                                                                         0xf,
                                                                     dt);
                                      if (!decode_neon.IsValid()) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      Alignment align = decode_neon.GetAlign();
                                      int lane = decode_neon.GetLane();
                                      SpacingType spacing =
                                          decode_neon.GetSpacing();
                                      unsigned first =
                                          ExtractDRegister(instr, 22, 12);
                                      unsigned length = 4;
                                      unsigned last =
                                          first +
                                          (length - 1) *
                                              (spacing == kSingle ? 1 : 2);
                                      unsigned rn = (instr >> 16) & 0xf;
                                      // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}] ; T1 NOLINT(whitespace/line_length)
                                      vld4(CurrentCond(),
                                           dt,
                                           NeonRegisterList(DRegister(first),
                                                            DRegister(last),
                                                            spacing,
                                                            lane),
                                           AlignedMemOperand(Register(rn),
                                                             align,
                                                             Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xc00) == 0xc00) ||
                                      ((instr & 0xd) == 0xd)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DataType dt =
                                      Dt_size_7_Decode((instr >> 10) & 0x3);
                                  if (dt.Is(kDataTypeValueInvalid)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  DecodeNeonAndAlign decode_neon =
                                      Align_index_align_3_Decode((instr >> 4) &
                                                                     0xf,
                                                                 dt);
                                  if (!decode_neon.IsValid()) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  Alignment align = decode_neon.GetAlign();
                                  int lane = decode_neon.GetLane();
                                  SpacingType spacing =
                                      decode_neon.GetSpacing();
                                  unsigned first =
                                      ExtractDRegister(instr, 22, 12);
                                  unsigned length = 4;
                                  unsigned last =
                                      first +
                                      (length - 1) *
                                          (spacing == kSingle ? 1 : 2);
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // VLD4{<c>}{<q>}.<dt> <list>, [<Rn>{:<align>}], <Rm> ; T1 NOLINT(whitespace/line_length)
                                  vld4(CurrentCond(),
                                       dt,
                                       NeonRegisterList(DRegister(first),
                                                        DRegister(last),
                                                        spacing,
                                                        lane),
                                       AlignedMemOperand(Register(rn),
                                                         align,
                                                         Register(rm),
                                                         PostIndex));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                  }
                  break;
                }
                case 0x10100000: {
                  // 0xf8100000
                  switch (instr & 0x01400000) {
                    case 0x00000000: {
                      // 0xf8100000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xf81f0000
                          switch (instr & 0x0000f000) {
                            case 0x0000f000: {
                              // 0xf81ff000
                              uint32_t U = (instr >> 23) & 0x1;
                              int32_t imm = instr & 0xfff;
                              if (U == 0) imm = -imm;
                              bool minus_zero = (imm == 0) && (U == 0);
                              Location location(imm, kT32PcDelta);
                              // PLD{<c>}{<q>} <label> ; T1
                              if (minus_zero) {
                                pld(CurrentCond(), MemOperand(pc, minus, 0));
                              } else {
                                pld(CurrentCond(), &location);
                              }
                              if (((instr & 0xff7ff000) != 0xf81ff000)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x00200000) {
                                case 0x00000000: {
                                  // 0xf81f0000
                                  if (((instr & 0xf000) == 0xf000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  uint32_t U = (instr >> 23) & 0x1;
                                  int32_t imm = instr & 0xfff;
                                  if (U == 0) imm = -imm;
                                  bool minus_zero = (imm == 0) && (U == 0);
                                  Location location(imm, kT32PcDelta);
                                  // LDRB{<c>}{<q>} <Rt>, <label> ; T1
                                  if (minus_zero) {
                                    ldrb(CurrentCond(),
                                         Best,
                                         Register(rt),
                                         MemOperand(pc, minus, 0));
                                  } else {
                                    ldrb(CurrentCond(),
                                         Register(rt),
                                         &location);
                                  }
                                  break;
                                }
                                case 0x00200000: {
                                  // 0xf83f0000
                                  if (((instr & 0xf000) == 0xf000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  uint32_t U = (instr >> 23) & 0x1;
                                  int32_t imm = instr & 0xfff;
                                  if (U == 0) imm = -imm;
                                  bool minus_zero = (imm == 0) && (U == 0);
                                  Location location(imm, kT32PcDelta);
                                  // LDRH{<c>}{<q>} <Rt>, <label> ; T1
                                  if (minus_zero) {
                                    ldrh(CurrentCond(),
                                         Best,
                                         Register(rt),
                                         MemOperand(pc, minus, 0));
                                  } else {
                                    ldrh(CurrentCond(),
                                         Register(rt),
                                         &location);
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x00a00000) {
                            case 0x00000000: {
                              // 0xf8100000
                              switch (instr & 0x00000d00) {
                                case 0x00000000: {
                                  // 0xf8100000
                                  switch (instr & 0x000002c0) {
                                    case 0x00000000: {
                                      // 0xf8100000
                                      switch (instr & 0x0000f000) {
                                        case 0x0000f000: {
                                          // 0xf810f000
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rn = (instr >> 16) & 0xf;
                                          Sign sign(plus);
                                          unsigned rm = instr & 0xf;
                                          Shift shift = LSL;
                                          uint32_t amount = (instr >> 4) & 0x3;
                                          AddrMode addrmode = Offset;
                                          // PLD{<c>}{<q>} [<Rn>, {+}<Rm>{, LSL #<amount>}] ; T1 NOLINT(whitespace/line_length)
                                          pld(CurrentCond(),
                                              MemOperand(Register(rn),
                                                         sign,
                                                         Register(rm),
                                                         shift,
                                                         amount,
                                                         addrmode));
                                          break;
                                        }
                                        default: {
                                          if (((instr & 0xf0000) == 0xf0000) ||
                                              ((instr & 0xf000) == 0xf000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rt = (instr >> 12) & 0xf;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          Sign sign(plus);
                                          unsigned rm = instr & 0xf;
                                          Shift shift = LSL;
                                          uint32_t amount = (instr >> 4) & 0x3;
                                          AddrMode addrmode = Offset;
                                          if ((rt < kNumberOfT32LowRegisters) &&
                                              (rn < kNumberOfT32LowRegisters) &&
                                              (rm < kNumberOfT32LowRegisters) &&
                                              shift.IsLSL() && (amount == 0) &&
                                              sign.IsPlus()) {
                                            // LDRB{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2 NOLINT(whitespace/line_length)
                                            ldrb(CurrentCond(),
                                                 Wide,
                                                 Register(rt),
                                                 MemOperand(Register(rn),
                                                            sign,
                                                            Register(rm),
                                                            addrmode));
                                          } else {
                                            // LDRB{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                                            ldrb(CurrentCond(),
                                                 Best,
                                                 Register(rt),
                                                 MemOperand(Register(rn),
                                                            sign,
                                                            Register(rm),
                                                            shift,
                                                            amount,
                                                            addrmode));
                                          }
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    default:
                                      UnallocatedT32(instr);
                                      break;
                                  }
                                  break;
                                }
                                case 0x00000900: {
                                  // 0xf8100900
                                  if (((instr & 0xf0000) == 0xf0000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  Sign sign((((instr >> 9) & 0x1) == 0) ? minus
                                                                        : plus);
                                  int32_t offset = instr & 0xff;
                                  // LDRB{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_2> ; T3 NOLINT(whitespace/line_length)
                                  ldrb(CurrentCond(),
                                       Best,
                                       Register(rt),
                                       MemOperand(Register(rn),
                                                  sign,
                                                  offset,
                                                  PostIndex));
                                  break;
                                }
                                case 0x00000c00: {
                                  // 0xf8100c00
                                  switch (instr & 0x00000200) {
                                    case 0x00000000: {
                                      // 0xf8100c00
                                      switch (instr & 0x0000f000) {
                                        case 0x0000f000: {
                                          // 0xf810fc00
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rn = (instr >> 16) & 0xf;
                                          int32_t offset = instr & 0xff;
                                          // PLD{<c>}{<q>} [<Rn>{, #-<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                          pld(CurrentCond(),
                                              MemOperand(Register(rn),
                                                         minus,
                                                         offset,
                                                         Offset));
                                          break;
                                        }
                                        default: {
                                          if (((instr & 0xf0000) == 0xf0000) ||
                                              ((instr & 0xf000) == 0xf000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rt = (instr >> 12) & 0xf;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          int32_t offset = instr & 0xff;
                                          // LDRB{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_2>}] ; T3 NOLINT(whitespace/line_length)
                                          ldrb(CurrentCond(),
                                               Best,
                                               Register(rt),
                                               MemOperand(Register(rn),
                                                          minus,
                                                          offset,
                                                          Offset));
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    case 0x00000200: {
                                      // 0xf8100e00
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      UnimplementedT32_32("LDRBT", instr);
                                      break;
                                    }
                                  }
                                  break;
                                }
                                case 0x00000d00: {
                                  // 0xf8100d00
                                  if (((instr & 0xf0000) == 0xf0000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  Sign sign((((instr >> 9) & 0x1) == 0) ? minus
                                                                        : plus);
                                  int32_t offset = instr & 0xff;
                                  // LDRB{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_2>}]! ; T3 NOLINT(whitespace/line_length)
                                  ldrb(CurrentCond(),
                                       Best,
                                       Register(rt),
                                       MemOperand(Register(rn),
                                                  sign,
                                                  offset,
                                                  PreIndex));
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                            case 0x00200000: {
                              // 0xf8300000
                              switch (instr & 0x00000d00) {
                                case 0x00000000: {
                                  // 0xf8300000
                                  switch (instr & 0x000002c0) {
                                    case 0x00000000: {
                                      // 0xf8300000
                                      switch (instr & 0x0000f000) {
                                        case 0x0000f000: {
                                          // 0xf830f000
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rn = (instr >> 16) & 0xf;
                                          Sign sign(plus);
                                          unsigned rm = instr & 0xf;
                                          Shift shift = LSL;
                                          uint32_t amount = (instr >> 4) & 0x3;
                                          AddrMode addrmode = Offset;
                                          // PLDW{<c>}{<q>} [<Rn>, {+}<Rm>{, LSL #<amount>}] ; T1 NOLINT(whitespace/line_length)
                                          pldw(CurrentCond(),
                                               MemOperand(Register(rn),
                                                          sign,
                                                          Register(rm),
                                                          shift,
                                                          amount,
                                                          addrmode));
                                          break;
                                        }
                                        default: {
                                          if (((instr & 0xf0000) == 0xf0000) ||
                                              ((instr & 0xf000) == 0xf000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rt = (instr >> 12) & 0xf;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          Sign sign(plus);
                                          unsigned rm = instr & 0xf;
                                          Shift shift = LSL;
                                          uint32_t amount = (instr >> 4) & 0x3;
                                          AddrMode addrmode = Offset;
                                          if ((rt < kNumberOfT32LowRegisters) &&
                                              (rn < kNumberOfT32LowRegisters) &&
                                              (rm < kNumberOfT32LowRegisters) &&
                                              shift.IsLSL() && (amount == 0) &&
                                              sign.IsPlus()) {
                                            // LDRH{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2 NOLINT(whitespace/line_length)
                                            ldrh(CurrentCond(),
                                                 Wide,
                                                 Register(rt),
                                                 MemOperand(Register(rn),
                                                            sign,
                                                            Register(rm),
                                                            addrmode));
                                          } else {
                                            // LDRH{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                                            ldrh(CurrentCond(),
                                                 Best,
                                                 Register(rt),
                                                 MemOperand(Register(rn),
                                                            sign,
                                                            Register(rm),
                                                            shift,
                                                            amount,
                                                            addrmode));
                                          }
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    default:
                                      UnallocatedT32(instr);
                                      break;
                                  }
                                  break;
                                }
                                case 0x00000900: {
                                  // 0xf8300900
                                  if (((instr & 0xf0000) == 0xf0000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  Sign sign((((instr >> 9) & 0x1) == 0) ? minus
                                                                        : plus);
                                  int32_t offset = instr & 0xff;
                                  // LDRH{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_2> ; T3 NOLINT(whitespace/line_length)
                                  ldrh(CurrentCond(),
                                       Best,
                                       Register(rt),
                                       MemOperand(Register(rn),
                                                  sign,
                                                  offset,
                                                  PostIndex));
                                  break;
                                }
                                case 0x00000c00: {
                                  // 0xf8300c00
                                  switch (instr & 0x00000200) {
                                    case 0x00000000: {
                                      // 0xf8300c00
                                      switch (instr & 0x0000f000) {
                                        case 0x0000f000: {
                                          // 0xf830fc00
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rn = (instr >> 16) & 0xf;
                                          int32_t offset = instr & 0xff;
                                          // PLDW{<c>}{<q>} [<Rn>{, #-<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                          pldw(CurrentCond(),
                                               MemOperand(Register(rn),
                                                          minus,
                                                          offset,
                                                          Offset));
                                          break;
                                        }
                                        default: {
                                          if (((instr & 0xf0000) == 0xf0000) ||
                                              ((instr & 0xf000) == 0xf000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rt = (instr >> 12) & 0xf;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          int32_t offset = instr & 0xff;
                                          // LDRH{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_2>}] ; T3 NOLINT(whitespace/line_length)
                                          ldrh(CurrentCond(),
                                               Best,
                                               Register(rt),
                                               MemOperand(Register(rn),
                                                          minus,
                                                          offset,
                                                          Offset));
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    case 0x00000200: {
                                      // 0xf8300e00
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      UnimplementedT32_32("LDRHT", instr);
                                      break;
                                    }
                                  }
                                  break;
                                }
                                case 0x00000d00: {
                                  // 0xf8300d00
                                  if (((instr & 0xf0000) == 0xf0000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  Sign sign((((instr >> 9) & 0x1) == 0) ? minus
                                                                        : plus);
                                  int32_t offset = instr & 0xff;
                                  // LDRH{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_2>}]! ; T3 NOLINT(whitespace/line_length)
                                  ldrh(CurrentCond(),
                                       Best,
                                       Register(rt),
                                       MemOperand(Register(rn),
                                                  sign,
                                                  offset,
                                                  PreIndex));
                                  break;
                                }
                                default:
                                  UnallocatedT32(instr);
                                  break;
                              }
                              break;
                            }
                            case 0x00800000: {
                              // 0xf8900000
                              switch (instr & 0x0000f000) {
                                case 0x0000f000: {
                                  // 0xf890f000
                                  if (((instr & 0xf0000) == 0xf0000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rn = (instr >> 16) & 0xf;
                                  int32_t offset = instr & 0xfff;
                                  // PLD{<c>}{<q>} [<Rn>{, #{+}<imm>}] ; T1
                                  pld(CurrentCond(),
                                      MemOperand(Register(rn),
                                                 plus,
                                                 offset,
                                                 Offset));
                                  break;
                                }
                                default: {
                                  if (((instr & 0xf0000) == 0xf0000) ||
                                      ((instr & 0xf000) == 0xf000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  int32_t offset = instr & 0xfff;
                                  if ((rt < kNumberOfT32LowRegisters) &&
                                      (rn < kNumberOfT32LowRegisters) &&
                                      ((offset >= 0) && (offset <= 31))) {
                                    // LDRB{<c>}.W <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                    ldrb(CurrentCond(),
                                         Wide,
                                         Register(rt),
                                         MemOperand(Register(rn),
                                                    plus,
                                                    offset,
                                                    Offset));
                                  } else {
                                    // LDRB{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                    ldrb(CurrentCond(),
                                         Best,
                                         Register(rt),
                                         MemOperand(Register(rn),
                                                    plus,
                                                    offset,
                                                    Offset));
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                            case 0x00a00000: {
                              // 0xf8b00000
                              switch (instr & 0x0000f000) {
                                case 0x0000f000: {
                                  // 0xf8b0f000
                                  if (((instr & 0xf0000) == 0xf0000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rn = (instr >> 16) & 0xf;
                                  int32_t offset = instr & 0xfff;
                                  // PLDW{<c>}{<q>} [<Rn>{, #{+}<imm>}] ; T1
                                  pldw(CurrentCond(),
                                       MemOperand(Register(rn),
                                                  plus,
                                                  offset,
                                                  Offset));
                                  break;
                                }
                                default: {
                                  if (((instr & 0xf0000) == 0xf0000) ||
                                      ((instr & 0xf000) == 0xf000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  int32_t offset = instr & 0xfff;
                                  if ((rt < kNumberOfT32LowRegisters) &&
                                      (rn < kNumberOfT32LowRegisters) &&
                                      ((offset >= 0) && (offset <= 62) &&
                                       ((offset & 1) == 0))) {
                                    // LDRH{<c>}.W <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                    ldrh(CurrentCond(),
                                         Wide,
                                         Register(rt),
                                         MemOperand(Register(rn),
                                                    plus,
                                                    offset,
                                                    Offset));
                                  } else {
                                    // LDRH{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                    ldrh(CurrentCond(),
                                         Best,
                                         Register(rt),
                                         MemOperand(Register(rn),
                                                    plus,
                                                    offset,
                                                    Offset));
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xf8500000
                      switch (instr & 0x00200000) {
                        case 0x00000000: {
                          // 0xf8500000
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xf85f0000
                              unsigned rt = (instr >> 12) & 0xf;
                              uint32_t U = (instr >> 23) & 0x1;
                              int32_t imm = instr & 0xfff;
                              if (U == 0) imm = -imm;
                              bool minus_zero = (imm == 0) && (U == 0);
                              Location location(imm, kT32PcDelta);
                              if ((imm >= -4095) && (imm <= 4095) &&
                                  ((rt < kNumberOfT32LowRegisters) &&
                                   (imm >= 0) && (imm <= 1020) &&
                                   ((imm & 3) == 0))) {
                                // LDR{<c>}.W <Rt>, <label> ; T2
                                if (minus_zero) {
                                  ldr(CurrentCond(),
                                      Wide,
                                      Register(rt),
                                      MemOperand(pc, minus, 0));
                                } else {
                                  ldr(CurrentCond(),
                                      Wide,
                                      Register(rt),
                                      &location);
                                }
                              } else if ((imm >= -4095) && (imm <= 4095)) {
                                // LDR{<c>}{<q>} <Rt>, <label> ; T2
                                if (minus_zero) {
                                  ldr(CurrentCond(),
                                      Best,
                                      Register(rt),
                                      MemOperand(pc, minus, 0));
                                } else {
                                  ldr(CurrentCond(),
                                      Best,
                                      Register(rt),
                                      &location);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x00800000) {
                                case 0x00000000: {
                                  // 0xf8500000
                                  switch (instr & 0x00000d00) {
                                    case 0x00000000: {
                                      // 0xf8500000
                                      if ((instr & 0x000002c0) == 0x00000000) {
                                        if (((instr & 0xf0000) == 0xf0000)) {
                                          UnallocatedT32(instr);
                                          return;
                                        }
                                        unsigned rt = (instr >> 12) & 0xf;
                                        unsigned rn = (instr >> 16) & 0xf;
                                        Sign sign(plus);
                                        unsigned rm = instr & 0xf;
                                        Shift shift = LSL;
                                        uint32_t amount = (instr >> 4) & 0x3;
                                        AddrMode addrmode = Offset;
                                        if ((rt < kNumberOfT32LowRegisters) &&
                                            (rn < kNumberOfT32LowRegisters) &&
                                            (rm < kNumberOfT32LowRegisters) &&
                                            shift.IsLSL() && (amount == 0) &&
                                            sign.IsPlus()) {
                                          // LDR{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2 NOLINT(whitespace/line_length)
                                          ldr(CurrentCond(),
                                              Wide,
                                              Register(rt),
                                              MemOperand(Register(rn),
                                                         sign,
                                                         Register(rm),
                                                         addrmode));
                                        } else {
                                          // LDR{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                                          ldr(CurrentCond(),
                                              Best,
                                              Register(rt),
                                              MemOperand(Register(rn),
                                                         sign,
                                                         Register(rm),
                                                         shift,
                                                         amount,
                                                         addrmode));
                                        }
                                      } else {
                                        UnallocatedT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000900: {
                                      // 0xf8500900
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      if (((Uint32((instr >> 16)) &
                                            Uint32(0xf)) == Uint32(0xd)) &&
                                          ((Uint32((instr >> 9)) &
                                            Uint32(0x1)) == Uint32(0x1)) &&
                                          ((Uint32(instr) & Uint32(0xff)) ==
                                           Uint32(0x4))) {
                                        unsigned rt = (instr >> 12) & 0xf;
                                        if ((rt <= 7) || (rt == kPCRegNum)) {
                                          // POP{<c>}.W <single_register_list> ; T4 NOLINT(whitespace/line_length)
                                          pop(CurrentCond(),
                                              Wide,
                                              Register(rt));
                                        } else {
                                          // POP{<c>}{<q>} <single_register_list> ; T4 NOLINT(whitespace/line_length)
                                          pop(CurrentCond(),
                                              Best,
                                              Register(rt));
                                        }
                                        return;
                                      }
                                      unsigned rt = (instr >> 12) & 0xf;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      Sign sign((((instr >> 9) & 0x1) == 0)
                                                    ? minus
                                                    : plus);
                                      int32_t offset = instr & 0xff;
                                      // LDR{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_2> ; T4 NOLINT(whitespace/line_length)
                                      ldr(CurrentCond(),
                                          Best,
                                          Register(rt),
                                          MemOperand(Register(rn),
                                                     sign,
                                                     offset,
                                                     PostIndex));
                                      break;
                                    }
                                    case 0x00000c00: {
                                      // 0xf8500c00
                                      switch (instr & 0x00000200) {
                                        case 0x00000000: {
                                          // 0xf8500c00
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rt = (instr >> 12) & 0xf;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          int32_t offset = instr & 0xff;
                                          // LDR{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_2>}] ; T4 NOLINT(whitespace/line_length)
                                          ldr(CurrentCond(),
                                              Best,
                                              Register(rt),
                                              MemOperand(Register(rn),
                                                         minus,
                                                         offset,
                                                         Offset));
                                          break;
                                        }
                                        case 0x00000200: {
                                          // 0xf8500e00
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          UnimplementedT32_32("LDRT", instr);
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    case 0x00000d00: {
                                      // 0xf8500d00
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned rt = (instr >> 12) & 0xf;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      Sign sign((((instr >> 9) & 0x1) == 0)
                                                    ? minus
                                                    : plus);
                                      int32_t offset = instr & 0xff;
                                      // LDR{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_2>}]! ; T4 NOLINT(whitespace/line_length)
                                      ldr(CurrentCond(),
                                          Best,
                                          Register(rt),
                                          MemOperand(Register(rn),
                                                     sign,
                                                     offset,
                                                     PreIndex));
                                      break;
                                    }
                                    default:
                                      UnallocatedT32(instr);
                                      break;
                                  }
                                  break;
                                }
                                case 0x00800000: {
                                  // 0xf8d00000
                                  if (((instr & 0xf0000) == 0xf0000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  int32_t offset = instr & 0xfff;
                                  if (((rt < kNumberOfT32LowRegisters) &&
                                       (rn < kNumberOfT32LowRegisters) &&
                                       ((offset >= 0) && (offset <= 124) &&
                                        ((offset & 3) == 0))) ||
                                      ((rt < kNumberOfT32LowRegisters) &&
                                       (rn == sp.GetCode()) &&
                                       ((offset >= 0) && (offset <= 1020) &&
                                        ((offset & 3) == 0)))) {
                                    // LDR{<c>}.W <Rt>, [<Rn>{, #{+}<imm_1>}] ; T3 NOLINT(whitespace/line_length)
                                    ldr(CurrentCond(),
                                        Wide,
                                        Register(rt),
                                        MemOperand(Register(rn),
                                                   plus,
                                                   offset,
                                                   Offset));
                                  } else {
                                    // LDR{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm_1>}] ; T3 NOLINT(whitespace/line_length)
                                    ldr(CurrentCond(),
                                        Best,
                                        Register(rt),
                                        MemOperand(Register(rn),
                                                   plus,
                                                   offset,
                                                   Offset));
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default:
                          UnallocatedT32(instr);
                          break;
                      }
                      break;
                    }
                    case 0x01000000: {
                      // 0xf9100000
                      switch (instr & 0x00200000) {
                        case 0x00000000: {
                          // 0xf9100000
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xf91f0000
                              switch (instr & 0x0000f000) {
                                case 0x0000f000: {
                                  // 0xf91ff000
                                  uint32_t U = (instr >> 23) & 0x1;
                                  int32_t imm = instr & 0xfff;
                                  if (U == 0) imm = -imm;
                                  bool minus_zero = (imm == 0) && (U == 0);
                                  Location location(imm, kT32PcDelta);
                                  // PLI{<c>}{<q>} <label> ; T3
                                  if (minus_zero) {
                                    pli(CurrentCond(),
                                        MemOperand(pc, minus, 0));
                                  } else {
                                    pli(CurrentCond(), &location);
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xf000) == 0xf000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  uint32_t U = (instr >> 23) & 0x1;
                                  int32_t imm = instr & 0xfff;
                                  if (U == 0) imm = -imm;
                                  bool minus_zero = (imm == 0) && (U == 0);
                                  Location location(imm, kT32PcDelta);
                                  // LDRSB{<c>}{<q>} <Rt>, <label> ; T1
                                  if (minus_zero) {
                                    ldrsb(CurrentCond(),
                                          Best,
                                          Register(rt),
                                          MemOperand(pc, minus, 0));
                                  } else {
                                    ldrsb(CurrentCond(),
                                          Register(rt),
                                          &location);
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x00800000) {
                                case 0x00000000: {
                                  // 0xf9100000
                                  switch (instr & 0x00000d00) {
                                    case 0x00000000: {
                                      // 0xf9100000
                                      switch (instr & 0x000002c0) {
                                        case 0x00000000: {
                                          // 0xf9100000
                                          switch (instr & 0x0000f000) {
                                            case 0x0000f000: {
                                              // 0xf910f000
                                              if (((instr & 0xf0000) ==
                                                   0xf0000)) {
                                                UnallocatedT32(instr);
                                                return;
                                              }
                                              unsigned rn = (instr >> 16) & 0xf;
                                              Sign sign(plus);
                                              unsigned rm = instr & 0xf;
                                              Shift shift = LSL;
                                              uint32_t amount =
                                                  (instr >> 4) & 0x3;
                                              AddrMode addrmode = Offset;
                                              // PLI{<c>}{<q>} [<Rn>, {+}<Rm>{, LSL #<amount>}] ; T1 NOLINT(whitespace/line_length)
                                              pli(CurrentCond(),
                                                  MemOperand(Register(rn),
                                                             sign,
                                                             Register(rm),
                                                             shift,
                                                             amount,
                                                             addrmode));
                                              break;
                                            }
                                            default: {
                                              if (((instr & 0xf0000) ==
                                                   0xf0000) ||
                                                  ((instr & 0xf000) ==
                                                   0xf000)) {
                                                UnallocatedT32(instr);
                                                return;
                                              }
                                              unsigned rt = (instr >> 12) & 0xf;
                                              unsigned rn = (instr >> 16) & 0xf;
                                              Sign sign(plus);
                                              unsigned rm = instr & 0xf;
                                              Shift shift = LSL;
                                              uint32_t amount =
                                                  (instr >> 4) & 0x3;
                                              AddrMode addrmode = Offset;
                                              if ((rt <
                                                   kNumberOfT32LowRegisters) &&
                                                  (rn <
                                                   kNumberOfT32LowRegisters) &&
                                                  (rm <
                                                   kNumberOfT32LowRegisters) &&
                                                  shift.IsLSL() &&
                                                  (amount == 0) &&
                                                  sign.IsPlus()) {
                                                // LDRSB{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2 NOLINT(whitespace/line_length)
                                                ldrsb(CurrentCond(),
                                                      Wide,
                                                      Register(rt),
                                                      MemOperand(Register(rn),
                                                                 sign,
                                                                 Register(rm),
                                                                 addrmode));
                                              } else {
                                                // LDRSB{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                                                ldrsb(CurrentCond(),
                                                      Best,
                                                      Register(rt),
                                                      MemOperand(Register(rn),
                                                                 sign,
                                                                 Register(rm),
                                                                 shift,
                                                                 amount,
                                                                 addrmode));
                                              }
                                              break;
                                            }
                                          }
                                          break;
                                        }
                                        default:
                                          UnallocatedT32(instr);
                                          break;
                                      }
                                      break;
                                    }
                                    case 0x00000900: {
                                      // 0xf9100900
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned rt = (instr >> 12) & 0xf;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      Sign sign((((instr >> 9) & 0x1) == 0)
                                                    ? minus
                                                    : plus);
                                      int32_t offset = instr & 0xff;
                                      // LDRSB{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_1> ; T2 NOLINT(whitespace/line_length)
                                      ldrsb(CurrentCond(),
                                            Best,
                                            Register(rt),
                                            MemOperand(Register(rn),
                                                       sign,
                                                       offset,
                                                       PostIndex));
                                      break;
                                    }
                                    case 0x00000c00: {
                                      // 0xf9100c00
                                      switch (instr & 0x00000200) {
                                        case 0x00000000: {
                                          // 0xf9100c00
                                          switch (instr & 0x0000f000) {
                                            case 0x0000f000: {
                                              // 0xf910fc00
                                              if (((instr & 0xf0000) ==
                                                   0xf0000)) {
                                                UnallocatedT32(instr);
                                                return;
                                              }
                                              unsigned rn = (instr >> 16) & 0xf;
                                              int32_t offset = instr & 0xff;
                                              // PLI{<c>}{<q>} [<Rn>{, #-<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                              pli(CurrentCond(),
                                                  MemOperand(Register(rn),
                                                             minus,
                                                             offset,
                                                             Offset));
                                              break;
                                            }
                                            default: {
                                              if (((instr & 0xf0000) ==
                                                   0xf0000) ||
                                                  ((instr & 0xf000) ==
                                                   0xf000)) {
                                                UnallocatedT32(instr);
                                                return;
                                              }
                                              unsigned rt = (instr >> 12) & 0xf;
                                              unsigned rn = (instr >> 16) & 0xf;
                                              int32_t offset = instr & 0xff;
                                              // LDRSB{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                              ldrsb(CurrentCond(),
                                                    Best,
                                                    Register(rt),
                                                    MemOperand(Register(rn),
                                                               minus,
                                                               offset,
                                                               Offset));
                                              break;
                                            }
                                          }
                                          break;
                                        }
                                        case 0x00000200: {
                                          // 0xf9100e00
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          UnimplementedT32_32("LDRSBT", instr);
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    case 0x00000d00: {
                                      // 0xf9100d00
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned rt = (instr >> 12) & 0xf;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      Sign sign((((instr >> 9) & 0x1) == 0)
                                                    ? minus
                                                    : plus);
                                      int32_t offset = instr & 0xff;
                                      // LDRSB{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_1>}]! ; T2 NOLINT(whitespace/line_length)
                                      ldrsb(CurrentCond(),
                                            Best,
                                            Register(rt),
                                            MemOperand(Register(rn),
                                                       sign,
                                                       offset,
                                                       PreIndex));
                                      break;
                                    }
                                    default:
                                      UnallocatedT32(instr);
                                      break;
                                  }
                                  break;
                                }
                                case 0x00800000: {
                                  // 0xf9900000
                                  switch (instr & 0x0000f000) {
                                    case 0x0000f000: {
                                      // 0xf990f000
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned rn = (instr >> 16) & 0xf;
                                      int32_t offset = instr & 0xfff;
                                      // PLI{<c>}{<q>} [<Rn>{, #{+}<imm>}] ; T1
                                      pli(CurrentCond(),
                                          MemOperand(Register(rn),
                                                     plus,
                                                     offset,
                                                     Offset));
                                      break;
                                    }
                                    default: {
                                      if (((instr & 0xf0000) == 0xf0000) ||
                                          ((instr & 0xf000) == 0xf000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned rt = (instr >> 12) & 0xf;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      int32_t offset = instr & 0xfff;
                                      // LDRSB{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1 NOLINT(whitespace/line_length)
                                      ldrsb(CurrentCond(),
                                            Best,
                                            Register(rt),
                                            MemOperand(Register(rn),
                                                       plus,
                                                       offset,
                                                       Offset));
                                      break;
                                    }
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00200000: {
                          // 0xf9300000
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xf93f0000
                              if (((instr & 0xf000) == 0xf000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rt = (instr >> 12) & 0xf;
                              uint32_t U = (instr >> 23) & 0x1;
                              int32_t imm = instr & 0xfff;
                              if (U == 0) imm = -imm;
                              bool minus_zero = (imm == 0) && (U == 0);
                              Location location(imm, kT32PcDelta);
                              // LDRSH{<c>}{<q>} <Rt>, <label> ; T1
                              if (minus_zero) {
                                ldrsh(CurrentCond(),
                                      Best,
                                      Register(rt),
                                      MemOperand(pc, minus, 0));
                              } else {
                                ldrsh(CurrentCond(), Register(rt), &location);
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x00800000) {
                                case 0x00000000: {
                                  // 0xf9300000
                                  switch (instr & 0x00000d00) {
                                    case 0x00000000: {
                                      // 0xf9300000
                                      if ((instr & 0x000002c0) == 0x00000000) {
                                        if (((instr & 0xf0000) == 0xf0000) ||
                                            ((instr & 0xf000) == 0xf000)) {
                                          UnallocatedT32(instr);
                                          return;
                                        }
                                        unsigned rt = (instr >> 12) & 0xf;
                                        unsigned rn = (instr >> 16) & 0xf;
                                        Sign sign(plus);
                                        unsigned rm = instr & 0xf;
                                        Shift shift = LSL;
                                        uint32_t amount = (instr >> 4) & 0x3;
                                        AddrMode addrmode = Offset;
                                        if ((rt < kNumberOfT32LowRegisters) &&
                                            (rn < kNumberOfT32LowRegisters) &&
                                            (rm < kNumberOfT32LowRegisters) &&
                                            shift.IsLSL() && (amount == 0) &&
                                            sign.IsPlus()) {
                                          // LDRSH{<c>}.W <Rt>, [<Rn>, #{+}<Rm>] ; T2 NOLINT(whitespace/line_length)
                                          ldrsh(CurrentCond(),
                                                Wide,
                                                Register(rt),
                                                MemOperand(Register(rn),
                                                           sign,
                                                           Register(rm),
                                                           addrmode));
                                        } else {
                                          // LDRSH{<c>}{<q>} <Rt>, [<Rn>, {+}<Rm>{, LSL #<imm>}] ; T2 NOLINT(whitespace/line_length)
                                          ldrsh(CurrentCond(),
                                                Best,
                                                Register(rt),
                                                MemOperand(Register(rn),
                                                           sign,
                                                           Register(rm),
                                                           shift,
                                                           amount,
                                                           addrmode));
                                        }
                                      } else {
                                        UnallocatedT32(instr);
                                      }
                                      break;
                                    }
                                    case 0x00000900: {
                                      // 0xf9300900
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned rt = (instr >> 12) & 0xf;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      Sign sign((((instr >> 9) & 0x1) == 0)
                                                    ? minus
                                                    : plus);
                                      int32_t offset = instr & 0xff;
                                      // LDRSH{<c>}{<q>} <Rt>, [<Rn>], #{+/-}<imm_1> ; T2 NOLINT(whitespace/line_length)
                                      ldrsh(CurrentCond(),
                                            Best,
                                            Register(rt),
                                            MemOperand(Register(rn),
                                                       sign,
                                                       offset,
                                                       PostIndex));
                                      break;
                                    }
                                    case 0x00000c00: {
                                      // 0xf9300c00
                                      switch (instr & 0x00000200) {
                                        case 0x00000000: {
                                          // 0xf9300c00
                                          if (((instr & 0xf0000) == 0xf0000) ||
                                              ((instr & 0xf000) == 0xf000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          unsigned rt = (instr >> 12) & 0xf;
                                          unsigned rn = (instr >> 16) & 0xf;
                                          int32_t offset = instr & 0xff;
                                          // LDRSH{<c>}{<q>} <Rt>, [<Rn>{, #-<imm_1>}] ; T2 NOLINT(whitespace/line_length)
                                          ldrsh(CurrentCond(),
                                                Best,
                                                Register(rt),
                                                MemOperand(Register(rn),
                                                           minus,
                                                           offset,
                                                           Offset));
                                          break;
                                        }
                                        case 0x00000200: {
                                          // 0xf9300e00
                                          if (((instr & 0xf0000) == 0xf0000)) {
                                            UnallocatedT32(instr);
                                            return;
                                          }
                                          UnimplementedT32_32("LDRSHT", instr);
                                          break;
                                        }
                                      }
                                      break;
                                    }
                                    case 0x00000d00: {
                                      // 0xf9300d00
                                      if (((instr & 0xf0000) == 0xf0000)) {
                                        UnallocatedT32(instr);
                                        return;
                                      }
                                      unsigned rt = (instr >> 12) & 0xf;
                                      unsigned rn = (instr >> 16) & 0xf;
                                      Sign sign((((instr >> 9) & 0x1) == 0)
                                                    ? minus
                                                    : plus);
                                      int32_t offset = instr & 0xff;
                                      // LDRSH{<c>}{<q>} <Rt>, [<Rn>{, #{+/-}<imm_1>}]! ; T2 NOLINT(whitespace/line_length)
                                      ldrsh(CurrentCond(),
                                            Best,
                                            Register(rt),
                                            MemOperand(Register(rn),
                                                       sign,
                                                       offset,
                                                       PreIndex));
                                      break;
                                    }
                                    default:
                                      UnallocatedT32(instr);
                                      break;
                                  }
                                  break;
                                }
                                case 0x00800000: {
                                  // 0xf9b00000
                                  if (((instr & 0xf0000) == 0xf0000) ||
                                      ((instr & 0xf000) == 0xf000)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rt = (instr >> 12) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  int32_t offset = instr & 0xfff;
                                  // LDRSH{<c>}{<q>} <Rt>, [<Rn>{, #{+}<imm>}] ; T1 NOLINT(whitespace/line_length)
                                  ldrsh(CurrentCond(),
                                        Best,
                                        Register(rt),
                                        MemOperand(Register(rn),
                                                   plus,
                                                   offset,
                                                   Offset));
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
              }
              break;
            }
            case 0x02000000: {
              // 0xea000000
              switch (instr & 0x11900000) {
                case 0x00000000: {
                  // 0xea000000
                  switch (instr & 0x00600000) {
                    case 0x00000000: {
                      // 0xea000000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xea000030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // AND{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          and_(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xea000030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // AND<c>.W {<Rd>}, <Rn>, <Rm> ; T2
                            and_(CurrentCond(),
                                 Wide,
                                 Register(rd),
                                 Register(rn),
                                 Register(rm));
                            if (((instr & 0xfff08000) != 0xea000000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // AND{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            and_(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rn),
                                 Operand(Register(rm),
                                         shift_operand.GetType(),
                                         shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xea000000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00200000: {
                      // 0xea200000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xea200030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // BIC{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          bic(CurrentCond(),
                              Best,
                              Register(rd),
                              Register(rn),
                              Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xea200030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // BIC<c>.W {<Rd>}, <Rn>, <Rm> ; T2
                            bic(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rn),
                                Register(rm));
                            if (((instr & 0xfff08000) != 0xea200000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // BIC{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            bic(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                Operand(Register(rm),
                                        shift_operand.GetType(),
                                        shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xea200000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xea400000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xea4f0000
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea4f0030
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              // RRX{<c>}{<q>} {<Rd>}, <Rm> ; T3
                              rrx(CurrentCond(), Register(rd), Register(rm));
                              if (((instr & 0xfffff0f0) != 0xea4f0030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x2))) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                if (amount == 0) amount = 32;
                                if (InITBlock() &&
                                    ((rd < kNumberOfT32LowRegisters) &&
                                     (rm < kNumberOfT32LowRegisters) &&
                                     ((amount >= 1) && (amount <= 32)))) {
                                  // ASR<c>.W {<Rd>}, <Rm>, #<imm> ; T3
                                  asr(CurrentCond(),
                                      Wide,
                                      Register(rd),
                                      Register(rm),
                                      amount);
                                  if (((instr & 0xffff8030) != 0xea4f0020)) {
                                    UnpredictableT32(instr);
                                  }
                                } else {
                                  // ASR{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                  asr(CurrentCond(),
                                      Best,
                                      Register(rd),
                                      Register(rm),
                                      amount);
                                  if (((instr & 0xffff8030) != 0xea4f0020)) {
                                    UnpredictableT32(instr);
                                  }
                                }
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x0)) &&
                                  ((instr & 0x000070c0) != 0x00000000)) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                if (InITBlock() &&
                                    ((rd < kNumberOfT32LowRegisters) &&
                                     (rm < kNumberOfT32LowRegisters) &&
                                     ((amount >= 1) && (amount <= 31)))) {
                                  // LSL<c>.W {<Rd>}, <Rm>, #<imm> ; T3
                                  lsl(CurrentCond(),
                                      Wide,
                                      Register(rd),
                                      Register(rm),
                                      amount);
                                  if (((instr & 0xffff8030) != 0xea4f0000)) {
                                    UnpredictableT32(instr);
                                  }
                                } else {
                                  // LSL{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                  lsl(CurrentCond(),
                                      Best,
                                      Register(rd),
                                      Register(rm),
                                      amount);
                                  if (((instr & 0xffff8030) != 0xea4f0000)) {
                                    UnpredictableT32(instr);
                                  }
                                }
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x1))) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                if (amount == 0) amount = 32;
                                if (InITBlock() &&
                                    ((rd < kNumberOfT32LowRegisters) &&
                                     (rm < kNumberOfT32LowRegisters) &&
                                     ((amount >= 1) && (amount <= 32)))) {
                                  // LSR<c>.W {<Rd>}, <Rm>, #<imm> ; T3
                                  lsr(CurrentCond(),
                                      Wide,
                                      Register(rd),
                                      Register(rm),
                                      amount);
                                  if (((instr & 0xffff8030) != 0xea4f0010)) {
                                    UnpredictableT32(instr);
                                  }
                                } else {
                                  // LSR{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                  lsr(CurrentCond(),
                                      Best,
                                      Register(rd),
                                      Register(rm),
                                      amount);
                                  if (((instr & 0xffff8030) != 0xea4f0010)) {
                                    UnpredictableT32(instr);
                                  }
                                }
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x3)) &&
                                  ((instr & 0x000070c0) != 0x00000000)) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                // ROR{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                ror(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    Register(rm),
                                    amount);
                                if (((instr & 0xffff8030) != 0xea4f0030)) {
                                  UnpredictableT32(instr);
                                }
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if ((instr & 0x00100000) == 0x00000000 &&
                                  (shift_operand.GetShift().IsLSL() &&
                                   (shift_operand.GetAmount() == 0))) {
                                // MOV<c>.W <Rd>, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                mov(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xea4f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if (InITBlock() &&
                                         (instr & 0x00100000) == 0x00000000 &&
                                         ((rd < kNumberOfT32LowRegisters) &&
                                          (rm < kNumberOfT32LowRegisters))) {
                                // MOV<c>.W <Rd>, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                mov(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xea4f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000) {
                                // MOV{<c>}{<q>} <Rd>, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                mov(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xea4f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea400030
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // ORR{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                              orr(CurrentCond(),
                                  Best,
                                  Register(rd),
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xea400030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000) ||
                                  ((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (InITBlock() &&
                                  (instr & 0x00100000) == 0x00000000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd == rn) &&
                                   (rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // ORR<c>.W {<Rd>}, <Rn>, <Rm> ; T2
                                orr(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    Register(rn),
                                    Register(rm));
                                if (((instr & 0xfff08000) != 0xea400000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000) {
                                // ORR{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                orr(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    Register(rn),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xfff08000) != 0xea400000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00600000: {
                      // 0xea600000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xea6f0000
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea6f0030
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              // MVN{<c>}{<q>} <Rd>, <Rm>, RRX ; T2
                              mvn(CurrentCond(),
                                  Best,
                                  Register(rd),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfffff0f0) != 0xea6f0030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (InITBlock() &&
                                  (instr & 0x00100000) == 0x00000000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // MVN<c>.W <Rd>, <Rm> ; T2
                                mvn(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    Register(rm));
                                if (((instr & 0xffff8000) != 0xea6f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000) {
                                // MVN{<c>}{<q>} <Rd>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                mvn(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xea6f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea600030
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // ORN{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T1
                              orn(CurrentCond(),
                                  Register(rd),
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xea600030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000) ||
                                  ((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              // ORN{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T1 NOLINT(whitespace/line_length)
                              orn(CurrentCond(),
                                  Register(rd),
                                  Register(rn),
                                  Operand(Register(rm),
                                          shift_operand.GetType(),
                                          shift_operand.GetAmount()));
                              if (((instr & 0xfff08000) != 0xea600000)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                  }
                  break;
                }
                case 0x00100000: {
                  // 0xea100000
                  switch (instr & 0x00600000) {
                    case 0x00000000: {
                      // 0xea100000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xea100f00
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea100f30
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // TST{<c>}{<q>} <Rn>, <Rm>, RRX ; T2
                              tst(CurrentCond(),
                                  Best,
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0fff0) != 0xea100f30)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rn < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // TST{<c>}.W <Rn>, <Rm> ; T2
                                tst(CurrentCond(),
                                    Wide,
                                    Register(rn),
                                    Register(rm));
                                if (((instr & 0xfff08f00) != 0xea100f00)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                // TST{<c>}{<q>} <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                tst(CurrentCond(),
                                    Best,
                                    Register(rn),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xfff08f00) != 0xea100f00)) {
                                  UnpredictableT32(instr);
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea100030
                              if (((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // ANDS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                              ands(CurrentCond(),
                                   Best,
                                   Register(rd),
                                   Register(rn),
                                   Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xea100030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30) ||
                                  ((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (OutsideITBlock() &&
                                  (instr & 0x00100000) == 0x00100000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd == rn) &&
                                   (rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // ANDS.W {<Rd>}, <Rn>, <Rm> ; T2
                                ands(Condition::None(),
                                     Wide,
                                     Register(rd),
                                     Register(rn),
                                     Register(rm));
                                if (((instr & 0xfff08000) != 0xea100000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00100000) {
                                // ANDS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                ands(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Register(rn),
                                     Operand(Register(rm),
                                             shift_operand.GetType(),
                                             shift_operand.GetAmount()));
                                if (((instr & 0xfff08000) != 0xea100000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00200000: {
                      // 0xea300000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xea300030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // BICS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          bics(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xea300030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (OutsideITBlock() &&
                              (instr & 0x00100000) == 0x00100000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // BICS.W {<Rd>}, <Rn>, <Rm> ; T2
                            bics(Condition::None(),
                                 Wide,
                                 Register(rd),
                                 Register(rn),
                                 Register(rm));
                            if (((instr & 0xfff08000) != 0xea300000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00100000) {
                            // BICS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            bics(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rn),
                                 Operand(Register(rm),
                                         shift_operand.GetType(),
                                         shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xea300000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xea500000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xea5f0000
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea5f0030
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              // RRXS{<c>}{<q>} {<Rd>}, <Rm> ; T3
                              rrxs(CurrentCond(), Register(rd), Register(rm));
                              if (((instr & 0xfffff0f0) != 0xea5f0030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x2))) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                if (amount == 0) amount = 32;
                                if (OutsideITBlock() &&
                                    ((rd < kNumberOfT32LowRegisters) &&
                                     (rm < kNumberOfT32LowRegisters) &&
                                     ((amount >= 1) && (amount <= 32)))) {
                                  // ASRS.W {<Rd>}, <Rm>, #<imm> ; T3
                                  asrs(Condition::None(),
                                       Wide,
                                       Register(rd),
                                       Register(rm),
                                       amount);
                                  if (((instr & 0xffff8030) != 0xea5f0020)) {
                                    UnpredictableT32(instr);
                                  }
                                } else {
                                  // ASRS{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                  asrs(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       Register(rm),
                                       amount);
                                  if (((instr & 0xffff8030) != 0xea5f0020)) {
                                    UnpredictableT32(instr);
                                  }
                                }
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x0)) &&
                                  ((instr & 0x000070c0) != 0x00000000)) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                if (OutsideITBlock() &&
                                    ((rd < kNumberOfT32LowRegisters) &&
                                     (rm < kNumberOfT32LowRegisters) &&
                                     ((amount >= 1) && (amount <= 31)))) {
                                  // LSLS.W {<Rd>}, <Rm>, #<imm> ; T3
                                  lsls(Condition::None(),
                                       Wide,
                                       Register(rd),
                                       Register(rm),
                                       amount);
                                  if (((instr & 0xffff8030) != 0xea5f0000)) {
                                    UnpredictableT32(instr);
                                  }
                                } else {
                                  // LSLS{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                  lsls(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       Register(rm),
                                       amount);
                                  if (((instr & 0xffff8030) != 0xea5f0000)) {
                                    UnpredictableT32(instr);
                                  }
                                }
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x1))) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                if (amount == 0) amount = 32;
                                if (OutsideITBlock() &&
                                    ((rd < kNumberOfT32LowRegisters) &&
                                     (rm < kNumberOfT32LowRegisters) &&
                                     ((amount >= 1) && (amount <= 32)))) {
                                  // LSRS.W {<Rd>}, <Rm>, #<imm> ; T3
                                  lsrs(Condition::None(),
                                       Wide,
                                       Register(rd),
                                       Register(rm),
                                       amount);
                                  if (((instr & 0xffff8030) != 0xea5f0010)) {
                                    UnpredictableT32(instr);
                                  }
                                } else {
                                  // LSRS{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                  lsrs(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       Register(rm),
                                       amount);
                                  if (((instr & 0xffff8030) != 0xea5f0010)) {
                                    UnpredictableT32(instr);
                                  }
                                }
                                return;
                              }
                              if (((Uint32((instr >> 4)) & Uint32(0x3)) ==
                                   Uint32(0x3)) &&
                                  ((instr & 0x000070c0) != 0x00000000)) {
                                unsigned rd = (instr >> 8) & 0xf;
                                unsigned rm = instr & 0xf;
                                uint32_t amount = ((instr >> 6) & 0x3) |
                                                  ((instr >> 10) & 0x1c);
                                // RORS{<c>}{<q>} {<Rd>}, <Rm>, #<imm> ; T3
                                rors(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Register(rm),
                                     amount);
                                if (((instr & 0xffff8030) != 0xea5f0030)) {
                                  UnpredictableT32(instr);
                                }
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (OutsideITBlock() &&
                                  (instr & 0x00100000) == 0x00100000 &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // MOVS.W <Rd>, <Rm> {, <shift> #<amount> } ; T3
                                movs(Condition::None(),
                                     Wide,
                                     Register(rd),
                                     Operand(Register(rm),
                                             shift_operand.GetType(),
                                             shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xea5f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00100000) {
                                // MOVS{<c>}{<q>} <Rd>, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                movs(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Operand(Register(rm),
                                             shift_operand.GetType(),
                                             shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xea5f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea500030
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // ORRS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                              orrs(CurrentCond(),
                                   Best,
                                   Register(rd),
                                   Register(rn),
                                   Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xea500030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000) ||
                                  ((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (OutsideITBlock() &&
                                  (instr & 0x00100000) == 0x00100000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd == rn) &&
                                   (rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // ORRS.W {<Rd>}, <Rn>, <Rm> ; T2
                                orrs(Condition::None(),
                                     Wide,
                                     Register(rd),
                                     Register(rn),
                                     Register(rm));
                                if (((instr & 0xfff08000) != 0xea500000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00100000) {
                                // ORRS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                orrs(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Register(rn),
                                     Operand(Register(rm),
                                             shift_operand.GetType(),
                                             shift_operand.GetAmount()));
                                if (((instr & 0xfff08000) != 0xea500000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00600000: {
                      // 0xea700000
                      switch (instr & 0x000f0000) {
                        case 0x000f0000: {
                          // 0xea7f0000
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea7f0030
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              // MVNS{<c>}{<q>} <Rd>, <Rm>, RRX ; T2
                              mvns(CurrentCond(),
                                   Best,
                                   Register(rd),
                                   Operand(Register(rm), RRX));
                              if (((instr & 0xfffff0f0) != 0xea7f0030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (OutsideITBlock() &&
                                  (instr & 0x00100000) == 0x00100000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // MVNS.W <Rd>, <Rm> ; T2
                                mvns(Condition::None(),
                                     Wide,
                                     Register(rd),
                                     Register(rm));
                                if (((instr & 0xffff8000) != 0xea7f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00100000) {
                                // MVNS{<c>}{<q>} <Rd>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                mvns(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Operand(Register(rm),
                                             shift_operand.GetType(),
                                             shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xea7f0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea700030
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // ORNS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T1
                              orns(CurrentCond(),
                                   Register(rd),
                                   Register(rn),
                                   Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xea700030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000) ||
                                  ((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              // ORNS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T1 NOLINT(whitespace/line_length)
                              orns(CurrentCond(),
                                   Register(rd),
                                   Register(rn),
                                   Operand(Register(rm),
                                           shift_operand.GetType(),
                                           shift_operand.GetAmount()));
                              if (((instr & 0xfff08000) != 0xea700000)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                  }
                  break;
                }
                case 0x00800000: {
                  // 0xea800000
                  switch (instr & 0x00600000) {
                    case 0x00000000: {
                      // 0xea800000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xea800030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // EOR{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          eor(CurrentCond(),
                              Best,
                              Register(rd),
                              Register(rn),
                              Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xea800030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // EOR<c>.W {<Rd>}, <Rn>, <Rm> ; T2
                            eor(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rn),
                                Register(rm));
                            if (((instr & 0xfff08000) != 0xea800000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // EOR{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            eor(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                Operand(Register(rm),
                                        shift_operand.GetType(),
                                        shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xea800000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xeac00000
                      switch (instr & 0x00000030) {
                        case 0x00000000: {
                          // 0xeac00000
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          uint32_t amount =
                              ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                          // PKHBT{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, LSL #<imm> } ; T1 NOLINT(whitespace/line_length)
                          pkhbt(CurrentCond(),
                                Register(rd),
                                Register(rn),
                                Operand(Register(rm), LSL, amount));
                          if (((instr & 0xfff08030) != 0xeac00000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        case 0x00000020: {
                          // 0xeac00020
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          uint32_t amount =
                              ((instr >> 6) & 0x3) | ((instr >> 10) & 0x1c);
                          if (amount == 0) amount = 32;
                          // PKHTB{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, ASR #<imm> } ; T1 NOLINT(whitespace/line_length)
                          pkhtb(CurrentCond(),
                                Register(rd),
                                Register(rn),
                                Operand(Register(rm), ASR, amount));
                          if (((instr & 0xfff08030) != 0xeac00020)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default:
                          UnallocatedT32(instr);
                          break;
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x00900000: {
                  // 0xea900000
                  switch (instr & 0x00600000) {
                    case 0x00000000: {
                      // 0xea900000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xea900f00
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea900f30
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // TEQ{<c>}{<q>} <Rn>, <Rm>, RRX ; T1
                              teq(CurrentCond(),
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0fff0) != 0xea900f30)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              // TEQ{<c>}{<q>} <Rn>, <Rm> {, <shift> #<amount> } ; T1 NOLINT(whitespace/line_length)
                              teq(CurrentCond(),
                                  Register(rn),
                                  Operand(Register(rm),
                                          shift_operand.GetType(),
                                          shift_operand.GetAmount()));
                              if (((instr & 0xfff08f00) != 0xea900f00)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xea900030
                              if (((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // EORS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                              eors(CurrentCond(),
                                   Best,
                                   Register(rd),
                                   Register(rn),
                                   Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xea900030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30) ||
                                  ((instr & 0xf00) == 0xf00)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (OutsideITBlock() &&
                                  (instr & 0x00100000) == 0x00100000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd == rn) &&
                                   (rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // EORS.W {<Rd>}, <Rn>, <Rm> ; T2
                                eors(Condition::None(),
                                     Wide,
                                     Register(rd),
                                     Register(rn),
                                     Register(rm));
                                if (((instr & 0xfff08000) != 0xea900000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00100000) {
                                // EORS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                eors(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Register(rn),
                                     Operand(Register(rm),
                                             shift_operand.GetType(),
                                             shift_operand.GetAmount()));
                                if (((instr & 0xfff08000) != 0xea900000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x01000000: {
                  // 0xeb000000
                  switch (instr & 0x00600000) {
                    case 0x00000000: {
                      // 0xeb000000
                      switch (instr & 0x000f0000) {
                        case 0x000d0000: {
                          // 0xeb0d0000
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xeb0d0030
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              // ADD{<c>}{<q>} {<Rd>}, SP, <Rm>, RRX ; T3
                              add(CurrentCond(),
                                  Best,
                                  Register(rd),
                                  sp,
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfffff0f0) != 0xeb0d0030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if ((instr & 0x00100000) == 0x00000000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  (((rd == rm)) || ((rd == sp.GetCode())))) {
                                // ADD{<c>}.W {<Rd>}, SP, <Rm> ; T3
                                add(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    sp,
                                    Register(rm));
                                if (((instr & 0xffff8000) != 0xeb0d0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000) {
                                // ADD{<c>}{<q>} {<Rd>}, SP, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                add(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    sp,
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xeb0d0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xeb000030
                              if (((instr & 0xf0000) == 0xd0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // ADD{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T3
                              add(CurrentCond(),
                                  Best,
                                  Register(rd),
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xeb000030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xd0000) ||
                                  ((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (InITBlock() &&
                                  (instr & 0x00100000) == 0x00000000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rn < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // ADD<c>.W {<Rd>}, <Rn>, <Rm> ; T3
                                add(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    Register(rn),
                                    Register(rm));
                                if (((instr & 0xfff08000) != 0xeb000000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000 &&
                                         shift_operand.GetShift().IsLSL() &&
                                         (shift_operand.GetAmount() == 0) &&
                                         ((rd == rn))) {
                                // ADD<c>.W {<Rd>}, <Rn>, <Rm> ; T3
                                add(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    Register(rn),
                                    Register(rm));
                                if (((instr & 0xfff08000) != 0xeb000000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000) {
                                // ADD{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                add(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    Register(rn),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xfff08000) != 0xeb000000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xeb400000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xeb400030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // ADC{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          adc(CurrentCond(),
                              Best,
                              Register(rd),
                              Register(rn),
                              Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xeb400030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // ADC<c>.W {<Rd>}, <Rn>, <Rm> ; T2
                            adc(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rn),
                                Register(rm));
                            if (((instr & 0xfff08000) != 0xeb400000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // ADC{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            adc(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                Operand(Register(rm),
                                        shift_operand.GetType(),
                                        shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xeb400000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00600000: {
                      // 0xeb600000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xeb600030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // SBC{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          sbc(CurrentCond(),
                              Best,
                              Register(rd),
                              Register(rn),
                              Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xeb600030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (InITBlock() &&
                              (instr & 0x00100000) == 0x00000000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // SBC<c>.W {<Rd>}, <Rn>, <Rm> ; T2
                            sbc(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rn),
                                Register(rm));
                            if (((instr & 0xfff08000) != 0xeb600000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00000000) {
                            // SBC{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            sbc(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rn),
                                Operand(Register(rm),
                                        shift_operand.GetType(),
                                        shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xeb600000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x01100000: {
                  // 0xeb100000
                  switch (instr & 0x00600000) {
                    case 0x00000000: {
                      // 0xeb100000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xeb100f00
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xeb100f30
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // CMN{<c>}{<q>} <Rn>, <Rm>, RRX ; T2
                              cmn(CurrentCond(),
                                  Best,
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0fff0) != 0xeb100f30)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rn < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // CMN{<c>}.W <Rn>, <Rm> ; T2
                                cmn(CurrentCond(),
                                    Wide,
                                    Register(rn),
                                    Register(rm));
                                if (((instr & 0xfff08f00) != 0xeb100f00)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                // CMN{<c>}{<q>} <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                cmn(CurrentCond(),
                                    Best,
                                    Register(rn),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xfff08f00) != 0xeb100f00)) {
                                  UnpredictableT32(instr);
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000f0000) {
                            case 0x000d0000: {
                              // 0xeb1d0000
                              switch (instr & 0x000070f0) {
                                case 0x00000030: {
                                  // 0xeb1d0030
                                  if (((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // ADDS{<c>}{<q>} {<Rd>}, SP, <Rm>, RRX ; T3
                                  adds(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       sp,
                                       Operand(Register(rm), RRX));
                                  if (((instr & 0xfffff0f0) != 0xeb1d0030)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0x70f0) == 0x30) ||
                                      ((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  ImmediateShiftOperand
                                      shift_operand((instr >> 4) & 0x3,
                                                    ((instr >> 6) & 0x3) |
                                                        ((instr >> 10) & 0x1c));
                                  // ADDS{<c>}{<q>} {<Rd>}, SP, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                  adds(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       sp,
                                       Operand(Register(rm),
                                               shift_operand.GetType(),
                                               shift_operand.GetAmount()));
                                  if (((instr & 0xffff8000) != 0xeb1d0000)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x000070f0) {
                                case 0x00000030: {
                                  // 0xeb100030
                                  if (((instr & 0xf0000) == 0xd0000) ||
                                      ((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // ADDS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T3
                                  adds(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       Register(rn),
                                       Operand(Register(rm), RRX));
                                  if (((instr & 0xfff0f0f0) != 0xeb100030)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xf0000) == 0xd0000) ||
                                      ((instr & 0x70f0) == 0x30) ||
                                      ((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  ImmediateShiftOperand
                                      shift_operand((instr >> 4) & 0x3,
                                                    ((instr >> 6) & 0x3) |
                                                        ((instr >> 10) & 0x1c));
                                  if (OutsideITBlock() &&
                                      (instr & 0x00100000) == 0x00100000 &&
                                      shift_operand.GetShift().IsLSL() &&
                                      (shift_operand.GetAmount() == 0) &&
                                      ((rd < kNumberOfT32LowRegisters) &&
                                       (rn < kNumberOfT32LowRegisters) &&
                                       (rm < kNumberOfT32LowRegisters))) {
                                    // ADDS.W {<Rd>}, <Rn>, <Rm> ; T3
                                    adds(Condition::None(),
                                         Wide,
                                         Register(rd),
                                         Register(rn),
                                         Register(rm));
                                    if (((instr & 0xfff08000) != 0xeb100000)) {
                                      UnpredictableT32(instr);
                                    }
                                  } else if ((instr & 0x00100000) ==
                                             0x00100000) {
                                    // ADDS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T3 NOLINT(whitespace/line_length)
                                    adds(CurrentCond(),
                                         Best,
                                         Register(rd),
                                         Register(rn),
                                         Operand(Register(rm),
                                                 shift_operand.GetType(),
                                                 shift_operand.GetAmount()));
                                    if (((instr & 0xfff08000) != 0xeb100000)) {
                                      UnpredictableT32(instr);
                                    }
                                  } else {
                                    UnallocatedT32(instr);
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xeb500000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xeb500030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // ADCS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          adcs(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xeb500030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (OutsideITBlock() &&
                              (instr & 0x00100000) == 0x00100000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // ADCS.W {<Rd>}, <Rn>, <Rm> ; T2
                            adcs(Condition::None(),
                                 Wide,
                                 Register(rd),
                                 Register(rn),
                                 Register(rm));
                            if (((instr & 0xfff08000) != 0xeb500000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00100000) {
                            // ADCS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            adcs(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rn),
                                 Operand(Register(rm),
                                         shift_operand.GetType(),
                                         shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xeb500000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00600000: {
                      // 0xeb700000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xeb700030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // SBCS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                          sbcs(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xeb700030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          if (OutsideITBlock() &&
                              (instr & 0x00100000) == 0x00100000 &&
                              shift_operand.GetShift().IsLSL() &&
                              (shift_operand.GetAmount() == 0) &&
                              ((rd == rn) && (rd < kNumberOfT32LowRegisters) &&
                               (rm < kNumberOfT32LowRegisters))) {
                            // SBCS.W {<Rd>}, <Rn>, <Rm> ; T2
                            sbcs(Condition::None(),
                                 Wide,
                                 Register(rd),
                                 Register(rn),
                                 Register(rm));
                            if (((instr & 0xfff08000) != 0xeb700000)) {
                              UnpredictableT32(instr);
                            }
                          } else if ((instr & 0x00100000) == 0x00100000) {
                            // SBCS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                            sbcs(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rn),
                                 Operand(Register(rm),
                                         shift_operand.GetType(),
                                         shift_operand.GetAmount()));
                            if (((instr & 0xfff08000) != 0xeb700000)) {
                              UnpredictableT32(instr);
                            }
                          } else {
                            UnallocatedT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x01800000: {
                  // 0xeb800000
                  switch (instr & 0x00600000) {
                    case 0x00200000: {
                      // 0xeba00000
                      switch (instr & 0x000f0000) {
                        case 0x000d0000: {
                          // 0xebad0000
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xebad0030
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              // SUB{<c>}{<q>} {<Rd>}, SP, <Rm>, RRX ; T1
                              sub(CurrentCond(),
                                  Best,
                                  Register(rd),
                                  sp,
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfffff0f0) != 0xebad0030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if ((instr & 0x00100000) == 0x00000000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0)) {
                                // SUB{<c>} {<Rd>}, SP, <Rm> ; T1
                                sub(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    sp,
                                    Register(rm));
                                if (((instr & 0xffff8000) != 0xebad0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000) {
                                // SUB{<c>}{<q>} {<Rd>}, SP, <Rm> {, <shift> #<amount> } ; T1 NOLINT(whitespace/line_length)
                                sub(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    sp,
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xffff8000) != 0xebad0000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xeba00030
                              if (((instr & 0xf0000) == 0xd0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // SUB{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                              sub(CurrentCond(),
                                  Best,
                                  Register(rd),
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0f0f0) != 0xeba00030)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xd0000) ||
                                  ((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (InITBlock() &&
                                  (instr & 0x00100000) == 0x00000000 &&
                                  shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rn < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // SUB<c>.W {<Rd>}, <Rn>, <Rm> ; T2
                                sub(CurrentCond(),
                                    Wide,
                                    Register(rd),
                                    Register(rn),
                                    Register(rm));
                                if (((instr & 0xfff08000) != 0xeba00000)) {
                                  UnpredictableT32(instr);
                                }
                              } else if ((instr & 0x00100000) == 0x00000000) {
                                // SUB{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                sub(CurrentCond(),
                                    Best,
                                    Register(rd),
                                    Register(rn),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xfff08000) != 0xeba00000)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                UnallocatedT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xebc00000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xebc00030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // RSB{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T1
                          rsb(CurrentCond(),
                              Best,
                              Register(rd),
                              Register(rn),
                              Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xebc00030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          // RSB{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T1 NOLINT(whitespace/line_length)
                          rsb(CurrentCond(),
                              Best,
                              Register(rd),
                              Register(rn),
                              Operand(Register(rm),
                                      shift_operand.GetType(),
                                      shift_operand.GetAmount()));
                          if (((instr & 0xfff08000) != 0xebc00000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x01900000: {
                  // 0xeb900000
                  switch (instr & 0x00600000) {
                    case 0x00200000: {
                      // 0xebb00000
                      switch (instr & 0x00000f00) {
                        case 0x00000f00: {
                          // 0xebb00f00
                          switch (instr & 0x000070f0) {
                            case 0x00000030: {
                              // 0xebb00f30
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              // CMP{<c>}{<q>} <Rn>, <Rm>, RRX ; T3
                              cmp(CurrentCond(),
                                  Best,
                                  Register(rn),
                                  Operand(Register(rm), RRX));
                              if (((instr & 0xfff0fff0) != 0xebb00f30)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0x70f0) == 0x30)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              ImmediateShiftOperand
                                  shift_operand((instr >> 4) & 0x3,
                                                ((instr >> 6) & 0x3) |
                                                    ((instr >> 10) & 0x1c));
                              if (shift_operand.GetShift().IsLSL() &&
                                  (shift_operand.GetAmount() == 0)) {
                                // CMP{<c>}.W <Rn>, <Rm> ; T3
                                cmp(CurrentCond(),
                                    Wide,
                                    Register(rn),
                                    Register(rm));
                                if (((instr & 0xfff08f00) != 0xebb00f00)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                // CMP{<c>}{<q>} <Rn>, <Rm>, <shift> #<amount> ; T3 NOLINT(whitespace/line_length)
                                cmp(CurrentCond(),
                                    Best,
                                    Register(rn),
                                    Operand(Register(rm),
                                            shift_operand.GetType(),
                                            shift_operand.GetAmount()));
                                if (((instr & 0xfff08f00) != 0xebb00f00)) {
                                  UnpredictableT32(instr);
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default: {
                          switch (instr & 0x000f0000) {
                            case 0x000d0000: {
                              // 0xebbd0000
                              switch (instr & 0x000070f0) {
                                case 0x00000030: {
                                  // 0xebbd0030
                                  if (((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // SUBS{<c>}{<q>} {<Rd>}, SP, <Rm>, RRX ; T1
                                  subs(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       sp,
                                       Operand(Register(rm), RRX));
                                  if (((instr & 0xfffff0f0) != 0xebbd0030)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0x70f0) == 0x30) ||
                                      ((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  ImmediateShiftOperand
                                      shift_operand((instr >> 4) & 0x3,
                                                    ((instr >> 6) & 0x3) |
                                                        ((instr >> 10) & 0x1c));
                                  // SUBS{<c>}{<q>} {<Rd>}, SP, <Rm> {, <shift> #<amount> } ; T1 NOLINT(whitespace/line_length)
                                  subs(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       sp,
                                       Operand(Register(rm),
                                               shift_operand.GetType(),
                                               shift_operand.GetAmount()));
                                  if (((instr & 0xffff8000) != 0xebbd0000)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                            default: {
                              switch (instr & 0x000070f0) {
                                case 0x00000030: {
                                  // 0xebb00030
                                  if (((instr & 0xf0000) == 0xd0000) ||
                                      ((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  // SUBS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T2
                                  subs(CurrentCond(),
                                       Best,
                                       Register(rd),
                                       Register(rn),
                                       Operand(Register(rm), RRX));
                                  if (((instr & 0xfff0f0f0) != 0xebb00030)) {
                                    UnpredictableT32(instr);
                                  }
                                  break;
                                }
                                default: {
                                  if (((instr & 0xf0000) == 0xd0000) ||
                                      ((instr & 0x70f0) == 0x30) ||
                                      ((instr & 0xf00) == 0xf00)) {
                                    UnallocatedT32(instr);
                                    return;
                                  }
                                  unsigned rd = (instr >> 8) & 0xf;
                                  unsigned rn = (instr >> 16) & 0xf;
                                  unsigned rm = instr & 0xf;
                                  ImmediateShiftOperand
                                      shift_operand((instr >> 4) & 0x3,
                                                    ((instr >> 6) & 0x3) |
                                                        ((instr >> 10) & 0x1c));
                                  if (OutsideITBlock() &&
                                      (instr & 0x00100000) == 0x00100000 &&
                                      shift_operand.GetShift().IsLSL() &&
                                      (shift_operand.GetAmount() == 0) &&
                                      ((rd < kNumberOfT32LowRegisters) &&
                                       (rn < kNumberOfT32LowRegisters) &&
                                       (rm < kNumberOfT32LowRegisters))) {
                                    // SUBS.W {<Rd>}, <Rn>, <Rm> ; T2
                                    subs(Condition::None(),
                                         Wide,
                                         Register(rd),
                                         Register(rn),
                                         Register(rm));
                                    if (((instr & 0xfff08000) != 0xebb00000)) {
                                      UnpredictableT32(instr);
                                    }
                                  } else if ((instr & 0x00100000) ==
                                             0x00100000) {
                                    // SUBS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T2 NOLINT(whitespace/line_length)
                                    subs(CurrentCond(),
                                         Best,
                                         Register(rd),
                                         Register(rn),
                                         Operand(Register(rm),
                                                 shift_operand.GetType(),
                                                 shift_operand.GetAmount()));
                                    if (((instr & 0xfff08000) != 0xebb00000)) {
                                      UnpredictableT32(instr);
                                    }
                                  } else {
                                    UnallocatedT32(instr);
                                  }
                                  break;
                                }
                              }
                              break;
                            }
                          }
                          break;
                        }
                      }
                      break;
                    }
                    case 0x00400000: {
                      // 0xebd00000
                      switch (instr & 0x000070f0) {
                        case 0x00000030: {
                          // 0xebd00030
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          // RSBS{<c>}{<q>} {<Rd>}, <Rn>, <Rm>, RRX ; T1
                          rsbs(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               Operand(Register(rm), RRX));
                          if (((instr & 0xfff0f0f0) != 0xebd00030)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                        default: {
                          if (((instr & 0x70f0) == 0x30)) {
                            UnallocatedT32(instr);
                            return;
                          }
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rn = (instr >> 16) & 0xf;
                          unsigned rm = instr & 0xf;
                          ImmediateShiftOperand
                              shift_operand((instr >> 4) & 0x3,
                                            ((instr >> 6) & 0x3) |
                                                ((instr >> 10) & 0x1c));
                          // RSBS{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, <shift> #<amount> } ; T1 NOLINT(whitespace/line_length)
                          rsbs(CurrentCond(),
                               Best,
                               Register(rd),
                               Register(rn),
                               Operand(Register(rm),
                                       shift_operand.GetType(),
                                       shift_operand.GetAmount()));
                          if (((instr & 0xfff08000) != 0xebd00000)) {
                            UnpredictableT32(instr);
                          }
                          break;
                        }
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x10000000: {
                  // 0xfa000000
                  switch (instr & 0x0000f080) {
                    case 0x0000f000: {
                      // 0xfa00f000
                      if ((instr & 0x00000070) == 0x00000000) {
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x2))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (InITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // ASR<c>.W {<Rd>}, <Rm>, <Rs> ; T2
                            asr(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          } else {
                            // ASR{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            asr(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          }
                          return;
                        }
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x0))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (InITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // LSL<c>.W {<Rd>}, <Rm>, <Rs> ; T2
                            lsl(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          } else {
                            // LSL{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            lsl(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          }
                          return;
                        }
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x1))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (InITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // LSR<c>.W {<Rd>}, <Rm>, <Rs> ; T2
                            lsr(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          } else {
                            // LSR{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            lsr(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          }
                          return;
                        }
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x3))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (InITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // ROR<c>.W {<Rd>}, <Rm>, <Rs> ; T2
                            ror(CurrentCond(),
                                Wide,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          } else {
                            // ROR{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            ror(CurrentCond(),
                                Best,
                                Register(rd),
                                Register(rm),
                                Register(rs));
                          }
                          return;
                        }
                        unsigned rd = (instr >> 8) & 0xf;
                        unsigned rm = (instr >> 16) & 0xf;
                        Shift shift((instr >> 21) & 0x3);
                        unsigned rs = instr & 0xf;
                        if (InITBlock() && (instr & 0x00100000) == 0x00000000 &&
                            ((rd < kNumberOfT32LowRegisters) &&
                             (rm < kNumberOfT32LowRegisters) &&
                             (rs < kNumberOfT32LowRegisters))) {
                          // MOV<c>.W <Rd>, <Rm>, <shift> <Rs> ; T2
                          mov(CurrentCond(),
                              Wide,
                              Register(rd),
                              Operand(Register(rm),
                                      shift.GetType(),
                                      Register(rs)));
                        } else if ((instr & 0x00100000) == 0x00000000) {
                          // MOV{<c>}{<q>} <Rd>, <Rm>, <shift> <Rs> ; T2
                          mov(CurrentCond(),
                              Best,
                              Register(rd),
                              Operand(Register(rm),
                                      shift.GetType(),
                                      Register(rs)));
                        } else {
                          UnallocatedT32(instr);
                        }
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x0000f080: {
                      // 0xfa00f080
                      switch (instr & 0x00600000) {
                        case 0x00000000: {
                          // 0xfa00f080
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xfa0ff080
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              if ((amount == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // SXTH{<c>}.W {<Rd>}, <Rm> ; T2
                                sxth(CurrentCond(),
                                     Wide,
                                     Register(rd),
                                     Register(rm));
                                if (((instr & 0xfffff0c0) != 0xfa0ff080)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                // SXTH{<c>}{<q>} {<Rd>}, <Rm> {, ROR #<amount> } ; T2 NOLINT(whitespace/line_length)
                                sxth(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Operand(Register(rm), ROR, amount));
                                if (((instr & 0xfffff0c0) != 0xfa0ff080)) {
                                  UnpredictableT32(instr);
                                }
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // SXTAH{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              sxtah(CurrentCond(),
                                    Register(rd),
                                    Register(rn),
                                    Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfff0f0c0) != 0xfa00f080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00200000: {
                          // 0xfa20f080
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xfa2ff080
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // SXTB16{<c>}{<q>} {<Rd>}, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              sxtb16(CurrentCond(),
                                     Register(rd),
                                     Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfffff0c0) != 0xfa2ff080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // SXTAB16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              sxtab16(CurrentCond(),
                                      Register(rd),
                                      Register(rn),
                                      Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfff0f0c0) != 0xfa20f080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00400000: {
                          // 0xfa40f080
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xfa4ff080
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              if ((amount == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // SXTB{<c>}.W {<Rd>}, <Rm> ; T2
                                sxtb(CurrentCond(),
                                     Wide,
                                     Register(rd),
                                     Register(rm));
                                if (((instr & 0xfffff0c0) != 0xfa4ff080)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                // SXTB{<c>}{<q>} {<Rd>}, <Rm> {, ROR #<amount> } ; T2 NOLINT(whitespace/line_length)
                                sxtb(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Operand(Register(rm), ROR, amount));
                                if (((instr & 0xfffff0c0) != 0xfa4ff080)) {
                                  UnpredictableT32(instr);
                                }
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // SXTAB{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              sxtab(CurrentCond(),
                                    Register(rd),
                                    Register(rn),
                                    Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfff0f0c0) != 0xfa40f080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default:
                          UnallocatedT32(instr);
                          break;
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x10100000: {
                  // 0xfa100000
                  switch (instr & 0x0000f080) {
                    case 0x0000f000: {
                      // 0xfa10f000
                      if ((instr & 0x00000070) == 0x00000000) {
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x2))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (OutsideITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // ASRS.W {<Rd>}, <Rm>, <Rs> ; T2
                            asrs(Condition::None(),
                                 Wide,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          } else {
                            // ASRS{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            asrs(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          }
                          return;
                        }
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x0))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (OutsideITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // LSLS.W {<Rd>}, <Rm>, <Rs> ; T2
                            lsls(Condition::None(),
                                 Wide,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          } else {
                            // LSLS{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            lsls(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          }
                          return;
                        }
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x1))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (OutsideITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // LSRS.W {<Rd>}, <Rm>, <Rs> ; T2
                            lsrs(Condition::None(),
                                 Wide,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          } else {
                            // LSRS{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            lsrs(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          }
                          return;
                        }
                        if (((Uint32((instr >> 21)) & Uint32(0x3)) ==
                             Uint32(0x3))) {
                          unsigned rd = (instr >> 8) & 0xf;
                          unsigned rm = (instr >> 16) & 0xf;
                          unsigned rs = instr & 0xf;
                          if (OutsideITBlock() &&
                              ((rd == rm) && (rd < kNumberOfT32LowRegisters) &&
                               (rs < kNumberOfT32LowRegisters))) {
                            // RORS.W {<Rd>}, <Rm>, <Rs> ; T2
                            rors(Condition::None(),
                                 Wide,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          } else {
                            // RORS{<c>}{<q>} {<Rd>}, <Rm>, <Rs> ; T2
                            rors(CurrentCond(),
                                 Best,
                                 Register(rd),
                                 Register(rm),
                                 Register(rs));
                          }
                          return;
                        }
                        unsigned rd = (instr >> 8) & 0xf;
                        unsigned rm = (instr >> 16) & 0xf;
                        Shift shift((instr >> 21) & 0x3);
                        unsigned rs = instr & 0xf;
                        if (OutsideITBlock() &&
                            (instr & 0x00100000) == 0x00100000 &&
                            ((rd < kNumberOfT32LowRegisters) &&
                             (rm < kNumberOfT32LowRegisters) &&
                             (rs < kNumberOfT32LowRegisters))) {
                          // MOVS.W <Rd>, <Rm>, <shift> <Rs> ; T2
                          movs(Condition::None(),
                               Wide,
                               Register(rd),
                               Operand(Register(rm),
                                       shift.GetType(),
                                       Register(rs)));
                        } else if ((instr & 0x00100000) == 0x00100000) {
                          // MOVS{<c>}{<q>} <Rd>, <Rm>, <shift> <Rs> ; T2
                          movs(CurrentCond(),
                               Best,
                               Register(rd),
                               Operand(Register(rm),
                                       shift.GetType(),
                                       Register(rs)));
                        } else {
                          UnallocatedT32(instr);
                        }
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x0000f080: {
                      // 0xfa10f080
                      switch (instr & 0x00600000) {
                        case 0x00000000: {
                          // 0xfa10f080
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xfa1ff080
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              if ((amount == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // UXTH{<c>}.W {<Rd>}, <Rm> ; T2
                                uxth(CurrentCond(),
                                     Wide,
                                     Register(rd),
                                     Register(rm));
                                if (((instr & 0xfffff0c0) != 0xfa1ff080)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                // UXTH{<c>}{<q>} {<Rd>}, <Rm> {, ROR #<amount> } ; T2 NOLINT(whitespace/line_length)
                                uxth(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Operand(Register(rm), ROR, amount));
                                if (((instr & 0xfffff0c0) != 0xfa1ff080)) {
                                  UnpredictableT32(instr);
                                }
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // UXTAH{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              uxtah(CurrentCond(),
                                    Register(rd),
                                    Register(rn),
                                    Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfff0f0c0) != 0xfa10f080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00200000: {
                          // 0xfa30f080
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xfa3ff080
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // UXTB16{<c>}{<q>} {<Rd>}, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              uxtb16(CurrentCond(),
                                     Register(rd),
                                     Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfffff0c0) != 0xfa3ff080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // UXTAB16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              uxtab16(CurrentCond(),
                                      Register(rd),
                                      Register(rn),
                                      Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfff0f0c0) != 0xfa30f080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        case 0x00400000: {
                          // 0xfa50f080
                          switch (instr & 0x000f0000) {
                            case 0x000f0000: {
                              // 0xfa5ff080
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              if ((amount == 0) &&
                                  ((rd < kNumberOfT32LowRegisters) &&
                                   (rm < kNumberOfT32LowRegisters))) {
                                // UXTB{<c>}.W {<Rd>}, <Rm> ; T2
                                uxtb(CurrentCond(),
                                     Wide,
                                     Register(rd),
                                     Register(rm));
                                if (((instr & 0xfffff0c0) != 0xfa5ff080)) {
                                  UnpredictableT32(instr);
                                }
                              } else {
                                // UXTB{<c>}{<q>} {<Rd>}, <Rm> {, ROR #<amount> } ; T2 NOLINT(whitespace/line_length)
                                uxtb(CurrentCond(),
                                     Best,
                                     Register(rd),
                                     Operand(Register(rm), ROR, amount));
                                if (((instr & 0xfffff0c0) != 0xfa5ff080)) {
                                  UnpredictableT32(instr);
                                }
                              }
                              break;
                            }
                            default: {
                              if (((instr & 0xf0000) == 0xf0000)) {
                                UnallocatedT32(instr);
                                return;
                              }
                              unsigned rd = (instr >> 8) & 0xf;
                              unsigned rn = (instr >> 16) & 0xf;
                              unsigned rm = instr & 0xf;
                              uint32_t amount = ((instr >> 4) & 0x3) * 8;
                              // UXTAB{<c>}{<q>} {<Rd>}, <Rn>, <Rm> {, ROR #<amount> } ; T1 NOLINT(whitespace/line_length)
                              uxtab(CurrentCond(),
                                    Register(rd),
                                    Register(rn),
                                    Operand(Register(rm), ROR, amount));
                              if (((instr & 0xfff0f0c0) != 0xfa50f080)) {
                                UnpredictableT32(instr);
                              }
                              break;
                            }
                          }
                          break;
                        }
                        default:
                          UnallocatedT32(instr);
                          break;
                      }
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x10800000: {
                  // 0xfa800000
                  switch (instr & 0x0060f0f0) {
                    case 0x0000f000: {
                      // 0xfa80f000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SADD8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      sadd8(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0000f010: {
                      // 0xfa80f010
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // QADD8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      qadd8(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0000f020: {
                      // 0xfa80f020
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SHADD8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      shadd8(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0000f040: {
                      // 0xfa80f040
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UADD8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uadd8(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0000f050: {
                      // 0xfa80f050
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UQADD8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uqadd8(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0000f060: {
                      // 0xfa80f060
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UHADD8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uhadd8(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0000f080: {
                      // 0xfa80f080
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rm = instr & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      // QADD{<c>}{<q>} {<Rd>}, <Rm>, <Rn> ; T1
                      qadd(CurrentCond(),
                           Register(rd),
                           Register(rm),
                           Register(rn));
                      break;
                    }
                    case 0x0000f090: {
                      // 0xfa80f090
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rm = instr & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      // QDADD{<c>}{<q>} {<Rd>}, <Rm>, <Rn> ; T1
                      qdadd(CurrentCond(),
                            Register(rd),
                            Register(rm),
                            Register(rn));
                      break;
                    }
                    case 0x0000f0a0: {
                      // 0xfa80f0a0
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rm = instr & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      // QSUB{<c>}{<q>} {<Rd>}, <Rm>, <Rn> ; T1
                      qsub(CurrentCond(),
                           Register(rd),
                           Register(rm),
                           Register(rn));
                      break;
                    }
                    case 0x0000f0b0: {
                      // 0xfa80f0b0
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rm = instr & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      // QDSUB{<c>}{<q>} {<Rd>}, <Rm>, <Rn> ; T1
                      qdsub(CurrentCond(),
                            Register(rd),
                            Register(rm),
                            Register(rn));
                      break;
                    }
                    case 0x0020f000: {
                      // 0xfaa0f000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SASX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      sasx(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           Register(rm));
                      break;
                    }
                    case 0x0020f010: {
                      // 0xfaa0f010
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // QASX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      qasx(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           Register(rm));
                      break;
                    }
                    case 0x0020f020: {
                      // 0xfaa0f020
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SHASX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      shasx(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0020f040: {
                      // 0xfaa0f040
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UASX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uasx(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           Register(rm));
                      break;
                    }
                    case 0x0020f050: {
                      // 0xfaa0f050
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UQASX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uqasx(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0020f060: {
                      // 0xfaa0f060
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UHASX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uhasx(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0020f080: {
                      // 0xfaa0f080
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SEL{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      sel(CurrentCond(),
                          Register(rd),
                          Register(rn),
                          Register(rm));
                      break;
                    }
                    case 0x0040f000: {
                      // 0xfac0f000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SSUB8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      ssub8(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0040f010: {
                      // 0xfac0f010
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // QSUB8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      qsub8(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0040f020: {
                      // 0xfac0f020
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SHSUB8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      shsub8(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0040f040: {
                      // 0xfac0f040
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // USUB8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      usub8(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0040f050: {
                      // 0xfac0f050
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UQSUB8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uqsub8(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0040f060: {
                      // 0xfac0f060
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UHSUB8{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uhsub8(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0040f080: {
                      // 0xfac0f080
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // CRC32B{<q>} <Rd>, <Rn>, <Rm> ; T1
                      crc32b(Condition::None(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0040f090: {
                      // 0xfac0f090
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // CRC32H{<q>} <Rd>, <Rn>, <Rm> ; T1
                      crc32h(Condition::None(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0040f0a0: {
                      // 0xfac0f0a0
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // CRC32W{<q>} <Rd>, <Rn>, <Rm> ; T1
                      crc32w(Condition::None(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0060f000: {
                      // 0xfae0f000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SSAX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      ssax(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           Register(rm));
                      break;
                    }
                    case 0x0060f010: {
                      // 0xfae0f010
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // QSAX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      qsax(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           Register(rm));
                      break;
                    }
                    case 0x0060f020: {
                      // 0xfae0f020
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SHSAX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      shsax(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0060f040: {
                      // 0xfae0f040
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // USAX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      usax(CurrentCond(),
                           Register(rd),
                           Register(rn),
                           Register(rm));
                      break;
                    }
                    case 0x0060f050: {
                      // 0xfae0f050
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UQSAX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uqsax(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    case 0x0060f060: {
                      // 0xfae0f060
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UHSAX{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uhsax(CurrentCond(),
                            Register(rd),
                            Register(rn),
                            Register(rm));
                      break;
                    }
                    default:
                      UnallocatedT32(instr);
                      break;
                  }
                  break;
                }
                case 0x10900000: {
                  // 0xfa900000
                  switch (instr & 0x0060f0f0) {
                    case 0x0000f000: {
                      // 0xfa90f000
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SADD16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      sadd16(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0000f010: {
                      // 0xfa90f010
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // QADD16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      qadd16(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0000f020: {
                      // 0xfa90f020
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // SHADD16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      shadd16(CurrentCond(),
                              Register(rd),
                              Register(rn),
                              Register(rm));
                      break;
                    }
                    case 0x0000f040: {
                      // 0xfa90f040
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UADD16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uadd16(CurrentCond(),
                             Register(rd),
                             Register(rn),
                             Register(rm));
                      break;
                    }
                    case 0x0000f050: {
                      // 0xfa90f050
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UQADD16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uqadd16(CurrentCond(),
                              Register(rd),
                              Register(rn),
                              Register(rm));
                      break;
                    }
                    case 0x0000f060: {
                      // 0xfa90f060
                      unsigned rd = (instr >> 8) & 0xf;
                      unsigned rn = (instr >> 16) & 0xf;
                      unsigned rm = instr & 0xf;
                      // UHADD16{<c>}{<q>} {<Rd>}, <Rn>, <Rm> ; T1
                      uhadd16(CurrentCond(),
                              Register(rd),
                              Register(rn),
                              Register(rm));
                      break;
                    }
                    case 0x0000f080: {
                      // 0xfa90f080
                      if (((instr >> 16) & 0xf) == (instr & 0xf)) {
                        unsigned rd = (instr >> 8) & 0xf;
                        unsigned rm = instr & 0xf;
                        if ((rd < kNumberOfT32LowRegisters) &&
                            (rm < kNumberOfT32LowRegisters)) {
                          // REV{<c>}.W <Rd>, <Rm> ; T2
                          rev(CurrentCond(), Wide, Register(rd), Register(rm));
                        } else {
                          // REV{<c>}{<q>} <Rd>, <Rm> ; T2
                          rev(CurrentCond(), Best, Register(rd), Register(rm));
                        }
                      } else {
                        UnallocatedT32(instr);
                      }
                      break;
                    }
                    case 0x0000f090: {
                      // 0xfa90f090
                      if (((instr >> 16) & 0xf) == (instr & 0xf)) {
                        unsigned rd = (instr >> 8) & 0xf;
                        unsigned rm = instr & 0xf;
                        if ((rd < kNumberOfT32LowRegisters) &&
                            (rm < kNumberOfT32LowRegisters)) {
                          // REV16{<c>}.W <Rd>, <Rm> ; T2
                          rev16(CurrentCond(),
                            