/*
 * Copyright (c) 2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: t-pmilter-1.c,v 1.27 2005/10/18 17:32:17 ca Exp $")
#include "sm/error.h"
#include "sm/assert.h"
#include "sm/types.h"
#include "sm/sysexits.h"
#include "sm/fcntl.h"
#include "sm/io.h"
#include "sm/ctype.h"
#include "sm/reccom.h"
#include "sm/mta.h"
#define PMILTER_DEBUG_DEFINE 1
#include "pmilter.h"
#include "sm/pmfdef.h"
#include "sm/pmfapi.h"
#include "sm/pmilter.h"
#include "util.h"
#include "t-pmilter.h"
#include "sm/test.h"

#if SM_USE_PMILTER

/* replace all occurrences of a in str with b... just a hack */
static void
strreplace(char *str, char a, char b)
{
	char *s;

	if (str == NULL)
		return;
	if (a == b)
		return;
	s = str;
	while (*s != '\0')
	{
		if (*s == a)
			*s = b;
		++s;
	}
}

/* hack: copy str, replace '_' with ' ', and append \r\n */
static char *
getreply(char *str)
{
	size_t len;
	char *new;

	if (str == NULL)
		return NULL;
	len = strlen(str);
	if (len > 1000)
		return NULL;
	new = malloc(len + 4);
	if (new == NULL)
		return NULL;
	sm_memcpy(new, str, len + 1);
	strreplace(new, '_', ' ');
	new[len] = '\r';
	new[len + 1] = '\n';
	new[len + 2] = '\0';
	return new;
}

/*
**  pmilter test program; uses "native" API
*/

static pmt_ctx_T pmt_ctx;

#define SETREPLY(str)						\
	do							\
	{							\
		if ((str) != NULL && *(str) != '\0')		\
			ret = sm_pmfi_setreply(pmse_ctx, str);	\
	} while (0)

#define T_DELAY(delay)			\
	do				\
	{				\
		if ((delay) > 0)	\
			sleep(delay);	\
	} while (0)

/* common function code; use only at the end of a function */
#define TPM_END(stage)					\
	if (PMT_EXIT(pmt_ctx.pmt_rcode[stage]))		\
		exit(0);				\
	SETREPLY(pmt_ctx.pmt_reply[stage]);		\
	T_DELAY(pmt_ctx.pmt_delay[stage]);		\
	return pmt_ctx.pmt_rcode[stage]

static sm_ret_T
tpm1_negotiate(pmss_ctx_P pmss_ctx, uint32_t srv_cap, uint32_t srv_fct, uint32_t srv_feat, uint32_t srv_misc, uint32_t *pm_cap, uint32_t *pm_fct, uint32_t *pm_feat, uint32_t *pm_misc)
{
	sm_ret_T ret;
	pmt_ctx_P pmt_ctx_l;

	SM_REQUIRE(pm_cap != NULL);
	SM_REQUIRE(pm_fct != NULL);
	SM_REQUIRE(pm_feat != NULL);
	SM_REQUIRE(pm_misc != NULL);

	pmt_ctx_l = (pmt_ctx_P) sm_pmfi_get_ctx_g_ss(pmss_ctx);
	SM_TEST(pmt_ctx_l != NULL);
	SM_TEST(pmt_ctx_l == &pmt_ctx);

	if (SM_IS_FLAG(pmt_ctx_l->pmt_flags, PMT_FL_M_HOSTN) ||
	    SM_IS_FLAG(pmt_ctx_l->pmt_flags, PMT_FL_M_SEID))
	{
		ret = sm_pmfi_setmaclist(pmss_ctx, PM_SMST_CONNECT,
					PMM_SRVHOSTNAME, PMM_SEID, PMM_END);
		SM_TEST(ret == SM_SUCCESS);
	}

	if (SM_IS_FLAG(pmt_ctx_l->pmt_flags, PMT_FL_M_MSGID))
	{
		ret = sm_pmfi_setmaclist(pmss_ctx, PM_SMST_DOT,
					PMM_DOT_MSGID, PMM_END);
		SM_TEST(ret == SM_SUCCESS);
	}

	if (SM_IS_FLAG(pmt_ctx_l->pmt_flags, PMT_FL_M_TAID))
	{
		ret = sm_pmfi_setmaclist(pmss_ctx, PM_SMST_MAIL,
					PMM_MAIL_TAID, PMM_END);
		SM_TEST(ret == SM_SUCCESS);
	}

	/* check that all bits in pmt_cap are also set in srv_cap... */
	if ((pmt_ctx.pmt_cap & srv_cap) != pmt_ctx.pmt_cap)
		sm_io_fprintf(smioerr,
			"sev=WARN, where=tpm1_negotiate, pm_cap=%#X, srv_cap=%#X\n",
			pmt_ctx_l->pmt_cap, srv_cap);

	*pm_cap = pmt_ctx_l->pmt_cap;
	*pm_fct = 0;
	*pm_feat = 0;
	*pm_misc = 0;
	sm_io_fprintf(smioerr, "sev=DBG, where=tpm1_negotiate, cap=%#X\n",
		pmt_ctx_l->pmt_cap);
	return SM_SUCCESS;
}

static sfsistat_T
tpm1_connect(pmse_ctx_P pmse_ctx, const char *hostname, sm_sockaddr_T *hostaddr)
{
	sfsistat_T ret;

	if (PMT_EXIT(pmt_ctx.pmt_rcode[SM_STAGE_NSEID]))
		exit(0);

	/* HACK; should get pmt_ctx from pmse_ctx (indirectly) */
	ret = sm_pmfi_set_ctx_se(pmse_ctx, &pmt_ctx);
	SM_TEST(ret == SM_SUCCESS);
	++pmt_ctx.pmt_se_cnt;

	if (SM_IS_FLAG(pmt_ctx.pmt_flags, PMT_FL_M_HOSTN))
	{
		char *srvhostname;

		ret = sm_pmfi_getmac(pmse_ctx, PMM_SRVHOSTNAME, &srvhostname);
		SM_TEST(ret == SM_SUCCESS);
		SM_TEST(srvhostname != NULL);
		sm_io_fprintf(smioerr,
			"sev=DBG, seid=%s, srvhostname=%s, where=connect, ret=%X\n",
			pmse_ctx->pmse_se_id, srvhostname, ret);
	}

	if (SM_IS_FLAG(pmt_ctx.pmt_flags, PMT_FL_M_SEID))
	{
		char *se_id;

		ret = sm_pmfi_getmac(pmse_ctx, PMM_SEID, &se_id);
		SM_TEST(ret == SM_SUCCESS);
		SM_TEST(se_id != NULL);
		sm_io_fprintf(smioerr,
			"sev=DBG, seid=%s, se_id=%s, where=connect, ret=%X\n",
			pmse_ctx->pmse_se_id, se_id, ret);
	}

	ret = pmt_ctx.pmt_rcode[SM_STAGE_NSEID];
	sm_io_fprintf(smioerr,
		"sev=DBG, seid=%s, host=%s, where=connect, rc=%d\n",
		pmse_ctx->pmse_se_id, hostname, ret);
	TPM_END(SM_STAGE_NSEID);
}

static sm_ret_T
tpm1_close(pmse_ctx_P pmse_ctx)
{
	void *actx;

	sm_io_fprintf(smioerr, "sev=DBG, seid=%s, where=close\n",
		pmse_ctx->pmse_se_id);
	actx = sm_pmfi_get_ctx_se(pmse_ctx);
	SM_TEST(actx == &pmt_ctx);
	return SM_SUCCESS;
}

static sfsistat_T
tpm1_helo(pmse_ctx_P pmse_ctx, const char *helohost)
{
	sfsistat_T ret;

	sm_io_fprintf(smioerr, "sev=DBG, seid=%s, helo=%s, rc=%d\n",
		pmse_ctx->pmse_se_id, helohost, pmt_ctx.pmt_rcode[SM_STAGE_HELO]);
	TPM_END(SM_STAGE_HELO);
}

static sfsistat_T
tpm1_mail(pmse_ctx_P pmse_ctx, const char *mail, char **argv)
{
	sfsistat_T ret;

	++pmt_ctx.pmt_ta_cnt;
	if (SM_IS_FLAG(pmt_ctx.pmt_flags, PMT_FL_M_TAID))
	{
		char *ta_id;

		ret = sm_pmfi_getmac(pmse_ctx, PMM_MAIL_TAID, &ta_id);
		SM_TEST(ret == SM_SUCCESS);
		SM_TEST(ta_id != NULL);
		sm_io_fprintf(smioerr,
			"sev=DBG, seid=%s, ta_id=%s, ret=%X\n",
			pmse_ctx->pmse_se_id, ta_id, ret);
	}

	if (SM_IS_FLAG(pmt_ctx.pmt_flags, PMT_FL_M_2ND) &&
	    pmt_ctx.pmt_ta_cnt > 1 &&
	    SM_IS_FLAG(pmt_ctx.pmt_flags, PMT_FL_M_MSGID))
	{
		char *msgid;

		ret = sm_pmfi_getmac(pmse_ctx, PMM_DOT_MSGID, &msgid);
		SM_TEST(ret == SM_SUCCESS);
		SM_TEST(msgid == NULL);
		sm_io_fprintf(smioerr,
			"sev=DBG, func=tpm1_mail, seid=%s, msgid=%s, ret=%X\n",
			pmse_ctx->pmse_se_id, msgid, ret);
	}

	sm_io_fprintf(smioerr,
		"sev=DBG, seid=%s, mail=%s, argv[0]=%s, rc=%d\n",
		pmse_ctx->pmse_se_id, mail,
		(argv != NULL && argv[0] != NULL) ? argv[0] : "NULL",
		pmt_ctx.pmt_rcode[SM_STAGE_MAIL]);
	TPM_END(SM_STAGE_MAIL);
}

static sfsistat_T
tpm1_rcpt(pmse_ctx_P pmse_ctx, const char *rcpt, char **argv)
{
	sm_ret_T ret, st;

	st = -1;
	if (SM_IS_FLAG(pmt_ctx.pmt_flags, PMT_FL_R_ST))
		ret = sm_pmfi_getstatus(pmse_ctx, &st);
	sm_io_fprintf(smioerr,
		"sev=DBG, seid=%s, rcpt=%s, argv[0]=%s, st=%d, rc=%d\n",
		pmse_ctx->pmse_se_id, rcpt,
		(argv != NULL && argv[0] != NULL) ? argv[0] : "NULL", st,
		pmt_ctx.pmt_rcode[SM_STAGE_RCPT]);
	TPM_END(SM_STAGE_RCPT);
}

static sfsistat_T
tpm1_data(pmse_ctx_P pmse_ctx)
{
	sm_ret_T ret;

	sm_io_fprintf(smioerr, "sev=DBG, seid=%s, where=data, rc=%d\n",
		pmse_ctx->pmse_se_id, pmt_ctx.pmt_rcode[SM_STAGE_DATA]);
	TPM_END(SM_STAGE_DATA);
}

static void
pm_open(pmse_ctx_P pmse_ctx)
{
	if (PMT_IS_FLAG(&pmt_ctx, PMT_FL_W2F) &&
	    !PMT_IS_FLAG(&pmt_ctx, PMT_FL_FD_OPEN) &&
	    !PMT_IS_FLAG(&pmt_ctx, PMT_FL_FD_FAIL))
	{
		pmt_ctx.pmt_fd = open(pmt_ctx.pmt_fname,
					O_WRONLY|O_APPEND|O_CREAT,
					0660);
		if (pmt_ctx.pmt_fd >= 0)
			PMT_SET_FLAG(&pmt_ctx, PMT_FL_FD_OPEN);
		else
			PMT_SET_FLAG(&pmt_ctx, PMT_FL_FD_FAIL);
		sm_io_fprintf(smioerr,
			"sev=DBG, seid=%s, where=pm_open, open=%d\n",
			pmse_ctx->pmse_se_id, pmt_ctx.pmt_fd);
	}
}

static sfsistat_T
tpm1_unknown(pmse_ctx_P pmse_ctx, const char *cmd)
{
	sm_io_fprintf(smioerr,
		"sev=DBG, seid=%s, where=unknown, cmd=%s\n",
		pmse_ctx->pmse_se_id, cmd);
	return SMTP_R_CONT;
}

static sm_ret_T
tpm1_abort(pmse_ctx_P pmse_ctx)
{
	sm_io_fprintf(smioerr,
		"sev=DBG, seid=%s, where=abort_ta\n",
		pmse_ctx->pmse_se_id);
	return SM_SUCCESS;
}

static sfsistat_T
tpm1_msg(pmse_ctx_P pmse_ctx, unsigned char *buf, size_t len)
{
	sm_ret_T ret;

	pm_open(pmse_ctx);
	if (PMT_IS_FLAG(&pmt_ctx, PMT_FL_W2F) &&
	    PMT_IS_FLAG(&pmt_ctx, PMT_FL_FD_OPEN) &&
	    !PMT_IS_FLAG(&pmt_ctx, PMT_FL_WR_FAIL) &&
	    buf != NULL && len > 0
	   )
	{
		ssize_t written;

		written = write(pmt_ctx.pmt_fd, buf, len);
		if (written == -1)
		{
			sm_io_fprintf(smioerr,
				"sev=ERROR, seid=%s, where=msg, write=%d\n",
				pmse_ctx->pmse_se_id, (int) written);
			PMT_SET_FLAG(&pmt_ctx, PMT_FL_WR_FAIL);
			close(pmt_ctx.pmt_fd);
			pmt_ctx.pmt_fd = INVALID_FD;
			PMT_CLR_FLAG(&pmt_ctx, PMT_FL_FD_OPEN);
		}
	}

	sm_io_fprintf(smioerr, "sev=DBG, seid=%s, where=msg, len=%d, rc=%d\n",
		pmse_ctx->pmse_se_id, (int) len, pmt_ctx.pmt_rcode[SM_STAGE_MSG]);
	TPM_END(SM_STAGE_MSG);
}

static sfsistat_T
tpm1_eom(pmse_ctx_P pmse_ctx)
{
	sm_ret_T ret;
	bool writefailed;
	int rcode;

	writefailed = PMT_IS_FLAG(&pmt_ctx, PMT_FL_W2F) &&
			PMT_IS_FLAG(&pmt_ctx, PMT_FL_WR_FAIL);
	if (PMT_IS_FLAG(&pmt_ctx, PMT_FL_W2F) &&
	    PMT_IS_FLAG(&pmt_ctx, PMT_FL_FD_OPEN) &&
	    !PMT_IS_FLAG(&pmt_ctx, PMT_FL_WR_FAIL))
	{
		int r;

		r = close(pmt_ctx.pmt_fd);
		sm_io_fprintf(smioerr,
			"sev=DBG, seid=%s, where=dot, write=%d\n",
			pmse_ctx->pmse_se_id, r);
		pmt_ctx.pmt_fd = INVALID_FD;
		PMT_CLR_FLAG(&pmt_ctx, PMT_FL_FD_OPEN);
	}

	if (SM_IS_FLAG(pmt_ctx.pmt_flags, PMT_FL_M_MSGID))
	{
		char *msgid;

		ret = sm_pmfi_getmac(pmse_ctx, PMM_DOT_MSGID, &msgid);
		SM_TEST(ret == SM_SUCCESS);
		SM_TEST(msgid != NULL);
		sm_io_fprintf(smioerr,
			"sev=DBG, seid=%s, msgid=%s, ret=%X\n",
			pmse_ctx->pmse_se_id, msgid, ret);
	}

	rcode = pmt_ctx.pmt_rcode[SM_STAGE_DOT];
	if (rcode == SMTP_R_OK && writefailed)
		rcode = SMTP_R_TEMP;

	sm_io_fprintf(smioerr, "sev=DBG, seid=%s, where=dot, rc=%d\n",
		pmse_ctx->pmse_se_id, pmt_ctx.pmt_rcode[SM_STAGE_DOT]);

	TPM_END(SM_STAGE_DOT);
}

static sm_ret_T
tpm1_signal(pmg_ctx_P pmg_ctx, int sig)
{
	sm_io_fprintf(smioerr, "sev=DBG, signal=%d\n", sig);
	return SM_SUCCESS;
}

/*
**  USAGE -- Print usage message to smioerr
**
**	Parameters:
**		prg -- program name
**
**	Returns:
**		exits
*/

static void
usage(const char *prg)
{
	sm_io_fprintf(smioerr, "usage: %s [options]\n"
		"-A cap		  add capabilities\n"
		"   R: get RCPT status\n"
		"-C cap		  disable capabilities\n"
		"   cap is a sequence of characters:\n"
		"   C: CNNCT\n"
		"   E: EHLO\n"
#if 0
		"   S: STTLS\n"
		"   A: AUTH\n"
#endif
		"   M: MAIL\n"
		"   R: RCPT\n"
		"   D: DATA\n"
		"   B: MSG\n"
		"-c n             set requested capabilities\n"
#if PMILTER_DEBUG
		"-d n             set debug level\n"
#endif
		"-f name          write msg to file\n"
		"-R stage=reply   set SMTP reply text for stage\n"
		"   stage:\n"
		"     c: new session (connect)\n"
		"     h: HELO\n"
		"     m: MAIL\n"
		"     r: RCPT\n"
		"     d: DATA\n"
		"     B: Body\n"
		"     b: End of Body\n"
		"-r stage=rcode   set SMTP reply code for stage to rcode\n"
		"   stage: see above\n"
		"   rcode=%d will invoke exit() to simulate a fatal error\n"
		"-w stage=delay   set a delay for replying in stage\n"
		"   stage: see above\n"
		, prg
		, PMT_EXIT_CODE
		);
	exit(EX_USAGE);
}

static pmilter_T
pmilter =
{
	"t-pmilter-1",
	LPMILTER_VERSION,
	SM_SCAP_PM_BASIC,
	0,
	0,
	0,
	tpm1_negotiate,
	tpm1_connect,
	tpm1_helo,
	tpm1_mail,
	tpm1_rcpt,
	tpm1_data,
	tpm1_msg,
	tpm1_eom,
	tpm1_abort,
	tpm1_close,
	tpm1_unknown,
	tpm1_signal
};

/*
**  MAIN -- PMILTER test server
**
**	Parameters:
**		argc -- number of arguments
**		argv -- vector of arguments
**
**	Returns:
**		exit code
*/

int
main(int argc, char *argv[])
{
	sm_ret_T ret;
	int c, ch, stage;
	unsigned int u, u2;
	uint32_t major, minor, patchlevel;
	pmg_ctx_P pmg_ctx;
	char *prg, *str;

	prg = argv[0];
	pmg_ctx = NULL;
	if (getuid() == 0 || geteuid() == 0)
	{
		sm_io_fprintf(smioerr,
			"%s: ERROR: do not run this as super-user!\n",
			prg);
		exit(EX_USAGE);
	}

	/* initialize test context */
	sm_memzero(&pmt_ctx, sizeof(pmt_ctx));
	for (c = 0; c < SM_STAGES; c++)
		pmt_ctx.pmt_rcode[c] = SMTP_R_CONT;
	pmt_ctx.pmt_fname[0] = '\0';
	pmt_ctx.pmt_fd = INVALID_FD;
	pmt_ctx.pmt_cap = SM_SCAP_PM_BASIC;

#if SM_HEAP_CHECK
	SmHeapCheck = 1;
	if (HEAP_CHECK)
		sm_heap_report(smioerr, 3);
#endif

	while ((c = getopt(argc, argv, "2A:C:c:d:f:m:R:r:w:")) != -1)
	{
		switch (c)
		{
		  case '2':
			pmt_ctx.pmt_flags |= PMT_FL_M_2ND;
			break;
		  case 'A':
			for (u = 0, u2 = 0; (ch = optarg[u]) != '\0'; u++)
			{
				switch (ch)
				{
				  case 'R': u2 |= SM_SCAP_PM_RCPT_ST;
					pmt_ctx.pmt_flags |= PMT_FL_R_ST;
					break;
				  default: usage(prg); break;
				}
			}
			pmt_ctx.pmt_cap |= u2;
			break;
		  case 'C':
			for (u = 0, u2 = 0; (ch = optarg[u]) != '\0'; u++)
			{
				switch (ch)
				{
				  case 'C': u2 |= SM_SCAP_PM_CNNCT; break;
				  case 'E': u2 |= SM_SCAP_PM_EHLO; break;
				  case 'S': u2 |= SM_SCAP_PM_STTLS; break;
				  case 'A': u2 |= SM_SCAP_PM_AUTH; break;
				  case 'M': u2 |= SM_SCAP_PM_MAIL; break;
				  case 'R': u2 |= SM_SCAP_PM_RCPT; break;
				  case 'D': u2 |= SM_SCAP_PM_DATA; break;
				  case 'B': u2 |= SM_SCAP_PM_MSG; break;
				  default: usage(prg); break;
				}
			}
			pmt_ctx.pmt_cap &= ~u2;
			break;
		  case 'c':
			pmt_ctx.pmt_cap = strtoul(optarg, NULL, 0);
			break;
		  case 'd':
#if PMILTER_DEBUG
			pm_debug = strtoul(optarg, NULL, 0);
#endif
			break;
		  case 'f':
			strlcpy(pmt_ctx.pmt_fname, optarg,
				sizeof(pmt_ctx.pmt_fname));
			PMT_SET_FLAG(&pmt_ctx, PMT_FL_W2F);
			break;
		  case 'm':
			for (u = 0; (ch = optarg[u]) != '\0'; u++)
			{
				switch (ch)
				{
				  case 'H':
					pmt_ctx.pmt_flags |= PMT_FL_M_HOSTN;
					break;
				  case 'M':
					pmt_ctx.pmt_flags |= PMT_FL_M_MSGID;
					break;
				  case 'S':
					pmt_ctx.pmt_flags |= PMT_FL_M_SEID;
					break;
				  case 'T':
					pmt_ctx.pmt_flags |= PMT_FL_M_TAID;
					break;
				  default: usage(prg); break;
				}
			}
			break;
		  case 'R':
			if (optarg != NULL && ISALPHA(optarg[0])
			    && optarg[1] == '=')
				u2 = 2;
			else
			{
				usage(prg);
				break;
			}

			str = getreply(optarg + u2);
			if (str == NULL)
			{
				sm_io_fprintf(smioerr, "sev=ERROR, strdup=NULL");
				exit(1);
			}
			stage = sm_getstage(optarg[0]);
			if (stage < 0 || stage > SM_STAGES)
			{
				usage(prg);
				/* NOTREACHED */
				SM_ASSERT(false);
			}
			pmt_ctx.pmt_reply[stage] = str;
			if (SM_STAGE_MSG == stage)
				pmt_ctx.pmt_cap |= SM_SCAP_PM_MSG_RC;
			break;

		  case 'r':
			if (optarg != NULL && ISALPHA(optarg[0])
			    && optarg[1] == '=')
				u2 = 2;
			else
			{
				usage(prg);
				break;
			}

			u = (unsigned int) strtoul(optarg + u2, NULL, 0);
			stage = sm_getstage(optarg[0]);
			if (stage < 0 || stage > SM_STAGES)
			{
				usage(prg);
				/* NOTREACHED */
				SM_ASSERT(false);
			}
			pmt_ctx.pmt_rcode[stage] = u;
			if (SM_STAGE_MSG == stage)
				pmt_ctx.pmt_cap |= SM_SCAP_PM_MSG_RC;
			break;

		  case 'w':
			if (optarg != NULL && ISALPHA(optarg[0])
			    && optarg[1] == '=')
				u2 = 2;
			else
			{
				usage(prg);
				break;
			}

			u = (unsigned int) strtoul(optarg + u2, NULL, 0);
			stage = sm_getstage(optarg[0]);
			if (stage < 0 || stage > SM_STAGES)
			{
				usage(prg);
				/* NOTREACHED */
				SM_ASSERT(false);
			}
			pmt_ctx.pmt_delay[stage] = u;
			break;


		  default:
			usage(prg);
			break;
		}
	}

	ret = sm_pmfi_init(&pmg_ctx);
	if (sm_is_err(ret))
	{
		sm_io_fprintf(smioerr, "sev=ERROR, sm_pmfi_init=%x\n", ret);
		goto error;
	}

	sm_test_begin(argc, argv, "test pmilter 1");

	argc -= optind;
	argv += optind;

	/* XXX HACK */
	if (argc > 0)
		pmg_ctx->pmg_sockname = argv[0];
	else
		pmg_ctx->pmg_sockname = "pmilter.sock";

	ret = sm_pmfi_version(pmg_ctx, &major, &minor, &patchlevel);
	SM_TEST(ret == SM_SUCCESS);
	SM_TEST(major == 1);
	SM_TEST(minor == 0);
	SM_TEST(patchlevel == 0);

	ret = sm_pmfi_set_ctx_g(pmg_ctx, &pmt_ctx);
	SM_TEST(ret == SM_SUCCESS);

	ret = sm_pmfi_start(pmg_ctx, &pmilter);
	if (sm_is_err(ret))
	{
		sm_io_fprintf(smioerr, "sev=ERROR, sm_pmfi_start=%x\n", ret);
		goto error;
	}

#if SM_HEAP_CHECK
	if (HEAP_CHECK)
		sm_heap_report(smioerr, 3);
#endif
	return sm_test_end();

  error:
	/* ignore shutdown errors... */
	(void) sm_pmilt_stop(pmg_ctx);

	/* select an appropriate error here... */
	return sm_error_value(ret);
}
#else /* SM_USE_PMILTER */
int
main(int argc, char *argv[])
{
	return 0;
}
#endif /* SM_USE_PMILTER */
