/*
 * Copyright (c) 2003-2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

/*
**  Send IPv4 address to SMAR for reverse resolution and access checks.
**  Used by t-smar-3.sh
*/

#include "sm/generic.h"
SM_RCSID("@(#)$Id: t-smar-3.c,v 1.29 2005/10/21 17:48:59 ca Exp $")
#include "sm/error.h"
#include "sm/assert.h"
#include "sm/memops.h"
#include "sm/io.h"
#include "sm/rcb.h"
#include "sm/qmgr.h"
#include "sm/qmgr-int.h"
#include "sm/dns.h"
#include "sm/smar.h"
#include "sm/reccom.h"
#include "sm/unixsock.h"
#include "sm/test.h"

/* context for requests */
struct t_req_S
{
	char		**rq_pa;
	int		  rq_count;
	rcbcom_ctx_P	  rq_qar_com;
	sm_evthr_task_P	  rq_wr_tsk;
};

#define SMAXLEN	256

typedef struct t_req_S	t_req_T, *t_req_P;

static pthread_mutex_t mutex;
static pthread_cond_t cond;

static int Verbose = 0;
static uint MaxCount = (DNS_TIMEOUT + 1) * 10;
static int Requests = 0;
static int aqt_rcpts_ar = 0;
static sm_evthr_ctx_P evthr_ctx;
static rcbcom_ctx_T rcb_com;

static uint32_t lfl = SMARA_LFL_IPV4|SMARA_LFL_RVRS4|SMARA_LFL_RVACC;

static sm_ret_T ar2qmgr(sm_evthr_task_P _tsk);
static sm_ret_T qmgr2ar(sm_evthr_task_P _tsk);

/*
**  FCTS -- "sleep" function: this function terminates the evthr system
**	either after a certain number of iterations or if there are no
**	more outstanding requests.
**
**	Parameters:
**		tsk -- evthr task
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
fcts(sm_evthr_task_P tsk)
{
	static uint count = 0;
	struct timeval sleept, delay;

	++count;
	if (Verbose > 3 && (count % 10 == 0))
		sm_io_fprintf(smioerr, "fcts: count=%u/%u, ", count, MaxCount);

	/* Theoretically Requests needs to be protected by a mutex... */
	if (Requests <= 0 || count > MaxCount)
		return EVTHR_DEL|EVTHR_TERM;
	sm_memzero(&sleept, sizeof(sleept));
	gettimeofday(&sleept, NULL);
	delay.tv_sec = 1;
	delay.tv_usec = 0;
	timeradd(&sleept, &delay, &(tsk->evthr_t_sleep));
	return EVTHR_SLPQ;
}

/*
**  QMGR_AR -- test program to AR interface
**
**	Parameters:
**		tsk -- evthr task
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
qmgr_ar(sm_evthr_task_P tsk)
{
	sm_ret_T ret;

	SM_IS_EVTHR_TSK(tsk);
	if (is_valid_fd(tsk->evthr_t_fd))
	{
		ret = EVTHR_WAITQ;
		if (evthr_got_wr(tsk))
		{
			if (Verbose > 3)
				sm_io_fprintf(smioerr, "write\n");
			ret = qmgr2ar(tsk);	/* XXX check ret here? */
			if (sm_is_err(ret) && Verbose > 0)
				fprintf(stderr, "qmgr2ar=%x\n", ret);
		}
		if (evthr_got_rd(tsk))
		{
			if (Verbose > 3)
				sm_io_fprintf(smioerr, "read\n");
			ret = ar2qmgr(tsk);
		}
		if (sm_is_err(ret))
		{
			if (Verbose > 1)
				sm_io_fprintf(smioerr, "qmgr_ar=%x\n", ret);
			return ret;
		}
		return ret;
	}
	return EVTHR_DEL;
}

/*
**  QR2AR -- Send IPv4 address to AR, i.e., put it into RCB
**
**	Parameters:
**		pa -- printable address (IPv4)
**		rcbe -- RCB entry
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
qr2ar(sm_str_P pa, sm_rcbe_P rcbe)
{
	sm_rcb_P rcb;
	sm_ret_T ret;
	sessta_id_T ta_id;

	ret = SM_SUCCESS;
	rcb = &(rcbe->rcbe_rcb);

	sm_snprintf(ta_id, sizeof(ta_id), SMTPS_STID_FORMAT, (ulonglong_T)0, 0);

	/* XXX send rcpt_idx and ta_id separately? */
	ret = sm_rcb_putv(rcb, RCB_PUTV_FIRST,
		SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
		SM_RCBV_INT, RT_S2A_ID, 1,
		SM_RCBV_BUF, RT_S2A_TAID, ta_id, SMTP_STID_SIZE,
		SM_RCBV_STR, RT_S2A_CLT_A, pa,
		SM_RCBV_INT2, RT_S2A_LTYPE, SMARA_LT_CLT_A_ACC, lfl,
		SM_RCBV_END);
	if (sm_is_err(ret))
		goto error;
	if (Verbose > 4)
		sm_io_fprintf(smioerr, "qr2ar, ret=%d, rw=%d, size=%d, len=%d, max=%d\n"
			, ret
			, rcb->sm_rcb_rw
			, rcb->sm_rcb_size
			, rcb->sm_rcb_len
			, rcb->sm_rcb_max
			);
	return ret;

  error:
	/* XXX leave rcb in a consistent state? */
	return ret;
}

/*
**  RCPT2AR -- send a recipient to SMAR
**
**	Parameters:
**		qar_tsk -- evthr task
**		pa -- printable address of recipient
**		qar_com -- RCBCOM context
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
rcpt2ar(sm_evthr_task_P qar_tsk, sm_str_P pa, rcbcom_ctx_P qar_com)
{
	sm_ret_T ret;
	sm_rcbe_P rcbe;

	ret = sm_rcbe_new_enc(&rcbe, -1, 0);
	if (sm_is_err(ret))
		goto error;
	ret = qr2ar(pa, rcbe);
	if (sm_is_err(ret))
		goto error;
	ret = sm_rcbcom_endrep(qar_com, qar_tsk, true, &rcbe);
	if (sm_is_err(ret))
		goto error;
	++aqt_rcpts_ar;
	return SM_SUCCESS;

  error:
	if (rcbe != NULL)
		sm_rcbe_free(rcbe);
	return ret;
}

/*
**  QMGR2AR -- Test program to AR interface
**
**	Parameters:
**		tsk -- evthr task
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
qmgr2ar(sm_evthr_task_P tsk)
{
	sm_ret_T ret;
	rcbcom_ctx_P qar_com;
	t_req_P t_req;

	SM_IS_EVTHR_TSK(tsk);
	t_req = (t_req_P) tsk->evthr_t_actx;
	qar_com = t_req->rq_qar_com;
	ret = sm_rcbcom2mod(tsk, qar_com);
	if (Verbose > 4)
		sm_io_fprintf(smioerr, "qmgr2ar, ret=%d\n", ret);
	return ret;
}

/*
**  REACT -- Decode data received from AR and act accordingly
**
**	Parameters:
**		tsk -- evthr task
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
react(sm_evthr_task_P tsk)
{
	uint32_t v, l, rt, tl;
	sm_ret_T ret;
	sm_rcb_P rcb;
	sessta_id_T ta_id;
	char statt[256];
	t_req_P t_req;
	rcbcom_ctx_P qar_com;
	sm_cstr_P cstr;

	/* decode rcb */
	t_req = (t_req_P) tsk->evthr_t_actx;
	qar_com = t_req->rq_qar_com;
	rcb = qar_com->rcbcom_rdrcb;
	cstr = NULL;
	ret = sm_rcb_open_dec(rcb);
	if (sm_is_err(ret))
		goto error;

	/* total length of record */
	ret = sm_rcb_getuint32(rcb, &tl);
	if (sm_is_err(ret) || tl > QM_AR_MAX_REC_LEN ||
	    tl > sm_rcb_getlen(rcb))
		goto err2;

	/* protocol header: version */
	ret = sm_rcb_get3uint32(rcb, &l, &rt, &v);
	if (sm_is_err(ret))
		goto err2;
	if (l != 4 || rt != RT_PROT_VER || v != PROT_VER_RT)
		goto err2;

	/* XXX define protocol first in smX docs! */

/*
*/

	/* XXX decode data, act accordingly... */
	ret = sm_rcb_get3uint32(rcb, &l, &rt, &v);
	if (sm_is_err(ret) || l != 4 || rt != RT_A2S_ID)
		goto err2;

	ret = sm_rcb_get2uint32(rcb, &l, &rt);
	if (sm_is_err(ret) || l != SMTP_STID_SIZE || rt != RT_A2S_TAID)
		goto err2;
	ret = sm_rcb_getn(rcb, (uchar *) ta_id, l);
	if (sm_is_err(ret))
		goto err2;

	ret = sm_rcb_get3uint32(rcb, &l, &rt, &v);
	if (sm_is_err(ret) || l != 4)
		goto err3;

	SM_TEST(rt == RT_A2S_MAP_RES);
	if (rt == RT_A2S_MAP_RES && v == SM_ACC_FOUND)
	{
		ret = sm_rcb_get3uint32(rcb, &l, &rt, &v);
		if (sm_is_err(ret) || l != 4)
			goto err3;
		SM_TEST(rt == RT_A2S_CLT_A_ST);
		if (rt == RT_A2S_CLT_A_ST)
			sm_io_fprintf(smioerr, "status=%d\n", v);
	}

	while (!SM_RCB_ISEOB(rcb))
	{
		ret = sm_rcb_get2uint32(rcb, &l, &rt);
		switch (rt)
		{
		  case RT_A2S_STATT:
			if (l < sizeof(statt))
			{
				sm_memzero(statt, sizeof(statt));
				ret = sm_rcb_getn(rcb, (uchar *) statt, l);
sm_io_fprintf(smioerr, "react=statt, ret=%x, statt='%.256s'\n", ret, statt);
				if (sm_is_err(ret))
					goto err2;
			}
			else
			{
sm_io_fprintf(smioerr, "ERROR: react=statt, stat=too-long\n");
				SM_TEST(false);
				goto err2;
			}
			break;
		  case RT_A2S_RVRS_ST:
			ret = sm_rcb_getuint32(rcb, &v);
sm_io_fprintf(smioerr, "react=rvrs_st, ret=%x, stat=%x\n", ret, v);
			break;
		  case RT_A2S_RVRS_NAME:
			ret = sm_rcb_getncstr(rcb, &cstr, l);
sm_io_fprintf(smioerr, "react=rvrs_name, ret=%x, name=%.256C\n", ret, cstr);
			break;
		  default:
sm_io_fprintf(smioerr, "ERROR: record-type=unknown, rt=%x, l=%d\n", rt, l);
			SM_TEST(false);
			break;
		}
	}

#if 0
	ret = sm_rcb_get3uint32(rcb, &l, &rt, &v);
	if (sm_is_err(ret) || l != 4 || rt != RT_R2Q_RCPT_NAR)
		goto err3;
#endif /* 0 */

	ret = sm_rcb_close_dec(qar_com->rcbcom_rdrcb);
	(void) sm_rcb_open_rcv(qar_com->rcbcom_rdrcb);
	--Requests;
	return ret;

  err3:
  err2:
	/* use rcb functions that don't do check the state */
	(void) sm_rcb_close_decn(qar_com->rcbcom_rdrcb);
  error:
	/* open rcb for receiving next record */
	(void) sm_rcb_open_rcvn(qar_com->rcbcom_rdrcb);
	sm_io_fprintf(smioerr, "ERROR: react=%x\n", ret);
	--Requests;
	return ret;
}

/*
**  AR2QMGR -- AR - test program interface
**
**	Parameters:
**		tsk -- evthr task
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
ar2qmgr(sm_evthr_task_P tsk)
{
	int fd;
	sm_ret_T ret;
	t_req_P t_req;
	rcbcom_ctx_P qar_com;

	SM_IS_EVTHR_TSK(tsk);

	t_req = (t_req_P) tsk->evthr_t_actx;
	qar_com = t_req->rq_qar_com;
	fd = tsk->evthr_t_fd;	/* checked in caller */
	ret = sm_rcb_rcv(fd, qar_com->rcbcom_rdrcb, QSS_RC_MINSZ);
	if (Verbose > 4)
		sm_io_fprintf(smioerr, "ar2qmgr, ret=%d, rw=%d, size=%d, len=%d, max=%d\n"
			, ret
			, qar_com->rcbcom_rdrcb->sm_rcb_rw
			, qar_com->rcbcom_rdrcb->sm_rcb_size
			, qar_com->rcbcom_rdrcb->sm_rcb_len
			, qar_com->rcbcom_rdrcb->sm_rcb_max
			);
	else if (Verbose > 1)
		sm_io_fprintf(smioerr, "ar2qmgr, ret=%d\n", ret);
	if (ret > 0)
	{
		return EVTHR_WAITQ;
	}
	else if (ret == 0)
	{
		ret = sm_rcb_close_rcv(qar_com->rcbcom_rdrcb);

		/* start appropriate function ... */
		ret = react(tsk);
		if (sm_is_err(ret))
			goto termit;	/* too harsh? */
		else if (ret == QMGR_R_WAITQ)
			return EVTHR_WAITQ;
		else if (ret == QMGR_R_ASYNC)
			return EVTHR_OK;
		else if (ret == EVTHR_DEL)
			goto termit;
		else
			return ret;
	}
	else if (ret == SM_IO_EOF)
	{
		ret = sm_rcb_close_rcv(qar_com->rcbcom_rdrcb);
  termit:
		close(fd);

		/* XXX see comment in qm_fr_ss() */
		tsk->evthr_t_fd = INVALID_FD;	/* make it invalid */

		return EVTHR_DEL;
	}
	else /* if (ret < 0) */
	{
	}
	return EVTHR_DEL;
}

/*
**  LOOKUP -- queue recipient address lookups to SMAR
**
**	Parameters:
**		tsk -- evthr task
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
lookup(sm_evthr_task_P tsk)
{
	t_req_P t_req;
	int n;
	sm_ret_T ret;
	char *addr;
	sm_str_P pa;

	SM_REQUIRE(tsk != NULL);
	t_req = (t_req_P) tsk->evthr_t_actx;
	if (t_req == NULL)
		return EVTHR_DEL|EVTHR_TERM;
	pa = NULL;

#if 0
	n = pthread_mutex_lock(&mutex);
	SM_TEST(n == 0);
	n = pthread_cond_wait(&cond, &mutex);
	SM_TEST(n == 0);
#endif /* 0 */
	if (Verbose > 1)
		sm_io_fprintf(smioerr, "lookup, count=%d\n", t_req->rq_count);
	for (n = 0; n < t_req->rq_count; n++)
	{
		addr = t_req->rq_pa[n];
		if (addr == NULL || *addr == '\0')
			return EVTHR_DEL|EVTHR_TERM;
		if (pa == NULL)
		{
			pa = sm_str_scpy(NULL, addr, SMAXLEN);
			SM_TEST(pa != NULL);
			if (pa == NULL)
				goto error;
		}
		else
		{
			sm_str_clr(pa);
			ret = sm_str_scat(pa, addr);
			SM_TEST(ret == SM_SUCCESS);
			if (sm_is_err(ret))
				goto error;
		}
		if (Verbose > 1)
			sm_io_fprintf(smioerr, "addr[%d]=%.256s\n", n, addr);
		ret = rcpt2ar(tsk, pa, t_req->rq_qar_com);
		if (sm_is_err(ret))
			goto error;
	}

	/* Wakeup write task */
	SM_TEST(t_req->rq_wr_tsk != NULL);
	if (t_req->rq_wr_tsk != NULL)
	{
		ret = evthr_en_wr(t_req->rq_wr_tsk);
		if (Verbose > 1)
			sm_io_fprintf(smioerr, "wakeup=%x\n", ret);
		SM_TEST(sm_is_success(ret));
	}

	/* Paranoia... */
	t_req->rq_count = 0;
	SM_STR_FREE(pa);

	/* We're done */
	return EVTHR_DEL;

  error:
	SM_STR_FREE(pa);
	return EVTHR_DEL|EVTHR_TERM;
}

/*
**  TESTSMAR -- Test SMAR
**
**	Parameters:
**		t_req -- test context
**
**	Returns:
**		none.
*/

static void
testsmar(t_req_P t_req)
{
	sm_ret_T ret;
	sm_evthr_task_P	task, task2, task3;
	struct timeval sleept;
	int lfd, n;
	rcbcom_ctx_P qar_com;

	evthr_ctx = NULL;
	task = task3 = NULL;

	ret = thr_init();
	SM_TEST(sm_is_success(ret));
	if (sm_is_err(ret))
		goto errq;
	ret = evthr_init(&evthr_ctx, 1, 6, 10);
	SM_TEST(sm_is_success(ret));
	if (sm_is_err(ret))
		goto errt1;
	SM_TEST(evthr_ctx != NULL);

	n = pthread_cond_init(&cond, NULL);
	SM_TEST(n == 0);
	n = pthread_mutex_init(&mutex, NULL);
	SM_TEST(n == 0);

	qar_com = t_req->rq_qar_com;
	ret = sm_rcbcom_open(qar_com);
	if (sm_is_err(ret))
		goto error;
	SM_IS_RCB(qar_com->rcbcom_rdrcb);

#if SMAR_TCP_NET
	(void) net_client_connect(smarip, smarport, &lfd);
#else
	(void) unix_client_connect(smarsock, &lfd);
#endif
	if (lfd < 0)
	{
		ret = sm_error_perm(SM_EM_AR, errno);
		goto error;
	}
	ret = sm_rcb_open_rcv(qar_com->rcbcom_rdrcb);
	if (sm_is_err(ret))
		goto error;

	ret = sm_fd_nonblock(lfd, true);
	if (sm_is_err(ret))
		goto error;	/* XXX COMPLAIN */
	ret = evthr_task_new(evthr_ctx, &task, EVTHR_EV_RD,
			lfd, NULL, qmgr_ar, (void *) t_req);
	if (sm_is_err(ret))
		goto error;
	t_req->rq_wr_tsk = task;

	n = gettimeofday(&sleept, NULL);
	SM_TEST(n == 0);
	sleept.tv_usec += 1000;
	ret = evthr_task_new(evthr_ctx, &task3, EVTHR_EV_SL, -1, &sleept,
				fcts, (void *) NULL);
	SM_TEST(sm_is_success(ret));
	SM_TEST(task3 != NULL);

	sleept.tv_usec += 1;
	ret = evthr_task_new(evthr_ctx, &task2, EVTHR_EV_SL, -1, &sleept,
				lookup, (void *) t_req);
	SM_TEST(sm_is_success(ret));
	SM_TEST(task2 != NULL);

	ret = evthr_loop(evthr_ctx);
	SM_TEST(sm_is_success(ret));
	goto done;

  error:
	sm_io_fprintf(smioerr, "ERROR\n");
  done:
	pthread_cond_destroy(&cond);
	pthread_mutex_destroy(&mutex);
	ret = evthr_stop(evthr_ctx);
	SM_TEST(sm_is_success(ret));
  errt1:
	ret = thr_stop();
	SM_TEST(sm_is_success(ret));
  errq:
	return;
}

/*
**  USAGE -- usage message
**
**	Parameters:
**		prg -- program name
**
**	Returns:
**		none
*/

static void
usage(const char *prg)
{
	fprintf(stderr, "usage: %s [options] IP-addresses...\n"
		"perform reverse lookup and access map check for IP addresses\n"
		"-d       query DNS blacklist\n"
		"-R n     wait n times 0.1s\n"
		"-V       increase verbosity\n"
		, prg
		);
	return;
}


/*
**  MAIN -- guess...
**
**	Parameters:
**		argc -- arg counter
**		argv -- arg vector
**
**	Returns:
**		usual exit code
*/

int
main(int argc, char *argv[])
{
	int r;
	t_req_T t_req;
	char **h;
	void *ptr;

	ptr = NULL;
	h = NULL;
	while ((r = getopt(argc, argv, "dR:V")) != -1)
	{
		switch (r)
		{
		  case 'd':
			lfl = SMARA_LFL_IPV4|SMARA_LFL_DNSBL;
			break;
		  case 'R':
			MaxCount = atoi(optarg);
			break;
		  case 'V':
			Verbose++;
#if RCBCOMM_DEBUG
			rcbcomm_debug++;
#endif
			break;
		  default:
			usage(argv[0]);
			return 1;
		}
	}

	sm_test_begin(argc, argv, "test smar 3");
	t_req.rq_count = argc - optind;
	h = (char **) sm_malloc(sizeof(char *) * t_req.rq_count);
	ptr = h;
	SM_TEST(h != NULL);
	if (h == NULL)
		goto error;
	t_req.rq_pa = h;
	t_req.rq_qar_com = &rcb_com;
	for (r = optind; r < argc; r++)
	{
		*h = argv[r];
		++h;
		++Requests;
	}
	testsmar(&t_req);
  error:
	if (ptr != NULL)
		sm_free(ptr);
	return sm_test_end();
}
