/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>

RCSID("$Id: xfs_vfsops-freebsd.c,v 1.16 2000/10/02 23:33:02 lha Exp $");

#include <xfs/xfs_common.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_vfsops.h>
#include <xfs/xfs_vfsops-bsd.h>
#include <xfs/xfs_vnodeops.h>
#include <xfs/xfs_node.h>

static vop_t **xfs_dead_vnodeop_p;

int
make_dead_vnode(struct mount *mp, struct vnode **vpp)
{
    XFSDEB(XDEBNODE, ("make_dead_vnode mp = %lx\n",
		      (unsigned long)mp));

    return getnewvnode(VT_NON, mp, xfs_dead_vnodeop_p, vpp);
}

static struct vnodeopv_entry_desc xfs_dead_vnodeop_entries[] = {
    {&vop_default_desc, (vop_t *) vop_defaultop},
    {&vop_lookup_desc,	(vop_t *) xfs_dead_lookup},
    {&vop_reclaim_desc, (vop_t *) xfs_returnzero},
    {&vop_lock_desc,	(vop_t *) vop_nolock},
    {&vop_unlock_desc,	(vop_t *) vop_nounlock},
    {&vop_islocked_desc,(vop_t *) vop_noislocked},
    {NULL, NULL}};

static struct vnodeopv_desc xfs_dead_vnodeop_opv_desc =
{&xfs_dead_vnodeop_p, xfs_dead_vnodeop_entries};

VNODEOP_SET(xfs_dead_vnodeop_opv_desc);

extern struct vnodeopv_desc xfs_vnodeop_opv_desc;

#if __FreeBSD_version >= 300000
static int
xfs_init(void)
{
    XFSDEB(XDEBVFOPS, ("xfs_init\n"));
#if defined(HAVE_KERNEL_VFS_OPV_INIT)
    vfs_opv_init(&xfs_vnodeop_opv_desc);
    vfs_opv_init(&xfs_dead_vnodeop_opv_desc);
#elif defined(HAVE_KERNEL_VFS_ADD_VNODEOPS) && !defined(KLD_MODULE)
    vfs_add_vnodeops (&xfs_vnodeop_opv_desc);
    vfs_add_vnodeops (&xfs_dead_vnodeop_opv_desc);
#endif
    return 0;
}

#else

static int
xfs_init(void)
{
    struct vnodeopv_desc *foo[] = {&xfs_vnodeop_opv_desc, NULL};
    struct vnodeopv_desc *dead_foo[] = {&xfs_dead_vnodeop_opv_desc, NULL};

    XFSDEB(XDEBVFOPS, ("xfs_init\n"));
    *(foo[0]->opv_desc_vector_p) = NULL;
    vfs_opv_init(foo);
    *(dead_foo[0]->opv_desc_vector_p) = NULL;
    vfs_opv_init(dead_foo);

    return 0;
}

#endif

static int
xfs_uninit(struct vfsconf *vfc)
{
    XFSDEB(XDEBVFOPS, ("xfs_uninit\n"));
    return 0;
}

struct vfsops xfs_vfsops = {
    xfs_mount,
    xfs_start,
    xfs_unmount,
    xfs_root,
    xfs_quotactl,
    xfs_statfs,
    xfs_sync,
    xfs_vget,
    xfs_fhtovp,
#ifdef HAVE_STRUCT_VFSOPS_VFS_CHECKEXP
    xfs_checkexp,
#endif
    xfs_vptofh,
    xfs_init
};

static struct vfsconf xfs_vfc = {
    &xfs_vfsops,
    "xfs",
    0,
    0,
    0
};

#ifndef HAVE_KERNEL_VFS_REGISTER

static int
vfs_register (struct vfsconf *vfs)
{
    int i;

    for (i = 0; i < MOUNT_MAXTYPE; i++)
	if (strcmp(vfsconf[i]->vfc_name, vfs->vfc_name) == 0)
	    return EEXIST;

    for (i = MOUNT_MAXTYPE - 1; i >= 0; --i)
	if (vfsconf[i] == &void_vfsconf)
	    break;

    if (i < 0) {
	XFSDEB(XDEBVFOPS, ("failed to find free VFS slot\n"));
	return EINVAL;
    }

    vfs->vfc_index = i;
    vfsconf[i] = vfs;

    vfssw[i] = vfs->vfc_vfsops;
    (*(vfssw[i]->vfs_init)) ();
    return 0;
}

static int
vfs_unregister (struct vfsconf *vfs)
{
    int i = vfs->vfc_index;

    if (vfs->vfc_refcount)
	return EBUSY;

    vfsconf[i] = &void_vfsconf;
    vfssw[i]   = NULL;
    return 0;
}

#endif

#if KLD_MODULE

int
xfs_install_filesys(void)
{
    return 0;
}

int
xfs_uninstall_filesys(void)
{
    return 0;
}

int
xfs_stat_filesys (void)
{
    return 0;
}

#else

int
xfs_install_filesys(void)
{
    return vfs_register(&xfs_vfc);
}

int
xfs_uninstall_filesys(void)
{
    return vfs_unregister(&xfs_vfc);
}

int
xfs_stat_filesys (void)
{
    return 0;
}

#endif /* KLD_MODULE */
