/* hey emacs! -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * $Id: Packet.cc,v 1.12 2002/03/08 05:54:30 benoit Exp $
 *
 * Copyright (c) 2000 Remi Lefebvre <remi@dhis.net>
 *                and Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#include <string.h>
#include "Packet.h"
#include "DdtCipher.h"
#include "DdtMD.h"
#include "DdtRand.h"

int Packet::encode (DdtpPacket* pkt, char* buf, int bufSize)
{
    XDR xdr;
    xdrmem_create(&xdr, buf, bufSize, XDR_ENCODE);
    if (xdr_DdtpPacket(&xdr, pkt) == FALSE)
    {
        return -1;
    }
    return 0;
}

int Packet::decode (DdtpPacket* pkt, char* buf, int bufSize)
{
    XDR xdr;
    xdrmem_create(&xdr, buf, bufSize, XDR_DECODE);
    if (xdr_DdtpPacket(&xdr, pkt) == FALSE)
    {
        return -1;
    }
    return 0;
}

int Packet::decrypt (DdtpPacket* pkt, char* md5_password)
{
    Logger* logger = Logger::Instance();

    switch (pkt->protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Packet* ddtpv1 = &(pkt->DdtpPacket_u.ddtpv1);
            switch (ddtpv1->encryptionType)
            {
                case PLAINTEXT:
                {
                    DDTPv1Plaintext* ptmsg = &(ddtpv1->DDTPv1Packet_u.pt);
                    return decryptDDTPv1Plaintext (ptmsg);
                    break;
                }
                case BLOWFISH:
                {
                    DDTPv1Blowfish* bfmsg = &(ddtpv1->DDTPv1Packet_u.bf);
                    return decryptDDTPv1Blowfish (bfmsg, md5_password);
                    break;
                }
                case ENCRYPTION_ERROR:
                {
                    switch (ddtpv1->DDTPv1Packet_u.status)
                    {
                        case INVALID_ENCRYPTION_TYPE:
                            break;
                        case ACCOUNT_INVALID:
                            logger->notice("account invalid");
                            break;
                        case PASSWORD_INVALID:
                            logger->notice("password invalid");
                            break;
                        case XDR_DECODE_FAILURE:
                            logger->debug("xdr decoding failed");
                            break;
                    }
                    break;
                }
                default:
                    logger->debug("invalid encryption type");
                    break;
            }
            break;
        }
        case DDTPv2:
            // protocol not yet developed
            break;
        case PROTOCOL_ERROR:
            logger->debug("protocol error");
        default:
            logger->debug("invalid protocol");
            break;
    }
    return -1;
}

int Packet::encrypt(DdtpPacket* pkt, char* md5_password)
{
    Logger* logger = Logger::Instance();

    switch (pkt->protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Packet* ddtpv1 = &(pkt->DdtpPacket_u.ddtpv1);
            switch (ddtpv1->encryptionType)
            {
                case PLAINTEXT:
                {
                    DDTPv1Plaintext* ptmsg = &(ddtpv1->DDTPv1Packet_u.pt);
                    return encryptDDTPv1Plaintext (ptmsg);
                    break;
                }
                case BLOWFISH:
                {
                    DDTPv1Blowfish* bfmsg = &(ddtpv1->DDTPv1Packet_u.bf);
                    return encryptDDTPv1Blowfish (bfmsg, md5_password);
                    break;
                }
                default:
                    logger->debug("invalid encryption type");
                    break;
            }
            break;
        }
        case DDTPv2:
            // protocol not yet developed
            break;
        default:
            logger->debug("invalid protocol");
            break;
    }
    return -1;
}

int Packet::getAccountId (DdtpPacket* pkt)
{
    Logger* logger = Logger::Instance();

    switch (pkt->protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Packet* ddtpv1 = &(pkt->DdtpPacket_u.ddtpv1);
            switch (ddtpv1->encryptionType)
            {
                case PLAINTEXT:
                    return (ddtpv1->DDTPv1Packet_u.pt.accountId);
                    break;
                case BLOWFISH:
                    return (ddtpv1->DDTPv1Packet_u.bf.accountId);
                    break;
                case ENCRYPTION_ERROR:
                    switch (ddtpv1->DDTPv1Packet_u.status)
                    {
                        case INVALID_ENCRYPTION_TYPE:
                            break;
                        case ACCOUNT_INVALID:
                            logger->notice("account invalid");
                            break;
                        case PASSWORD_INVALID:
                            logger->notice("password invalid");
                            break;
                        case XDR_DECODE_FAILURE:
                            logger->debug("xdr decoding failed");
                            break;
                    }
                    break;
                default:
                    logger->debug("invalid encryption type");
                    break;
            }
            break;
        }
        case DDTPv2:
            // protocol not yet developed
            break;
        case PROTOCOL_ERROR:
            logger->debug("protocol error");
        default:
            logger->debug("invalid protocol");
            break;
    }
    return -1;
}

int Packet::createUpdateQuery (
    DdtpPacket*             pkt,
    ProtocolVersion         protocolVersion,
    EncryptionType          encryptionType,
    int                     accountId,
    DDTPv1UpdateQueryOpCode opCode,
    unsigned long           ipAddress,
    char*                   updatePassword)
{
    pkt->protocolVersion = protocolVersion;
    switch (protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Packet* ddtpv1 = &(pkt->DdtpPacket_u.ddtpv1);
            ddtpv1->encryptionType                     = encryptionType;
            ddtpv1->DDTPv1Packet_u.pt.accountId        = accountId;
            ddtpv1->DDTPv1Packet_u.pt.msg.messageType  = ALIVE_QUERY;

            DDTPv1Message* msg = &(ddtpv1->DDTPv1Packet_u.pt.msg);
            msg->messageType = UPDATE_QUERY;

            DDTPv1UpdateQuery* updateQuery =&(msg->DDTPv1Message_u.updateQuery);
            updateQuery->opCode     = opCode;
            updateQuery->ipAddress  = ipAddress;
            strncpy (updateQuery->updatePassword, updatePassword, 9);
            
            DdtCipher cipher(DdtCipher::ALGO_ROT13, DdtCipher::MODE_NONE);
            cipher.encrypt((unsigned char*)updateQuery->updatePassword, strlen(updateQuery->updatePassword), 
                          (unsigned char*)updateQuery->updatePassword, strlen(updateQuery->updatePassword));
            break;
        }
        case DDTPv2:
            // not implemented
            return -1;
            break;
        default:
            return -1;
            break;
    }
    return 0;
}

int Packet::createAliveQuery (
    DdtpPacket*             pkt,
    ProtocolVersion         protocolVersion,
    int                     accountId)
{
    pkt->protocolVersion = protocolVersion;
    switch (protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Packet* ddtpv1 = &(pkt->DdtpPacket_u.ddtpv1);
            ddtpv1->encryptionType                     = PLAINTEXT;
            ddtpv1->DDTPv1Packet_u.pt.accountId        = accountId;
            ddtpv1->DDTPv1Packet_u.pt.msg.messageType  = ALIVE_QUERY;
            break;
        }
        case DDTPv2:
            // not implemented
            return -1;
            break;
        default:
            return -1;
            break;
    }
    return 0;
}

int Packet::createAliveReply (
    DdtpPacket*             pkt,
    ProtocolVersion         protocolVersion,
    int                     accountId)
{
    pkt->protocolVersion = protocolVersion;
    switch (protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Packet* ddtpv1 = &(pkt->DdtpPacket_u.ddtpv1);
            ddtpv1->encryptionType                     = PLAINTEXT;
            ddtpv1->DDTPv1Packet_u.pt.accountId        = accountId;
            ddtpv1->DDTPv1Packet_u.pt.msg.messageType  = ALIVE_REPLY;
            break;
        }
        case DDTPv2:
            // not implemented
            return -1;
            break;
        default:
            return -1;
            break;
    }
    return 0;
}

int Packet::decryptDDTPv1Plaintext (DDTPv1Plaintext* pt)
{
    // do nothing
    return 0;
};

int Packet::encryptDDTPv1Plaintext (DDTPv1Plaintext* pt)
{
    // do nothing
    return 0;
};

int Packet::decryptDDTPv1Blowfish (DDTPv1Blowfish* bf, char* md5_password)
{
#ifdef DDT_SERVER
    DdtCipher *cipher = NULL;
    Logger* logger = Logger::Instance();
    logger->debug("in decrypt using blowfish");

    // BLOWFISH INITIALIZATION VECTOR
    // retrieve the iv from the packet
    unsigned char cbc_iv[8];
    memcpy (cbc_iv, bf->iv, 8);

    // BLOWFISH DECRYPTION OF THE CHECK
    unsigned char check[8]; 

    cipher = new DdtCipher(DdtCipher::ALGO_BLOWFISH, DdtCipher::MODE_ECB);
    cipher->setKey((unsigned char*)md5_password, DdtMD::getAlgoDigestLength(DdtMD::MD_MD5));
    cipher->setIV(cbc_iv, sizeof(cbc_iv));
    cipher->decrypt((unsigned char*)(&bf->check), sizeof(bf->check), check, sizeof(check));
    delete cipher; cipher = NULL;
    
    if (memcmp(check, cbc_iv, 8) != 0)
    {
        logger->notice("%d password invalid", bf->accountId);
        *((DDTPv1PacketStatus*)bf) = PASSWORD_INVALID;
        return -1;
    }
    
    // BLOWFISH DECRYPTION OF THE MESSAGE
    // decrypt the DDTPv1Message portion of the DdtpPacket into a buffer
    unsigned char buf[56];
    cipher = new DdtCipher(DdtCipher::ALGO_BLOWFISH, DdtCipher::MODE_CBC);
    cipher->setKey((unsigned char*)md5_password, DdtMD::getAlgoDigestLength(DdtMD::MD_MD5));
    cipher->setIV(cbc_iv, sizeof(cbc_iv));
    cipher->decrypt((unsigned char*)bf->msg, 56, buf, 56);
    delete cipher; cipher = NULL;

    // XDR DECODE
    // decode the buffer (in the XDR sense) back into the DdtpPacket!
    XDR xdr;
    xdrmem_create (&xdr, (char*)buf, 56, XDR_DECODE);
    if (xdr_DDTPv1Message (&xdr, (DDTPv1Message*)(&bf->msg)) == FALSE)
    {
        *((DDTPv1PacketStatus*)bf) = XDR_DECODE_FAILURE;
        logger->notice("xdr decoding failed");
        logger->debug("xdr decoding failed");
        return -1;
    }
    xdr_destroy (&xdr);
#endif // DDT_SERVER
    return 0;
}

int Packet::encryptDDTPv1Blowfish (DDTPv1Blowfish* bf, char* md5_password)
{
    DdtCipher *cipher = NULL;
    Logger* logger = Logger::Instance();

    // BLOWFISH INITIALIZATION VECTOR
    // generate a random initialization vector
    unsigned char iv[8];
    unsigned char cbc_iv[8];
    DdtRand::randomBytes(iv, 8);
    memcpy (cbc_iv, iv, 8);
    memcpy (bf->iv, iv, 8);

    // XDR ENCODE
    // encode the DDTPv1Message portion into a buffer
    unsigned char buf[56];
    XDR xdr;
    xdrmem_create (&xdr, (char*)buf, 56, XDR_ENCODE);
    if (xdr_DDTPv1Message (&xdr, (DDTPv1Message*)(&bf->msg)) == FALSE)
    {
        return -1;
    }
    xdr_destroy (&xdr);

    // BLOWFISH ENCRYPTION
    // encrypt the buffer back into the DdtpPacket!
    cipher = new DdtCipher(DdtCipher::ALGO_BLOWFISH, DdtCipher::MODE_CBC);
    int md5Length = DdtMD::getAlgoDigestLength(DdtMD::MD_MD5);
    cipher->setKey((unsigned char*)md5_password, md5Length);
    cipher->setIV(cbc_iv, sizeof(cbc_iv));
    cipher->encrypt(buf, 56, (unsigned char*)bf->msg, 56);
    delete cipher; cipher = NULL;

    //BF_cbc_encrypt_w (logger, buf, (unsigned char*)bf->msg, 56,
    //                  &key, cbc_iv, BF_ENCRYPT);

    // encrypt the iv into the check 
    cipher = new DdtCipher(DdtCipher::ALGO_BLOWFISH, DdtCipher::MODE_ECB);
    cipher->setKey((unsigned char*)md5_password, md5Length);
    //cipher->setIV(cbc_iv, sizeof(cbc_iv));
    cipher->encrypt(iv, 8, (unsigned char*)bf->check, 8);
    delete cipher; cipher = NULL;

    //BF_ecb_encrypt_w (logger, iv, (unsigned char*)(&bf->check), &key, BF_ENCRYPT);

    // perform a check to ensure that
    // nothing has been overwritten
    if (memcmp(bf->iv, iv, 8) != 0)
    {
        return -1;
    }
    return 0;
}

/*
int PacketProcessor::processDDTPv1Packet (DDTPv1Packet* ddtpv1)
{
    int rval = -1;

    switch (ddtpv1->encryptionType)
    {
        case PLAINTEXT:
            rval = processDDTPv1Plaintext (&(ddtpv1->DDTPv1Packet_u.pt));
            rval = processDDTPv1Message (&(ddtpv1->DDTPv1Packet_u.pt.msg),
                                           ddtpv1->DDTPv1Packet_u.pt.accountId);
            ddtpv1->encryptionType = PLAINTEXT;
            break;
        case BLOWFISH:
            rval = processDDTPv1Blowfish (&(ddtpv1->DDTPv1Packet_u.bf));
            if (rval == -1)
            {
                // only packets from the client are encrypted...
                replyRequired_ = true;
                ddtpv1->encryptionType = ENCRYPTION_ERROR;
                break;
            }
            rval = processDDTPv1Message (&(ddtpv1->DDTPv1Packet_u.pt.msg),
                                           ddtpv1->DDTPv1Packet_u.pt.accountId);
            ddtpv1->encryptionType = PLAINTEXT;
            break;
        case ENCRYPTION_ERROR:
            // set the replyStatus_ variable
            switch (ddtpv1->DDTPv1Packet_u.status)
            {
                case ACCOUNT_INVALID:
                    log_->notice("account invalid");
                    replyStatus_ = INVALID_ACCOUNT;
                    break;
                case PASSWORD_INVALID:
                    log_->notice("password invalid");
                    replyStatus_ = INVALID_PASSWORD;
                    break;
                case XDR_DECODE_FAILURE:
                    log_->debug("xdr decoding failed");
                    break;
            }
            break;
        default:
            ddtpv1->encryptionType = ENCRYPTION_ERROR;
            ddtpv1->DDTPv1Packet_u.status = INVALID_ENCRYPTION_TYPE;
            log_->debug("invalid encryption type");
            rval = -1;
            break;
    }
    return rval;
}

int PacketProcessor::processDDTPv1Plaintext (DDTPv1Plaintext* pt)
{
    // do nothing
    return 0;
};

int PacketProcessor::processDDTPv1Blowfish (DDTPv1Blowfish* bf)
{
#ifdef DDT_SERVER
    // BLOWFISH KEY
    // generate the blowfish key
    // this is a two step process:
    // 1) retrieve the 32-byte 7-bit string that is the MD5-hash of the password
    // 2) set up the blowfish key

 
    // step 1: retrieve the 32-byte 7-bit MD5-hash from the database
    int md5Length = DdtMD::getAlgoDigestLength(DdtMD::MD_MD5);
    char md5_password[2 * md5Length + 1];
    if (db_->fetchAccountInfo (bf->accountId, "updatePassword", md5_password,
                               (2 * md5Length + 1)) == false)
    {
        *((DDTPv1PacketStatus*)bf) = ACCOUNT_INVALID;
        log_->notice("%d account invalid", bf->accountId);
        return -1;
    }

    // step 2: set up the blowfish key
    BF_KEY key;
    BF_set_key_w (log_, &key, MD5_DIGEST_LENGTH, (unsigned char*)md5_password);

    // BLOWFISH INITIALIZATION VECTOR
    // retrieve the iv from the packet
    unsigned char cbc_iv[8];
    memcpy (cbc_iv, bf->iv, 8);

    // BLOWFISH DECRYPTION OF THE CHECK
    unsigned char check[8];
    BF_ecb_encrypt_w (log_, (unsigned char*)(&bf->check), check, &key, BF_DECRYPT);
    if (memcmp(check, cbc_iv, 8) != 0)
    {
        log_->notice("%d password invalid", bf->accountId);
        *((DDTPv1PacketStatus*)bf) = PASSWORD_INVALID;
        return -1;
    }
    
    // BLOWFISH DECRYPTION OF THE MESSAGE
    // decrypt the DDTPv1Message portion of the DdtpPacket into a buffer
    unsigned char buf[56];
    BF_cbc_encrypt_w (log_, (unsigned char*)bf->msg, buf, 56,
                    &key, cbc_iv, BF_DECRYPT);

    // XDR DECODE
    // decode the buffer (in the XDR sense) back into the DdtpPacket!
    XDR xdr;
    xdrmem_create (&xdr, (char*)buf, 56, XDR_DECODE);
    if (xdr_DDTPv1Message (&xdr, (DDTPv1Message*)(&bf->msg)) == FALSE)
    {
        *((DDTPv1PacketStatus*)bf) = XDR_DECODE_FAILURE;
        log_->debug("xdr decoding failed");
        return -1;
    }
    xdr_destroy (&xdr);
#endif // DDT_SERVER
    return 0;
}

int PacketProcessor::processDDTPv1Message (DDTPv1Message* msg, int accountId)
{
    int rval = -1;

    messageType_ = msg->messageType;

    if (amServer_)
    {
        switch (msg->messageType)
        {
            case UPDATE_QUERY:
                rval = processDDTPv1UpdateQuery (
                        &(msg->DDTPv1Message_u.updateQuery),
                        &(msg->DDTPv1Message_u.updateReply),
                        accountId);
                msg->messageType = UPDATE_REPLY;
                replyRequired_ = true;
                break;
            case ALIVE_REPLY:
                rval = processDDTPv1AliveReply (
                        &(msg->DDTPv1Message_u.aliveReply),
                        accountId);
                replyRequired_ = false;
                break;
            case INVALID_ENCRYPTION_TYPE:
            default:
                msg->messageType = MESSAGE_ERROR;
                msg->DDTPv1Message_u.status = INVALID_MESSAGE_TYPE;
                log_->debug("invalid message type");
                rval = -1;
                replyRequired_ = false;
                break;
        }
    }
    else
    {
        switch (msg->messageType)
        {
            case UPDATE_REPLY:
                rval = processDDTPv1UpdateReply (
                        &(msg->DDTPv1Message_u.updateReply));
                replyRequired_ = false;
                break;
            case ALIVE_QUERY:
                rval =  processDDTPv1AliveQuery (
                        &(msg->DDTPv1Message_u.aliveQuery),
                        &(msg->DDTPv1Message_u.aliveReply));
                msg->messageType = ALIVE_REPLY;
                replyRequired_ = true;
                break;
            default:
                msg->messageType = MESSAGE_ERROR;
                msg->DDTPv1Message_u.status = INVALID_MESSAGE_TYPE;
                log_->debug("invalid message type");
                rval = -1;
                replyRequired_ = false;
                break;
        }
    }
    return rval;
};

int PacketProcessor::processDDTPv1UpdateQuery (DDTPv1UpdateQuery* updateQuery,
                                               DDTPv1UpdateReply* updateReply,
                                               int accountId)
{
    log_->debug("received update query");
#ifdef DDT_SERVER

    // validation of the password
    // first, get the update password for the accountId from the database
    char db_password[2 * MD5_DIGEST_LENGTH + 1];
    if (db_->fetchAccountInfo (accountId, "updatePassword", db_password,
                               (2 * MD5_DIGEST_LENGTH + 1)) == false)
    {
        log_->notice("%d account invalid", accountId);
        updateReply->status = INVALID_ACCOUNT;
        return -1;
    }

    // retreive the password that's in the message and MD5 hash it
    char          msg_password[MD5_DIGEST_LENGTH * 2 + 1];
    unsigned char       digest[MD5_DIGEST_LENGTH    ];

    char updatePassword[9];
    strncpy  (updatePassword, updateQuery->updatePassword, 9);
    strrot13 (updatePassword);
    MD5_w (log_, (unsigned char*)updatePassword, strlen(updatePassword), digest);

    for (int i = 0; i < MD5_DIGEST_LENGTH; i++)
    {
        sprintf(&(msg_password[i*2]), "%02x", digest[i]);
    }

    // validate the message password against the database password
    if (strncasecmp(msg_password, db_password, 2 * MD5_DIGEST_LENGTH))
    {
        log_->notice("%d password invalid", accountId);
        updateReply->status = INVALID_PASSWORD;
        return -1;
    }

    // process the query
    switch (updateQuery->opCode)
    {
        // FIXME: here we should set the hoststatus to WAITING_ONLINE
        //        in the database and wait a first keepalive reply
        //        before actually changing the IP to avoid bad IPs.
        case MARK_ONLINE:
            if (db_->markOnline(accountId, ntohl (updateQuery->ipAddress)) == -1)
            {
                log_->debug("%d could not be marked online", accountId);
                updateReply->status = UPDATE_FAILED;
                return -1;
            }

            struct in_addr addr;
            addr.s_addr = htonl (updateQuery->ipAddress);
            log_->notice("%d marked online at %s", accountId, inet_ntoa (addr));

            updateReply->status = UPDATE_SUCCEEDED;
            return 0;
            break;
        case MARK_OFFLINE:
            if (db_->markOffline(accountId) == -1)
            {
                log_->debug("%d could not be marked offline", accountId);
                updateReply->status = UPDATE_FAILED;
                return -1;
            }
            log_->notice("%d marked offline", accountId);
            updateReply->status = UPDATE_SUCCEEDED;
            return 0;
            break;
        default:
            updateReply->status = INVALID_OPCODE;
            log_->debug("invalid op code");
            return -1;
            break;
    }
#endif // DDT_SERVER    
    return -1;
};

// this could eventually be done in the client with the new getUpdateReply()
// function
int PacketProcessor::processDDTPv1UpdateReply (DDTPv1UpdateReply* updateReply)
{
    log_->debug("received update reply");

    replyStatus_ = updateReply->status;

    switch (updateReply->status)
    {
        case UPDATE_SUCCEEDED:
            log_->debug("online/offline update query succeeded");
            return 0;
            // shouldn't there be a break here ?
        case UPDATE_FAILED:
            log_->debug("online/offline update query failed");
            return -1;
            break;
        // FIXME: will never ever reach there .. we need restructure this a tiny bit
        case INVALID_PASSWORD:
            log_->notice("invalid password");
            return -1;
            break;
        case INVALID_ACCOUNT:
            log_->notice("invalid account");
            return -1;
            break;
        case INVALID_OPCODE:
            log_->debug("invalid opcode");
            return -1;
            break;
    }
    return 0;
}

int PacketProcessor::processDDTPv1AliveQuery (DDTPv1AliveQuery* aliveQuery,
                                              DDTPv1AliveReply* aliveReply)
{
    log_->debug("received alive query");
    return 0;
}

int PacketProcessor::processDDTPv1AliveReply (DDTPv1AliveReply* aliveReply, int accountId)
{
    log_->debug("received alive reply");
#ifdef DDT_SERVER
    if (db_->markAlive(accountId) == -1)
    {
        log_->debug("%d could not be marked alive", accountId);
        return -1;
    }
    log_->info("%d is alive", accountId);
#endif // DDT_SERVER
    return 0;
}
*/
