/********************************************************************************
 *   Jabberoo/Judo -- C++ Jabber Library                                        *
 *                                                                              * 
 *   Copyright (C) 1999-2000 Dave Smith (dave@jabber.org)                       *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or              *
 *   modify it under the terms of the GNU Lesser General Public                 *
 *   License as published by the Free Software Foundation; either               *
 *   version 2.1 of the License, or (at your option) any later version.         *
 *                                                                              *
 *   This library is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 *   Lesser General Public License for more details.                            *
 *                                                                              *
 *   You should have received a copy of the GNU Lesser General Public           *
 *   License along with this library; if not, write to the Free Software        *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 ********************************************************************************
 */

#include "judo.hh"

using namespace judo;

void TagStream::push(const char* data)
{
    // Ensure expat is initialized
    initExpat();
    // Attempt to parse the data
    if (!XML_Parse(*_Parser, data, strlen(data), 0))
	 throw XCP_ParserFailure();
    // See if expat needs to be reset
    if (_DocumentEnded)
    {
        cleanupExpat();
	_DocumentEnded = false;
    }
}

TagStream::TagStream()
    : _Parser(NULL), _DocumentStarted(false), _DocumentEnded(false)
{}

TagStream::~TagStream()
{
    cleanupExpat();
}

// Expat callbacks
void TagStream::OnStartElement(void* pUserdata, const char* pszElementName, const char** apszAttribs)
{
    TagStream* strm = (TagStream*)pUserdata;

    // Document has started
    if (strm->_DocumentStarted)
    {
	// If the stack isn't empty (we're already building a packet), push
	// a new child tag onto the stack (using the top of the stack as the parent)
	if (!strm->_TagStack.empty())
	    strm->_TagStack.push(&(strm->_TagStack.top()->addTag(pszElementName, 
								 apszAttribs)));
	// Otherwise, this is the first element, so we just allocate a fresh tag
	else
	    // Allocate the first element and push it on the stack
	    strm->_TagStack.push(new Tag(pszElementName, apszAttribs));
    }
    // Document hasn't started yet...
    else
    {
	Tag* root = new Tag(pszElementName, apszAttribs);
	strm->OnDocumentStart(root);
	strm->_DocumentStarted = true;
    }
}

void TagStream::OnEndElement(void* pUserdata, const char* pszElementName)
{
    TagStream* strm    = (TagStream*)pUserdata;
    Tag*       current = NULL;

    switch (strm->_TagStack.size())
    {
    case 1:
	// There is only one element on the stack, this must be the closing
	// element for the packet
	current = strm->_TagStack.top();
	strm->_TagStack.pop();
	strm->OnTag(current);
	break;
    case 0:
	// There are no elements left on the stack, this must be the closing
	// element for the document
	strm->OnDocumentEnd();
	strm->_DocumentEnded = true;
	strm->_DocumentStarted = false;
	break;
    default:
	// Otherwise, we just pop the topmost element off
	// and continue on our merry way
	strm->_TagStack.pop();
    }
}

void TagStream::OnCDATA(void* pUserdata, const char* pszCDATA, int iCDATASz)
{
    TagStream* strm = (TagStream*)pUserdata;

    // Add the CDATA section to the tag on top of the stack;
    // this will automatically merge adjacent CDATA sections
    strm->_TagStack.top()->addCDATA(pszCDATA, iCDATASz);
}

void TagStream::initExpat()
{
    if (_Parser == NULL)
    {
        // Setup expat parser
        _Parser = new XML_Parser;
        *_Parser = XML_ParserCreate(NULL);

        // Initialize callbacks
        XML_SetUserData(*_Parser, this);
        XML_SetElementHandler(*_Parser, &TagStream::OnStartElement, &TagStream::OnEndElement);
        XML_SetCharacterDataHandler(*_Parser, &TagStream::OnCDATA);
    }
}

void TagStream::cleanupExpat()
{
    if (_Parser != NULL)
    {
        XML_ParserFree(*_Parser);
        delete _Parser;
        _Parser = NULL;
    }
}
