/* 
 *  Copyright (C) 2001 Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 * Saving and loading bookmarks from Netscape and Mozilla
 */

#include "bookmarks_io.h"
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>


/* These are the only exported functions. */
BookmarkTranslator *bookmarks_io_netscape_init (void);
BookmarkTranslator *bookmarks_io_mozilla_init (void);

/* private functions */
static BookmarkItem *load_from_file_ns (const gchar *filename, 
					BookmarkItem **new_default_folder);
static BookmarkItem *load_from_file_moz (const gchar *filename, 
					 BookmarkItem **new_default_folder);
static BookmarkItem *load_from_string (const gchar *data, 
				       BookmarkItem **new_default_folder);
static gboolean save_to_file_ns (BookmarkItem *root,
				 BookmarkItem *default_folder,
				 const gchar *file);
static gboolean save_to_file_moz (BookmarkItem *root,
				  BookmarkItem *default_folder,
				  const gchar *file);
static gboolean save_to_string (BookmarkItem *root,
				BookmarkItem *default_folder,
				gchar **data);

/* netscape.c, these functions are kept there for historic reasons */
BookmarkItem *netscape_import_bookmarks (const gchar *filename, 
					 gboolean use_locale);
gboolean netscape_export_bookmarks (const gchar *filename, BookmarkItem *root,
				    gboolean use_locale);

/**
 * Info about the translator
 */

static gchar *extensions[] = { "html", NULL };

static BookmarkTranslator nsbm = {
	N_("Netscape's bookmarks format"),
	extensions,
	load_from_file_ns,
	load_from_string,
	save_to_file_ns,
	save_to_string
};

static BookmarkTranslator mozbm = {
	N_("Mozilla's bookmarks format"),
	extensions,
	load_from_file_moz,
	load_from_string,
	save_to_file_moz,
	save_to_string
};

BookmarkTranslator *
bookmarks_io_netscape_init (void)
{
	if (g_list_find (bookmarks_translators, &nsbm) == NULL)
		bookmarks_translators = g_list_prepend (bookmarks_translators,
							&nsbm);
	return &nsbm;
}

BookmarkTranslator *
bookmarks_io_mozilla_init (void)
{
	if (g_list_find (bookmarks_translators, &mozbm) == NULL)
		bookmarks_translators = g_list_prepend (bookmarks_translators,
							&mozbm);
	return &mozbm;
}

static gboolean
save_to_file_ns (BookmarkItem *root, BookmarkItem *default_folder, 
		 const gchar *filename)
{
        g_return_val_if_fail (filename != NULL, FALSE);
	netscape_export_bookmarks (filename, root, TRUE);
        return TRUE;
}

static gboolean
save_to_file_moz (BookmarkItem *root, BookmarkItem *default_folder, 
		  const gchar *filename)
{
        g_return_val_if_fail (filename != NULL, FALSE);
	netscape_export_bookmarks (filename, root, FALSE);
        return TRUE;
}

static gboolean
save_to_string (BookmarkItem *root,
		BookmarkItem *default_folder,
		gchar **data)
{
	return FALSE; // not implemented
}

static BookmarkItem *
load_from_file_ns (const char *file, BookmarkItem **default_bookmarks_root)
{
	BookmarkItem *b;

	if (!(g_file_exists (file))) {
		/* no bookmarks */
		return NULL;
	}

	b = netscape_import_bookmarks (file, TRUE);
	
	if (default_bookmarks_root) *default_bookmarks_root = NULL;

	return b;
}

static BookmarkItem *
load_from_file_moz (const char *file, BookmarkItem **default_bookmarks_root)
{
	BookmarkItem *b;
	
	if (!(g_file_exists (file))) {
		/* no bookmarks */
		return NULL;
	}

	b = netscape_import_bookmarks (file, FALSE);
	
	if (default_bookmarks_root) *default_bookmarks_root = NULL;

	return b;
}

static BookmarkItem *
load_from_string (const char *data, BookmarkItem **default_bookmarks_root)
{
	return NULL; // not implemented
}
