/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc_callbacks.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "misc_string.h"
#include "mozilla.h"
#include "link_interfaces.h"
#include "embed.h"
#include "window.h"
#include "eel-gconf-extensions.h"
#include "prefs_strings.h"

#include <string.h>
#include <stdlib.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gtkpixmapmenuitem.h>

typedef struct
{
	gchar *label;
	const gchar *icon;
} LinkProperties;

/* local function prototypes */
static gboolean link_menu_release_event_cb (GtkButton *button,
						      GdkEventButton *event,
				      		      GaleonWindow *window);
static void link_interfaces_cleanup (GList *items);
static LinkProperties *link_interfaces_get_props (LinkInterfaceItem *lti);
static void link_menu_activate_cb (GtkWidget *widget, GaleonWindow *window);
static void link_menu_select_cb (GtkWidget *widget, GaleonWindow *window);
static void link_menu_deselect_cb (GtkWidget *widget, GaleonWindow *window);

/**
 * link_interfaces_set_sensitivity: update the link toolbar intefaces'
 * sensitivity
 */
void
link_interfaces_set_sensitivity (GaleonEmbed *embed)
{
	GList *items = NULL;
	GaleonWindow *window = embed->parent_window;

	return_if_not_embed (embed);
	return_if_not_window (window);
	if (!embed->mozembed || !embed->wrapper) return;

	/* get usable link elemenets */
	items = mozilla_get_link_interface_items (embed);

	/* set relevant links menu button in the toolbar sensitivity */
	if (window->link_menu_button)
	{
		gtk_widget_set_sensitive (window->link_menu_button,
					  (g_list_length (items) != 0));
	}

	gtk_widget_set_sensitive (window->related_links_menuitem,
				  (g_list_length (items) != 0));

	link_interfaces_cleanup (items);
}

/** 
 * link_menu_release_event_cb: link menuitem released
 */
static gboolean
link_menu_release_event_cb (GtkButton *button, GdkEventButton *event,
				      GaleonWindow *window)
{
	GtkWidget *menu = GTK_WIDGET (button)->parent;
	
	if (event->button == 1 && GTK_IS_BUTTON (button) &&
	    !GTK_BUTTON (button)->button_down)
		return FALSE;
	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	
	return_val_if_not_window (window, FALSE);

	if (GTK_IS_MENU (menu))
	{
		gtk_menu_shell_deactivate (GTK_MENU_SHELL (menu));
	}

	embed_activate_link (window->active_embed, NULL,
		(gchar *) gtk_object_get_data (GTK_OBJECT (button), "url"),
		misc_general_mouse_state_to_link_state (event->button,
							event->state));

	return TRUE;
}

/**
 * link_interfaces_cleanup: free the items list
 */
static void
link_interfaces_cleanup (GList *items)
{
	GList *l;

	if (items)
	{
		for (l = items; l; l = g_list_next (l))
		{
			LinkInterfaceItem *lti =
				(LinkInterfaceItem *) l->data;
			if (lti->href)  g_free (lti->href);
			if (lti->title) g_free (lti->title);
			g_free (lti);
		}

		g_list_free (items);
	}
}

/**
 * link_menu_button_press_event_cb: link menu button in the main toolbar
 * pressed
 */
gboolean
link_menu_button_press_event_cb (GtkButton *button,
				 GdkEventButton *event,
				 GaleonWindow *window)
{
	GtkWidget *menu;

	if (event->button == 3)
	{
		return window_generic_button_press_event_cb
				(button, event, window);
	}

	gtk_signal_emit_stop_by_name (GTK_OBJECT (button),
				      "button-press-event");

	menu = gtk_menu_new ();

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);

	link_interfaces_build_menu (window, GTK_MENU (menu), TRUE);
	
	gnome_popup_menu_do_popup_modal (menu,
					 misc_gui_menu_position_under_widget,
					 GTK_WIDGET (button), event, NULL);
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), FALSE);
	gtk_widget_unref (menu);
	
	return TRUE;
}

/**
 * link_interfaces_get_props: build the appriopate label for a LinkInterfaceItem
 */
static LinkProperties*
link_interfaces_get_props (LinkInterfaceItem *lti)
{
	LinkProperties *ret = g_new0 (LinkProperties, 1);

	/* fill in props */
	switch (lti->type)
	{
	case LINK_ITEM_HOME:
   		/* fall through */
	case LINK_ITEM_START:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Start"));
		ret->icon = GNOME_STOCK_MENU_HOME;
		break;
	case LINK_ITEM_NEXT:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Next"));
		ret->icon = GNOME_STOCK_MENU_FORWARD;
		break;
	case LINK_ITEM_PREV:
		/* fall through */
	case LINK_ITEM_PREVIOUS:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Previous"));
		ret->icon = GNOME_STOCK_MENU_BACK;
		break;
	case LINK_ITEM_CONTENTS:
		/* fall through */
	case LINK_ITEM_TOC:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Contents"));
		ret->icon = GNOME_STOCK_MENU_INDEX;
		break;
	case LINK_ITEM_INDEX:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Index"));
		ret->icon = GNOME_STOCK_MENU_INDEX;
		break;
	case LINK_ITEM_GLOSSARY:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Glossary"));
		ret->icon = GNOME_STOCK_MENU_INDEX;
		break;
	case LINK_ITEM_COPYRIGHT:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Copyright"));
		ret->icon = GNOME_STOCK_MENU_ABOUT;
		break;
	case LINK_ITEM_CHAPTER:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Chapter"));
		ret->icon = GNOME_STOCK_MENU_BOOK_OPEN;
		break;
	case LINK_ITEM_SECTION:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Section"));
		ret->icon = GNOME_STOCK_MENU_BOOK_OPEN;
		break;
	case LINK_ITEM_SUBSECTION:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Subsection"));
		ret->icon = GNOME_STOCK_MENU_BOOK_OPEN;
		break;
	case LINK_ITEM_APPENDIX:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Appendix"));
		ret->icon = GNOME_STOCK_MENU_ATTACH;
		break;
	case LINK_ITEM_HELP:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Help"));
		ret->icon = GNOME_STOCK_MENU_BOOK_BLUE;
		break;
	case LINK_ITEM_TOP:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Top"));
		ret->icon = GNOME_STOCK_MENU_UP;
		break;
	case LINK_ITEM_SEARCH:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Search"));
		ret->icon = GNOME_STOCK_MENU_SEARCH;
		break;
	case LINK_ITEM_MADE:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Author"));
		ret->icon = GNOME_STOCK_MENU_ABOUT;
		break;
	default:
		if (lti->title)
			ret->label = g_strdup (lti->title);
		else
			ret->label = g_strdup (_("Untitled"));
		ret->icon = GNOME_STOCK_MENU_JUMP_TO;
		break;
	}

	return ret;
}

/**
 * link_menu_activate_cb: item in the link menu activated
 */
static void
link_menu_activate_cb (GtkWidget *widget, GaleonWindow *window)
{
	window_load_url (window,
		(gchar *) gtk_object_get_data (GTK_OBJECT (widget), "url"));
}

/**
 * link_menu_select_cb: menuitem selected, update statusbar
 */
static void
link_menu_select_cb (GtkWidget *widget, GaleonWindow *window)
{
	if (eel_gconf_get_integer (CONF_BOOKMARKS_TOOLTIPS) == 1)
	{
		gnome_appbar_push (GNOME_APPBAR (window->appbar), (gchar *)
			gtk_object_get_data (GTK_OBJECT (widget), "url"));
	}
}

/**
 * link_menu_deselect_cb: menuitem deselected, update statusbar
 */
static void
link_menu_deselect_cb (GtkWidget *widget, GaleonWindow *window)
{
	if (eel_gconf_get_integer (CONF_BOOKMARKS_TOOLTIPS) == 1)
	{
		gnome_appbar_pop (GNOME_APPBAR (window->appbar));
	}
}

/**
 * link_interfaces_build_menu: build a menu with relevant <link> elements
 */
void
link_interfaces_build_menu (GaleonWindow *window, GtkMenu *menu,
			    gboolean status)
{
	GList *l, *items;

	items = mozilla_get_link_interface_items (window->active_embed);

	for (l = items; l; l = g_list_next (l))
	{
		LinkInterfaceItem *lti = (LinkInterfaceItem *) l->data;
		LinkProperties *props = link_interfaces_get_props (lti);
		GtkWidget *m = gtk_pixmap_menu_item_new ();
		GtkWidget *label = gtk_label_new ("");
		GtkWidget *hb = gtk_hbox_new (FALSE, 0);
		gchar *tmp;

		/* set pixmap */
		gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (m),
			gnome_stock_pixmap_widget (GTK_WIDGET (m),
				props->icon));

		/* set label */
		tmp = misc_string_escape_uline_accel (props->label);
		misc_gui_label_set_accel_text (tmp, label,
					       GTK_WIDGET (menu), m);
		gtk_box_pack_start (GTK_BOX (hb), label, FALSE, FALSE, 0);
		gtk_container_add (GTK_CONTAINER (m), hb);
		g_free (tmp);

		/* connect signals */
		gtk_object_set_data_full (GTK_OBJECT (m), "url",
					  lti->href, g_free);
		gtk_object_set_data_full (GTK_OBJECT (m), "title",
					  props->label, g_free);
		gtk_object_set_data (GTK_OBJECT (m), "type",
				     GINT_TO_POINTER (lti->type));

		gtk_signal_connect (GTK_OBJECT (m), "button-press-event",
				    GTK_SIGNAL_FUNC
				   	(generic_link_button_press_event_cb),
				    window);
		gtk_signal_connect (GTK_OBJECT (m), "button-release-event",
				    GTK_SIGNAL_FUNC
				   	(link_menu_release_event_cb),
				    window);
		gtk_signal_connect (GTK_OBJECT (m), "activate",
				    GTK_SIGNAL_FUNC (link_menu_activate_cb),
				    window);
		if (status)
		{
			gtk_signal_connect (GTK_OBJECT (m), "select",
					    GTK_SIGNAL_FUNC
					    (link_menu_select_cb),
					    window);
			gtk_signal_connect (GTK_OBJECT (m), "deselect",
					    GTK_SIGNAL_FUNC
					    (link_menu_deselect_cb),
					    window);
		}

		gtk_widget_lock_accelerators (m);

		gtk_menu_append (menu, m);
		gtk_widget_show_all (m);

		if (lti->title) g_free (lti->title);
		g_free (lti);
		g_free (props);
	}

	g_list_free (items);
}
