/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "EventContext.h"
#include "nsIDOMEventTarget.h"
#include "nsIDocument.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDOMHTMLObjectElement.h"
#include "nsIInterfaceRequestor.h"
#include "nsIDOMHTMLImageElement.h"
#include "nsIDOMElement.h"
#include "nsIDOMXULDocument.h"
#include "nsIURI.h"
#include "nsIDOMNSDocument.h"
#include "nsReadableUtils.h"
#include "nsGUIEvent.h"
#include "nsIDOMNSEvent.h"
#include "nsIDOMCharacterData.h"
#include "nsIDOMHTMLButtonElement.h"
#include "nsIDOMHTMLLabelElement.h"
#include "nsIDOMHTMLLegendElement.h"
#include "nsIDOMHTMLTextAreaElement.h"

#include "misc_string.h"

EventContext::EventContext ()
{
}

EventContext::~EventContext ()
{
}

nsresult EventContext::Init (nsIDOMEvent *event, GaleonWrapper *wrapper)
{
	mEvent = event;
	mWrapper = wrapper;
	mDOMDocument = nsnull;

	return NS_OK;
}

#define MOZILLA_KEYS_LENGHT 16

static const struct 
{
	gint modifier;
	gint key;
}
mozilla_keys [] =
{
	{ CTRL_KEY, NS_VK_A + 32 },
	{ CTRL_KEY, NS_VK_E + 32 },
	{ CTRL_KEY, NS_VK_B + 32 },
	{ CTRL_KEY, NS_VK_F + 32 },
	{ CTRL_KEY, NS_VK_D + 32 },
	{ CTRL_KEY, NS_VK_H + 32 },
	{ CTRL_KEY, NS_VK_W + 32 },
	{ CTRL_KEY, NS_VK_U + 32 },
	{ CTRL_KEY, NS_VK_K + 32 },
	{ CTRL_KEY, NS_VK_N + 32 },
	{ CTRL_KEY, NS_VK_P + 32 },
	{ CTRL_KEY, NS_VK_V + 32 },
	{ CTRL_KEY + KEY_CODE, NS_VK_LEFT },
	{ CTRL_KEY + KEY_CODE, NS_VK_RIGHT },
	{ ALT_KEY + KEY_CODE, NS_VK_LEFT },
	{ ALT_KEY + KEY_CODE, NS_VK_RIGHT }
};

nsresult EventContext::GetKeyEventInfo (WrapperKeyEventInfo *info)
{
	PRBool mod_key;
	nsresult res;
	nsIDOMKeyEvent *keyEvent = (nsIDOMKeyEvent*)mEvent;

	nsCOMPtr<nsIDOMEventTarget>  targetNode;
	res = mEvent->GetTarget(getter_AddRefs(targetNode));
	if (NS_FAILED(res) || !targetNode) return NS_ERROR_FAILURE;
	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(targetNode);
	if (!node) return NS_ERROR_FAILURE;

	/* Get the context */
	res = GetEventContext (targetNode, &info->ctx);
	if (NS_FAILED(res)) return NS_ERROR_FAILURE;
	
	/* Get the key/modifier */
	info->modifier = 0;

	keyEvent->GetCharCode((PRUint32*)&info->key);

	if (info->key==0) 
	{
		keyEvent->GetKeyCode((PRUint32*)&info->key);
		info->modifier |= KEY_CODE;
	}

	keyEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	keyEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	keyEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	keyEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;

	/* Hack to make access keys working ... I hate this*/
	if (info->modifier & ALT_KEY)
	{
		nsAutoString accesskey;

		nsCOMPtr<nsIDOMHTMLAnchorElement> el1 = do_QueryInterface (node);
		if (el1) el1->GetAccessKey (accesskey);
		
		nsCOMPtr<nsIDOMHTMLAreaElement> el2 = do_QueryInterface (node);
		if (el2) el2->GetAccessKey (accesskey);

		nsCOMPtr<nsIDOMHTMLButtonElement> el3 = do_QueryInterface (node);
		if (el3) el3->GetAccessKey (accesskey);

		nsCOMPtr<nsIDOMHTMLInputElement> el4 = do_QueryInterface (node);
		if (el4) el4->GetAccessKey (accesskey);

		nsCOMPtr<nsIDOMHTMLLabelElement> el5 = do_QueryInterface (node);
		if (el5) el5->GetAccessKey (accesskey);

		nsCOMPtr<nsIDOMHTMLLegendElement> el6 = do_QueryInterface (node);
		if (el6) el6->GetAccessKey (accesskey);

		nsCOMPtr<nsIDOMHTMLTextAreaElement> el7 = do_QueryInterface (node);
		if (el7) el7->GetAccessKey (accesskey);

		gchar* akey;
		akey = ToNewCString (accesskey);

		if (akey[0] >=97 && akey[0] <=122)
		{
			if (akey[0] - 97 + 32 + NS_VK_A == info->key)
				return NS_ERROR_FAILURE;
		}
		else if (akey[0] >= 65 && akey[0] <= 90)
		{
			if (akey[0] - 65 + 32 + NS_VK_A == info->key)
				return NS_ERROR_FAILURE;
		}

		nsMemory::Free (akey);
	}

	if (info->ctx.context & CONTEXT_INPUT)
	{
		for (int i=0; i<MOZILLA_KEYS_LENGHT; i++)
		{
			if ((mozilla_keys[i].modifier == info->modifier) &&
			    (mozilla_keys[i].key == info->key))
			{
				return NS_ERROR_FAILURE;
			}
		}
	}

	return NS_OK;
}

nsresult EventContext::GetEventContext (nsIDOMEventTarget *EventTarget,
					WrapperContextInfo *info)
{
	nsresult rv;

	info->context = CONTEXT_DOCUMENT;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(EventTarget, &rv);
	if (NS_FAILED(rv) || !node) return NS_ERROR_FAILURE;

        /* Is page xul ? then do not display context menus
	 * FIXME I guess there is an easier way ... */
	/* From philipl: This test needs to be here otherwise we
	 * arrogantly assume we can QI to a HTMLElement, which is
	 * not true for xul content. */ 

	nsCOMPtr<nsIDOMDocument> domDoc;
	rv = node->GetOwnerDocument(getter_AddRefs(domDoc));
	if (NS_FAILED(rv) || !domDoc) return NS_ERROR_FAILURE;

	mDOMDocument = domDoc;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc, &rv);
	if (NS_FAILED(rv) || !doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMXULDocument> xul_document = do_QueryInterface(domDoc);
	if (xul_document)
	{
		info->context = CONTEXT_NONE;
		return NS_ERROR_FAILURE;
	}

	// Now we know that the page isn't a xul window, we can try and
	// do something useful with it.

	PRUint16 type;
	rv = node->GetNodeType(&type);
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMHTMLElement> element = do_QueryInterface(node);
	if ((nsIDOMNode::ELEMENT_NODE == type) && element)
	{
		nsAutoString tag;
		rv = element->GetTagName(tag);
		if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

		if (tag.EqualsWithConversion("img", PR_TRUE))
		{
			info->context |= CONTEXT_IMAGE;

			nsAutoString img;
			nsCOMPtr <nsIDOMHTMLImageElement> image = 
						do_QueryInterface(node, &rv);
			if (NS_FAILED(rv) || !image) return NS_ERROR_FAILURE;			

			rv = image->GetSrc (img);
			if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
			info->img = ToNewCString(img);

			rv = image->GetAlt (img);
			if (NS_SUCCEEDED(rv))
			{
				gchar *tmp = ToNewCString (img);
				info->imgalt = misc_string_strip_newline (tmp);
				nsMemory::Free (tmp);
			}

			rv = image->GetLongDesc (img);
			if (NS_SUCCEEDED(rv) && !img.IsEmpty())
			{
				nsCString src, imglongdesc;
				src.AssignWithConversion(img);

				nsCOMPtr<nsIURI> uri;
				doc->GetDocumentURL(getter_AddRefs(uri));

				rv = uri->Resolve (src, imglongdesc);

				info->imglongdesc = nsCRT::strdup(imglongdesc.get());
			}

			rv = image->GetWidth (&info->imgwidth);
			rv = image->GetHeight (&info->imgheight);

			rv = element->GetTitle (img);
                        if (NS_SUCCEEDED(rv))
			{
				gchar *tmp = ToNewCString (img);
                                info->imgtitle =
					misc_string_strip_newline (tmp);
				nsMemory::Free (tmp);
			}
		}
		else if (tag.EqualsWithConversion("input", PR_TRUE))
		{
			nsCOMPtr<nsIDOMElement> element;
			element = do_QueryInterface (node);
			if (!element) return NS_ERROR_FAILURE;

			nsAutoString attr;
			attr.AssignWithConversion("type");
			nsAutoString value;
			element->GetAttribute (attr, value);

			if (value.EqualsWithConversion("image", PR_TRUE))
			{
				info->context |= CONTEXT_IMAGE;
				nsCOMPtr<nsIDOMHTMLInputElement> input;
				input = do_QueryInterface (node);
				if (!input) return NS_ERROR_FAILURE;

				nsAutoString img;
				rv = input->GetSrc (img);
				if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

				nsCOMPtr<nsIURI> uri;
				doc->GetDocumentURL(getter_AddRefs(uri));

				nsCString src, cImg;
				src.AssignWithConversion(img);

				rv = uri->Resolve (src, cImg);
				info->img = nsCRT::strdup(cImg.get());

				if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
			}
			else if (!value.EqualsWithConversion("radio", PR_TRUE) &&
				 !value.EqualsWithConversion("submit", PR_TRUE) &&
				 !value.EqualsWithConversion("reset", PR_TRUE) &&
				 !value.EqualsWithConversion("hidden", PR_TRUE) &&
				 !value.EqualsWithConversion("button", PR_TRUE) &&
				 !value.EqualsWithConversion("checkbox", PR_TRUE))
			{
				info->context |= CONTEXT_INPUT;
			}
		}
		else if (tag.EqualsWithConversion("textarea", PR_TRUE))
		{
			info->context |= CONTEXT_INPUT;
		}
		else if (tag.EqualsWithConversion("object", PR_TRUE))
		{
			nsCOMPtr<nsIDOMHTMLObjectElement> object;
			object = do_QueryInterface (node);
			if (!element) return NS_ERROR_FAILURE;

			nsAutoString value;
			object->GetType(value);
			
			if (value.Find("image/") == 0)
			{
				info->context |= CONTEXT_IMAGE;
				
				nsAutoString img;
				
				rv = object->GetData (img);
				if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

				nsCOMPtr<nsIURI> uri;
				doc->GetDocumentURL(getter_AddRefs(uri));
				
				nsCString src, cImg;
				src.AssignWithConversion(img);

				rv = uri->Resolve (src, cImg);
				info->img = nsCRT::strdup(cImg.get());

				if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
			}
			else
			{
				info->context = CONTEXT_NONE;
				return NS_OK;
			}
		}
	}

	/* Is page framed ? */
	PRBool framed;
	IsPageFramed (node, &framed);
	info->framed_page = (gboolean)framed;

	/* Bubble out, looking for items of interest */
	while (node)
	{
		nsCOMPtr <nsIDOMElement> dom_elem = do_QueryInterface(node);
		if (dom_elem)
		{
			nsAutoString nspace;
			nsAutoString localname;
			nsAutoString value;
			nspace.AssignWithConversion ("http://www.w3.org/1999/xlink");
			localname.AssignWithConversion ("type");

			dom_elem->GetAttributeNS (nspace, localname, value);
			if (value.EqualsWithConversion("simple", PR_TRUE))
			{
				info->context |= CONTEXT_LINK;
				localname.AssignWithConversion ("href");
				dom_elem->GetAttributeNS (nspace, localname, value);
				
				info->link = ToNewCString(value);
			}
		}

		PRUint16 type;
		rv = node->GetNodeType(&type);
		if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

		element = do_QueryInterface(node);
		if ((nsIDOMNode::ELEMENT_NODE == type) && element)
		{
			nsAutoString tag;
			rv = element->GetTagName(tag);
			if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

			/* Link */
			if (tag.EqualsWithConversion("a", PR_TRUE))
			{
				nsCOMPtr <nsIDOMHTMLAnchorElement> anchor =
					do_QueryInterface(node);
				nsAutoString tmp;
				rv = anchor->GetHref (tmp);
				if (NS_FAILED(rv))
					return NS_ERROR_FAILURE;

				if (anchor && !tmp.IsEmpty()) 
				{
					info->context |= CONTEXT_LINK;

					info->link = ToNewCString(tmp);
					rv = anchor->GetHreflang (tmp);
					if (NS_SUCCEEDED(rv))
						info->linklang = ToNewCString(tmp);
					rv = anchor->GetTarget (tmp);
					if (NS_SUCCEEDED(rv))
						info->linktarget = ToNewCString(tmp);
					rv = anchor->GetType (tmp);
					if (NS_SUCCEEDED(rv))
						info->linktype = ToNewCString(tmp);
					rv = anchor->GetRel (tmp);
					if (NS_SUCCEEDED(rv))
						info->linkrel = ToNewCString(tmp);
					rv = anchor->GetRev (tmp);
					if (NS_SUCCEEDED(rv))
						info->linkrev = ToNewCString(tmp);
					rv = element->GetTitle (tmp);
		                        if (NS_SUCCEEDED(rv))
					{
                		                gchar *s = ToNewCString (tmp);;
						info->linktitle = misc_string_strip_newline (s);
						nsMemory::Free (s);
					}

					if (strcmp(info->linktype, 
						   "text/smartbookmark") == 0)
					{
						info->is_smart = TRUE;

						if (!info->img)
						{
							nsCOMPtr<nsIDOMNode> childNode;
							node->GetFirstChild (getter_AddRefs(childNode));

							if (childNode)
							{
								nsCOMPtr <nsIDOMHTMLImageElement> image = 
									do_QueryInterface(childNode, &rv);

								if (image)
								{
									nsAutoString img;
									rv = image->GetSrc (img);
									if (!NS_FAILED(rv))
									{
										info->img = ToNewCString(img);
									}
								}
							}
						}
					}

					/* Get the text of the link */
					info->linktext = mozilla_get_link_text (node);
				}
			
			}
			else if (tag.EqualsWithConversion("option", PR_TRUE))
			{
				info->context = CONTEXT_NONE;
				return NS_OK;
			}
			if (tag.EqualsWithConversion("area", PR_TRUE))
			{
				info->context |= CONTEXT_LINK;
				nsCOMPtr <nsIDOMHTMLAreaElement> area =
						do_QueryInterface(node, &rv);
				if (NS_SUCCEEDED(rv) && area)
				{
					nsAutoString href;
					rv = area->GetHref (href);
					if (NS_FAILED(rv))
						return NS_ERROR_FAILURE;
					
					info->link = ToNewCString(href);
				}
			}
			else if (tag.EqualsWithConversion("textarea", PR_TRUE))
			{
				info->context |= CONTEXT_INPUT;
			}

			if (info->bgimg == NULL)
			{
				nsCOMPtr<nsIDOMElement> domelement;
				domelement = do_QueryInterface (node);
				if (!domelement) return NS_ERROR_FAILURE;

				nsAutoString attr;
				attr.AssignWithConversion("background");
				nsAutoString value;
				domelement->GetAttribute (attr, value);
				
				if (!value.IsEmpty())
				{
					nsCString tmp, bgimg;
					tmp.AssignWithConversion(value);

					nsIURI *uri;
					doc->GetBaseURL(uri);
					rv = uri->Resolve 
						(tmp, bgimg);
					info->bgimg = nsCRT::strdup(bgimg.get());
					if (NS_FAILED (rv))
						return NS_ERROR_FAILURE;
				}
				else
				{
					info->bgimg = NULL;
				}
			}
			else if (info->bgimg == NULL)
			{
				nsCOMPtr<nsIDOMHTMLBodyElement> bgelement;
				bgelement = do_QueryInterface (node);
				if (bgelement)
				{
					nsAutoString value;
					bgelement->GetBackground (value);

					if (!value.IsEmpty())
					{
						nsCString tmp, bgimg;
						tmp.AssignWithConversion(value);

						nsIURI *uri;
						doc->GetBaseURL(uri);
						rv = uri->Resolve 
							(tmp, bgimg);
						info->bgimg = nsCRT::strdup(bgimg.get());
						if (NS_FAILED (rv))
							return NS_ERROR_FAILURE;
					}
					else
					{
						info->bgimg = NULL;
					}
				}
			}

			if (info->bgimg == NULL)
			{
				nsAutoString cssurl;
				rv = GetCSSBackground (node, cssurl);
				if (NS_SUCCEEDED (rv))
				{
					nsCString tmp, bgimg;
					tmp.AssignWithConversion(cssurl);

					nsIURI *uri;
					doc->GetBaseURL(uri);
					rv = uri->Resolve 
						(tmp, bgimg);
					info->bgimg = nsCRT::strdup(bgimg.get());
					if (NS_FAILED (rv))
						return NS_ERROR_FAILURE;
				}
			}
		}
		
		nsCOMPtr<nsIDOMNode> parentNode;
		node->GetParentNode (getter_AddRefs(parentNode));
		node = parentNode;
	}
	
	return NS_OK;
}

nsresult EventContext::GetCSSBackground (nsIDOMNode *node, nsAutoString& url)
{
	nsresult result;

	nsCOMPtr<nsIDOMElementCSSInlineStyle> style;
	style = do_QueryInterface (node);
	if (!style) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMCSSStyleDeclaration> decl;
	result = style->GetStyle (getter_AddRefs(decl));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsAutoString prop;
	prop.AssignWithConversion("background-image");
	nsAutoString value;
	decl->GetPropertyValue (prop, value);
	if (value.IsEmpty())
	{
		prop.AssignWithConversion("background");
		decl->GetPropertyValue (prop, value);
		if (value.IsEmpty())
		{
			prop.AssignWithConversion("background-repeat");
			decl->GetPropertyValue (prop, value);
			if (value.IsEmpty())
				return NS_ERROR_FAILURE;
		}
	}

	PRInt32 start, end;
	nsAutoString startsub;
	nsAutoString cssurl;
	startsub.AssignWithConversion("url(");
	nsAutoString endsub;
	endsub.AssignWithConversion(")");
	start = value.Find (startsub) + 4;
	end = value.Find (endsub);

	if (start == -1 || end == -1)
		return NS_ERROR_FAILURE;

	url = Substring (value, start, end - start);

	return NS_OK;
}

nsresult EventContext::GetMouseEventInfo (WrapperMouseEventInfo *info)
{
	nsresult result;
	DOMTimeStamp ts;
	nsIDOMMouseEvent *aMouseEvent = (nsIDOMMouseEvent*)mEvent;

	aMouseEvent->GetButton ((PRUint16*)&info->button);

	aMouseEvent->GetTimeStamp(&ts);
	info->timestamp = ts;

	/* be sure we are not clicking on the scroolbars */

	nsCOMPtr<nsIDOMNSEvent> nsEvent = do_QueryInterface(aMouseEvent, &result);
	if (NS_FAILED(result) || !nsEvent) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventTarget> OriginalTarget;
	result = nsEvent->GetOriginalTarget(getter_AddRefs(OriginalTarget));
	if (NS_FAILED(result) || !OriginalTarget) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNode> OriginalNode = do_QueryInterface(OriginalTarget);
	if (!OriginalNode) return NS_ERROR_FAILURE;

	nsString nodename;
	OriginalNode->GetNodeName(nodename);

	if (nodename.EqualsWithConversion ("xul:scrollbarbutton") ||
	    nodename.EqualsWithConversion ("xul:thumb") ||
	    nodename.EqualsWithConversion ("xul:vbox") ||
	    nodename.EqualsWithConversion ("xul:spacer") ||
	    nodename.EqualsWithConversion ("xul:slider"))
		return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventTarget> EventTarget;
	result = aMouseEvent->GetTarget(getter_AddRefs(EventTarget));
	if (NS_FAILED(result) || !EventTarget) return NS_ERROR_FAILURE;

	result = GetEventContext (EventTarget, &info->ctx);
	if (NS_FAILED(result)) return result;

	/* Get the modifier */

	PRBool mod_key;

	info->modifier = 0;

	aMouseEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	aMouseEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	aMouseEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	aMouseEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;

	return NS_OK;
}

nsresult EventContext::IsPageFramed (nsIDOMNode *node, PRBool *Framed)
{
	nsresult result;
	
	nsCOMPtr<nsIDOMDocument> mainDocument;
	result = mWrapper->GetMainDOMDocument (getter_AddRefs(mainDocument));
	if (NS_FAILED(result) || !mainDocument) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsIDOMDocument> nodeDocument;
	result = node->GetOwnerDocument (getter_AddRefs(nodeDocument));
	if (NS_FAILED(result) || !nodeDocument) return NS_ERROR_FAILURE;
 
	*Framed = (mainDocument != nodeDocument);

        return NS_OK;
}

nsresult EventContext::GetTargetDocument (nsIDOMDocument **domDoc)
{
	if (!mDOMDocument) return NS_ERROR_FAILURE;

	*domDoc = mDOMDocument.get();

	NS_IF_ADDREF(*domDoc);

	return NS_OK;
}
