/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

#include "ProgressListener.h"

#include "mozilla.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "glade.h"
#include "dialog.h"

#include <unistd.h>
#include <gtk/gtk.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkprogress.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmain.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-i18n.h>

#include "nsXPIDLString.h"
#include "nsIChannel.h"
#include "nsIFTPChannel.h"

extern "C" {
gint progress_dialog_cancel_cb (GtkButton *button,
				GIProgressCallbacks *aProgress);
void progress_dialog_pause_cb (GtkButton *button,
			       GIProgressCallbacks *aProgress);
}


static gboolean timeout_listener (GProgressListener *Progress);

NS_IMPL_ISUPPORTS3(GProgressListener, nsIWebProgressListener,
		   GIProgressCallbacks, nsISupportsWeakReference)

//------------------------------------------------------------------------------

GProgressListener::GProgressListener (void) : mLauncher(nsnull),
					      mPersist(nsnull),
					      mHandler(nsnull)
{
	NS_INIT_ISUPPORTS ();

	// Increment window count while progress is under way.
	window_count++;
}

GProgressListener::~GProgressListener()
{
	/* destructor code */
}

NS_METHOD GProgressListener::InitForPersist (nsIWebBrowserPersist *aPersist,
					       nsIDOMWindow *aParent, nsIURI *aURI,
					       nsIFile *aFile, DownloadAction aAction,
					       gpointer info,
					       PRInt64 aTimeDownloadStarted)
{
	nsresult rv;

	mInterval = 500000; //in microsecs == 500ms == 0.5s
	mPriorKRate = 0;
	mRateChanges = 0;
	mRateChangeLimit = 2; //Only update rate every second
	
	mPersist = aPersist;
	mParent = aParent;
	mUri = aURI;
	mFile = aFile;
	mAction = aAction;
        mCallbackInfo = info;
	mTimeDownloadStarted = aTimeDownloadStarted;

        if (mAction == ACTION_FAVICON || mAction == ACTION_FAVICON_EDITOR ||
	    mAction == ACTION_SET_PIXMAP)
        {
                mTimer = g_timer_new ();
                g_timer_start (mTimer);
                mTimeoutFunc = gtk_timeout_add
                        (10000, (GtkFunction) timeout_listener, this);
        }


	GtkWidget *aParentWidget = mozilla_find_gtk_parent (mParent);

	nsCOMPtr<nsIWebProgressListener> progress =
			NS_STATIC_CAST(nsIWebProgressListener *, this);
	nsCOMPtr<GIProgressCallbacks> callbacks =
			do_QueryInterface (progress);
	GladeXML *gxml = glade_widget_new ("galeon.glade", 
					   "download_progress_dialog",
					   &mProgressDialog, callbacks);

	mProgressBar = glade_xml_get_widget (gxml, "progressbar");
	mLocation = glade_xml_get_widget (gxml, "location_entry");
	mFileName = glade_xml_get_widget (gxml, "filename_entry");
	mStatus = glade_xml_get_widget (gxml, "status_entry");
	mTimeElapsed = glade_xml_get_widget (gxml, "elapsed_entry");
	mTimeRemaining = glade_xml_get_widget (gxml, "remaining_entry");
	mPauseButton = glade_xml_get_widget (gxml, "pause_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	mCheckedCanPause = PR_FALSE;
	mCanPause = PR_FALSE;
	mIsPaused = PR_FALSE;

	PRInt64 now = PR_Now ();
	mLastUpdate = now;
	if (mTimeDownloadStarted != 0)
		mStartTime = mTimeDownloadStarted;
	else
		mStartTime = now;
	mElapsed = now - mStartTime;

	nsCString spec;
	rv = mUri->GetSpec (spec);
	gtk_entry_set_text (GTK_ENTRY(mLocation),spec.get());
	
	char *text;
	rv = mFile->GetPath (&text);
	gtk_entry_set_text (GTK_ENTRY(mFileName),text);
	g_free (text);
	
	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));
	gtk_label_set_text (GTK_LABEL(mTimeRemaining),FormatTime(0));

	dialog_set_parent (mProgressDialog, aParentWidget);

	aPersist->SetProgressListener (this);

	return NS_OK;
}

NS_METHOD GProgressListener::InitForDownload(nsIHelperAppLauncher *aLauncher,
						nsISupports *aContext,
						GContentHandler *aHandler,
						DownloadAction aAction)
{
	nsresult rv;
	
	mInterval = 500000; //in microsecs == 500ms == 0.5s
	mPriorKRate = 0;
	mRateChanges = 0;
	mRateChangeLimit = 2; //Only update rate every second
	
	mLauncher = aLauncher;
	mParent = do_QueryInterface (aContext);
	mHandler = aHandler;
	mAction = aAction;

	rv = mLauncher->GetDownloadInfo(getter_AddRefs(mUri),
					&mTimeDownloadStarted,
					getter_AddRefs(mFile));

	GtkWidget *aParentWidget = mozilla_find_gtk_parent (mParent);

	nsCOMPtr<nsIWebProgressListener> progress =
			NS_STATIC_CAST(nsIWebProgressListener *, this);
	nsCOMPtr<GIProgressCallbacks> callbacks =
			do_QueryInterface (progress);
	GladeXML *gxml = glade_widget_new ("galeon.glade", 
					   "download_progress_dialog",
					   &mProgressDialog, callbacks);

	mProgressBar = glade_xml_get_widget (gxml, "progressbar");
	mLocation = glade_xml_get_widget (gxml, "location_entry");
	mFileName = glade_xml_get_widget (gxml, "filename_entry");
	mStatus = glade_xml_get_widget (gxml, "status_entry");
	mTimeElapsed = glade_xml_get_widget (gxml, "elapsed_entry");
	mTimeRemaining = glade_xml_get_widget (gxml, "remaining_entry");
	mPauseButton = glade_xml_get_widget (gxml, "pause_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	mCheckedCanPause = PR_FALSE;
	mCanPause = PR_FALSE;
	mIsPaused = PR_FALSE;

	PRInt64 now = PR_Now ();
	mLastUpdate = now;

//	mStartTime = mTimeDownloadStarted;
	mStartTime = now;

	mElapsed = now - mStartTime;

	nsCString spec;
	rv = mUri->GetSpec (spec);
	gtk_entry_set_text (GTK_ENTRY(mLocation),spec.get());
	
	char *text;
	rv = mFile->GetPath (&text);
	gtk_entry_set_text (GTK_ENTRY(mFileName),text);
	g_free (text);
	
	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));
	gtk_label_set_text (GTK_LABEL(mTimeRemaining),FormatTime(0));

	dialog_set_parent (mProgressDialog, aParentWidget);

	aLauncher->SetWebProgressListener (this);

	return NS_OK;
}

/* void onStateChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aStateFlags, in unsigned long aStatus); */
NS_IMETHODIMP GProgressListener::
			OnStateChange(nsIWebProgress *aWebProgress,
				      nsIRequest *aRequest,PRInt32 aStateFlags,
				      PRUint32 aStatus)
{
	if (aStateFlags & nsIWebProgressListener::STATE_STOP)
	{
		if (GTK_IS_WIDGET(mProgressDialog))
			gtk_widget_destroy (mProgressDialog);

		nsXPIDLCString filename;
		mFile->GetPath(getter_Copies(filename));

		switch (mAction)
		{
		case ACTION_VIEWSOURCE:
			misc_general_launch_external_viewer (filename.get());
			break;
		case ACTION_SETBACKGROUND:
			gchar *command;

			/* build command */
			command = g_strconcat ("background-properties-capplet "
					       "--init-session-settings --ignore "
					       "--background-image=",
					       filename.get(), NULL);

			/* execute it synchronously */
			gnome_execute_shell (NULL, command);

			/* free */
			g_free (command);
			break;
		case ACTION_SAVEFORHELPER:
			if (mLauncher)
				mLauncher->CloseProgressWindow ();
			if (mHandler)
				mHandler->LaunchHelperApp ();
			break;

		case ACTION_NONE:
			if (mLauncher)
				mLauncher->CloseProgressWindow ();
			break;

                case ACTION_FAVICON_EDITOR:
                case ACTION_FAVICON:
			if (mTimer) g_timer_destroy (mTimer);
			gtk_timeout_remove (mTimeoutFunc);
			favicon_download_completed
				((FaviconInfo *) mCallbackInfo);
			break;

		case ACTION_SET_PIXMAP:
			{
				if (mTimer) g_timer_destroy (mTimer);
				gtk_timeout_remove (mTimeoutFunc);

				SetPixmapInfo *i =
					(SetPixmapInfo *) mCallbackInfo;

				gchar *fn;
				mFile->GetPath(&fn);

				if (i->pixmap == NULL ||
				    strcmp (i->url, i->selected_url))
				{
					unlink (fn);
					g_free (fn);
					g_free (i);
					break;
				}

				/* build a pixmap and update the given
				 * widget */
				PixmapData *pix =
					misc_gui_pixmap_data_new_from_file (fn,
									FALSE);

				if (pix)
				{
					gtk_pixmap_set
						(GTK_PIXMAP (i->pixmap),
						 pix->pixmap, pix->mask);
					gtk_widget_show
						(GTK_WIDGET (i->pixmap));
					if (pix->pixmap)
						gdk_pixmap_unref (pix->pixmap);
					if (pix->mask)
						gdk_bitmap_unref (pix->mask);
					g_free (pix);
				}

				/* remove file since we dont need it
				 * anymore */
				unlink (fn);
				g_free (fn);
				g_free (i);
			}
			break;

		default:
			break;
		}

		// Decrement window count as this progess is finished.
		window_count--;

		// If there are no more windows left, exit galeon.
		if (g_list_length (all_windows) == 0 && !galeon_server_mode
		    && window_count ==0)
		{
			galeon_exit (TRUE, TRUE);
		}
	}

	return NS_OK;
}

/* void onProgressChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aCurSelfProgress, in long aMaxSelfProgress, in long aCurTotalProgress, in long aMaxTotalProgress); */
NS_IMETHODIMP GProgressListener::
			OnProgressChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest,
					 PRInt32 aCurSelfProgress,
					 PRInt32 aMaxSelfProgress,
					 PRInt32 aCurTotalProgress,
					 PRInt32 aMaxTotalProgress)
{
	char *text, *rateStr, *totalStr;

        if (mAction == ACTION_FAVICON || mAction == ACTION_FAVICON_EDITOR)
        {
                PRInt32 currentKBytes =
                        (PRInt32)(aCurTotalProgress / 1024.0 + 0.5);
                PRInt32 totalKBytes =
                        (PRInt32)(aMaxTotalProgress / 1024.0 + 0.5);
 
                if (currentKBytes >= 40 || totalKBytes >= 40)
                {
                        CancelHelperProgress ();
                }

                return NS_OK;
        }


	if (!mCheckedCanPause)
	{
		mCheckedCanPause = PR_TRUE;

		nsresult rv;
		nsCOMPtr<nsIFTPChannel> aChannel = do_QueryInterface (aRequest,
								      &rv);
		if (NS_SUCCEEDED(rv))
		{
			mCanPause = PR_TRUE;
			gtk_widget_show (mPauseButton);
		}
		else
		{
			mCanPause = PR_FALSE;
			gtk_widget_hide (mPauseButton);
		}
	}
	mRequest = aRequest;

	PRInt64 now = PR_Now ();

	if ((now - mLastUpdate < mInterval) && 
	     (aMaxTotalProgress != -1) &&  
	     (aCurTotalProgress < aMaxTotalProgress))
		return NS_OK;

	mLastUpdate = now;
	mElapsed = now - mStartTime;

	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));

	PRInt32 currentKBytes = (PRInt32)(aCurTotalProgress/1024 +.5);

	PRInt32 totalKBytes = (PRInt32)(aMaxTotalProgress/1024 +.5);
	if (aMaxTotalProgress > 0)
	{
		gfloat progress = (gfloat)aCurTotalProgress/
				  (gfloat)aMaxTotalProgress;
		gchar *strper = g_strdup_printf (_("%.2f%% - Downloading"), 
						 progress*100);

		if (progress > 0 && progress < 1)
			gtk_window_set_title (GTK_WINDOW(mProgressDialog), 
					      strper);

		if (progress >= 0 && progress <= 1.0)
		{
			gtk_progress_set_percentage (GTK_PROGRESS(mProgressBar),
						     progress);
		}
		else
		{
			gtk_progress_set_format_string
					(GTK_PROGRESS(mProgressBar), "?? %%");
		}
		totalStr = g_strdup_printf ("%d",totalKBytes);
		g_free (strper);
	}
	else
	{
		gtk_progress_set_format_string (GTK_PROGRESS(mProgressBar),
						"?? %%");
		totalStr = g_strdup ("??");
	}

	PRInt64 currentRate;
	if (mElapsed)
		currentRate = ((PRInt64)(aCurTotalProgress))*1000000 / mElapsed;
	else
		currentRate = 0;
		
	if (currentRate)
	{
		PRFloat64 currentKRate = ((PRFloat64)currentRate)/1024;
		if (currentKRate != mPriorKRate)
		{
			if (mRateChanges++ == mRateChangeLimit)
			{
				mPriorKRate = currentKRate;
				mRateChanges = 0;
			}
			else
			{
				currentKRate = mPriorKRate;
			}
		}
		else
		{
			mRateChanges = 0;
		}
		rateStr = g_strdup_printf("%.2f",currentKRate);
	}
	else
	{
		rateStr = g_strdup ("??.??");
	}

	if (mIsPaused)
	{
		text = g_strdup (_("Download Paused"));
	}
	else
	{
		text = g_strdup_printf (_("%dK of %sK bytes at %sK bytes/sec"),
					currentKBytes, totalStr, rateStr);
	}
	gtk_label_set_text (GTK_LABEL(mStatus),text);
	g_free (text);
	g_free (totalStr);
	g_free (rateStr);

	if (currentRate && (aMaxTotalProgress > 0))
	{
		PRInt32 remaining = 
			(PRInt32)((aMaxTotalProgress - aCurTotalProgress)
				   /currentRate +.5);
		gtk_label_set_text (GTK_LABEL(mTimeRemaining),
				    FormatTime(remaining));
	}
	else
	{
		gtk_label_set_text (GTK_LABEL(mTimeRemaining), _("Unknown"));
	}

	return NS_OK;
}

/* void onLocationChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsIURI location); */
NS_IMETHODIMP GProgressListener::
			OnLocationChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, nsIURI *location)
{
    return NS_OK;
}

/* void onStatusChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsresult aStatus, in wstring aMessage); */
NS_IMETHODIMP GProgressListener::
			OnStatusChange(nsIWebProgress *aWebProgress,
				       nsIRequest *aRequest, nsresult aStatus,
				       const PRUnichar *aMessage)
{
    return NS_OK;
}

/* void onSecurityChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long state); */
NS_IMETHODIMP GProgressListener::
			OnSecurityChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, PRInt32 state)
{
    return NS_OK;
}

NS_IMETHODIMP GProgressListener::CancelHelperProgress (void)
{
	mAction = ACTION_NONE;
	if (mIsPaused)
		TogglePause ();

	if (mPersist)
		return mPersist->CancelSave ();

	if (mLauncher)
		return mLauncher->Cancel ();

	return NS_ERROR_FAILURE;
}

NS_IMETHODIMP GProgressListener::TogglePause (void)
{
	if (!mCheckedCanPause) return NS_ERROR_FAILURE;

	nsresult rv;
	if (mIsPaused)
	{
		rv = mRequest->Resume ();
		if (NS_SUCCEEDED(rv))
		{
			gtk_label_set_text (GTK_LABEL(GTK_BIN(mPauseButton)
					    ->child),
					    _("Pause"));
		}
	}
	else
	{
		rv = mRequest->Suspend ();
		if (NS_SUCCEEDED(rv))
		{
			gtk_label_set_text (GTK_LABEL(mStatus),
					    _("Download Paused"));
			gtk_label_set_text (GTK_LABEL(GTK_BIN(mPauseButton)
					    ->child),
					    _("Resume"));
		}
	}
	mIsPaused = !mIsPaused;
	return rv;
}

char *GProgressListener::FormatTime (PRUint32 aTime)
{
	PRUint32 secs = (PRUint32)(aTime+.5);
	PRUint32 hours = secs/3600;
	secs -= hours*3600;
	PRUint32 mins = secs/60;
	secs -= mins*60;
	char *result;
	if (hours)
		result = g_strdup_printf ("%u:%02u.%02u",hours,mins,secs);
	else
		result = g_strdup_printf ("%02u.%02u",mins,secs);
	return result;
}

//////////////////////////////////////////////////////////////////////////////
// begin ProgressDialog callbacks.
//////////////////////////////////////////////////////////////////////////////

gint
progress_dialog_cancel_cb (GtkButton *button,
			   GIProgressCallbacks *aProgress)
{
	aProgress->CancelHelperProgress ();
	return 0; /* FIXME: philipl, is this the right thing to return? */
}

void progress_dialog_pause_cb (GtkButton *button,
			       GIProgressCallbacks *aProgress)
{
	aProgress->TogglePause ();
}


static gboolean
timeout_listener (GProgressListener *Progress)
{
        /* a timeout of 2 minutes */
        if (g_timer_elapsed (Progress->mTimer, NULL) >= 120)
        {   
                g_timer_destroy (Progress->mTimer);
                Progress->CancelHelperProgress ();
                return FALSE;               
        }
        
        return TRUE;
}
        
