/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "eel-gconf-extensions.h"
#include "prefs.h"
#include "prefs_mime.h"
#include "mime.h"
#include "misc_gui.h"

#include <gtk/gtkradiobutton.h>
#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <libgnome/gnome-i18n.h>

/* default MIME types */
extern struct
{
	gchar *name;
	MimeAction action;
} default_mimetypes[];

extern PreferencesDialog *pd;

/* Callbacks */
void
prefs_mime_new_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_mime_delete_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void prefs_mime_list_select_row_cb (GtkCList *clist, 
				    gint row, 
				    gint column, 
				    GdkEventButton *event, 
				    PreferencesDialog *pd);
void 
prefs_mime_list_unselect_row_cb (GtkCList *clist, gint row, gint column,
				 GdkEventButton *event, PreferencesDialog *pd);
void prefs_mime_list_sort_cb (GtkCList *clist, gint column,
			      PreferencesDialog *pd);
void
prefs_mime_type_changed_cb (GtkWidget *entry, PreferencesDialog *pd);
void
prefs_mime_action_clicked_cb (GtkWidget *button, PreferencesDialog *pd);
void
prefs_mime_helper_changed_cb (GtkWidget *entry, PreferencesDialog *pd);
void
prefs_mime_helper_buttons_toggled_cb (GtkWidget *button,
				      PreferencesDialog *pd);

/**
 * prefs_mime_set_dialog_to_config: update the prefs dialog to reflect the
 * current mime setup
 */
void
prefs_mime_set_dialog_to_config (void)
{
	gchar *text[5] = { NULL, NULL, NULL, NULL, NULL };
	GtkWidget *clist = pd->mime_list;
	gint row;
	GList *l;

	/* freeze and empty the list */
	gtk_clist_freeze (GTK_CLIST(clist));
	gtk_clist_clear (GTK_CLIST(clist));

	/* re-insert every element */
	for (l = mime_types; l != NULL; l = g_list_next(l))
	{
		MimeItem *m = (MimeItem *) (l->data);

		/* insert the row */
		text[0] = g_strdup (m->name);
		switch (m->action) {
		case MIME_SAVE_TO_DISK:
			text[1] = _("Save to disk");
			break;
		case MIME_RUN_PROGRAM:  
			text[1] = _("Run with Helper App");
			break;
		case MIME_UNKNOWN:
			text[1] = _("Unknown");
			break;
		case MIME_IGNORE:
			text[1] = _("Ignored");
			break;
		case MIME_ASK_ACTION:
			text[1] = _("Ask the user");
			break;
		}
		text[2] = g_strdup(m->helperApp);
		if (m->alwaysUse)
			text[3] = _("True");
		else
			text[3] = _("False");

		row = gtk_clist_append (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), row, m);

		/* gtk copies these and its safe to free them right after they 
		   are appended */
		g_free (text[0]);
		g_free (text[2]);
	}

	/* sort */
	gtk_clist_set_sort_column (GTK_CLIST(clist), 0);
	gtk_clist_set_sort_type (GTK_CLIST(clist), GTK_SORT_ASCENDING);
	gtk_clist_sort (GTK_CLIST(clist));

	/* thaw the list -- make changes visible */
	gtk_clist_thaw (GTK_CLIST(clist));	
}

/* new mime type button clicked */
void
prefs_mime_new_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	MimeItem *newitem = g_new0 (MimeItem, 1);
	GtkCList *clist = GTK_CLIST (pd->mime_list);
	gchar *text[5];
	gint row;

	newitem->name = g_strdup (_("New mime item"));
	newitem->action = MIME_UNKNOWN;
	newitem->helperApp = g_strdup (_("None"));
	newitem->terminalHelper = FALSE;
	newitem->urlHelper = FALSE;
	newitem->alwaysUse = FALSE;
	mime_types = g_list_append (mime_types, newitem);

	text[0] = newitem->name;
	text[1] = _("Unknown");
	text[2] = newitem->helperApp;
	text[3] = _("False");
	text[4] = NULL;
	row = gtk_clist_append (clist, text);
	gtk_clist_set_row_data (clist, row, newitem);
	gtk_clist_select_row (clist, row, 0);
	gtk_clist_moveto (clist, row, 0, 0, 0);
}

/* mime delete button clicked */
void
prefs_mime_delete_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkCList *clist = GTK_CLIST (pd->mime_list);
	gint row;
	
	if (pd->selected_mime == NULL)
	{
		return;
	}

	mime_item_free (pd->selected_mime);
	mime_types = g_list_remove (mime_types, pd->selected_mime);
	row = gtk_clist_find_row_from_data (clist, pd->selected_mime);
	pd->selected_mime = NULL;
	gtk_clist_remove (clist, row);

	if (row >= clist->rows) row = clist->rows - 1;
	gtk_clist_select_row (clist, row, 0);
	gtk_clist_moveto (clist, row, 0, 0, 0);
}

/* mime type selected */
void prefs_mime_list_select_row_cb (GtkCList *clist, 
				    gint row, 
				    gint column, 
				    GdkEventButton *event, 
				    PreferencesDialog *pd)
{
	gint act_num = 0;
	
	pd->enable_edit_mime = FALSE;
	pd->selected_mime = (MimeItem*) gtk_clist_get_row_data (clist, row);

	switch (pd->selected_mime->action)
	{
	case MIME_SAVE_TO_DISK:
		act_num = 0;	/* save */
		break;
	case MIME_RUN_PROGRAM:	/* run with helper app */
		act_num = 1;
		break;
	case MIME_ASK_ACTION:	/* Ask */
	case MIME_UNKNOWN:	/* default to ask */
		act_num = 2;
		break;
	case MIME_IGNORE:
		break;
	}

	gtk_widget_set_sensitive (pd->mime_action_frame, TRUE);
	gtk_widget_set_sensitive (pd->mime_delete_button, TRUE);
	gtk_widget_set_sensitive (pd->mime_type_label, TRUE);
	gtk_widget_set_sensitive (pd->mime_type_entry, TRUE);
	gtk_widget_set_sensitive (pd->mime_helper_frame, act_num);

	misc_gui_gtk_radio_button_set (GTK_RADIO_BUTTON (pd->mime_save_to_disk),
			      	       act_num);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_type_entry), 
			    pd->selected_mime->name);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_helper_entry),
			    pd->selected_mime->helperApp);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_always_use),
				      pd->selected_mime->alwaysUse);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_terminal),
				      pd->selected_mime->terminalHelper);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_url_helper),
				      pd->selected_mime->urlHelper);

	pd->enable_edit_mime = TRUE;
}

/* mime type unselected */
void 
prefs_mime_list_unselect_row_cb (GtkCList *clist, gint row, gint column,
				 GdkEventButton *event, PreferencesDialog *pd)
{
	pd->selected_mime = NULL;
	pd->enable_edit_mime = FALSE;

	/* The default is Ask */
	misc_gui_gtk_radio_button_set (GTK_RADIO_BUTTON (
						pd->mime_save_to_disk), 2);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_type_entry), "");
	gtk_entry_set_text (GTK_ENTRY (pd->mime_helper_entry), "");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_always_use),
				      FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_terminal),
				      FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_url_helper),
				      FALSE);
	gtk_widget_set_sensitive (pd->mime_action_frame, FALSE);
	gtk_widget_set_sensitive (pd->mime_helper_frame, FALSE);
	gtk_widget_set_sensitive (pd->mime_type_entry, FALSE);
	gtk_widget_set_sensitive (pd->mime_delete_button, FALSE);
	gtk_widget_set_sensitive (pd->mime_type_label, FALSE);
}

/* sort the mime listing */
void prefs_mime_list_sort_cb (GtkCList *clist, gint column,
			      PreferencesDialog *pd)
{
	if (column == pd->mime_last_sort_column)
		pd->mime_last_sort_type = !pd->mime_last_sort_type;
	else
		pd->mime_last_sort_type = GTK_SORT_ASCENDING;
	gtk_clist_set_sort_type (clist, pd->mime_last_sort_type);
	pd->mime_last_sort_column = column;
	gtk_clist_set_sort_column (clist, column);
	gtk_clist_sort (clist);
}

/* handles changes in the mime type entry */
void
prefs_mime_type_changed_cb (GtkWidget *entry, PreferencesDialog *pd)
{
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	if (pd->selected_mime->name)
		g_free (pd->selected_mime->name);
	pd->selected_mime->name = gtk_editable_get_chars 
		(GTK_EDITABLE (pd->mime_type_entry), 0, -1);

	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 0,
			    pd->selected_mime->name);
}

/* handles changes in the mime action */
void
prefs_mime_action_clicked_cb (GtkWidget *button, PreferencesDialog *pd)
{
	gint state;
	gchar *name;
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	state = misc_gui_gtk_radio_button_get (GTK_RADIO_BUTTON (
					       pd->mime_save_to_disk));
	switch (state)
	{
		case 0:
			pd->selected_mime->action = MIME_SAVE_TO_DISK;
			name = _("Save to disk");
			break;
		case 1:
			pd->selected_mime->action = MIME_RUN_PROGRAM;
			name = _("Run with Helper App");
			break;
		case 2:
			pd->selected_mime->action = MIME_ASK_ACTION;
			name = _("Ask the user");
			break;
		default:
			pd->selected_mime->action = MIME_UNKNOWN;
			name = _("Unknown");
			break;
	}

	/* update the clist */
	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 1, name);

	/* set the helper frame sensitive if the type is run or ask */
	gtk_widget_set_sensitive (pd->mime_helper_frame, state);
}

/* handles changes in the helper app entry */
void
prefs_mime_helper_changed_cb (GtkWidget *entry, PreferencesDialog *pd)
{
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	if (pd->selected_mime->helperApp)
	{
		g_free (pd->selected_mime->helperApp);
	}
	pd->selected_mime->helperApp = gtk_editable_get_chars 
		(GTK_EDITABLE (pd->mime_helper_entry), 0, -1);

	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 2,
			    pd->selected_mime->helperApp);
}

/* handles changes in the mime helper buttons' states */
void
prefs_mime_helper_buttons_toggled_cb (GtkWidget *button,
				      PreferencesDialog *pd)
{
	gchar *text;
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	pd->selected_mime->terminalHelper = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_terminal));
	pd->selected_mime->urlHelper = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_url_helper));
	pd->selected_mime->alwaysUse = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_always_use));

	if (pd->selected_mime->alwaysUse)
		text = _("True");
	else
		text = _("False");

	/* update clist */
	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 3, text);
}
