// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "mediasearch.h"
#include "mediaview.h"
#include "song.h"

#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qregexp.h>

static MediaSearch *INSTANCE = 0;


MediaSearch::MediaSearch(QWidget *parent, const char *name)
    : QWidget(parent, name), useregexp(true), currentmatch(0)
{
    searchcombo = new QComboBox(true, this, "search combobox");
    connect(searchcombo, SIGNAL(textChanged(const QString &)),
	    SLOT(search(const QString &)));
    connect(searchcombo, SIGNAL(activated(const QString &)),
	    SLOT(search(const QString &)));

    searchlabel = new QLabel(searchcombo, tr("&Search"), this);
    searchlabel->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Minimum));
    searchprev = new QPushButton(tr("&Prev"), this);
    searchprev->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
    searchnext = new QPushButton(tr("&Next"), this);
    searchnext->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));

    searchprev->setEnabled(false);
    searchnext->setEnabled(false);
    connect(searchprev, SIGNAL(clicked()), SLOT(searchPrev()));
    connect(searchnext, SIGNAL(clicked()), SLOT(searchNext()));

    QHBoxLayout *h1 = new QHBoxLayout(this, 0, 4, "search layout");
    h1->addWidget(searchlabel);
    h1->addWidget(searchcombo);
    h1->addWidget(searchprev);
    h1->addWidget(searchnext);

    INSTANCE = this;
}

MediaSearch::~MediaSearch()
{
    clear();
}

void MediaSearch::clear()
{
    if (! searchcombo)
	return;

    bool found = false;
    for (int i = 0; i < searchcombo->count(); i++) {
	if (searchcombo->text(i) == searchcombo->lineEdit()->text()) {
	    found = true;
	    break;
	}
    }
    if (! found)
	searchcombo->insertItem(searchcombo->lineEdit()->text());

    searchcombo->lineEdit()->clear();
    matchlist.clear();
    currentmatch = 0;
}

void MediaSearch::search(const QString &pattern)
{
    // clear old matches when starting a new search
    if (matchlist.count())
	matchlist.clear();
    currentmatch = 0;

    if (pattern.isNull() || pattern.isEmpty()) {
	searchnext->setEnabled(false);
	searchprev->setEnabled(false);
	return;
    }

    QRegExp r(pattern);

    QListViewItem *item = MediaView::instance()->firstChild();
    while (item) {
	if (item->rtti() == Song::RTTI) {
	    // this is a song
	    Song *s = (Song *) item;
	    if (useregexp) {
		if (s->title().contains(r) > 0)
		    matchlist.append(s);
	    } else {
		if (s->title().contains(pattern))
		    matchlist.append(s);
	    }
	}

	if (item->firstChild()) {
	    item = item->firstChild();
	} else if (item->nextSibling()) {
	    item = item->nextSibling();
	} else {
	    while (item && ! item->nextSibling())
		item = item->parent();

	    if (item)
		item = item->nextSibling();
	}
    }

    if (! matchlist.isEmpty()) {
	MediaView::instance()->setCurrentItem(matchlist[currentmatch]);
	MediaView::instance()->ensureItemVisible(matchlist[currentmatch]);
	searchnext->setEnabled(true);
	searchprev->setEnabled(true);
    } else {
	MediaView::instance()->clearSelection();
	searchnext->setEnabled(false);
	searchprev->setEnabled(false);
    }
}


void MediaSearch::searchNext()
{
    currentmatch = ++currentmatch % matchlist.count();
    MediaView::instance()->setCurrentItem(matchlist[currentmatch]);
    MediaView::instance()->ensureItemVisible(matchlist[currentmatch]);
}

void MediaSearch::searchPrev()
{
    if (--currentmatch < 0)
	currentmatch = matchlist.count() - 1;
    currentmatch %= matchlist.count();
    MediaView::instance()->setCurrentItem(matchlist[currentmatch]);
    MediaView::instance()->ensureItemVisible(matchlist[currentmatch]);
}

MediaSearch *MediaSearch::instance()
{
    return INSTANCE;
}
