// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "mediaview.h"
#include "collection.h"
#include "collectionview.h"
#include "constants.h"
#include "decoder.h"
#include "dragobjects.h"
#include "equalizer.h"
#include "group.h"
#include "groupedit.h"
#include "icons.h"
#include "organizer.h"
#include "song.h"
#include "songedit.h"

#include <qaction.h>
#include <qapplication.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qcursor.h>
#include <qdir.h>
#include <qdom.h>
#include <qfiledialog.h>
#include <qfileinfo.h>
#include <qgroupbox.h>
#include <qheader.h>
#include <qinputdialog.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qstatusbar.h>
#include <qtimer.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qurl.h>


class RealSongEditor : public SongEditor
{
    Q_OBJECT
public:
    RealSongEditor(Song *);


public slots:
    void browse();
};

RealSongEditor::RealSongEditor(Song *s)
    : SongEditor(Organizer::instance(), 0, true)
{
    if (Organizer::instance()->icon())
	setIcon(*Organizer::instance()->icon());

    nameLineEdit->setText(s->text(0));
    pathLineEdit->setText(s->location());
    enabledCheckBox->setChecked(! s->isOn());
    enabledCheckBox->setChecked(s->isOn());
    prioSpinBox->setValue(s->priority());

    QStringList presets(Equalizer::instance()->presets());
    if (! presets.contains(s->equalizer()))
	s->setEqualizer("Default");
    eqComboBox->insertStringList(presets);
    eqComboBox->setCurrentText(s->equalizer());

    connect(browseButton, SIGNAL(clicked()), SLOT(browse()));
}

void RealSongEditor::browse()
{
    QUrl url(pathLineEdit->text());
    QFileInfo fi(url.toString(false, false));

    QString filter = Decoder::filter();
    QString filename = QFileDialog::getOpenFileName(fi.dirPath(),
						    filter,
						    this);

    if (filename.isNull())
	return;

    pathLineEdit->setText(filename);
}

static MediaView *INSTANCE = 0;


MediaView::MediaView(QWidget *parent, const char *name)
    : QListView(parent, name), oldcurrent(0), dropitem(0), dragitem(0),
      lengthTimer( 0 ), pressed(false), supportedgroup(false), supportedsong(false),
      dragging(false), modified(false)
{
    setShowSortIndicator(false);
    setMouseTracking(false);
    viewport()->setMouseTracking(false);
    setAllColumnsShowFocus(true);
    setRootIsDecorated(true);
    setAcceptDrops(true);
    viewport()->setAcceptDrops(true);
    addColumn(tr("Song ti1tle"));
    addColumn(tr("Length"));
    addColumn(tr("Priority"));
    addColumn(tr("Equalizer"));

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), SLOT(timeout()));

    connect(this, SIGNAL(doubleClicked(QListViewItem *)),
	    SLOT(itemSelected(QListViewItem *)));
    connect(this, SIGNAL(returnPressed(QListViewItem *)),
	    SLOT(itemSelected(QListViewItem *)));
    connect(this, SIGNAL(currentChanged(QListViewItem *)),
	    SLOT(itemHighlighted(QListViewItem *)));
    connect(this, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
	    SLOT(popupContextMenu(QListViewItem *, const QPoint &, int)));

    INSTANCE = this;
}

MediaView::~MediaView()
{
}

void MediaView::setupActions()
{
    actionAddSong = new QAction(tr("Add Song"),
				tr("Add &Song"),
				0,
				this);
    actionAddSong->setStatusTip(tr("Add a song to the current group."));
    actionAddSong->
	setWhatsThis(tr("<b>Add a song to the current group.</b>"
			"<p>Use the file dialog to select the file you want "
			"to add.  The selected song is added to the selected "
			"group.</p>"));
    connect(actionAddSong, SIGNAL(activated()), this, SLOT(addSong()));

    actionAddGroup = new QAction(tr("Add Group"),
				 tr("Add &Group"),
				 0,
				 this);
    actionAddGroup->setStatusTip(tr("Add a group to the current group."));
    actionAddGroup->
	setWhatsThis(tr("<b>Add a group to the current group.</b>"
			"<p>Use the input dialog to input the name you "
			"want to give the new group.  The group is added "
			"as a sub-group to the selected group.  If there "
			"is no selected group, the new group is added "
			"without a parent.</p>"));
    connect(actionAddGroup, SIGNAL(activated()), this, SLOT(addGroup()));

    actionAddRootGroup = new QAction(tr("Add Root Group"),
				     tr("Add &Root Group"),
				     0,
				     this);
    actionAddRootGroup->setStatusTip(tr("Add a group with no parent."));
    actionAddRootGroup->
	setWhatsThis(tr("<b>Add a group with no parent.</b>"
			"<p>Use the input dialog to input the name you "
			"want to give the new group.  The group is added "
			"without a parent.</p>"));
    connect(actionAddRootGroup, SIGNAL(activated()), this, SLOT(addRootGroup()));

    actionBldFromDir = new QAction(tr("Build from Directory"),
				   tr("Build from Directory"),
				   0,
				   this);
    actionBldFromDir->setStatusTip(tr("Build your Media from a Directory."));
    actionBldFromDir->
	setWhatsThis(tr("<b>Build your Media from a Directory.</b>"
			"<p>Use the file dialog to select the "
			"directory holding the files you wish to "
			"add.  All files in the selected directory "
			"and its sub-directories will be added to "
			"your Media.  This is the fastest way to load "
			"media into MQ3.</p>"));
    connect(actionBldFromDir, SIGNAL(activated()), SLOT(buildFromDir()));

    actionRemSong = new QAction(tr("Remove Song"),
				tr("&Remove Song"),
				0,
				this);
    actionRemSong->setStatusTip(tr("Remove the selected song."));
    actionRemSong->
	setWhatsThis(tr("<b>Remove the selected song.</b>"
			"<p>Removes the selected song from the tree.  No "
			"files are deleted from the harddisk by this "
			"operation.</p>"));
    connect(actionRemSong, SIGNAL(activated()), this, SLOT(remSong()));

    actionRemGroup = new QAction(tr("Remove Group"),
				 tr("Re&move Group"),
				 0,
				 this);
    actionRemGroup->setStatusTip(tr("Remove the selected group."));
    actionRemGroup->
	setWhatsThis(tr("<b>Remove the selected group.</b>"
			"<p>Removes the selected group (and all the songs "
			"in the group) from the tree.  No files are "
			"deleted from the harddisk by this operation.</p>"));
    connect(actionRemGroup, SIGNAL(activated()), this, SLOT(remGroup()));

    actionRemAll = new QAction(tr("Remove all Media"),
			       tr("Remove all Media"),
			       0,
			       this);
    actionRemAll->setStatusTip(tr("Remove all Media."));
    actionRemAll->
	setWhatsThis(tr("<b>Remove all Media.</b>"
			"<p>Removes all your Media from the MediaView."
			"No files are deleted from the harddisk "
			"by this operation.</p>"));
    connect(actionRemAll, SIGNAL(activated()), this, SLOT(remAll()));

    actionEditSong = new QAction(tr("Edit Song"),
				 tr("&Edit Song"),
				 0,
				 this);
    actionEditSong->setStatusTip(tr("Edit properties of the selected song."));
    actionEditSong->
	setWhatsThis(tr("<b>Edit properties of the selected song.</b>"
			"<p>Use the presented dialog to change various "
			"properties of the selected song.</p>"));
    connect(actionEditSong, SIGNAL(activated()), this, SLOT(editSong()));

    actionEditGroup = new QAction(tr("Edit Group"),
				  tr("E&dit Group"),
				  0,
				  this);
    actionEditGroup->setStatusTip(tr("Edit properties of the selected group."));
    actionEditGroup->
	setWhatsThis(tr("<b>Edit properties of the selected group.</b>"
			"<p>Use the presented dialog to change various "
			"properties of the selected group.</p>"));
    connect(actionEditGroup, SIGNAL(activated()), this, SLOT(editGroup()));


    actionGroupEnableAll = new QAction(tr("Enable all"),
				       tr("Enable all"),
				       0,
				       this);
    actionGroupEnableAll->setStatusTip(tr("Enable all songs in the selected group."));
    connect(actionGroupEnableAll, SIGNAL(activated()), SLOT(enableAll()));

    actionGroupDisableAll = new QAction(tr("Disable all"),
					tr("Disable all"),
					0,
					this);
    actionGroupDisableAll->setStatusTip(tr("Disable all songs in the selected group."));
    connect(actionGroupDisableAll, SIGNAL(activated()), SLOT(disableAll()));

    actionMediaEnableAll = new QAction(tr("Enable all"),
				       tr("Enable all"),
				       0,
				       this);
    actionMediaEnableAll->setStatusTip(tr("Enable all songs in your Media."));
    connect(actionMediaEnableAll, SIGNAL(activated()), SLOT(enableAll()));

    actionMediaDisableAll = new QAction(tr("Disable all"),
					tr("Disable all"),
					0,
					this);
    actionMediaDisableAll->setStatusTip(tr("Disable all songs in your Media."));
    connect(actionMediaDisableAll, SIGNAL(activated()), SLOT(disableAll()));

    actionIncPrio = new QAction(tr("+ 1"),
				tr("+ 1"),
				0,
				this);
    actionIncPrio->setStatusTip(tr("Priority + 1"));
    connect(actionIncPrio, SIGNAL(activated()), SLOT(incPrio()));

    actionDecPrio = new QAction(tr("- 1"),
				tr("- 1"),
				0,
				this);
    actionDecPrio->setStatusTip(tr("Priority - 1"));
    connect(actionDecPrio, SIGNAL(activated()), SLOT(decPrio()));

    actionIncPrio5 = new QAction(tr("+ 5"),
				 tr("+ 5"),
				 0,
				 this);
    actionIncPrio5->setStatusTip(tr("Priority + 5"));
    connect(actionIncPrio5, SIGNAL(activated()), SLOT(incPrio5()));

    actionDecPrio5 = new QAction(tr("- 5"),
				 tr("- 5"),
				 0,
				 this);
    actionDecPrio5->setStatusTip(tr("Priority - 5"));
    connect(actionDecPrio5, SIGNAL(activated()), SLOT(decPrio5()));

    itemHighlighted(0);
}

void MediaView::setupMenus()
{
    mainmenu = new QPopupMenu(this);
    priomenu = new QPopupMenu(this);
    songmenu = new QPopupMenu(this);
    groupmenu = new QPopupMenu(this);

    // media menu (menubar)
    actionAddSong->addTo(mainmenu);
    actionEditSong->addTo(mainmenu);
    actionRemSong->addTo(mainmenu);
    mainmenu->insertSeparator();
    actionAddGroup->addTo(mainmenu);
    actionAddRootGroup->addTo(mainmenu);
    actionEditGroup->addTo(mainmenu);
    actionRemGroup->addTo(mainmenu);
    mainmenu->insertSeparator();
    actionBldFromDir->addTo(mainmenu);
    actionRemAll->addTo(mainmenu);

    // priority menu
    actionIncPrio->addTo(priomenu);
    actionDecPrio->addTo(priomenu);
    actionIncPrio5->addTo(priomenu);
    actionDecPrio5->addTo(priomenu);

    // song context menu
    actionEditSong->addTo(songmenu);
    actionRemSong->addTo(songmenu);
    songmenu->insertSeparator();
    songmenu->insertItem(tr("Priority"), priomenu);

    // group context menu
    actionAddSong->addTo(groupmenu);
    groupmenu->insertSeparator();
    actionAddGroup->addTo(groupmenu);
    actionAddRootGroup->addTo(groupmenu);
    actionRemGroup->addTo(groupmenu);
    actionEditGroup->addTo(groupmenu);
    groupmenu->insertSeparator();
    actionGroupEnableAll->addTo(groupmenu);
    actionGroupDisableAll->addTo(groupmenu);
    groupmenu->insertSeparator();
    groupmenu->insertItem(tr("Priority"), priomenu);
}

void MediaView::setupToolbars()
{
    QToolBar *toolbar = new QToolBar(tr("Media"), Organizer::instance());

    // media
    actionAddSong->addTo(toolbar);
    actionEditSong->addTo(toolbar);
    actionRemSong->addTo(toolbar);
    toolbar->addSeparator();
    actionAddGroup->addTo(toolbar);
    actionAddRootGroup->addTo(toolbar);
    actionEditGroup->addTo(toolbar);
    actionRemGroup->addTo(toolbar);
    toolbar->addSeparator();
    actionBldFromDir->addTo(toolbar);
    actionRemAll->addTo(toolbar);
}

void MediaView::loadIcons()
{
    setColumnText(0, IconLoader::load("media"), tr("Media"));
    actionAddSong->setIconSet(IconLoader::load("addsong"));
    actionAddGroup->setIconSet(IconLoader::load("addgroup"));
    actionAddRootGroup->setIconSet(IconLoader::load("addrootgroup"));
    actionBldFromDir->setIconSet(IconLoader::load("buildfromdir"));
    actionRemSong->setIconSet(IconLoader::load("removesong"));
    actionRemGroup->setIconSet(IconLoader::load("removegroup"));
    actionRemAll->setIconSet(IconLoader::load("removemedia"));
    actionEditSong->setIconSet(IconLoader::load("editsong"));
    actionEditGroup->setIconSet(IconLoader::load("editgroup"));
    iconset = IconLoader::load("song");
}

void MediaView::load()
{
    bool mod = modified;

    QString filename = QDir::homeDirPath() + "/.mq3/playtree";
    QFile file(filename);
    if (! file.open(IO_ReadOnly))
	return;

    setUpdatesEnabled(false);

    QDomDocument doc;

    if (doc.setContent(&file) &&
	doc.firstChild().toElement().tagName() == "playtree") {
	QDomElement e = doc.firstChild().toElement().firstChild().toElement();

	while (! e.isNull()) {
	    (void) Group::create(0, e);

	    e = e.nextSibling().toElement();
	}
    }

    file.close();

    setUpdatesEnabled(true);

    setModified(mod);

    updateSongLengths();
}

void MediaView::save()
{
    if (! modified)
	return;

    Organizer::instance()->statusBar()->message(tr("Saving media changes..."));
    QApplication::setOverrideCursor(QCursor(WaitCursor));

    qApp->processEvents(0);

    QFileInfo fileinfo(QDir::homeDirPath() + "/.mq3");
    if (! fileinfo.exists()) {
	QDir::home().mkdir(".mq3");
    }

    QFile file(QDir::homeDirPath() + "/.mq3/playtree");
    if (! file.open(IO_WriteOnly)) {
	qWarning("MediaView: error opening output file");
	return;
    }

    QTextStream ts(&file);
    ts.setEncoding(QTextStream::UnicodeUTF8);
    xmlDescription(ts);
    file.close();

    setModified(false);

    QApplication::restoreOverrideCursor();
    Organizer::instance()->statusBar()->message(tr("Saved media changes."), 2000);
}

void MediaView::xmlDescription(QTextStream &stream) const
{
    stream << "<!DOCTYPE MQ3-Playtree>" << endl << "<playtree>" << endl;

    QListViewItem *c = firstChild();
    while (c) {
	if (c->rtti() == Group::RTTI)
	    ((Group *) c)->xmlDescription(stream);
	else if (c->rtti() == Song::RTTI)
	    ((Song *) c)->xmlDescription(stream);

	c = c->nextSibling();
    }

    stream << "</playtree>" << endl;
}

void MediaView::contentsMousePressEvent(QMouseEvent *e)
{
    QListView::contentsMousePressEvent(e);
    QPoint p(contentsToViewport(e->pos()));
    QListViewItem *item = itemAt(p);

    if (! item)
	return;

    if (p.x() > (header()->cellPos(header()->mapToActual(0)) +
		 treeStepSize() * (item->depth() + 1) + itemMargin()) ||
	p.x() < header()->cellPos(header()->mapToActual(0))) {
	pressed = true;
	presspos = e->pos();
    }
}

void MediaView::contentsMouseReleaseEvent(QMouseEvent *e)
{
    pressed = false;
    QListView::contentsMouseReleaseEvent(e);
}

void MediaView::contentsMouseMoveEvent(QMouseEvent *e)
{
    if (! pressed)
	return;

    if ((presspos - e->pos()).manhattanLength() > QApplication::startDragDistance()) {
	pressed = false;

	QListViewItem *item = itemAt(contentsToViewport(presspos));
	if (! item)
	    return;

	if (item->rtti() == Group::RTTI) {
	    // drag group
	    GroupDrag *gd = new GroupDrag((Group *) item, viewport());
	    gd->setPixmap(IconLoader::load("addgroup").
			  pixmap(QIconSet::Small, QIconSet::Normal));

	    dragitem = item;
	    dragging = true;
	    if (gd->drag() && gd->target() != this) {
		// we should delete the original
		delete dragitem;
	    }
	    dragging = false;
	    dragitem = 0;
	} else if (item->rtti() == Song::RTTI) {
	    // drag song
	    SongDrag *sd = new SongDrag((Song*) item, viewport());
	    sd->setPixmap(IconLoader::load("addsong").
			  pixmap(QIconSet::Small, QIconSet::Normal));

	    dragitem = item;
	    dragging = true;
	    if (sd->drag() && sd->target() != this) {
		// we should delete the original
		delete dragitem;
	    }
	    dragging = false;
	    dragitem = 0;
	}
    }
}

void MediaView::contentsDragEnterEvent(QDragEnterEvent *e)
{
    if (SongDrag::canDecode(e))
	supportedsong = true;
    else if (GroupDrag::canDecode(e))
	supportedgroup = true;
    else if (QUriDrag::canDecode(e)) {
	supporteduri = true;
	QStringList l;
	QUriDrag::decodeToUnicodeUris(e, l);

	QStringList::Iterator it = l.begin();
	while (it != l.end()) {
	    if (! Decoder::supports(*it++)) {
		supporteduri = false;
		break;
	    }
	}
    }

    if (! supportedsong && ! supportedgroup && ! supporteduri) {
        e->ignore();
	return;
    }

    oldcurrent = currentItem();

    QListViewItem *item = itemAt(contentsToViewport(e->pos()));

    if (item &&
	item->rtti() != Group::RTTI &&
	item->rtti() != Song::RTTI) {
	e->ignore();
	return;
    }

    dropitem = item;

    // if dragging a group, make sure that we are not trying to move a parent into
    // a child
    if (supportedgroup && dragging && dragitem) {
	if (dragitem == dropitem || isBChildOfA(dragitem, dropitem)) {
	    e->ignore();
	    return;
	}
    }

    timer->start(groupOpenTimeout, true);

    if (! supporteduri && (e->action() == QDropEvent::Copy ||
			   e->action() == QDropEvent::Move))
	e->acceptAction();
}

void MediaView::contentsDragMoveEvent(QDragMoveEvent *e)
{
    if (! supportedsong && ! supportedgroup && ! supporteduri) {
	e->ignore();
	return;
    }

    QListViewItem *item = itemAt(contentsToViewport(e->pos()));
    timer->stop();

    if (item &&
	item->rtti() != Group::RTTI &&
	item->rtti() != Song::RTTI) {
	e->ignore();
	return;
    }

    dropitem = item;
    if (dropitem)
	timer->start(groupOpenTimeout, true);

    setSelected(dropitem, true);

    if (supportedgroup && dragging && dragitem) {
	// if dragging a group, make sure that we are not trying to move a parent into
	// a child
	if (dragitem == dropitem || isBChildOfA(dragitem, dropitem)) {
	    e->ignore();
	    return;
	}
    }

    e->accept();
    if (! supporteduri && (e->action() == QDropEvent::Copy ||
			   e->action() == QDropEvent::Move))
	e->acceptAction();
}

void MediaView::contentsDragLeaveEvent(QDragLeaveEvent *)
{
    supportedsong = false;
    supportedgroup = false;

    timer->stop();
    dropitem = 0;

    setCurrentItem(oldcurrent);
    setSelected(oldcurrent, true);
}

void MediaView::contentsDropEvent(QDropEvent *e)
{
    if (! supportedsong && ! supportedgroup && ! supporteduri) {
	e->ignore();
	return;
    }

    QListViewItem *item = itemAt(contentsToViewport(e->pos()));

    if (item &&
	item->rtti() != Group::RTTI &&
	item->rtti() != Song::RTTI) {
	e->ignore();
	return;
    }

    dropitem = item;

    if (supportedgroup && e->action() == QDropEvent::Move &&
	dragging && dragitem) {
	// if dragging a group, make sure that we are not trying to move a parent into
	// a child
	if (dragitem == dropitem || isBChildOfA(dragitem, dropitem)) {
	    e->ignore();
	    return;
	}
    }

    Group *dropgroup = 0;
    while (dropitem && dropitem->rtti() != Group::RTTI)
	dropitem = dropitem->parent();

    if (dropitem && dropitem->rtti() == Group::RTTI)
	dropgroup = (Group *) dropitem;

    if (supporteduri) {
	QStringList l;
	if (QUriDrag::decodeToUnicodeUris(e, l)) {
	    QStringList::Iterator it = l.begin();

	    while (it != l.end()) {
		QUrl url(*it++);
		if (url.isLocalFile() || url.protocol() == "mqp")
		    (void) new Song(dropgroup, url, url, "Default");
	    }
	}
    } else if (supportedsong) {
	SongDrag::decode(e, dropgroup);
    } else if (supportedgroup) {
	GroupDrag::decode(e, dropgroup);
    }

    if (dropgroup)
	dropgroup->setOpen(true);
    timer->stop();
    e->accept();

    if (! supporteduri && (e->action() == QDropEvent::Copy ||
			   e->action() == QDropEvent::Move)) {
	e->acceptAction();

	if (dragging && dragitem && e->action() == QDropEvent::Move) {
	    // delete the local copy
	    delete dragitem;
	    dragitem = 0;
	}
    }
}

void MediaView::timeout()
{
    if (dropitem && ! dropitem->isOpen()) {
	dropitem->setOpen(true);
	dropitem->repaint();
    }
}

bool MediaView::isBChildOfA(QListViewItem *A, QListViewItem *B) const
{
    QListViewItem *item = A->firstChild();
    while (item) {
	if (item == B)
	    return true;
	else if (isBChildOfA(item, B))
	    return true;

	item = item->nextSibling();
    }

    return false;
}

void MediaView::popupContextMenu(QListViewItem *item, const QPoint &point, int)
{
    if (! item)
	return;

    if (item->rtti() == Group::RTTI)
	groupmenu->popup(point);
    else if (item->rtti() == Song::RTTI)
	songmenu->popup(point);
}



void MediaView::itemHighlighted(QListViewItem *item)
{
    if (! item) {
	actionAddSong->setEnabled(false);
	actionAddGroup->setEnabled(true);
	actionAddRootGroup->setEnabled(true);
	actionBldFromDir->setEnabled(true);

	actionRemSong->setEnabled(false);
	actionRemGroup->setEnabled(false);
	actionRemAll->setEnabled(false);

	actionEditSong->setEnabled(false);
	actionEditGroup->setEnabled(false);
    } else if (item->rtti() == Group::RTTI) {
	// group
	actionAddSong->setEnabled(true);
	actionAddGroup->setEnabled(true);
	actionAddRootGroup->setEnabled(true);
	actionBldFromDir->setEnabled(false);

	actionRemSong->setEnabled(false);
	actionRemGroup->setEnabled(true);
	actionRemAll->setEnabled((item->parent() == 0));

	actionEditSong->setEnabled(false);
	actionEditGroup->setEnabled(true);
    } else if (item->rtti() == Song::RTTI) {
	// song
	actionAddSong->setEnabled(true);
	actionAddGroup->setEnabled(false);
	actionAddRootGroup->setEnabled(false);
	actionBldFromDir->setEnabled(false);

	actionRemSong->setEnabled(true);
	actionRemGroup->setEnabled(false);
	actionRemAll->setEnabled(false);

	actionEditSong->setEnabled(true);
	actionEditGroup->setEnabled(false);
    } else {
	// something we don't understand - shouldn't happen
	actionAddSong->setEnabled(false);
	actionAddGroup->setEnabled(true);
	actionAddRootGroup->setEnabled(true);
	actionBldFromDir->setEnabled(true);

	actionRemSong->setEnabled(false);
	actionRemGroup->setEnabled(false);
	actionRemAll->setEnabled(false);

	actionEditSong->setEnabled(false);
	actionEditGroup->setEnabled(false);
    }
}

void MediaView::itemSelected(QListViewItem *item)
{
    if (! item || item->rtti() != Song::RTTI)
	return;

    Collection *c = CollectionView::instance()->currentCollection();
    if (! c)
	return;

    if (c->currentSong && c->currentSong->isOn()) {
	c->history.push(c->currentSong);

	while (c->history.count() > historySize)
	    c->history.remove(c->history.begin());
    }

    Song *song = (Song *) item;
    c->setCurrentSong(song);

    emit songSelected();
}

void MediaView::addSong()
{
    QListViewItem *item = currentItem();

    Group *parent = 0;
    if (item) {
	if (item->rtti() == Group::RTTI)
	    parent = (Group *) item;
	else if (item->rtti() == Song::RTTI) {
	    QListViewItem *item2 = item->parent();
	    if (item2->rtti() == Group::RTTI)
		parent = (Group *) item2;
	}
    }
    if (! parent)
	return;

    QString filter = Decoder::filter();
    QStringList filenames = QFileDialog::getOpenFileNames(filter,
							  QString::null,
							  this);

    QStringList::Iterator it = filenames.begin();
    Song *song;
    while (it != filenames.end()) {
	QUrl url(*it++);
	song = new Song(parent, url.fileName(), url.toString(), "Default");
	lengthlist.append(song);
    }

    setModified(true);
    updateSongLengths();
}

void MediaView::addGroup()
{
    QListViewItem *item = currentItem();

    Group *parent = 0;
    if (item && item->rtti() == Group::RTTI)
	parent = (Group *) item;

    QString name = QInputDialog::getText(tr("Add Group"),
					 tr("Enter the name of the new Group:"),
					 QLineEdit::Normal, QString::null,
					 0, this);

    if (name.isNull() || name.isEmpty())
	return;

    if (parent)
	(void) new Group(parent, name);
    else
	(void) new Group(name);

    setModified(true);
}

void MediaView::addRootGroup()
{
    QString name = QInputDialog::getText(tr("Add Root Group"),
					 tr("Enter the name of the new Group:"),
					 QLineEdit::Normal, QString::null,
					 0, this);

    if (name.isNull() || name.isEmpty())
	return;
    (void) new Group(name);
    setModified(true);
}

void MediaView::buildFromDir()
{
    QString dirname = QFileDialog::getExistingDirectory(QString::null, this, 0,
							tr("Add a Collection"));

    if (dirname.isNull())
	return;

    QFileInfo fi(dirname);
    if (fi.isDir() && fi.isReadable()) {
	QDir dir(fi.absFilePath());

	setUpdatesEnabled(false);

	const QFileInfoList *files = dir.entryInfoList();

	if (files) {
	    QFileInfoListIterator it(*files);
	    QFileInfo *info;

	    while ((info = it.current()) != 0) {
		++it;

		if (info->fileName() == "." || info->fileName() == "..") {
		    ;
		} else if (info->isDir()) {
		    Group *g = new Group(info->fileName());
		    g->propogate(info->absFilePath());
		}
	    }
	}

	setUpdatesEnabled(true);
    }

    setModified(true);

    // see which songs need length calculations
    QListViewItem *item = firstChild();
    while (item) {
	if (item->rtti() == Song::RTTI) {
	    // this is a song
	    Song *song = (Song *) item;
	    if (song->length() == -1.0)
		lengthlist.append(song);
	}

	if (item->firstChild()) {
	    item = item->firstChild();
	} else if (item->nextSibling()) {
	    item = item->nextSibling();
	} else {
	    while (item && ! item->nextSibling())
		item = item->parent();

	    if (item)
		item = item->nextSibling();
	}
    }

    updateSongLengths();
}

void MediaView::remSong()
{
    QListViewItem *item = currentItem();

    if (! item || item->rtti() != Song::RTTI)
	return;

    if (QMessageBox::warning(this,
			     tr("Confirm Remove"),
			     tr("Are you sure you want to remove this song?"),
			     tr("&Yes"),
			     tr("&No"),
			     QString::null, 0, 1))
	return;

    Song *s = (Song *) item;
    delete s;

    setModified(true);
}

void MediaView::remGroup()
{
    QListViewItem *item = currentItem();

    // find a group
    while (item && item->rtti() != Group::RTTI)
	item = item->parent();

    if (! item)
	return;

    if (QMessageBox::warning(this,
			     tr("Confirm Remove"),
			     tr("Are you sure you want to remove this group?"),
			     tr("&Yes"),
			     tr("&No"),
			     QString::null, 0, 1))
	return;

    Group *g = (Group *) item;
    delete g;

    setModified(true);
}

void MediaView::remAll()
{
    if (QMessageBox::warning(this, "Confirm Remove",
			     "Are you sure you want to remove all media?",
			     "&Yes", "&No", QString::null, 0, 1))
	return;

    QListViewItem *child = firstChild(), *tmp;
    while (child) {
	tmp = child;
	child = child->nextSibling();
	delete tmp;
    }

    setModified(true);
}

void MediaView::editSong()
{
    QListViewItem *item = currentItem();
    if (! item || item->rtti() != Song::RTTI)
	return;

    Song *s = (Song *) item;

    RealSongEditor se(s);
    if (se.exec() != QDialog::Accepted)
 	return;

    if (se.nameLineEdit->text() != s->text(0))
 	s->setText(0, se.nameLineEdit->text());

    if (se.pathLineEdit->text() != s->location()) {
 	s->setLocation(se.pathLineEdit->text());

 	setModified(true);
 	s->setLength(-1.0);
 	lengthlist.append(s);
    }

    if (se.enabledCheckBox->isChecked() != s->isOn())
 	s->setOn(se.enabledCheckBox->isChecked());

    if (se.prioSpinBox->value() != s->priority())
 	s->setPriority(se.prioSpinBox->value());

    if (se.eqComboBox->currentText() != s->equalizer())
	s->setEqualizer(se.eqComboBox->currentText());

    if (isModified())
	s->repaint();
    updateSongLengths();
}

void MediaView::editGroup()
{
    QListViewItem *item = currentItem();
    if (! item || item->rtti() != Group::RTTI)
	return;

    Group *g = (Group *) item;

    GroupEditor ge(this, "organizer group editor", true);
    if (icon())
     	ge.setIcon(*icon());
    ge.nameLineEdit->setText(g->text(0));
    QStringList presets(Equalizer::instance()->presets());
    ge.eqComboBox->insertStringList(presets);

    QString eq;
    int songcount = 0, enabledcount = 0;
    groupStatus(g, songcount, enabledcount, eq);

    if (songcount > 0) {
	if (songcount == enabledcount) {
     	    // all songs enabled
	    ge.enabledCheckBox->setChecked(true);
	    ge.enabledCheckBox->setTristate(false);
	} else if (enabledcount == 0) {
	    // none enabled
	    ge.enabledCheckBox->setChecked(false);
	    ge.enabledCheckBox->setTristate(false);
	} else {
	    ge.enabledCheckBox->setTristate(true);
	    ge.enabledCheckBox->setNoChange();
	}
    }

    if (eq.isNull())
	eq = "Default";
    if (! presets.contains(eq))
	ge.eqComboBox->insertItem(eq);
    ge.eqComboBox->setCurrentText(eq);

    if (ge.exec() != QDialog::Accepted)
	return;

    if (ge.nameLineEdit->text() != g->text(0)) {
	g->setText(0, ge.nameLineEdit->text());
	g->repaint();
    }

    if (ge.prioSpinBox->value())
	addGroupPrio(g, ge.prioSpinBox->value());

    if (ge.enabledCheckBox->state() != QButton::NoChange)
        setGroupEnabled(g, ge.enabledCheckBox->isChecked());
    if (ge.eqComboBox->currentText() != eq)
	setGroupEqualizer(g, ge.eqComboBox->currentText());
}


void MediaView::enableAll()
{
    QListViewItem *item = currentItem();
    if (! item)
	return;

    Group *g = 0;
    if (item->rtti() == Group::RTTI)
	g = (Group *) item;

    setGroupEnabled(g, true);
}

void MediaView::disableAll()
{
    QListViewItem *item = currentItem();
    if (! item)
	return;

    Group *g = 0;
    if (item->rtti() == Group::RTTI)
	g = (Group *) item;

    setGroupEnabled(g, false);
}

void MediaView::incPrio()
{
    QListViewItem *item = currentItem();
    if (! item)
	return;

    if (item->rtti() == Group::RTTI) {
	// group
	addGroupPrio((Group *) item, 1);
    } else if (item->rtti() == Song::RTTI) {
	// song
	Song *song = (Song *) item;
	song->setPriority(song->priority() + 1);
		    song->repaint();

    }
}

void MediaView::decPrio()
{
    QListViewItem *item = currentItem();
    if (! item)
	return;

    if (item->rtti() == Group::RTTI) {
	// group
	addGroupPrio((Group *) item, -1);
    } else if (item->rtti() == Song::RTTI) {
	// song
	Song *song = (Song *) item;
	int newprio = song->priority() - 1;
	song->setPriority(newprio < 1 ? 1 : newprio);
	song->repaint();
    }
}

void MediaView::incPrio5()
{
    QListViewItem *item = currentItem();
    if (! item)
	return;

    if (item->rtti() == Group::RTTI) {
	// group
	addGroupPrio((Group *) item, 5);
    } else if (item->rtti() == Song::RTTI) {
	// song
	Song *song = (Song *) item;
	song->setPriority(song->priority() + 5);
	song->repaint();
    }
}

void MediaView::decPrio5()
{
    QListViewItem *item = currentItem();
    if (! item)
	return;

    if (item->rtti() == Group::RTTI) {
	// group
	addGroupPrio((Group *) item, -5);
    } else if (item->rtti() == Song::RTTI) {
	// song
	Song *song = (Song *) item;
	int newprio = song->priority() - 5;
	song->setPriority(newprio < 1 ? 1 : newprio);
	song->repaint();
    }
}

void MediaView::addGroupPrio(Group *group, int prio)
{
    QListViewItem *item = group->firstChild();
    while (item) {
	if (item->rtti() == Group::RTTI) {
	    // recurse groups
	    addGroupPrio((Group *) item, prio);
	} else if (item->rtti() == Song::RTTI) {
	    // song
	    Song *song = (Song *) item;
	    int newprio = song->priority() + prio;

	    if (newprio < 1)
		newprio = 1;
	    song->setPriority(newprio);
	    song->repaint();
	}

	item = item->nextSibling();
    }
}

void MediaView::setGroupEnabled(Group *group, bool enabled)
{
    QListViewItem *item;
    if (group)
	item = group->firstChild();
    else
	item = firstChild();
    while (item) {
	if (item->rtti() == Group::RTTI) {
	    setGroupEnabled((Group *) item, enabled);
	} else if (item->rtti() == Song::RTTI) {
	    Song *song = (Song *) item;
	    if (song->isOn() != enabled)
		song->setOn(enabled);
	}

	item = item->nextSibling();
    }
}

void MediaView::setGroupEqualizer(Group *group, const QString &eq)
{
    QListViewItem *item;
    if (group)
	item = group->firstChild();
    else
	item = firstChild();
    while (item) {
	if (item->rtti() == Group::RTTI) {
	    setGroupEqualizer((Group *) item, eq);
	} else if (item->rtti() == Song::RTTI) {
	    Song *song = (Song *) item;
	    if (song->equalizer() != eq)
		song->setEqualizer(eq);
	}

	item = item->nextSibling();
    }
}

void MediaView::groupStatus(Group *group, int &songcount, int &enabledcount, QString &eq)
{
    QListViewItem *item = group->firstChild();
    while (item) {
	if (item->rtti() == Group::RTTI) {
	    groupStatus((Group *) item, songcount, enabledcount, eq);
       	} else if (item->rtti() == Song::RTTI) {
	    Song *song = (Song *) item;
	    songcount++;
	    if (song->isOn())
		enabledcount++;
	    if (song->equalizer() != eq) {
		if (eq.isNull())
		    eq = song->equalizer();
		else
		    eq = tr("Multiple");
	    }
	}

	item = item->nextSibling();
    }
}

void MediaView::updateSongLengths()
{
    if (lengthlist.count()) {
	if (! lengthTimer) {
	    lengthTimer = new QTimer(this);
	    connect(lengthTimer, SIGNAL(timeout()), SLOT(doLengthCalc()));
	}
	lengthTimer->start(10);
    }
}

void MediaView::doLengthCalc()
{
    if (! lengthlist.count()) {
	if (lengthTimer) {
	    lengthTimer->stop();
	    delete lengthTimer;
	    lengthTimer = 0;
	}

	return;
    }

    Song *song = lengthlist.first();
    lengthlist.remove(lengthlist.begin());

    if (! song)
	return;

    QUrl url(song->location());
    if (url.isLocalFile()) {
	Organizer::instance()->statusBar()->message(tr("Getting length for ") +
						    song->title(), 2000);

	QFile file(url.toString(false, false));
	double newlen = -2.0;
	Decoder *decoder = Decoder::create(song->location(), &file, 0, true);
	if (decoder) {
	    if (decoder->initialize()) {
		newlen = decoder->lengthInSeconds();
		setModified(true);
	    }
	    delete decoder;
	}

	song->setLength(newlen);
	song->repaint();
    }
}

MediaView *MediaView::instance()
{
    return INSTANCE;
}

#include "mediaview.moc"
