/*
 * ratCode.c --
 *
 *	This file contains basic support for decoding and encoding of
 *	strings coded in various MIME-encodings.
 *
 * TkRat software and its included text is Copyright 1996-2000 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "rat.h"

/*
 * Used for list of saved encodings
 */
typedef struct RatEncoding {
    char *name;
    Tcl_Encoding enc;
    struct RatEncoding *nextPtr;
} RatEncoding;

/*
 * List used when decoding QP
 */
char alphabetHEX[17] = "0123456789ABCDEF";

/*
 * List used when decoding base64
 * It consists of 64 chars plus '=' and null
 */
static char alphabet64[66] =
	    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";

/*
 * List used when decoding modified base64
 * It consists of 64 chars plus '=' and null
 */
static char modified64[66] =
	    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+,=";


static int FindMimeHdr(Tcl_Interp *interp, unsigned char *hdr,
	unsigned char **sPtr, unsigned char **ePtr, Tcl_Encoding *encoding,
	int *code, unsigned char **data, int *length);
static int RatUtf8to16(const unsigned char *src, unsigned char *dst);
static int RatUtf16to8(const unsigned char *src, unsigned char *dst);

#ifdef MEM_DEBUG
static RatEncoding *mem_encPtr = NULL;
#endif /* MEM_DEBUG */

/*
 *----------------------------------------------------------------------
 *
 * FindMimeHdr --
 *
 *      Find a string encoded according to rfc2047
 *
 * Results:
 *	Returns data in most arguments.
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

static int
FindMimeHdr(Tcl_Interp *interp, unsigned char *hdr, unsigned char **sPtr,
	unsigned char **ePtr, Tcl_Encoding *encoding, int *code,
	unsigned char **data, int *length)
{
    unsigned char *sCharset, *eCharset, *cPtr, c;

    for (cPtr = hdr; *cPtr; cPtr++) {
	if ('=' == cPtr[0] && '?' == cPtr[1]) {
	    *sPtr = cPtr;
	    sCharset = cPtr+2;
	    for (cPtr+=2; '?' != *cPtr && *cPtr; cPtr++);
	    if ('?' != *cPtr) return 0;
	    if ('?' != cPtr[2]) continue;
	    switch (cPtr[1]) {
		case 'b':
		case 'B':
		    *code = ENCBASE64;
		    break;
		case 'q':
		case 'Q':
		    *code = ENCQUOTEDPRINTABLE;
		    break;
		default:
		    continue;
	    }
	    eCharset = cPtr;
	    *data = cPtr+3;
	    for (cPtr+=3, *length = 0;
		    *cPtr && ('?' != *cPtr || '=' != cPtr[1]);
		    cPtr++, (*length)++);
	    if ('?' != *cPtr) return 0;
	    *ePtr = cPtr+2;
	    c = *eCharset;
	    *eCharset = '\0';
	    *encoding = RatGetEncoding(interp, (char*)sCharset);
	    *eCharset = c;
	    return 1;
	}
    }
    return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * RatDecodeHeader --
 *
 *      Decodes a header line encoded according to rfc1522.
 *
 * Results:
 *	Returns a pointer to a static storage area
 *
 * Side effects:
 *	None
 *
 * TODO, handle address entries correct
 *
 *----------------------------------------------------------------------
 */

char*
RatDecodeHeader(Tcl_Interp *interp, char *data, int adr)
{
    static Tcl_DString ds, tmp;
    static int initialized = 0;
    char buf[1024];
    unsigned char *sPtr, *ePtr, *decoded, *text, *cPtr,
	    *point = (unsigned char*)data;
    int length, code, first = 1;
    unsigned long dlen;
    unsigned int i;
    Tcl_Encoding encoding;
    Tcl_DString *myPtr = NULL;

    if (!data || !*data) {
	return "";
    }

    if (!initialized) {
	Tcl_DStringInit(&ds);
	initialized = 1;
    } else {
	Tcl_DStringSetLength(&ds, 0);
    }

    /*
     * Check for headers from buggy programs (with raw eight-bit data
     * in them)
     */
    for (cPtr = (unsigned char*)data; *cPtr; cPtr++) {
	if (*cPtr & 0x80) {
	    myPtr = (Tcl_DString*)ckalloc(sizeof(Tcl_DString));
	    Tcl_DStringInit(myPtr);
	    Tcl_ExternalToUtfDString(NULL, data, -1, myPtr);
	    data = Tcl_DStringValue(myPtr);
	    point = (unsigned char*)data;
	    break;
	}
    }

    while (FindMimeHdr(interp, point, &sPtr, &ePtr, &encoding, &code, &text,
	    &length)) {
	if (sPtr != point) {
	    if (!first) {
		for (cPtr = point; cPtr < sPtr && isspace(*cPtr); cPtr++);
		if (cPtr < sPtr) {
		    Tcl_DStringAppend(&ds, (char*)point, sPtr-point);
		}
	    } else {
		Tcl_DStringAppend(&ds, (char*)point, sPtr-point);
		first = 0;
	    }
	}
	point = ePtr;
	if (NULL == encoding) {
	    Tcl_DStringAppend(&ds, (char*)sPtr, ePtr-sPtr);
	    continue;
	}
	if (ENCBASE64 == code) {
	    decoded = rfc822_base64(text, length, &dlen);
	    memmove(buf, decoded, dlen);
	    ckfree(decoded);
	    buf[dlen] = '\0';
	} else {
	    for (i=0, cPtr = text; cPtr-text < length; i++, cPtr++) {
		if ('_' == *cPtr) {
		    buf[i] = ' ';
		} else if ('=' == *cPtr) {
		    buf[i] =
			    ((strchr(alphabetHEX, cPtr[1])-alphabetHEX)<<4) +
			     (strchr(alphabetHEX, cPtr[2])-alphabetHEX);
		    cPtr += 2;
		} else {
		    buf[i] = *cPtr;
		}
	    }
	    buf[i] = '\0';
	    dlen = i;
	}
	Tcl_ExternalToUtfDString(encoding, buf, dlen, &tmp);
	Tcl_DStringAppend(&ds,
		Tcl_DStringValue(&tmp), Tcl_DStringLength(&tmp));
	Tcl_DStringFree(&tmp);
    }
    if (*point) {
	Tcl_DStringAppend(&ds, (char*)point, -1);
    }
    if (myPtr) {
	Tcl_DStringFree(myPtr);
	ckfree(myPtr);
    }
    return Tcl_DStringValue(&ds);
}

/*
 *----------------------------------------------------------------------
 *
 * RatDecode --
 *
 *	General decoding interface. It takes as arguments a chunk of data,
 *	the encoding the data is in. And returns a new ckalloced block of
 *	decoded data. The decoded data will not have any \r or \0 in it
 *	\0 will be changed to the string \0, unless the toCharset parameter
 *	is NULL. If that is the case the data is assumed to be wanted
 *	in raw binary form.
 *	It is also possible to get this routine to do some character set
 *	transformation, but this is not yet implemented.
 *
 * Results:
 *	A block of decoded data. It is the callers responsibility to free
 *	this data.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_DString*
RatDecode(Tcl_Interp *interp, int cte, char *data, int length, char *charset)
{
    char *src, *dst, buf[64], lbuf[4];
    int allocated, dataIndex = 0, index, srcLength, len;
    Tcl_Encoding enc = NULL;
    Tcl_DString *dsPtr = (Tcl_DString*)ckalloc(sizeof(Tcl_DString)),
		tmpDs;

    Tcl_DStringInit(dsPtr);
    if (charset) {
        enc = RatGetEncoding(interp, charset);
    }
    while (dataIndex < length) {
	if (cte == ENCBASE64) {
	    src = buf;
	    for (srcLength = 0; dataIndex < length
		    && srcLength < sizeof(buf)-2;) {
		for (index=0; dataIndex<length && index<4; dataIndex++) {
		    if (strchr(alphabet64, data[dataIndex])) {
			lbuf[index++] = strchr(alphabet64, data[dataIndex])
				- alphabet64;
		    }
		}
		if (0 == index) {
		    continue;
		}
		src[srcLength++] = lbuf[0] << 2 | ((lbuf[1]>>4)&0x3);
		if (strchr(alphabet64, '=')-alphabet64 != lbuf[2]) {
		    src[srcLength++] = lbuf[1] << 4 | ((lbuf[2]>>2)&0xf);
		    if (strchr(alphabet64, '=')-alphabet64 != lbuf[3]) {
			src[srcLength++] = lbuf[2] << 6 | (lbuf[3]&0x3f);
		    }
		}
	    }
	} else if (cte == ENCQUOTEDPRINTABLE) {
	    src = buf;
	    for (srcLength = 0; dataIndex < length &&
		    srcLength < sizeof(buf); ) {
		if ('=' == data[dataIndex]) {
		    if ('\r' == data[dataIndex+1]) {
			dataIndex += 3;
		    } else if ('\n' == data[dataIndex+1]) {
			dataIndex += 2;
		    } else {
			src[srcLength++] = 16*(strchr(alphabetHEX,
				data[dataIndex+1])-alphabetHEX)
				+ strchr(alphabetHEX,
				data[dataIndex+2])-alphabetHEX;
			dataIndex += 3;
		    }
		} else {
		    src[srcLength++] = data[dataIndex++];
		}
	    }
	} else {
	    src = data;
	    srcLength = length;
	    dataIndex = length;
	    allocated = 0;
	}
	if (charset) {
	    Tcl_ExternalToUtfDString(enc, src, srcLength, &tmpDs);
	    Tcl_DStringAppend(dsPtr,
		    Tcl_DStringValue(&tmpDs), Tcl_DStringLength(&tmpDs));
	    Tcl_DStringFree(&tmpDs);
	} else {
	    Tcl_DStringAppend(dsPtr, src, srcLength);
	}
    }
    if (charset) {
	len = Tcl_DStringLength(dsPtr);
	for (src = dst = Tcl_DStringValue(dsPtr); *src; src++) {
	    if (*src != '\r') {
		*dst++ = *src;
	    } else {
		len--;
	    }
	}
	Tcl_DStringSetLength(dsPtr, len);
    }
    return dsPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatEncodeHeaderLine --
 *
 *	Encodes one header line according to MIME.
 *	The nameLength argument should tell how long the header name is in
 *	characters. This is so that the line folding can do its job properly.
 *
 * Results:
 *	A block of encoded header line. It is the callers responsibility to
 *	free this block later with a call to ckfree().
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatEncodeHeaderLine (Tcl_Interp *interp, Tcl_Obj *line, int nameLength)
{
    unsigned char *buf, *srcPtr, *dstPtr, *startPtr, *endPtr;
    int length, dstLen, used, l;
    Tcl_DString ds;
    char *charset;

    if (NULL == line) {
	return NULL;
    }

    /* TODO, handle arbitrarly long lines */
    Tcl_GetStringFromObj(line, &dstLen);
    dstLen += 1024;
    dstPtr = buf = (unsigned char*)ckalloc(dstLen);

    /*
     * TODO, should select charset instead
     */
    charset = Tcl_GetVar2(interp, "option", "charset", TCL_GLOBAL_ONLY);

    /*
     * Do nothing if we are not initialized first
     */
    if (NULL == charset) {
	return NULL;
    }

    /*
     * Now do the encoding
     */
    Tcl_DStringInit(&ds);
    length = nameLength;
    used = 0;
    for (srcPtr = (unsigned char*)Tcl_GetString(line); *srcPtr;) {
	if (0x80 & *srcPtr) {
	    startPtr = (unsigned char*)Tcl_GetString(line);
	    dstPtr = buf;
	    used = 0;
	    endPtr = startPtr + strlen((char*)startPtr);
	    length = nameLength;
	    sprintf((char*)dstPtr, "=?%s?Q?", charset);
	    length += strlen((char*)dstPtr);
	    if (length > 70) {
		sprintf((char*)dstPtr, "\n =?%s?Q?", charset);
		length = -1;
	    }
	    dstPtr += strlen((char*)dstPtr);
	    Tcl_UtfToExternalDString(RatGetEncoding(interp, charset),
		    (char*)startPtr, endPtr-startPtr, &ds);
	    for (srcPtr = (unsigned char*)Tcl_DStringValue(&ds),
		    l = Tcl_DStringLength(&ds); l > 0; srcPtr++, l--) {
		if (length > 70) {
		    sprintf((char*)dstPtr, "?=\n =?%s?Q?", charset);
		    length = strlen((char*)dstPtr+3);
		    dstPtr += strlen((char*)dstPtr);
		}
		if (' ' == *srcPtr) {
		    *dstPtr++ = '_';
		    length++;
		} else if (*srcPtr & 0x80 || *srcPtr == '_'
			|| *srcPtr == '?' || *srcPtr == '=') {
		    *dstPtr++ = '=';
		    *dstPtr++ = alphabetHEX[*srcPtr >> 4];
		    *dstPtr++ = alphabetHEX[*srcPtr & 0x0f];
		    length += 3;
		} else {
		    *dstPtr++ = *srcPtr;
		    length++;
		}
	    }
	    Tcl_DStringSetLength(&ds, 0);
	    srcPtr = endPtr;
	    *dstPtr++ = '?';
	    *dstPtr++ = '=';
	} else {
	    *dstPtr++ = *srcPtr++;
	    length++;
	}
    }
    *dstPtr = '\0';
    return (char*)buf;
}

/*
 *----------------------------------------------------------------------
 *
 * RatEncodeAddresses --
 *
 *	Encodes the fullname portions of a bunch of addreses.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fullnames of the addresses may change.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatEncodeAddresses(Tcl_Interp *interp, ADDRESS *adrPtr)
{
    Tcl_Obj *oPtr;
    char *cPtr;

    while (adrPtr) {
	if (adrPtr->personal) {
	    for (cPtr = adrPtr->personal; *cPtr; cPtr++) {
		if (*cPtr & 0x80) {
		    oPtr = Tcl_NewStringObj(adrPtr->personal, -1);
		    cPtr = RatEncodeHeaderLine(interp, oPtr, 0);
		    Tcl_DecrRefCount(oPtr);
		    ckfree(adrPtr->personal);
		    adrPtr->personal = cpystr(cPtr);
		}
	    }
	}
	adrPtr = adrPtr->next;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatCheckEncodingsCmd --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      See above
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Encoding
RatGetEncoding(Tcl_Interp *interp, char *name)
{
    static RatEncoding *encPtr = NULL;
    RatEncoding *ePtr;
    Tcl_Encoding tenc;
    char *tclName, lname[256];

    if (NULL == name) {
	return NULL;
    }

    RatStrNCpy(lname, name, sizeof(lname));
    lcase(lname);
    tclName = Tcl_GetVar2(interp, "charsetMapping", lname, TCL_GLOBAL_ONLY);
    if (NULL == tclName) {
	tclName = lname;
    }

    for (ePtr = encPtr; ePtr; ePtr = ePtr->nextPtr) {
	if (!strcmp(ePtr->name, tclName)) {
	    return ePtr->enc;
	}
    }

    tenc = Tcl_GetEncoding(interp, tclName);
    if (NULL == tenc) {
	return NULL;
    }
    ePtr = (RatEncoding*)ckalloc(sizeof(RatEncoding));
    ePtr->name = cpystr(tclName);
    ePtr->enc = tenc;
    ePtr->nextPtr = encPtr;
    encPtr = ePtr;
#ifdef MEM_DEBUG
    mem_encPtr = encPtr;
#endif /* MEM_DEBUG */
    return ePtr->enc;
}


/*
 *----------------------------------------------------------------------
 *
 * RatCheckEncodingsCmd --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      See above
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatCheckEncodingsCmd(ClientData dummy, Tcl_Interp *interp, int objc,
	Tcl_Obj *const objv[])
{
    int i, listLength, srcLen, in, ret;
    Tcl_Obj *oPtr, *vPtr;
    Tcl_EncodingState state;
    char buf[1024], *src;
    Tcl_Encoding enc;

    if (3 != objc) {
	Tcl_AppendResult(interp, "Usage: ", Tcl_GetString(objv[0]), \
		" variable charsets", (char*) NULL);
	return TCL_ERROR;
    }
    vPtr = Tcl_GetVar2Ex(interp, Tcl_GetString(objv[1]), NULL, 0);
    Tcl_ListObjLength(interp, objv[2], &listLength);
    for (i=0; i<listLength; i++) {
	Tcl_ListObjIndex(interp, objv[2], i, &oPtr);
	if (NULL == (enc = RatGetEncoding(interp, Tcl_GetString(oPtr)))) {
	    continue;
	}
	src = Tcl_GetStringFromObj(vPtr, &srcLen);
	state = 0;
	ret = 0;
	while (srcLen && TCL_CONVERT_UNKNOWN != ret) {
	    ret = Tcl_UtfToExternal(interp, enc, src, srcLen,
			TCL_ENCODING_STOPONERROR, &state, buf, sizeof(buf),
			&in, NULL, NULL);
	    src += in;
	    srcLen -= in;
	}
	if (TCL_CONVERT_UNKNOWN != ret) {
	    Tcl_SetObjResult(interp, oPtr);
	    return TCL_OK;
	}
    }
    Tcl_SetResult(interp, "", TCL_STATIC);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatCode64 --
 *
 *	Encode the given object in base64
 *
 * Results:
 *      A new Tcl_Obj
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
RatCode64(Tcl_Obj *sPtr)
{
    Tcl_Obj *dPtr = Tcl_NewObj();
    unsigned char *cPtr, buf[4];
    int l, ll;

    cPtr = Tcl_GetStringFromObj(sPtr, &l);

    for (ll = 0; l > 0; l -= 3, cPtr += 3) {
	buf[0] = alphabet64[cPtr[0] >> 2];
	buf[1] = alphabet64[((cPtr[0] << 4) + (l>1 ? (cPtr[1]>>4) : 0)) & 0x3f];
	buf[2] = l > 1 ?
	    alphabet64[((cPtr[1]<<2) + (l>2 ? (cPtr[2]>>6) : 0)) & 0x3f] : '=';
	buf[3] = l > 2 ? alphabet64[cPtr[2] & 0x3f] : '=';
	Tcl_AppendToObj(dPtr, buf, 4);
	if (18 == ++ll || l < 4) {
	    Tcl_AppendToObj(dPtr, "\n", 1);
	    ll = 0;
	}
    }
    return dPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatUtf8to16 --
 *
 *	Convert the given utf-8 character to UCS-2
 *
 * Results:
 *      Returns the number of characters consumed from src
 *	On failure a negative number is returned.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatUtf8to16(const unsigned char *src, unsigned char *dst)
{
    if (0 == (*src & 0x80)) {
	dst[0] = 0;
	dst[1] = *src;
	return 1;
    } else if (0xc0 == (*src & 0xe0)) {       
        if (!(src[1] & 0x80)) {
            return 1;
        }
        dst[0] = (src[0] & 0x1f) >> 2;
        dst[1] = ((src[0] & 0x03) << 6) + (src[1] & 0x3f);
        return 2;
    } else if (0xe0 == (*src & 0xf0)) {
        if (!(src[1] & 0x80) && !(src[2] & 0x80)) {
            return 1;
        }
        dst[0] = ((src[0] & 0x0f) << 4) + ((src[1] & 0x3f) >> 2);
        dst[1] = ((src[1] & 0x03) << 6) + (src[2] & 0x3f);
        return 3;
    } else {
	dst[0] = 0;
	dst[1] = *src;
	return 1;
    }
}   

/*
 *----------------------------------------------------------------------
 *
 * RatUtf16to8 --
 *
 *	Convert the given UCS-2 character to utf-8
 *
 * Results:
 *      Returns the length of the generated string on success.
 *	On failure a negative number is returned.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatUtf16to8(const unsigned char *src, unsigned char *dst)
{
    if (src[0] >= 0x08) {
	dst[0] = 0xe0 | (src[0] >> 4);
	dst[1] = 0x80 | ((src[0] & 0x0f) << 2) | (src[1] >> 6);
	dst[2] = 0x80 | (src[1] & 0x3f);
	return 3;
    } else if (src[0] || src[1] > 0x7f) {
	dst[0] = 0xc0 | (src[0] << 2) | (src[1] >> 6);
	dst[1] = 0x80 | (src[1] & 0x3f);
	return 2;
    } else {
	dst[0] = src[1];
	return 1;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * RatUtf8toMutf7 --
 *
 *	Convert the given utf-8 encoded text to modified utf-7
 *
 * Results:
 *      Returns the length of the generated string on success.
 *	On failure a negative number is returned.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

unsigned char*
RatUtf8toMutf7(const unsigned char *src)
{
    static unsigned char *dst = NULL;
    static int dstlen = 0;
    unsigned char buf[3];
    int len = 0, overflow = 0;

    if (dstlen < strlen(src)*3) {
	dstlen = strlen(src)*3;
	dst = (unsigned char *)ckrealloc(dst, dstlen);
    }
    while (*src) {
	if ('&' == *src) {
	    if (dstlen <= len+2) {
		dstlen += 128;
		dst = (unsigned char *)ckrealloc(dst, dstlen);
	    }
	    dst[len++] = '&';
	    dst[len++] = '-';
	    src++;
	} else if (*src & 0x80) {
	    if (dstlen <= len+6) {
		dstlen += 128;
		dst = (unsigned char *)ckrealloc(dst, dstlen);
	    }
	    dst[len++] = '&';
	    do {
		if (dstlen <= len+5) {
		    dstlen += 128;
		    dst = (unsigned char *)ckrealloc(dst, dstlen);
		}
		if (overflow) {
		    buf[0] = buf[3];
		    if (*src & 0x80) {
			src += RatUtf8to16(src, buf+1);
		    } else {
			buf[1] = buf[2] = 0;
		    }
		    overflow = 0;
		} else {
		    src += RatUtf8to16(src, buf);
		    if (*src & 0x80) {
			src += RatUtf8to16(src, buf+2);
			overflow = 1;
		    } else {
			buf[2] = buf[3] = 0;
		    }
		}
		dst[len++] = modified64[buf[0] >> 2];
		dst[len++] = modified64[((buf[0] << 4) + (buf[1]>>4)) & 0x3f];
		if (buf[1] || buf[2]) {
		    dst[len++] =
			modified64[((buf[1]<<2) + (buf[2]>>6)) & 0x3f];
		    if (buf[2]) {
			dst[len++] = modified64[buf[2] & 0x3f];
		    }
		}
	    } while (*src & 0x80 || overflow);
	    if (strchr(modified64, *src) || '\0' == *src) {
		dst[len++] = '-';
	    }
	} else {
	    if (dstlen <= len+1) {
		dstlen += 128;
		dst = (unsigned char *)ckrealloc(dst, dstlen);
	    }
	    dst[len++] = *src++;
	}
    }
    dst[len] = '\0';
    return dst;
}

/*
 *----------------------------------------------------------------------
 *
 * RatMutf7toUtf8 --
 *
 *	Convert the given modified utf-7 encoded text to utf-8
 *
 * Results:
 *      Returns the length of the generated string on success.
 *	On failure a negative number is returned.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

unsigned char*
RatMutf7toUtf8(const unsigned char *src)
{
    static unsigned char *dst = NULL;
    static int dstlen = 0;
    unsigned char utf16[2], lbuf[4];
    int i, l, len=0, odd;

    if (dstlen < strlen(src)*3) {
	dstlen = strlen(src)*3;
	dst = (unsigned char *)ckrealloc(dst, dstlen);
    }
    while (*src) {
	if (len >= dstlen) {
	    dstlen += 128;
	    dst = (unsigned char *)ckrealloc(dst, dstlen);
	}
	if ('&' == *src && '-' == src[1]) {
	    dst[len++] = '&';
	    src += 2;
	} else if ('&' == *src) {
	    src++;
	    odd = 0;
	    do {
		for (i=0; i<4; i++) {
		    if (strchr(modified64, *src)) {
			lbuf[i] = strchr(modified64, *src++) - modified64;
		    } else {
			lbuf[i] = 0;
		    }
		}
		if (odd) {
		    odd = 0;
		    if (len >= dstlen+6) {
			dstlen += 128;
			dst = (unsigned char *)ckrealloc(dst, dstlen);
		    }
		    utf16[1] = (lbuf[0] << 2) | (lbuf[1] >> 4);
		    len += RatUtf16to8(utf16, dst+len);
		    utf16[0] = (lbuf[1] << 4) | (lbuf[2] >> 2);
		    utf16[1] = (lbuf[2] << 6) | lbuf[3];
		    if (utf16[0] != 0 || utf16[1] != 0) {
			l = RatUtf16to8(utf16, dst+len);
			len += l;
		    }
		} else {
		    if (len >= dstlen+3) {
			dstlen += 128;
			dst = (unsigned char *)ckrealloc(dst, dstlen);
		    }
		    utf16[0] = (lbuf[0] << 2) | (lbuf[1] >> 4);
		    utf16[1] = (lbuf[1] << 4) | (lbuf[2] >> 2);
		    len += RatUtf16to8(utf16, dst+len);
		    utf16[0] = (lbuf[2] << 6) | lbuf[3];
		    odd = 1;
		}
	    } while (strchr(modified64, *src));
	    if ('-' == *src) {
		src++;
	    }
	} else {
	    dst[len++] = *src++;
	}
    }
    dst[len] = '\0';
    return dst;
}

/*
 * Test code for Mutf7 <-> utf8 functions
static void
Test(unsigned char *in)
{
    unsigned char stage1[1024], stage2[1024];

    printf("In:     %s\n", in); fflush(stdin);
    RatUtf8toMutf7(in, stage1, sizeof(stage1));
    printf("Stage1: %s\n", stage1); fflush(stdin);
    RatMutf7toUtf8(stage1, stage2, sizeof(stage2));
    printf("Stage2: %s\n", stage2); fflush(stdin);
    if (strcmp(stage2, in)) {
	printf("ERROR\n");
    }
    printf("\n");
}

int main()
{
    Test("får");
    Test("Räksmörgås");
    Test("å");
    Test("åä");
    Test("åäö");
    Test("åäöå");
    Test("åäöåä");
    Test("åäöåäö");

    return 0;
} */

#ifdef MEM_DEBUG
void ratCodeCleanup()
{
    RatEncoding *e, *n;

    for (e = mem_encPtr; e; e = n) {
	n = e->nextPtr;
	ckfree(e->name);
	ckfree(e);
    }
}
#endif /* MEM_DEBUG */
