# tkfbox.tcl --
#       This is a special version of the tk file selection box. It has been
#       subject to the following modifications:
#	- There is a new argument -mode (open, save, anyopen, dirok)
#       - There is a new argument -ok which sets the text of the ok button
#       - The file and cancel texts are fetched from the global variables
#         t(file) and t(cancel)
#	- The original tk icon list is used
#       - The whole thing is now contained in a separate namespace rat_fbox
#       The original comments follows:
#
#	Implements the "TK" standard file selection dialog box. This
#	dialog box is used on the Unix platforms whenever the tk_strictMotif
#	flag is not set.
#
#	The "TK" standard file selection dialog box is similar to the
#	file selection dialog box on Win95(TM). The user can navigate
#	the directories by clicking on the folder icons or by
#	selectinf the "Directory" option menu. The user can select
#	files by clicking on the file icons or by entering a filename
#	in the "Filename:" entry.
#
# RCS: @(#) $Id: rat_fbox.tcl,v 1.17.2.3 2000/11/22 16:33:09 maf Exp $
#
# Copyright (c) 1994-1998 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

package provide rat_fbox 1.1

namespace eval rat_fbox {
    namespace export run
    variable state
    variable previous {}
}

#----------------------------------------------------------------------
#
#		      F I L E   D I A L O G
#
#----------------------------------------------------------------------

# rat_fbox::run --
#
#	Implements the TK file selection dialog. This dialog is used when
#	the tk_strictMotif flag is set to false. This procedure shouldn't
#	be called directly. Call tk_getOpenFile or tk_getSaveFile instead.
#
# Arguments:
#	args		Options parsed by the procedure.
#

proc rat_fbox::run {args} {
    variable state
    set dataName __rat_filedialog
    upvar #0 $dataName data

    rat_fbox::config $dataName $args

    if {![string compare $data(-parent) .]} {
        set w .$dataName
    } else {
        set w $data(-parent).$dataName
    }

    # (re)create the dialog box if necessary
    #
    if {![winfo exists $w]} {
	set data(showDotfiles) 0
	rat_fbox::create $w
    } else {
	set data(dirMenuBtn) $w.f1.menu
	set data(dirMenu) $w.f1.menu.menu
	set data(upBtn) $w.f1.up
	set data(icons) $w.icons
	set data(ent) $w.f2.ent
	set data(rclBtn) $w.f2.rcl
	set data(prevMenuLab) $w.f3.lab
	set data(prevMenuBtn) $w.f3.menu
	set data(prevMenu) $w.f3.menu.menu
	set data(okBtn) $w.f4.ok
	set data(cancelBtn) $w.f4.cancel
    }

    # Initialize recall button
    #
    set data(storedFile) $data(selectFile)
    if {"" == $data(storedFile)} {
	$data(rclBtn) config -state disabled
    } else {
	$data(rclBtn) config -state normal
    }

    # Update previous directories menu
    #
    variable previous
    $data(prevMenu) delete 0 end
    if [llength $previous] {
	set var [format %s(selectPath) $dataName]
	foreach path $previous {
	    $data(prevMenu) add command -label $path \
		    -command [list set $var $path]
	}
	set data(prevPath) [lindex $previous 0]
	$data(prevMenuBtn) config -state normal
	$data(prevMenuLab) config -state normal
    } else {
	set data(prevPath) ""
	$data(prevMenuBtn) config -state disabled
	$data(prevMenuLab) config -state disabled
    }

    rat_fbox::updateWhenIdle $w

    # Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr {[winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]}]
    set y [expr {[winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]}]
    wm geom $w [winfo reqwidth $w]x[winfo reqheight $w]+$x+$y
    wm title $w $data(-title)
    wm transient $w $data(-parent)
    wm deiconify $w

    # Set a grab and claim the focus too.

    ModalGrab $w $data(ent)
    $data(ent) delete 0 end
    $data(ent) insert 0 $data(selectFile)
    $data(ent) select from 0
    $data(ent) select to   end
    $data(ent) icursor end

    trace variable data(selectPath) w "rat_fbox::setPath $w"

    # Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.

    tkwait variable rat_fbox::state(selectFilePath)
    if { "" != $state(selectFilePath)} {
        set dir [file dirname $state(selectFilePath)]
        set i [lsearch -exact $previous $dir]
        if {-1 == $i} {
	    set previous [lrange [linsert $previous 0 $dir] 0 10]
        } else {
	    set previous [lrange \
		    [linsert [lreplace $previous $i $i] 0 $dir] 0 10]
	}
    }
    destroy $w

    return $state(selectFilePath)
}

# rat_fbox::config --
#
#	Configures the TK filedialog according to the argument list
#
proc rat_fbox::config {dataName argList} {
    upvar #0 $dataName data

    # 0: Delete all variable that were set on data(selectPath) the
    # last time the file dialog is used. The traces may cause troubles
    # if the dialog is now used with a different -parent option.

    foreach trace [trace vinfo data(selectPath)] {
	trace vdelete data(selectPath) [lindex $trace 0] [lindex $trace 1]
    }

    # 1: the configuration specs
    #
    set specs {
	{-defaultextension "" "" ""}
	{-filetypes "" "" ""}
	{-initialdir "" "" ""}
	{-initialfile "" "" ""}
	{-parent "" "" "."}
	{-title "" "" ""}
	{-ok "" "" ""}
	{-mode "" "" ""}
    }

    # 2: default values depending on the type of the dialog
    #
    if {![info exists data(selectPath)]} {
	# first time the dialog has been popped up
	set data(selectPath) [pwd]
	set data(selectFile) ""
    }

    # 3: parse the arguments
    #
    tclParseConfigSpec $dataName $specs "" $argList

    if {![string compare $data(-title) ""]} {
	if {![string compare $data(-mode) "open"]} {
	    set data(-title) "Open"
	} else {
	    set data(-title) "Save As"
	}
    }

    # 4: set the default directory and selection according to the -initial
    #    settings
    #
    if {[string compare $data(-initialdir) ""]} {
	if {[file isdirectory $data(-initialdir)]} {
	    set data(selectPath) [lindex [glob $data(-initialdir)] 0]
	} else {
	    set data(selectPath) [pwd]
	}

	# Convert the initialdir to an absolute path name.

	set old [pwd]
	cd $data(selectPath)
	set data(selectPath) [pwd]
	cd $old
    }
    set data(selectFile) $data(-initialfile)

    # 5. Parse the -filetypes option
    #
    set data(-filetypes) [tkFDGetFileTypes $data(-filetypes)]

    if {![winfo exists $data(-parent)]} {
	error "bad window path name \"$data(-parent)\""
    }
}

proc rat_fbox::create {w} {
    set dataName [lindex [split $w .] end]
    upvar #0 $dataName data
    global tk_library t
    variable state

    toplevel $w -class TkRat

    # f1: the frame with the directory option menu
    #
    set f1 [frame $w.f1]
    label $f1.lab -text $t(directory): -under 0
    set data(dirMenuBtn) $f1.menu
    set data(dirMenu) [tk_optionMenu $f1.menu [format %s(selectPath) $dataName] ""]
    $f1.menu configure -width 25
    checkbutton $f1.dot -text . -variable ${dataName}(showDotfiles) \
		-command "rat_fbox::updateWhenIdle $w"
    set data(upBtn) [button $f1.up]
    if {![info exists state(updirImage)]} {
	set state(updirImage) [image create bitmap -data {
#define updir_width 28
#define updir_height 16
static char updir_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x80, 0x1f, 0x00, 0x00, 0x40, 0x20, 0x00, 0x00,
   0x20, 0x40, 0x00, 0x00, 0xf0, 0xff, 0xff, 0x01, 0x10, 0x00, 0x00, 0x01,
   0x10, 0x02, 0x00, 0x01, 0x10, 0x07, 0x00, 0x01, 0x90, 0x0f, 0x00, 0x01,
   0x10, 0x02, 0x00, 0x01, 0x10, 0x02, 0x00, 0x01, 0x10, 0x02, 0x00, 0x01,
   0x10, 0xfe, 0x07, 0x01, 0x10, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x01,
   0xf0, 0xff, 0xff, 0x01};}]
    }
    $data(upBtn) config -image $state(updirImage)

    $f1.menu config -takefocus 1 -highlightthickness 2
 
    pack $data(upBtn) -side right -padx 4 -fill both
    pack $f1.dot -side right -padx 4 -fill both
    pack $f1.lab -side left -padx 4 -fill both
    pack $f1.menu -expand yes -fill both -padx 4

    # data(icons): the IconList that list the files and directories.
    #
    set data(icons) [tkIconList $w.icons \
	-browsecmd "rat_fbox::listBrowse $w" \
	-command   "rat_fbox::okCmd $w iconList"]

    # f2: the frame with the recall button and the "file name" field
    #
    set f2 [frame $w.f2 -bd 0]
    label $f2.lab -text $t(filename): -anchor e -width 14 -pady 0
    set data(ent) [entry $f2.ent]
    set data(rclBtn) [button $f2.rcl -text $t(recall) -under 0 -width 8 ]
    pack $f2.rcl -side right
    pack $f2.lab -side left
    pack $f2.ent -expand yes -fill x -pady 0

    # The font to use for the icons. The default Canvas font on Unix
    # is just deviant.
    global $w.icons
    set $w.icons(font) [$data(ent) cget -font]

    # f3: the frame with the Previous directory field
    #
    set f3 [frame $w.f3 -bd 0]

    set data(prevMenuLab) [button $f3.lab -text $t(previous): \
        -anchor e -width 14 \
        -bd [$f2.lab cget -bd] \
        -highlightthickness [$f2.lab cget -highlightthickness] \
        -relief [$f2.lab cget -relief] \
        -padx 0 ]
    bindtags $data(prevMenuLab) [list $data(prevMenuLab) Label \
            [winfo toplevel $data(prevMenuLab)] all]
    set data(prevMenuBtn) $f3.menu
    set data(prevMenu) \
	    [tk_optionMenu $f3.menu [format %s(prevPath) $dataName] ""]
    $f3.menu config -width 25 -takefocus 1 -highlightthickness 2

    # The "File of types:" label needs to be grayed-out when
    # -filetypes are not specified. The label widget does not support
    # grayed-out text on monochrome displays. Therefore, we have to
    # use a button widget to emulate a label widget (by setting its
    # bindtags)

    pack $data(prevMenuLab) -side left
    pack $data(prevMenuBtn) -expand yes -fill x -side right

    # f4: the frame with the buttons
    set f4 [frame $w.f4 -bd 0]
    set data(okBtn)     [button $f4.ok     -text $t(ok)     -under 0 -width 6 \
	-default active -pady 3]
    set data(cancelBtn) [button $f4.cancel -text $t(cancel) -under 0 -width 6\
	-default normal -pady 3]
    pack $data(okBtn) -side left -padx 20
    pack $data(cancelBtn) -side right -padx 20

    # Pack all the frames together. We are done with widget construction.
    #
    pack $f1 -side top -fill x -pady 4
    pack $f4 -side bottom -fill x -padx 40
    pack $f3 -side bottom -fill x -padx 3
    pack $f2 -side bottom -fill x -padx 3
    pack $data(icons) -expand yes -fill both -padx 4 -pady 1

    # Set up the event handlers
    #
    bind $data(ent) <Return>  "rat_fbox::activateEnt $w return"
    
    $data(upBtn)     config -command "rat_fbox::upDirCmd $w"
    $data(okBtn)     config -command "rat_fbox::okCmd $w okBtn"
    $data(cancelBtn) config -command "rat_fbox::cancelCmd $w"
    $data(rclBtn)    config -command "rat_fbox::rclCmd $w"

    bind $w <Alt-d> "focus $data(dirMenuBtn)"
    bind $w <Alt-n> "focus $data(ent)"
    bind $w <KeyPress-Escape> "tkButtonInvoke $data(cancelBtn)"
    bind $w <Alt-c> "tkButtonInvoke $data(cancelBtn)"
    bind $w <Alt-o> "rat_fbox::invokeBtn $w Open"
    bind $w <Alt-s> "rat_fbox::invokeBtn $w Save"

    wm protocol $w WM_DELETE_WINDOW "rat_fbox::cancelCmd $w"

    # Build the focus group for all the entries
    #
    tkFocusGroup_Create $w
    tkFocusGroup_BindIn $w  $data(ent) "rat_fbox::entFocusIn $w"
    tkFocusGroup_BindOut $w $data(ent) "rat_fbox::entFocusOut $w"
}

# rat_fbox::updateWhenIdle --
#
#	Creates an idle event handler which updates the dialog in idle
#	time. This is important because loading the directory may take a long
#	time and we don't want to load the same directory for multiple times
#	due to multiple concurrent events.
#
proc rat_fbox::updateWhenIdle {w} {
    upvar #0 [winfo name $w] data

    if {[info exists data(updateId)]} {
	return
    } else {
	set data(updateId) [after idle rat_fbox::doUpdate $w]
    }
}

# rat_fbox::doUpdate --
#
#	Loads the files and directories into the IconList widget. Also
#	sets up the directory option menu for quick access to parent
#	directories.
#
proc rat_fbox::doUpdate {w} {

    # This proc may be called within an idle handler. Make sure that the
    # window has not been destroyed before this proc is called
    if ![winfo exists $w] {
	return
    }

    set dataName [winfo name $w]
    upvar #0 $dataName data
    global tk_library 
    variable state
    catch {unset data(updateId)}

    if {![info exists state(folderImage)]} {
	set state(folderImage) [image create photo -data {
R0lGODlhEAAMAKEAAAD//wAAAPD/gAAAACH5BAEAAAAALAAAAAAQAAwAAAIghINhyycvVFsB
QtmS3rjaH1Hg141WaT5ouprt2HHcUgAAOw==}]
	set state(fileImage)   [image create photo -data {
R0lGODlhDAAMAKEAALLA3AAAAP//8wAAACH5BAEAAAAALAAAAAAMAAwAAAIgRI4Ha+IfWHsO
rSASvJTGhnhcV3EJlo3kh53ltF5nAhQAOw==}]
    }
    set folder $state(folderImage)
    set file   $state(fileImage)

    set appPWD [pwd]
    if {[catch {
	cd $data(selectPath)
    }]} {
	# We cannot change directory to $data(selectPath). $data(selectPath)
	# should have been checked before rat_fbox::doUpdate is called, so
	# we normally won't come to here. Anyways, give an error and abort
	# action.
	tk_messageBox -type ok -parent $data(-parent) -message \
	    "Cannot change to the directory \"$data(selectPath)\".\nPermission denied."\
	    -icon warning
	cd $appPWD
	return
    }

    # Turn on the busy cursor. BUG?? We haven't disabled X events, though,
    # so the user may still click and cause havoc ...
    #
    set entCursor [$data(ent) cget -cursor]
    set dlgCursor [$w         cget -cursor]
    $data(ent) config -cursor watch
    $w         config -cursor watch
    update idletasks
    
    tkIconList_DeleteAll $data(icons)

    # Make the dir list
    #
    if $data(showDotfiles) {
	set fl [glob -nocomplain .* *]
    } else {
	set fl [glob -nocomplain *]
    }
    foreach f [lsort -dictionary $fl] {
	if {![string compare $f .]} {
	    continue
	}
	if {![string compare $f ..]} {
	    continue
	}
	if {[file isdir ./$f]} {
	    if {![info exists hasDoneDir($f)]} {
		tkIconList_Add $data(icons) $folder $f
		set hasDoneDir($f) 1
	    }
	}
    }
    # Make the file list
    #
    set files [lsort -dictionary $fl]

    set top 0
    foreach f $files {
	if {![file isdir ./$f]} {
	    if {![info exists hasDoneFile($f)]} {
		tkIconList_Add $data(icons) $file $f
		set hasDoneFile($f) 1
	    }
	}
    }

    tkIconList_Arrange $data(icons)

    # Update the Directory: option menu
    #
    set list ""
    set dir ""
    foreach subdir [file split $data(selectPath)] {
	set dir [file join $dir $subdir]
	lappend list $dir
    }

    $data(dirMenu) delete 0 end
    set var [format %s(selectPath) $dataName]
    foreach path $list {
	$data(dirMenu) add command -label $path -command [list set $var $path]
    }

    # Restore the PWD to the application's PWD
    #
    cd $appPWD

    # Restore the Open/Save Button
    #
    $data(okBtn) config -text $data(-ok)

    # turn off the busy cursor.
    #
    $data(ent) config -cursor $entCursor
    $w         config -cursor $dlgCursor
}

# rat_fbox::setPathSilently --
#
# 	Sets data(selectPath) without invoking the trace procedure
#
proc rat_fbox::setPathSilently {w path} {
    upvar #0 [winfo name $w] data
    
    trace vdelete  data(selectPath) w "rat_fbox::setPath $w"
    set data(selectPath) $path
    trace variable data(selectPath) w "rat_fbox::setPath $w"
}


# This proc gets called whenever data(selectPath) is set
#
proc rat_fbox::setPath {w name1 name2 op} {
    if {[winfo exists $w]} {
	upvar #0 [winfo name $w] data
	set data(selectPath) $data($name2)
	rat_fbox::updateWhenIdle $w
    }
}


# rat_fbox::resolveFile --
#
#	Interpret the user's text input in a file selection dialog.
#	Performs:
#
#	(1) ~ substitution
#	(2) resolve all instances of . and ..
#	(3) check for non-existent files/directories
#	(4) check for chdir permissions
#
# Arguments:
#	context:  the current directory you are in
#	text:	  the text entered by the user
#	defaultext: the default extension to add to files with no extension
#
# Return vaue:
#	[list $flag $directory $file]
#
#	 flag = OK	: valid input
#	      = PATH	: the directory does not exist
#	      = FILE	: the directory exists by the file doesn't
#			  exist
#	      = CHDIR	: Cannot change to the directory
#	      = ERROR	: Invalid entry
#
#	 directory      : valid only if flag = OK or FILE
#	 file           : valid only if flag = OK
#
#	directory may not be the same as context, because text may contain
#	a subdirectory name
#
proc rat_fbox::resolveFile {context text defaultext} {

    set appPWD [pwd]

    set path [rat_fbox::joinFile $context $text]

    if {[file ext $path] == ""} {
	set path "$path$defaultext"
    }


    if {[catch {file exists $path}]} {
	# This "if" block can be safely removed if the following code
	# stop generating errors.
	#
	#	file exists ~nonsuchuser
	#
	return [list ERROR $path ""]
    }

    if {[file exists $path]} {
	if {[file isdirectory $path]} {
	    if {[catch {
		cd $path
	    }]} {
		return [list CHDIR $path ""]
	    }
	    set directory [pwd]
	    set file ""
	    set flag OK
	    cd $appPWD
	} else {
	    if {[catch {
		cd [file dirname $path]
	    }]} {
		return [list CHDIR [file dirname $path] ""]
	    }
	    set directory [pwd]
	    set file [file tail $path]
	    set flag OK
	    cd $appPWD
	}
    } else {
	set dirname [file dirname $path]
	if {[file exists $dirname]} {
	    if {[catch {
		cd $dirname
	    }]} {
		return [list CHDIR $dirname ""]
	    }
	    set directory [pwd]
	    set file [file tail $path]
	    set flag FILE
	    cd $appPWD
	} else {
	    set directory $dirname
	    set file [file tail $path]
	    set flag PATH
	}
    }

    return [list $flag $directory $file]
}


# Gets called when the entry box gets keyboard focus. We clear the selection
# from the icon list . This way the user can be certain that the input in the 
# entry box is the selection.
#
proc rat_fbox::entFocusIn {w} {
    upvar #0 [winfo name $w] data

    if {[string compare [$data(ent) get] ""]} {
	$data(ent) selection from 0
	$data(ent) selection to   end
	$data(ent) icursor end
    } else {
	$data(ent) selection clear
    }

    tkIconList_Unselect $data(icons)

    $data(okBtn) config -text $data(-ok)
}

proc rat_fbox::entFocusOut {w} {
    upvar #0 [winfo name $w] data

    $data(ent) selection clear
}


# Gets called when user presses Return in the "File name" entry.
#
proc rat_fbox::activateEnt {w from} {
    upvar #0 [winfo name $w] data

    set text [string trim [$data(ent) get]]
    set list [rat_fbox::resolveFile $data(selectPath) $text \
		  $data(-defaultextension)]
    set flag [lindex $list 0]
    set path [lindex $list 1]
    set file [lindex $list 2]

    switch -- $flag {
	OK {
	    if {![string compare $file ""] && "$data(-mode)" != "dirok"} {
		# user has entered an existing (sub)directory
		set data(selectPath) $path
		$data(ent) delete 0 end
	    } else {
		rat_fbox::setPathSilently $w $path
		set data(selectFile) $file
		rat_fbox::done $w
	    }
	}
	FILE {
	    if {![string compare $data(-mode) open]} {
		tk_messageBox -icon warning -type ok -parent $data(-parent) \
		    -message "File \"[file join $path $file]\" does not exist."
		$data(ent) select from 0
		$data(ent) select to   end
		$data(ent) icursor end
	    } else {
		rat_fbox::setPathSilently $w $path
		set data(selectFile) $file
		rat_fbox::done $w
	    }
	}
	PATH {
	    tk_messageBox -icon warning -type ok -parent $data(-parent) \
		-message "Directory \"$path\" does not exist."
	    $data(ent) select from 0
	    $data(ent) select to   end
	    $data(ent) icursor end
	}
	CHDIR {
	    tk_messageBox -type ok -parent $data(-parent) -message \
	       "Cannot change to the directory \"$path\".\nPermission denied."\
		-icon warning
	    $data(ent) select from 0
	    $data(ent) select to   end
	    $data(ent) icursor end
	}
	ERROR {
	    tk_messageBox -type ok -parent $data(-parent) -message \
	       "Invalid file name \"$path\"."\
		-icon warning
	    $data(ent) select from 0
	    $data(ent) select to   end
	    $data(ent) icursor end
	}
    }
}

# Gets called when user presses the Alt-s or Alt-o keys.
#
proc rat_fbox::invokeBtn {w key} {
    upvar #0 [winfo name $w] data

    if {![string compare [$data(okBtn) cget -text] $key]} {
	tkButtonInvoke $data(okBtn)
    }
}

# Gets called when user presses the "parent directory" button
#
proc rat_fbox::upDirCmd {w} {
    upvar #0 [winfo name $w] data

    if {[string compare $data(selectPath) "/"]} {
	set data(selectPath) [file dirname $data(selectPath)]
    }
}

# Gets called when user presses the "recall" button
#
proc rat_fbox::rclCmd {w} {
    upvar #0 [winfo name $w] data

    $data(ent) delete 0 end
    $data(ent) insert 0 $data(storedFile)
}

# Join a file name to a path name. The "file join" command will break
# if the filename begins with ~
#
proc rat_fbox::joinFile {path file} {
    if {[string match {~*} $file] && [file exists $path/$file]} {
	return [file join $path ./$file]
    } else {
	return [file join $path $file]
    }
}



# Gets called when user presses the "OK" button
#
proc rat_fbox::okCmd {w from} {
    upvar #0 [winfo name $w] data

    set text [tkIconList_Get $data(icons)]
    if {[string compare $text ""]} {
	set file [rat_fbox::joinFile $data(selectPath) $text]
	if {[file isdirectory $file] && ([string compare $data(-mode) dirok] ||
		![string compare $from iconList])} {
	    rat_fbox::listInvoke $w $text
	    return
	}
    }

    rat_fbox::activateEnt $w $from
}

# Gets called when user presses the "Cancel" button
#
proc rat_fbox::cancelCmd {w} {
    upvar #0 [winfo name $w] data
    variable state

    set state(selectFilePath) ""
}

# Gets called when user browses the IconList widget (dragging mouse, arrow
# keys, etc)
#
proc rat_fbox::listBrowse {w text} {
    upvar #0 [winfo name $w] data
    global t

    if {$text == ""} {
	return
    }

    set file [rat_fbox::joinFile $data(selectPath) $text]
    if {![file isdirectory $file]} {
	$data(ent) delete 0 end
	$data(ent) insert 0 $text

	$data(okBtn) config -text $data(-ok)
    } else {
	$data(okBtn) config -text $t(open)
    }
}

# Gets called when user invokes the IconList widget (double-click, 
# Return key, etc)
#
proc rat_fbox::listInvoke {w text} {
    upvar #0 [winfo name $w] data

    if {$text == ""} {
	return
    }

    set file [rat_fbox::joinFile $data(selectPath) $text]

    if {[file isdirectory $file]} {
	set appPWD [pwd]
	if {[catch {cd $file}]} {
	    tk_messageBox -type ok -parent $data(-parent) -message \
	       "Cannot change to the directory \"$file\".\nPermission denied."\
		-icon warning
	} else {
	    cd $appPWD
	    set data(selectPath) $file
	}
    } else {
	set data(selectFile) $file
	rat_fbox::done $w
    }
}

# rat_fbox::done --
#
#	Gets called when user has input a valid filename.  Pops up a
#	dialog box to confirm selection when necessary. Sets the
#	state(selectFilePath) variable, which will break the "tkwait"
#	loop in tkFDialog and return the selected filename to the
#	script that calls tk_getOpenFile or tk_getSaveFile
#
proc rat_fbox::done {w {selectFilePath ""}} {
    upvar #0 [winfo name $w] data
    variable state

    if {![string compare $selectFilePath ""]} {
	set selectFilePath [rat_fbox::joinFile $data(selectPath) \
		$data(selectFile)]
	set state(selectFile)     $data(selectFile)
	set state(selectPath)     $data(selectPath)

	if {[file exists $selectFilePath] && 
	    ![string compare $data(-mode) save]} {

		set reply [tk_messageBox -icon warning -type yesno\
			-parent $data(-parent) -message "File\
			\"$selectFilePath\" already exists.\nDo\
			you want to overwrite it?"]
		if {![string compare $reply "no"]} {
		    return
		}
	}
    }
    set state(selectFilePath) $selectFilePath
}

