/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.quarkus.component.jta;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import io.quarkus.test.QuarkusUnitTest;
import jakarta.inject.Inject;
import jakarta.inject.Named;
import jakarta.transaction.Status;
import jakarta.transaction.TransactionManager;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

public class RequiresNewJtaTransactionPolicyTest {

    @RegisterExtension
    static final QuarkusUnitTest CONFIG = new QuarkusUnitTest()
            .setArchiveProducer(() -> ShrinkWrap.create(JavaArchive.class)
                    .addClasses(MockTransactionManagerProducer.class, MockTransaction.class));

    @Inject
    TransactionManager transactionManager;

    @Inject
    @Named("PROPAGATION_REQUIRES_NEW")
    RequiresNewJtaTransactionPolicy transactionPolicy;

    @AfterEach
    public void afterEach() {
        reset(transactionManager);
    }

    @Test
    public void runTransactionPolicy() throws Exception {
        CountDownLatch latch = new CountDownLatch(1);
        MockTransaction transaction = new MockTransaction();
        when(transactionManager.suspend()).thenReturn(transaction);
        when(transactionManager.getStatus()).thenReturn(Status.STATUS_NO_TRANSACTION);
        try {
            transactionPolicy.run(() -> latch.countDown());
        } catch (Throwable throwable) {
            fail("Expected transaction policy to run successfully");
        }
        assertTrue(latch.await(5, TimeUnit.SECONDS));
        verify(transactionManager, times(1)).begin();
        verify(transactionManager, times(1)).commit();
        verify(transactionManager, times(1)).resume(transaction);
    }

    @Test
    public void runTransactionPolicyWithException() throws Exception {
        CountDownLatch latch = new CountDownLatch(1);
        when(transactionManager.getStatus()).thenReturn(Status.STATUS_NO_TRANSACTION);
        try {
            transactionPolicy.run(() -> {
                latch.countDown();
                throw new Exception("Simulated exception");
            });
        } catch (Throwable throwable) {
            // Expected
        }
        assertTrue(latch.await(5, TimeUnit.SECONDS));
        verify(transactionManager, times(1)).rollback();
    }
}
