/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.index.sai.disk.v1;

import com.google.common.base.Stopwatch;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.index.sai.StorageAttachedIndex;
import org.apache.cassandra.index.sai.analyzer.AbstractAnalyzer;
import org.apache.cassandra.index.sai.disk.PerColumnIndexWriter;
import org.apache.cassandra.index.sai.disk.format.IndexComponent;
import org.apache.cassandra.index.sai.disk.format.IndexDescriptor;
import org.apache.cassandra.index.sai.disk.v1.ColumnCompletionMarkerUtil;
import org.apache.cassandra.index.sai.disk.v1.MetadataWriter;
import org.apache.cassandra.index.sai.disk.v1.segment.SegmentBuilder;
import org.apache.cassandra.index.sai.disk.v1.segment.SegmentMetadata;
import org.apache.cassandra.index.sai.utils.NamedMemoryLimiter;
import org.apache.cassandra.index.sai.utils.PrimaryKey;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class SSTableIndexWriter
implements PerColumnIndexWriter {
    private static final Logger logger = LoggerFactory.getLogger(SSTableIndexWriter.class);
    private final IndexDescriptor indexDescriptor;
    private final StorageAttachedIndex index;
    private final long nowInSec = FBUtilities.nowInSeconds();
    private final AbstractAnalyzer analyzer;
    private final NamedMemoryLimiter limiter;
    private final BooleanSupplier isIndexValid;
    private final List<SegmentMetadata> segments = new ArrayList<SegmentMetadata>();
    private boolean aborted = false;
    private SegmentBuilder currentBuilder;

    public SSTableIndexWriter(IndexDescriptor indexDescriptor, StorageAttachedIndex index, NamedMemoryLimiter limiter, BooleanSupplier isIndexValid) {
        this.indexDescriptor = indexDescriptor;
        this.index = index;
        this.analyzer = index.hasAnalyzer() ? index.analyzer() : null;
        this.limiter = limiter;
        this.isIndexValid = isIndexValid;
    }

    @Override
    public void addRow(PrimaryKey key, Row row, long sstableRowId) throws IOException {
        if (this.maybeAbort()) {
            return;
        }
        if (this.index.termType().isNonFrozenCollection()) {
            Iterator<ByteBuffer> valueIterator = this.index.termType().valuesOf(row, this.nowInSec);
            if (valueIterator != null) {
                while (valueIterator.hasNext()) {
                    ByteBuffer value = valueIterator.next();
                    this.addTerm(this.index.termType().asIndexBytes(value.duplicate()), key, sstableRowId);
                }
            }
        } else {
            ByteBuffer value = this.index.termType().valueOf(key.partitionKey(), row, this.nowInSec);
            if (value != null) {
                this.addTerm(this.index.termType().asIndexBytes(value.duplicate()), key, sstableRowId);
            }
        }
    }

    @Override
    public void onSSTableWriterSwitched(Stopwatch stopwatch) throws IOException {
        if (this.maybeAbort()) {
            return;
        }
        boolean emptySegment = this.currentBuilder == null || this.currentBuilder.isEmpty();
        logger.debug(this.index.identifier().logMessage("Flushing index with {}buffered data on SSTable writer switched..."), (Object)(emptySegment ? "no " : ""));
        if (!emptySegment) {
            this.flushSegment();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void complete(Stopwatch stopwatch) throws IOException {
        if (this.maybeAbort()) {
            return;
        }
        long start = stopwatch.elapsed(TimeUnit.MILLISECONDS);
        boolean emptySegment = this.currentBuilder == null || this.currentBuilder.isEmpty();
        logger.debug(this.index.identifier().logMessage("Completing index flush with {}buffered data..."), (Object)(emptySegment ? "no " : ""));
        try {
            if (!emptySegment) {
                this.flushSegment();
                long elapsed = stopwatch.elapsed(TimeUnit.MILLISECONDS);
                logger.debug(this.index.identifier().logMessage("Completed flush of final segment for SSTable {}. Duration: {} ms. Total elapsed: {} ms"), new Object[]{this.indexDescriptor.sstableDescriptor, elapsed - start, elapsed});
            }
            if (this.currentBuilder != null) {
                long bytesAllocated = this.currentBuilder.totalBytesAllocated();
                long globalBytesUsed = this.currentBuilder.release();
                logger.debug(this.index.identifier().logMessage("Flushing final segment for SSTable {} released {}. Global segment memory usage now at {}."), new Object[]{this.indexDescriptor.sstableDescriptor, FBUtilities.prettyPrintMemory(bytesAllocated), FBUtilities.prettyPrintMemory(globalBytesUsed)});
            }
            this.writeSegmentsMetadata();
            ColumnCompletionMarkerUtil.create(this.indexDescriptor, this.index.identifier(), this.segments.isEmpty());
        }
        finally {
            this.index.indexMetrics().segmentsPerCompaction.update(this.segments.size());
            this.segments.clear();
            this.index.indexMetrics().compactionCount.inc();
        }
    }

    @Override
    public void abort(Throwable cause) {
        this.aborted = true;
        String message = this.index.identifier().logMessage("Aborting SSTable index flush for {}...");
        if (cause == null) {
            logger.debug(message, (Object)this.indexDescriptor.sstableDescriptor);
        } else {
            logger.warn(message, (Object)this.indexDescriptor.sstableDescriptor, (Object)cause);
        }
        if (this.currentBuilder != null) {
            long allocated = this.currentBuilder.totalBytesAllocated();
            long globalBytesUsed = this.currentBuilder.release();
            logger.debug(this.index.identifier().logMessage("Aborting index writer for SSTable {} released {}. Global segment memory usage now at {}."), new Object[]{this.indexDescriptor.sstableDescriptor, FBUtilities.prettyPrintMemory(allocated), FBUtilities.prettyPrintMemory(globalBytesUsed)});
        }
        this.indexDescriptor.deleteColumnIndex(this.index.termType(), this.index.identifier());
    }

    private boolean maybeAbort() {
        if (this.aborted) {
            return true;
        }
        if (this.isIndexValid.getAsBoolean()) {
            return false;
        }
        this.abort(new RuntimeException(String.format("index %s is dropped", this.index.identifier())));
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addTerm(ByteBuffer term, PrimaryKey key, long sstableRowId) throws IOException {
        if (!this.index.validateTermSize(key.partitionKey(), term, false, null)) {
            return;
        }
        if (this.currentBuilder == null) {
            this.currentBuilder = this.newSegmentBuilder();
        } else if (this.shouldFlush(sstableRowId)) {
            this.flushSegment();
            this.currentBuilder = this.newSegmentBuilder();
        }
        if (term.remaining() == 0 && this.index.termType().skipsEmptyValue()) {
            return;
        }
        if (this.analyzer == null || !this.index.termType().isLiteral()) {
            this.limiter.increment(this.currentBuilder.add(term, key, sstableRowId));
        } else {
            this.analyzer.reset(term);
            try {
                while (this.analyzer.hasNext()) {
                    ByteBuffer tokenTerm = this.analyzer.next();
                    this.limiter.increment(this.currentBuilder.add(tokenTerm, key, sstableRowId));
                }
            }
            finally {
                this.analyzer.end();
            }
        }
    }

    private boolean shouldFlush(long sstableRowId) {
        boolean reachMemoryLimit;
        boolean bl = reachMemoryLimit = this.limiter.usageExceedsLimit() && this.currentBuilder.hasReachedMinimumFlushSize();
        if (reachMemoryLimit) {
            logger.debug(this.index.identifier().logMessage("Global limit of {} and minimum flush size of {} exceeded. Current builder usage is {} for {} cells. Global Usage is {}. Flushing..."), new Object[]{FBUtilities.prettyPrintMemory(this.limiter.limitBytes()), FBUtilities.prettyPrintMemory(this.currentBuilder.getMinimumFlushBytes()), FBUtilities.prettyPrintMemory(this.currentBuilder.totalBytesAllocated()), this.currentBuilder.getRowCount(), FBUtilities.prettyPrintMemory(this.limiter.currentBytesUsed())});
        }
        return reachMemoryLimit || this.currentBuilder.exceedsSegmentLimit(sstableRowId);
    }

    private void flushSegment() throws IOException {
        long start = Clock.Global.nanoTime();
        try {
            long bytesAllocated = this.currentBuilder.totalBytesAllocated();
            SegmentMetadata segmentMetadata = this.currentBuilder.flush(this.indexDescriptor);
            long flushMillis = Math.max(1L, TimeUnit.NANOSECONDS.toMillis(Clock.Global.nanoTime() - start));
            if (segmentMetadata != null) {
                this.segments.add(segmentMetadata);
                double rowCount = segmentMetadata.numRows;
                this.index.indexMetrics().compactionSegmentCellsPerSecond.update((long)(rowCount / (double)flushMillis * 1000.0));
                double segmentBytes = segmentMetadata.componentMetadatas.indexSize();
                this.index.indexMetrics().compactionSegmentBytesPerSecond.update((long)(segmentBytes / (double)flushMillis * 1000.0));
                logger.debug(this.index.identifier().logMessage("Flushed segment with {} cells for a total of {} in {} ms."), new Object[]{(long)rowCount, FBUtilities.prettyPrintMemory((long)segmentBytes), flushMillis});
            }
            long globalBytesUsed = this.currentBuilder.release();
            this.currentBuilder = null;
            logger.debug(this.index.identifier().logMessage("Flushing index segment for SSTable {} released {}. Global segment memory usage now at {}."), new Object[]{this.indexDescriptor.sstableDescriptor, FBUtilities.prettyPrintMemory(bytesAllocated), FBUtilities.prettyPrintMemory(globalBytesUsed)});
        }
        catch (Throwable t) {
            logger.error(this.index.identifier().logMessage("Failed to build index for SSTable {}."), (Object)this.indexDescriptor.sstableDescriptor, (Object)t);
            this.indexDescriptor.deleteColumnIndex(this.index.termType(), this.index.identifier());
            this.index.indexMetrics().segmentFlushErrors.inc();
            throw t;
        }
    }

    private void writeSegmentsMetadata() throws IOException {
        if (this.segments.isEmpty()) {
            return;
        }
        try (MetadataWriter writer = new MetadataWriter(this.indexDescriptor.openPerIndexOutput(IndexComponent.META, this.index.identifier()));){
            SegmentMetadata.write(writer, this.segments);
        }
        catch (IOException e) {
            this.abort(e);
            throw e;
        }
    }

    private SegmentBuilder newSegmentBuilder() {
        SegmentBuilder builder = this.index.termType().isVector() ? new SegmentBuilder.VectorSegmentBuilder(this.index, this.limiter) : new SegmentBuilder.TrieSegmentBuilder(this.index, this.limiter);
        long globalBytesUsed = this.limiter.increment(builder.totalBytesAllocated());
        logger.debug(this.index.identifier().logMessage("Created new segment builder while flushing SSTable {}. Global segment memory usage now at {}."), (Object)this.indexDescriptor.sstableDescriptor, (Object)FBUtilities.prettyPrintMemory(globalBytesUsed));
        return builder;
    }
}

