/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSortedSet;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.concurrent.ExecutorFactory;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.PartitionPosition;
import org.apache.cassandra.dht.AbstractBounds;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.RequestExecutionException;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.SystemDistributedKeyspace;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.service.reads.range.RangeCommands;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.NoSpamLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PartitionDenylist {
    private static final Logger logger = LoggerFactory.getLogger(PartitionDenylist.class);
    private static final NoSpamLogger AVAILABILITY_LOGGER = NoSpamLogger.getLogger(logger, 1L, TimeUnit.MINUTES);
    private final ExecutorService executor = ExecutorFactory.Global.executorFactory().pooled("DenylistCache", 2);
    private volatile LoadingCache<TableId, DenylistEntry> denylist = this.buildEmptyCache();
    private int loadAttempts = 0;
    private int loadSuccesses = 0;

    public synchronized int getLoadAttempts() {
        return this.loadAttempts;
    }

    public synchronized int getLoadSuccesses() {
        return this.loadSuccesses;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void initialLoad() {
        if (!DatabaseDescriptor.getPartitionDenylistEnabled()) {
            return;
        }
        PartitionDenylist partitionDenylist = this;
        synchronized (partitionDenylist) {
            ++this.loadAttempts;
        }
        String retryReason = "Insufficient nodes";
        try {
            if (this.checkDenylistNodeAvailability()) {
                this.load();
                return;
            }
        }
        catch (Throwable tr) {
            logger.error("Failed to load partition denylist", tr);
            retryReason = "Exception";
        }
        int retryInSeconds = DatabaseDescriptor.getDenylistInitialLoadRetrySeconds();
        logger.info("{} while loading partition denylist cache. Scheduled retry in {} seconds.", (Object)retryReason, (Object)retryInSeconds);
        ScheduledExecutors.optionalTasks.schedule(this::initialLoad, (long)retryInSeconds, TimeUnit.SECONDS);
    }

    private boolean checkDenylistNodeAvailability() {
        boolean sufficientNodes = RangeCommands.sufficientLiveNodesForSelectStar(SystemDistributedKeyspace.PartitionDenylistTable, DatabaseDescriptor.getDenylistConsistencyLevel());
        if (!sufficientNodes) {
            AVAILABILITY_LOGGER.warn("Attempting to load denylist and not enough nodes are available for a {} refresh. Reload the denylist when unavailable nodes are recovered to ensure your denylist remains in sync.", new Object[]{DatabaseDescriptor.getDenylistConsistencyLevel()});
        }
        return sufficientNodes;
    }

    private LoadingCache<TableId, DenylistEntry> buildEmptyCache() {
        return Caffeine.newBuilder().refreshAfterWrite((long)DatabaseDescriptor.getDenylistRefreshSeconds(), TimeUnit.SECONDS).executor((Executor)this.executor).build((CacheLoader)new CacheLoader<TableId, DenylistEntry>(){

            public DenylistEntry load(TableId tid) {
                PartitionDenylist.this.checkDenylistNodeAvailability();
                return PartitionDenylist.this.getDenylistForTableFromCQL(tid);
            }

            public DenylistEntry reload(TableId tid, DenylistEntry oldValue) {
                DenylistEntry newEntry;
                if (PartitionDenylist.this.checkDenylistNodeAvailability() && (newEntry = PartitionDenylist.this.getDenylistForTableFromCQL(tid)) != null) {
                    return newEntry;
                }
                if (oldValue != null) {
                    return oldValue;
                }
                return new DenylistEntry();
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void load() {
        long start = Clock.Global.currentTimeMillis();
        Map<TableId, DenylistEntry> allDenylists = this.getDenylistForAllTablesFromCQL();
        LoadingCache<TableId, DenylistEntry> newDenylist = this.buildEmptyCache();
        newDenylist.putAll(allDenylists);
        PartitionDenylist partitionDenylist = this;
        synchronized (partitionDenylist) {
            ++this.loadSuccesses;
        }
        this.denylist = newDenylist;
        logger.info("Loaded partition denylist cache in {}ms", (Object)(Clock.Global.currentTimeMillis() - start));
    }

    public boolean addKeyToDenylist(String keyspace, String table, ByteBuffer key) {
        if (!this.canDenylistKeyspace(keyspace)) {
            return false;
        }
        String insert = String.format("INSERT INTO system_distributed.partition_denylist (ks_name, table_name, key) VALUES ('%s', '%s', 0x%s)", keyspace, table, ByteBufferUtil.bytesToHex(key));
        try {
            QueryProcessor.process(insert, DatabaseDescriptor.getDenylistConsistencyLevel());
            return this.refreshTableDenylist(keyspace, table);
        }
        catch (RequestExecutionException e) {
            logger.error("Failed to denylist key [{}] in {}/{}", new Object[]{ByteBufferUtil.bytesToHex(key), keyspace, table, e});
            return false;
        }
    }

    public boolean removeKeyFromDenylist(String keyspace, String table, ByteBuffer key) {
        String delete = String.format("DELETE FROM system_distributed.partition_denylist WHERE ks_name = '%s' AND table_name = '%s' AND key = 0x%s", keyspace, table, ByteBufferUtil.bytesToHex(key));
        try {
            QueryProcessor.process(delete, DatabaseDescriptor.getDenylistConsistencyLevel());
            return this.refreshTableDenylist(keyspace, table);
        }
        catch (RequestExecutionException e) {
            logger.error("Failed to remove key from denylist: [{}] in {}/{}", new Object[]{ByteBufferUtil.bytesToHex(key), keyspace, table, e});
            return false;
        }
    }

    private boolean canDenylistKeyspace(String keyspace) {
        return !"system_distributed".equals(keyspace) && !"system".equals(keyspace) && !"system_traces".equals(keyspace) && !"system_virtual_schema".equals(keyspace) && !"system_views".equals(keyspace) && !"system_auth".equals(keyspace);
    }

    public boolean isKeyPermitted(String keyspace, String table, ByteBuffer key) {
        return this.isKeyPermitted(this.getTableId(keyspace, table), key);
    }

    public boolean isKeyPermitted(TableId tid, ByteBuffer key) {
        TableMetadata tmd = Schema.instance.getTableMetadata(tid);
        if (!DatabaseDescriptor.getPartitionDenylistEnabled() || tid == null || tmd == null || !this.canDenylistKeyspace(tmd.keyspace)) {
            return true;
        }
        try {
            DenylistEntry entry = (DenylistEntry)this.denylist.get((Object)tid);
            if (entry == null) {
                return true;
            }
            return !entry.keys.contains((Object)key);
        }
        catch (Exception e) {
            this.logAccessFailure(tid, e);
            return true;
        }
    }

    private void logAccessFailure(TableId tid, Throwable e) {
        TableMetadata tmd = Schema.instance.getTableMetadata(tid);
        if (tmd == null) {
            logger.debug("Failed to access partition denylist cache for unknown table id {}", (Object)tid.toString(), (Object)e);
        } else {
            logger.debug("Failed to access partition denylist cache for {}/{}", new Object[]{tmd.keyspace, tmd.name, e});
        }
    }

    public int getDeniedKeysInRangeCount(String keyspace, String table, AbstractBounds<PartitionPosition> range) {
        return this.getDeniedKeysInRangeCount(this.getTableId(keyspace, table), range);
    }

    public int getDeniedKeysInRangeCount(TableId tid, AbstractBounds<PartitionPosition> range) {
        TableMetadata tmd = Schema.instance.getTableMetadata(tid);
        if (!DatabaseDescriptor.getPartitionDenylistEnabled() || tid == null || tmd == null || !this.canDenylistKeyspace(tmd.keyspace)) {
            return 0;
        }
        try {
            Token endToken;
            DenylistEntry denylistEntry = (DenylistEntry)this.denylist.get((Object)tid);
            if (denylistEntry == null || denylistEntry.tokens.size() == 0) {
                return 0;
            }
            Token startToken = ((PartitionPosition)range.left).getToken();
            if (startToken.compareTo(endToken = ((PartitionPosition)range.right).getToken()) <= 0 || endToken.isMinimum()) {
                Object subSet = denylistEntry.tokens.tailSet((Object)startToken, PartitionPosition.Kind.MIN_BOUND == ((PartitionPosition)range.left).kind());
                if (!endToken.isMinimum()) {
                    subSet = subSet.headSet(endToken, PartitionPosition.Kind.MAX_BOUND == ((PartitionPosition)range.right).kind());
                }
                return subSet.size();
            }
            return denylistEntry.tokens.tailSet((Object)startToken, PartitionPosition.Kind.MIN_BOUND == ((PartitionPosition)range.left).kind()).size() + denylistEntry.tokens.headSet((Object)endToken, PartitionPosition.Kind.MAX_BOUND == ((PartitionPosition)range.right).kind()).size();
        }
        catch (Exception e) {
            this.logAccessFailure(tid, e);
            return 0;
        }
    }

    private DenylistEntry getDenylistForTableFromCQL(TableId tid) {
        return this.getDenylistForTableFromCQL(tid, DatabaseDescriptor.getDenylistMaxKeysPerTable());
    }

    private DenylistEntry getDenylistForTableFromCQL(TableId tid, int limit) {
        TableMetadata tmd = Schema.instance.getTableMetadata(tid);
        if (tmd == null) {
            return null;
        }
        String readDenylist = String.format("SELECT * FROM %s.%s WHERE ks_name='%s' AND table_name='%s' LIMIT %d", "system_distributed", "partition_denylist", tmd.keyspace, tmd.name, limit + 1);
        try {
            UntypedResultSet results = QueryProcessor.process(readDenylist, DatabaseDescriptor.getDenylistConsistencyLevel());
            if (results == null || results.isEmpty()) {
                return new DenylistEntry();
            }
            if (results.size() > limit) {
                boolean globalLimit = limit != DatabaseDescriptor.getDenylistMaxKeysPerTable();
                String violationType = globalLimit ? "global" : "per-table";
                int errorLimit = globalLimit ? DatabaseDescriptor.getDenylistMaxKeysTotal() : limit;
                logger.error("Partition denylist for {}/{} has exceeded the {} allowance of ({}). Remaining keys were ignored; please reduce the total number of keys denied or increase the denylist_max_keys_per_table param in cassandra.yaml to avoid inconsistency in denied partitions across nodes.", new Object[]{tmd.keyspace, tmd.name, violationType, errorLimit});
            }
            HashSet<ByteBuffer> keys = new HashSet<ByteBuffer>();
            TreeSet<Token> tokens = new TreeSet<Token>();
            int processed = 0;
            for (UntypedResultSet.Row row : results) {
                ByteBuffer key = row.getBlob("key");
                keys.add(key);
                tokens.add(StorageService.instance.getTokenMetadata().partitioner.getToken(key));
                if (++processed < limit) continue;
                break;
            }
            return new DenylistEntry((ImmutableSet<ByteBuffer>)ImmutableSet.copyOf(keys), (ImmutableSortedSet<Token>)ImmutableSortedSet.copyOf(tokens));
        }
        catch (RequestExecutionException e) {
            logger.error("Error reading partition_denylist table for {}/{}. Returning empty denylist.", new Object[]{tmd.keyspace, tmd.name, e});
            return new DenylistEntry();
        }
    }

    private Map<TableId, DenylistEntry> getDenylistForAllTablesFromCQL() {
        this.checkDenylistNodeAvailability();
        String allDeniedTables = String.format("SELECT DISTINCT ks_name, table_name FROM %s.%s", "system_distributed", "partition_denylist");
        try {
            UntypedResultSet deniedTableResults = QueryProcessor.process(allDeniedTables, DatabaseDescriptor.getDenylistConsistencyLevel());
            if (deniedTableResults == null || deniedTableResults.isEmpty()) {
                return Collections.emptyMap();
            }
            int totalProcessed = 0;
            HashMap<TableId, DenylistEntry> results = new HashMap<TableId, DenylistEntry>();
            for (UntypedResultSet.Row row : deniedTableResults) {
                String ks = row.getString("ks_name");
                String table = row.getString("table_name");
                TableId tid = this.getTableId(ks, table);
                if (DatabaseDescriptor.getDenylistMaxKeysTotal() - totalProcessed <= 0) {
                    logger.error("Hit limit on allowable denylisted keys in total. Processed {} total entries. Not adding all entries to denylist for {}/{}. Remove denylist entries in system_distributed.{} or increase your denylist_max_keys_total param in cassandra.yaml.", new Object[]{totalProcessed, ks, table, "partition_denylist"});
                    results.put(tid, new DenylistEntry());
                    continue;
                }
                int allowedTableRecords = Math.min(DatabaseDescriptor.getDenylistMaxKeysPerTable(), DatabaseDescriptor.getDenylistMaxKeysTotal() - totalProcessed);
                DenylistEntry tableDenylist = this.getDenylistForTableFromCQL(tid, allowedTableRecords);
                if (tableDenylist != null) {
                    totalProcessed += tableDenylist.keys.size();
                }
                results.put(tid, tableDenylist);
            }
            return results;
        }
        catch (RequestExecutionException e) {
            logger.error("Error reading full partition denylist from system_distributed.partition_denylist. Partition Denylisting will be compromised. Exception: " + e);
            return Collections.emptyMap();
        }
    }

    private boolean refreshTableDenylist(String keyspace, String table) {
        this.checkDenylistNodeAvailability();
        TableId tid = this.getTableId(keyspace, table);
        if (tid == null) {
            logger.warn("Got denylist mutation for unknown ks/cf: {}/{}. Skipping refresh.", (Object)keyspace, (Object)table);
            return false;
        }
        DenylistEntry newEntry = this.getDenylistForTableFromCQL(tid);
        this.denylist.put((Object)tid, (Object)newEntry);
        return true;
    }

    private TableId getTableId(String keyspace, String table) {
        TableMetadata tmd = Schema.instance.getTableMetadata(keyspace, table);
        return tmd == null ? null : tmd.id;
    }

    private static class DenylistEntry {
        public final ImmutableSet<ByteBuffer> keys;
        public final ImmutableSortedSet<Token> tokens;

        public DenylistEntry() {
            this.keys = ImmutableSet.of();
            this.tokens = ImmutableSortedSet.of();
        }

        public DenylistEntry(ImmutableSet<ByteBuffer> keys, ImmutableSortedSet<Token> tokens) {
            this.keys = keys;
            this.tokens = tokens;
        }
    }
}

