/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.pipeline.transforms.csvinput;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import org.apache.hop.core.file.TextFileInputField;
import org.apache.hop.junit.rules.RestoreHopEngineEnvironmentExtension;
import org.apache.hop.pipeline.PipelineTestingUtil;
import org.apache.hop.pipeline.transforms.mock.TransformMockHelper;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

class CsvInputRowNumberTest extends CsvInputUnitTestBase {
  @RegisterExtension
  static RestoreHopEngineEnvironmentExtension env = new RestoreHopEngineEnvironmentExtension();

  private CsvInput csvInput;
  private TransformMockHelper<CsvInputMeta, CsvInputData> transformMockHelper;

  @BeforeEach
  void setUp() throws Exception {
    transformMockHelper =
        TransformMockUtil.getTransformMockHelper(
            CsvInputMeta.class, CsvInputData.class, "CsvInputRowNumberTest");
  }

  @AfterEach
  void cleanUp() {
    transformMockHelper.cleanUp();
  }

  @Test
  void hasNotEnclosures_HasNotNewLine() throws Exception {
    File tmp = createTestFile("utf-8", "a,b\na,");
    try {
      doTest(tmp);
    } finally {
      tmp.delete();
    }
  }

  public void doTest(File file) throws Exception {
    CsvInputData data = new CsvInputData();
    CsvInputMeta meta = createMeta(file, createInputFileFields("a", "b"));
    csvInput =
        new CsvInput(
            transformMockHelper.transformMeta,
            meta,
            data,
            0,
            transformMockHelper.pipelineMeta,
            transformMockHelper.pipeline);

    List<Object[]> actual;
    try {
      csvInput.init();
      actual = PipelineTestingUtil.execute(csvInput, 2, false);
    } finally {
      csvInput.dispose();
    }

    List<Object[]> expected =
        Arrays.asList(new Object[] {"a", "b", 1L}, new Object[] {"a", null, 2L});
    PipelineTestingUtil.assertResult(expected, actual);
  }

  @Override
  CsvInputMeta createMeta(File file, TextFileInputField[] fields) {
    CsvInputMeta meta = super.createMeta(file, fields);
    meta.setRowNumField("rownum");
    return meta;
  }
}
