/*
 * COPYRIGHT (c) International Business Machines Corp. 2021-2024
 *
 * This program is provided under the terms of the Common Public License,
 * version 1.0 (CPL-1.0). Any use, reproduction or distribution for this
 * software constitutes recipient's acceptance of CPL-1.0 terms which can be
 * found in the file LICENSE file or at
 * https://opensource.org/licenses/cpl1.0.php
 */

#define _DEFAULT_SOURCE

#include <string.h>
#include <errno.h>

#include "names.h"
#include "utils.h"

static const struct kmip_enum kmip_tags[] = {
	{ .val = KMIP_TAG_ACTIVATION_DATE, .name = "ActivationDate" },
	{ .val = KMIP_TAG_APPLICATION_DATA, .name = "ApplicationData" },
	{ .val = KMIP_TAG_APPLICATION_NAMESPACE,
			.name = "ApplicationNamespace" },
	{ .val = KMIP_TAG_APPLICATION_SPECIFIC_INFORMATION,
			.name = "ApplicationSpecificInformation" },
	{ .val = KMIP_TAG_ARCHIVE_DATE, .name = "ArchiveDate" },
	{ .val = KMIP_TAG_ASYNCHRONOUS_CORRELATION_VALUE,
			.name = "AsynchronousCorrelationValue" },
	{ .val = KMIP_TAG_ASYNCHRONOUS_INDICATOR,
			.name = "AsynchronousIndicator" },
	{ .val = KMIP_TAG_ATTRIBUTE, .name = "Attribute" },
	{ .val = KMIP_TAG_ATTRIBUTE_INDEX, .name = "AttributeIndex" },
	{ .val = KMIP_TAG_ATTRIBUTE_NAME, .name = "AttributeName" },
	{ .val = KMIP_TAG_ATTRIBUTE_VALUE, .name = "AttributeValue" },
	{ .val = KMIP_TAG_AUTHENTICATION, .name = "Authentication" },
	{ .val = KMIP_TAG_BATCH_COUNT, .name = "BatchCount" },
	{ .val = KMIP_TAG_BATCH_ERROR_CONTINUATION_OPTION,
			.name = "BatchErrorContinuationOption" },
	{ .val = KMIP_TAG_BATCH_ITEM, .name = "BatchItem" },
	{ .val = KMIP_TAG_BATCH_ORDER_OPTION, .name = "BatchOrderOption" },
	{ .val = KMIP_TAG_BLOCK_CIPHER_MODE, .name = "BlockCipherMode" },
	{ .val = KMIP_TAG_CANCELATION_RESULT, .name = "CancelationResult" },
	{ .val = KMIP_TAG_CERTIFICATE, .name = "Certificate" },
	{ .val = KMIP_TAG_CERTIFICATE_IDENTIFIER,
			.name = "CertificateIndentifyer" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER, .name = "CertificateIssuer" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_ALTERNATIVE_NAME,
			.name = "CertificateIssuerAlternativeName" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_DISTINGUISHED_NAME,
			.name = "CertificateIssuerDistinguishedName" },
	{ .val = KMIP_TAG_CERTIFICATE_REQUEST,
			.name = "CertificateRequest" },
	{ .val = KMIP_TAG_CERTIFICATE_REQUEST_TYPE,
			.name = "CertificateRequestType" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT, .name = "CertificateSubject" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_ALTERNATIVE_NAME,
			.name = "CertificateSubjectAlternativeName" },
	{ .val = KMIP_TAG_CERTIFICATE_TYPE, .name = "CertificateType" },
	{ .val = KMIP_TAG_CERTIFICATE_VALUE, .name = "CertificateValue" },
	{ .val = KMIP_TAG_COMMON_TEMPLATE_ATTRIBUTE,
			.name = "CommonTemplateAttribute" },
	{ .val = KMIP_TAG_COMPROMIZE_DATE, .name = "CompromizeDate" },
	{ .val = KMIP_TAG_COMPROMISE_OCCURRENCE_DATE,
			.name = "CompromiseOccurrenceDate" },
	{ .val = KMIP_TAG_CONTACT_INFORMATION, .name = "ContactInformation" },
	{ .val = KMIP_TAG_CREDENTIAL, .name = "Credential" },
	{ .val = KMIP_TAG_CREDENTIAL_TYPE, .name = "CredentialType" },
	{ .val = KMIP_TAG_CREDENTIAL_VALUE, .name = "CredentialValue" },
	{ .val = KMIP_TAG_CRITICALITY_INDICATOR,
			.name = "CriticalityIndicator" },
	{ .val = KMIP_TAG_CRT_Coefficient, .name = "CrtCoefficient" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_ALGORITHM,
			.name = "CryptographicAlgorithm" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_DOMAIN_PARAMETERS,
			.name = "CryptographicDomainParameters" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_LENGTH, .name = "CryptographicLength" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_PARAMETERS,
			.name = "CryptographicParameters" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_USAGE_MASK,
			.name = "CryptographicUsageMask" },
	{ .val = KMIP_TAG_CUSTOM_ATTRIBUTE, .name = "CustomAttribute" },
	{ .val = KMIP_TAG_D, .name = "D" },
	{ .val = KMIP_TAG_DEACTIVATION_DATE, .name = "DeactivationDate" },
	{ .val = KMIP_TAG_DERIVATION_DATE, .name = "DerivationDate" },
	{ .val = KMIP_TAG_DERIVATION_DATA, .name = "DerivationData" },
	{ .val = KMIP_TAG_DERIVATION_PARAMETERS,
			.name = "DerivationParameters" },
	{ .val = KMIP_TAG_DESTROY_DATE, .name = "DestroyDate" },
	{ .val = KMIP_TAG_DIGEST, .name = "Digest" },
	{ .val = KMIP_TAG_DIGEST_VALUE, .name = "DigestValue" },
	{ .val = KMIP_TAG_ENCRYPTION_KEY_INFORMATION,
			.name = "EncryptionKeyInformation" },
	{ .val = KMIP_TAG_G, .name = "G" },
	{ .val = KMIP_TAG_HASHING_ALGORITHM, .name = "HashingAlgorithm" },
	{ .val = KMIP_TAG_INITIAL_DATE, .name = "InitialDate" },
	{ .val = KMIP_TAG_INITIALIZATION_VECTOR,
			.name = "InitializationVector" },
	{ .val = KMIP_TAG_ISSUER, .name = "Issuer" },
	{ .val = KMIP_TAG_ITERATION_COUNT, .name = "IterationCount" },
	{ .val = KMIP_TAG_IV_COUNTER_NONCE, .name = "IvCounterNonce" },
	{ .val = KMIP_TAG_J, .name = "J" },
	{ .val = KMIP_TAG_KEY, .name = "Key" },
	{ .val = KMIP_TAG_KEY_BLOCK, .name = "KeyBlock" },
	{ .val = KMIP_TAG_KEY_COMPRESSION_TYPE, .name = "KeyCompressionType" },
	{ .val = KMIP_TAG_KEY_FORMAT_TYPE, .name = "KeyFormatType" },
	{ .val = KMIP_TAG_KEY_MATERIAL, .name = "KeyMaterial" },
	{ .val = KMIP_TAG_KEY_PART_IDENTIFIER, .name = "KeyPartIdentifier" },
	{ .val = KMIP_TAG_KEY_VALUE, .name = "KeyValue" },
	{ .val = KMIP_TAG_KEY_WRAPPING_DATA, .name = "KeyWrappingData" },
	{ .val = KMIP_TAG_KEY_WRAPPING_SPECIFICATION,
			.name = "KeyWrappingSpecification" },
	{ .val = KMIP_TAG_LAST_CHANGE_DATE, .name = "LastChangeDate" },
	{ .val = KMIP_TAG_LEASE_TIME, .name = "LeaseTime" },
	{ .val = KMIP_TAG_LINK, .name = "Link" },
	{ .val = KMIP_TAG_LINK_TYPE, .name = "LinkType" },
	{ .val = KMIP_TAG_LINKED_OBJECT_IDENTIFIER,
			.name = "LinkedObjectIdentifier" },
	{ .val = KMIP_TAG_MAC_SIGNATURE, .name = "MACSignature" },
	{ .val = KMIP_TAG_MAC_SIGNATURE_KEY_INFORMATION,
			.name = "MACSignatureKeyInformation" },
	{ .val = KMIP_TAG_MAXIMUM_ITEMS, .name = "MaximumItems" },
	{ .val = KMIP_TAG_MAXIMUM_RESPONSE_SIZE,
			.name = "MaximumResponseSize" },
	{ .val = KMIP_TAG_MESSAGE_EXTENSION, .name = "MessageExtension" },
	{ .val = KMIP_TAG_MODULUS, .name = "Modulus" },
	{ .val = KMIP_TAG_NAME, .name = "Name" },
	{ .val = KMIP_TAG_NAME_TYPE, .name = "NameType" },
	{ .val = KMIP_TAG_NAME_VALUE, .name = "NameValue" },
	{ .val = KMIP_TAG_OBJECT_GROUP, .name = "ObjectGroup" },
	{ .val = KMIP_TAG_OBJECT_TYPE, .name = "ObjectType" },
	{ .val = KMIP_TAG_OFFSET, .name = "Offset" },
	{ .val = KMIP_TAG_OPAQUE_DATA_TYPE, .name = "OpaqueDataType" },
	{ .val = KMIP_TAG_OPAQUE_DATA_VALUE, .name = "OpaqueDataValue" },
	{ .val = KMIP_TAG_OPAQUE_OBJECT, .name = "OpaqueObject" },
	{ .val = KMIP_TAG_OPERATION, .name = "Operation" },
	{ .val = KMIP_TAG_OPERATION_POLICY_NAME,
			.name = "OperationPolicyName" },
	{ .val = KMIP_TAG_P, .name = "P" },
	{ .val = KMIP_TAG_PADDING_METHOD, .name = "PaddingMethod" },
	{ .val = KMIP_TAG_PRIME_EXPONENT_P, .name = "PrimeExponentP" },
	{ .val = KMIP_TAG_PRIME_EXPONENT_Q, .name = "PrimeExponentQ" },
	{ .val = KMIP_TAG_PRIME_FIELD_SIZE, .name = "PrimeFieldSize" },
	{ .val = KMIP_TAG_PRIVATE_EXPONENT, .name = "PrivateExponent" },
	{ .val = KMIP_TAG_PRIVATE_KEY, .name = "PrivateKey" },
	{ .val = KMIP_TAG_PRIVATE_KEY_TEMPLATE_ATTRIBUTE,
			.name = "PrivateKeyTemplateAttribute" },
	{ .val = KMIP_TAG_PRIVATE_KEY_UNIQUE_IDENTIFIER,
			.name = "PrivateKeyUniqueIdentifier" },
	{ .val = KMIP_TAG_PROCESS_START_DATE, .name = "ProcessStartDate" },
	{ .val = KMIP_TAG_PROTECT_STOP_DATE, .name = "ProtectStopDate" },
	{ .val = KMIP_TAG_PROTOCOL_VERSION, .name = "ProtocolVersion" },
	{ .val = KMIP_TAG_PROTOCOL_VERSION_MAJOR,
			.name = "ProtocolVersionMajor" },
	{ .val = KMIP_TAG_PROTOCOL_VERSION_MINOR,
			.name = "ProtocolVersionMinor" },
	{ .val = KMIP_TAG_PUBLIC_EXPONENT, .name = "PublicExponent" },
	{ .val = KMIP_TAG_PUBLIC_KEY, .name = "PublicKey" },
	{ .val = KMIP_TAG_PUBLIC_KEY_TEMPLATE_ATTRIBUTE,
			.name = "PublicKeyTemplateAttribute" },
	{ .val = KMIP_TAG_PUBLIC_KEY_UNIQUE_IDENTIFIER,
			.name = "PublicKeyUniqueIdentifier" },
	{ .val = KMIP_TAG_PUT_FUNCTION, .name = "PutFunction" },
	{ .val = KMIP_TAG_Q, .name = "Q" },
	{ .val = KMIP_TAG_Q_STRING, .name = "QString" },
	{ .val = KMIP_TAG_Q_LENGTH, .name = "QLength" },
	{ .val = KMIP_TAG_QUERY_FUNCTION, .name = "QueryFunction" },
	{ .val = KMIP_TAG_RECOMMENDED_CURVE, .name = "RecommendedCurve" },
	{ .val = KMIP_TAG_REPLACED_UNIQUE_IDENTIFIER,
			.name = "ReplacedUniqueIdentifier" },
	{ .val = KMIP_TAG_REQUEST_HEADER, .name = "RequestHeader" },
	{ .val = KMIP_TAG_REQUEST_MESSAGE, .name = "RequestMessage" },
	{ .val = KMIP_TAG_REQUEST_PAYLOAD, .name = "RequestPayload" },
	{ .val = KMIP_TAG_RESPONSE_HEADER, .name = "ResponseHeader" },
	{ .val = KMIP_TAG_RESPONSE_MESSAGE, .name = "ResponseMessage" },
	{ .val = KMIP_TAG_RESPONSE_PAYLOAD, .name = "ResponsePayload" },
	{ .val = KMIP_TAG_RESULT_MESSAGE, .name = "ResultMessage" },
	{ .val = KMIP_TAG_RESULT_REASON, .name = "ResultReason" },
	{ .val = KMIP_TAG_RESULT_STATUS, .name = "ResultStatus" },
	{ .val = KMIP_TAG_REVOCATION_MESSAGE, .name = "RevocationMessage" },
	{ .val = KMIP_TAG_REVOCATION_REASON, .name = "RevocationReason" },
	{ .val = KMIP_TAG_REVOCATION_REASON_CODE,
			.name = "RevocationReasonCode" },
	{ .val = KMIP_TAG_KEY_ROLE_TYPE, .name = "KeyRoleType" },
	{ .val = KMIP_TAG_SALT, .name = "Salt" },
	{ .val = KMIP_TAG_SECRET_DATA, .name = "SecretData" },
	{ .val = KMIP_TAG_SECRET_DATA_TYPE, .name = "SecretDataType" },
	{ .val = KMIP_TAG_SERIAL_NUMBER, .name = "SerialNumber" },
	{ .val = KMIP_TAG_SERVER_INFORMATION, .name = "ServerInformation" },
	{ .val = KMIP_TAG_SPLIT_KEY, .name = "SplitKey" },
	{ .val = KMIP_TAG_SPLIT_KEY_METHOD, .name = "SplitKeyMethod" },
	{ .val = KMIP_TAG_SPLIT_KEY_PARTS, .name = "SplitKeyParts" },
	{ .val = KMIP_TAG_SPLIT_KEY_THRESHOLD, .name = "SplitKeyThreshold" },
	{ .val = KMIP_TAG_STATE, .name = "State" },
	{ .val = KMIP_TAG_STORAGE_STATUS_MASK, .name = "StorageStatusMask" },
	{ .val = KMIP_TAG_SYMMETRIC_KEY, .name = "SymmetricKey" },
	{ .val = KMIP_TAG_TEMPLATE, .name = "Template" },
	{ .val = KMIP_TAG_TEMPLATE_ATTRIBUTE, .name = "TemplateAttribute" },
	{ .val = KMIP_TAG_TIME_STAMP, .name = "TimeStamp" },
	{ .val = KMIP_TAG_UNIQUE_BATCH_ITEM_ID, .name = "UniqueBatchItemId" },
	{ .val = KMIP_TAG_UNIQUE_IDENTIFIER, .name = "UniqueIdentifier" },
	{ .val = KMIP_TAG_USAGE_LIMITS, .name = "UsageLimits" },
	{ .val = KMIP_TAG_USAGE_LIMITS_COUNT, .name = "UsageLimitsCount" },
	{ .val = KMIP_TAG_USAGE_LIMITS_TOTAL, .name = "UsageLimitsTotal" },
	{ .val = KMIP_TAG_USAGE_LIMITS_UNIT, .name = "UsageLimitsUnit" },
	{ .val = KMIP_TAG_USERNAME, .name = "Username" },
	{ .val = KMIP_TAG_VALIDITY_DATE, .name = "ValidityDate" },
	{ .val = KMIP_TAG_VALIDITY_INDICATOR, .name = "ValidityIndicator" },
	{ .val = KMIP_TAG_VENDOR_EXTENSION, .name = "VendorExtension" },
	{ .val = KMIP_TAG_VENDOR_IDENTIFICATION,
			.name = "VendorIdentification" },
	{ .val = KMIP_TAG_WRAPPING_METHOD, .name = "WrappingMethod" },
	{ .val = KMIP_TAG_X, .name = "X" },
	{ .val = KMIP_TAG_Y, .name = "Y" },
	{ .val = KMIP_TAG_PASSWORD, .name = "Password" },
	{ .val = KMIP_TAG_DEVICE_IDENTIFIER, .name = "DeviceIdentifier" },
	{ .val = KMIP_TAG_ENCODING_OPTION, .name = "EncodingOption" },
	{ .val = KMIP_TAG_EXTENSION_INFORMATION,
			.name = "ExtensionInformation" },
	{ .val = KMIP_TAG_EXTENSION_NAME, .name = "ExtensionName" },
	{ .val = KMIP_TAG_EXTENSION_TAG, .name = "ExtensionTag" },
	{ .val = KMIP_TAG_EXTENSION_TYPE, .name = "ExtensionType" },
	{ .val = KMIP_TAG_FRESH, .name = "Fresh" },
	{ .val = KMIP_TAG_MACHINE_IDENTIFIER, .name = "MachineIdentifier" },
	{ .val = KMIP_TAG_MEDIA_IDENTIFIER, .name = "MediaIdentifier" },
	{ .val = KMIP_TAG_NETWORK_IDENTIFIER, .name = "NetworkIdentifier" },
	{ .val = KMIP_TAG_OBJECT_GROUP_MEMBER, .name = "ObjectGroupMember" },
	{ .val = KMIP_TAG_CERTIFICATE_LENGTH, .name = "CertificateLength" },
	{ .val = KMIP_TAG_DIGITAL_SIGNATURE_ALGORITHM,
			.name = "DigitalSignatureAlgorithm" },
	{ .val = KMIP_TAG_CERTIFICATE_SERIAL_NUMBER,
			.name = "CertificateSerialNumber" },
	{ .val = KMIP_TAG_DEVICE_SERIAL_NUMBER,
			.name = "DeviceSerialNumber" },
	{ .val = KMIP_TAG_ISSUER_ALTERNATE_NAME,
			.name = "IssuerAlternateName" },
	{ .val = KMIP_TAG_ISSUER_DISTINGUISHED_NAME,
			.name = "IssuerDistinguishedName" },
	{ .val = KMIP_TAG_SUBJECT_ALTERNATE_NAME,
			.name = "SubjectAlternateName" },
	{ .val = KMIP_TAG_SUBJECT_DISTINGUISHED_NAME,
			.name = "SubjectDistinguishedName" },
	{ .val = KMIP_TAG_X_509_CERTIFICATE_IDENTIFIER,
			.name = "X_509CertificateIdentifier" },
	{ .val = KMIP_TAG_X_509_CERTIFICATE_ISSUER,
			.name = "X_509CertificateIssuer" },
	{ .val = KMIP_TAG_X_509_CERTIFICATE_SUBJECT,
			.name = "X_509CertificateSubject" },
	{ .val = KMIP_TAG_KEY_VALUE_LOCATION,
			.name = "KeyValueLocation" },
	{ .val = KMIP_TAG_KEY_VALUE_LOCATION_VALUE,
			.name = "KeyValueLocationValue" },
	{ .val = KMIP_TAG_KEY_VALUE_LOCATION_TYPE,
			.name = "KeyValueLocationType" },
	{ .val = KMIP_TAG_KEY_VALUE_PRESENT, .name = "KeyValuePresent" },
	{ .val = KMIP_TAG_ORIGINAL_CREATION_DATE,
			.name = "OriginalCreationDate" },
	{ .val = KMIP_TAG_PGP_KEY, .name = "PGPKey" },
	{ .val = KMIP_TAG_PGP_KEY_VERSION, .name = "PGPKeyVersion" },
	{ .val = KMIP_TAG_ALTERNATE_NAME, .name = "AlternateName" },
	{ .val = KMIP_TAG_ALTERNATE_NAME_VALUE, .name = "AlternateNameValue" },
	{ .val = KMIP_TAG_ALTERNATE_NAME_TYPE, .name = "AlternateNameType" },
	{ .val = KMIP_TAG_DATA, .name = "Data" },
	{ .val = KMIP_TAG_SIGNATURE_DATA, .name = "SignatureData" },
	{ .val = KMIP_TAG_DATA_LENGTH, .name = "DataLength" },
	{ .val = KMIP_TAG_RANDOM_IV, .name = "RandomIV" },
	{ .val = KMIP_TAG_MAC_DATA, .name = "MACData" },
	{ .val = KMIP_TAG_ATTESTATION_TYPE, .name = "AttestationType" },
	{ .val = KMIP_TAG_NONCE, .name = "Nonce" },
	{ .val = KMIP_TAG_NONCE_ID, .name = "NonceId" },
	{ .val = KMIP_TAG_NONCE_VALUE, .name = "NonceValue" },
	{ .val = KMIP_TAG_ATTESTATION_MEASUREMENT,
			.name = "AttestationMeasurement" },
	{ .val = KMIP_TAG_ATTESTATION_ASSERTION,
			.name = "AttestationAssertion" },
	{ .val = KMIP_TAG_IV_LENGTH, .name = "IVLength" },
	{ .val = KMIP_TAG_TAG_LENGTH, .name = "TagLength" },
	{ .val = KMIP_TAG_FIXED_FIELD_LENGTH, .name = "FixedFieldLength" },
	{ .val = KMIP_TAG_COUNTER_LENGTH,  .name = "CounterLength" },
	{ .val = KMIP_TAG_INITIAL_COUNTER_VALUE,
			.name = "InitialCounterValue" },
	{ .val = KMIP_TAG_INVOCATION_FIELD_LENGTH,
			.name = "InvocationFieldLength" },
	{ .val = KMIP_TAG_ATTESTATION_CAPABLE_INDICATOR,
			.name = "AttestationCapableIndicator" },
	{ .val = KMIP_TAG_OFFSET_ITEMS, .name = "OffsetItems" },
	{ .val = KMIP_TAG_LOCATED_ITEMS, .name = "LocatedItems" },
	{ .val = KMIP_TAG_CORRELATION_VALUE, .name = "CorrelationValue" },
	{ .val = KMIP_TAG_INIT_INDICATOR, .name = "InitIndicator" },
	{ .val = KMIP_TAG_FINAL_INDICATOR, .name = "FinalIndicator" },
	{ .val = KMIP_TAG_RNG_PARAMETERS, .name = "RNGParameters" },
	{ .val = KMIP_TAG_RNG_ALGORITHM, .name = "RNGAlgorithm" },
	{ .val = KMIP_TAG_DRBG_ALGORITHM, .name = "DRBGAlgorithm" },
	{ .val = KMIP_TAG_FIPS186_VARIANT, .name = "Fips186Variant" },
	{ .val = KMIP_TAG_PREDICTION_RESISTANCE,
			.name = "PredictionResistance" },
	{ .val = KMIP_TAG_RANDOM_NUMBER_GENERATOR,
			.name = "RandomNumberGenerator" },
	{ .val = KMIP_TAG_VALIDATION_INFORMATION,
			.name = "ValidationInformation" },
	{ .val = KMIP_TAG_VALIDATION_AUTHORITY_TYPE,
			.name = "ValidationAuthorityType" },
	{ .val = KMIP_TAG_VALIDATION_AUTHORITY_COUNTRY,
			.name = "ValidationAuthorityCountry" },
	{ .val = KMIP_TAG_VALIDATION_AUTHORITY_URI,
			.name = "ValidationAuthorityURI" },
	{ .val = KMIP_TAG_VALIDATION_VERSION_MAJOR,
			.name = "ValidationVersionMajor" },
	{ .val = KMIP_TAG_VALIDATION_VERSION_MINOR,
			.name = "ValidationVersionMinor" },
	{ .val = KMIP_TAG_VALIDATION_TYPE, .name = "ValidationType" },
	{ .val = KMIP_TAG_VALIDATION_LEVEL, .name = "ValidationLevel" },
	{ .val = KMIP_TAG_VALIDATION_CERTIFICATE_IDENTIFIER,
			.name = "ValidationCertificateIdentifier" },
	{ .val = KMIP_TAG_VALIDATION_CERTIFICATE_URI,
			.name = "ValidationCertificateURI" },
	{ .val = KMIP_TAG_VALIDATION_VENDOR_URI,
			.name = "ValidationVendorURI" },
	{ .val = KMIP_TAG_VALIDATION_PROFILE, .name = "ValidationProfile" },
	{ .val = KMIP_TAG_PROFILE_INFORMATION, .name = "ProfileInformation" },
	{ .val = KMIP_TAG_PROFILE_NAME, .name = "ProfileName" },
	{ .val = KMIP_TAG_SERVER_URI, .name = "ServerURI" },
	{ .val = KMIP_TAG_SERVER_PORT, .name = "ServerPort" },
	{ .val = KMIP_TAG_STREAMING_CAPABILITY, .name = "StreamingCapability" },
	{ .val = KMIP_TAG_ASYNCHRONOUS_CAPABILITY,
			.name = "AsynchronousCapability" },
	{ .val = KMIP_TAG_ATTESTATION_CAPABILITY,
			.name = "AttestationCapability" },
	{ .val = KMIP_TAG_UNWRAP_MODE, .name = "UnwrapMode" },
	{ .val = KMIP_TAG_DESTROY_ACTION, .name = "DestroyAction" },
	{ .val = KMIP_TAG_SHREDDING_ALGORITHM, .name = "ShreddingAlgorithm" },
	{ .val = KMIP_TAG_RNG_MODE, .name = "RNGMode" },
	{ .val = KMIP_TAG_CLIENT_REGISTRATION_METHOD,
			.name = "ClientRegistrationMethod" },
	{ .val = KMIP_TAG_CAPABILITY_INFORMATION,
			.name = "CapabilityInformation" },
	{ .val = KMIP_TAG_KEY_WRAP_TYPE, .name = "KeyWrapType" },
	{ .val = KMIP_TAG_BATCH_UNDO_CAPABILITY,
			.name = "BatchUndoCapability" },
	{ .val = KMIP_TAG_BATCH_CONTINUE_CAPABILITY,
			.name = "BatchContinueCapability" },
	{ .val = KMIP_TAG_PKCS_12_FRIENDLY_NAME,
			.name = "PKCS_12FriendlyName" },
	{ .val = KMIP_TAG_DESCRIPTION, .name = "Description" },
	{ .val = KMIP_TAG_COMMENT, .name = "Comment" },
	{ .val = KMIP_TAG_AUTHENTICATED_ENCRYPTION_ADDITIONAL_DATA,
			.name = "AuthenticatedEncryptionAdditionalDat" },
	{ .val = KMIP_TAG_AUTHENTICTAED_ENCRYPTION_TAG,
			.name = "AuthentictaedEncryptionTag" },
	{ .val = KMIP_TAG_SALT_LENGTH, .name = "SaltLength" },
	{ .val = KMIP_TAG_MASK_GENERATOR, .name = "MaskGenerator" },
	{ .val = KMIP_TAG_MASK_GENERATOR_HASHING_ALGORITHM,
			.name = "MaskGeneratorHashingAlgorithm" },
	{ .val = KMIP_TAG_P_SOURCE, .name = "PSource" },
	{ .val = KMIP_TAG_TRAILER_FIELD, .name = "TrailerField" },
	{ .val = KMIP_TAG_CLIENT_CORRELATION_VALUE,
			.name = "ClientCorrelationValue" },
	{ .val = KMIP_TAG_SERVER_CORRELATION_VALUE,
			.name = "ServerCorrelationValue" },
	{ .val = KMIP_TAG_DIGESTED_DATA, .name = "DigestedData" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_CN,
			.name = "CertificateSubjectCN" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_O,
			.name = "CertificateSubjectO" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_OU,
			.name = "CertificateSubjectOU" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_EMAIL,
			.name = "CertificateSubjectEmail" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_C,
			.name = "CertificateSubjectC" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_ST,
			.name = "CertificateSubjectST" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_L,
			.name = "CertificateSubjectL" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_UID,
			.name = "CertificateSubjectUID" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_SERIAL_NUMBER,
			.name = "CertificateSubjectSerialNumber" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_TITLE,
			.name = "CertificateSubjectTitle" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_DC,
			.name = "CertificateSubjectDC" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_DN_QUALIFIER,
			.name = "CertificateSubjectDNQualifier" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_CN,
			.name = "CertificateIssuerCN" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_O, .name = "CertificateIssuerO" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_OU,
			.name = "CertificateIssuerOU" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_EMAIL,
			.name = "CertificateIssuerEmail" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_C, .name = "CertificateIssuerC" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_ST,
			.name = "CertificateIssuerST" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_L, .name = "CertificateIssuerL" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_UID,
			.name = "CertificateIssuerUID" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_SERIAL_NUMBER,
			.name = "CertificateIssuerSerialNumber" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_TITLE,
			.name = "CertificateIssuerTitle" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_DC,
			.name = "CertificateIssuerDC" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_DN_QUALIFIER,
			.name = "CertificateIssuerDNQualifier" },
	{ .val = KMIP_TAG_SENSITIVE, .name = "Sensitive" },
	{ .val = KMIP_TAG_ALWAYS_SENSITIVE, .name = "AlwaysSensitive" },
	{ .val = KMIP_TAG_EXTRACTABLE, .name = "Extractable" },
	{ .val = KMIP_TAG_NEVER_EXTRACTABLE, .name = "NeverExtractable" },
	{ .val = KMIP_TAG_REPLACE_EXISTING, .name = "ReplaceExisting" },
	{ .val = KMIP_TAG_ATTRIBUTES, .name = "Attributes" },
	{ .val = KMIP_TAG_COMMON_ATTRIBUTES, .name = "CommonAttributes" },
	{ .val = KMIP_TAG_PRIVATE_KEY_ATTRIBUTES,
			.name = "PrivateKeyAttributes" },
	{ .val = KMIP_TAG_PUBLIC_KEY_ATTRIBUTES,
			.name = "PublicKeyAttributes" },
	{ .val = KMIP_TAG_EXTENSION_ENUMERATION,
			.name = "ExtensionEnumeration" },
	{ .val = KMIP_TAG_EXTENSION_ATTRIBUTE,
			.name = "ExtensionAttribute" },
	{ .val = KMIP_TAG_EXTENSION_PARENT_STRUCTURE_TAG,
			.name = "ExtensionParentStructureTag" },
	{ .val = KMIP_TAG_EXTENSION_DESCRIPTION,
			.name = "ExtensionDescription" },
	{ .val = KMIP_TAG_SERVER_NAME, .name = "ServerName" },
	{ .val = KMIP_TAG_SERVER_SERIAL_NUMBER,
			.name = "ServerSerialNumber" },
	{ .val = KMIP_TAG_SERVER_VERSION, .name = "ServerVersion" },
	{ .val = KMIP_TAG_SERVER_LOAD, .name = "ServerLoad" },
	{ .val = KMIP_TAG_PRODUCT_NAME, .name = "ProductName" },
	{ .val = KMIP_TAG_BUILD_LEVEL, .name = "BuildLevel" },
	{ .val = KMIP_TAG_BUILD_DATE, .name = "BuildDate" },
	{ .val = KMIP_TAG_CLUSTER_INFO, .name = "ClusterInfo" },
	{ .val = KMIP_TAG_ALTERNATE_FAILOVER_ENDPOINTS,
			.name = "AlternateFailoverEndpoints" },
	{ .val = KMIP_TAG_SHORT_UNIQUE_IDENTIFIER,
			.name = "ShortUniqueIdentifier" },
	{ .val = KMIP_TAG_TAG, .name = "Tag" },
	{ .val = KMIP_TAG_CERTIFICATE_REQUEST_UNIQUE_IDENTIFIER,
			.name = "CertificateRequestUniqueIdentifier" },
	{ .val = KMIP_TAG_NIST_KEY_TYPE, .name = "NISTKeyType" },
	{ .val = KMIP_TAG_ATTRIBUTE_REFERENCE, .name = "AttributeReference" },
	{ .val = KMIP_TAG_CURRENT_ATTRIBUTE, .name = "CurrentAttribute" },
	{ .val = KMIP_TAG_NEW_ATTRIBUTE, .name = "NewAttribute" },
	{ .val = KMIP_TAG_CERTIFICATE_REQUEST_VALUE,
			.name = "CertificateRequestValue" },
	{ .val = KMIP_TAG_LOG_MESSAGE, .name = "LogMessage" },
	{ .val = KMIP_TAG_PROFILE_VERSION, .name = "ProfileVersion" },
	{ .val = KMIP_TAG_PROFILE_VERSION_MAJOR,
			.name = "ProfileVersionMajor" },
	{ .val = KMIP_TAG_PROFILE_VERSION_MINOR,
			.name = "ProfileVersionMinor" },
	{ .val = KMIP_TAG_PROTECTION_LEVEL, .name = "ProtectionLevel" },
	{ .val = KMIP_TAG_PROTECTION_PERIOD, .name = "ProtectionPeriod" },
	{ .val = KMIP_TAG_QUANTUM_SAFE, .name = "QuantumSafe" },
	{ .val = KMIP_TAG_QUANTUM_SAFE_CAPABILITY,
			.name = "QuantumSafeCapability" },
	{ .val = KMIP_TAG_TICKET, .name = "Ticket" },
	{ .val = KMIP_TAG_TICKET_TYPE, .name = "TicketType" },
	{ .val = KMIP_TAG_TICKET_VALUE, .name = "TicketValue" },
	{ .val = KMIP_TAG_REQUEST_COUNT, .name = "RequestCount" },
	{ .val = KMIP_TAG_RIGHTS, .name = "Rights" },
	{ .val = KMIP_TAG_OBJECTS, .name = "Objects" },
	{ .val = KMIP_TAG_OPERATIONS, .name = "Operations" },
	{ .val = KMIP_TAG_RIGHT, .name = "Right" },
	{ .val = KMIP_TAG_ENDPOINT_ROLE, .name = "EndpointRole" },
	{ .val = KMIP_TAG_DEFAULTS_INFORMATION, .name = "DefaultsInformation" },
	{ .val = KMIP_TAG_OBJECT_DEFAULTS, .name = "ObjectDefaults" },
	{ .val = KMIP_TAG_EPHEMERAL, .name = "Ephemeral" },
	{ .val = KMIP_TAG_SERVER_HASHED_PASSWORD,
			.name = "ServerHashedPassword" },
	{ .val = KMIP_TAG_ONE_TIME_PASSWORD, .name = "OneTimePassword" },
	{ .val = KMIP_TAG_HASHED_PASSWORD, .name = "HashedPassword" },
	{ .val = KMIP_TAG_ADJUSTMENT_TYPE, .name = "AdjustmentType" },
	{ .val = KMIP_TAG_PKCS_11_INTERFACE, .name = "PKCS_11Interface" },
	{ .val = KMIP_TAG_PKCS_11_FUNCTION, .name = "PKCS_11Function" },
	{ .val = KMIP_TAG_PKCS_11_INPUT_PARAMETERS,
			.name = "PKCS_11InputParameters" },
	{ .val = KMIP_TAG_PKCS_11_OUTPUT_PARAMETERS,
			.name = "PKCS_11OutputParameters" },
	{ .val = KMIP_TAG_PKCS_11_RETURN_CODE,
			.name = "PKCS_11ReturnCode" },
	{ .val = KMIP_TAG_PROTECTION_STORAGE_MASK,
			.name = "ProtectionStorageMask" },
	{ .val = KMIP_TAG_PROTECTION_STORAGE_MASKS,
			.name = "ProtectionStorageMasks" },
	{ .val = KMIP_TAG_INTEROP_FUNCTION, .name = "InteropFunction" },
	{ .val = KMIP_TAG_INTEROP_IDENTIFIER, .name = "InteropIdentifier" },
	{ .val = KMIP_TAG_ADJUSTMENT_VALUE, .name = "AdjustmentValue" },
	{ .val = KMIP_TAG_COMMON_PROTECTION_STORAGE_MASKS,
			.name = "CommonProtectionStorageMasks" },
	{ .val = KMIP_TAG_PRIVATE_PROTECTION_STORAGE_MASKS,
			.name = "PrivateProtectionStorageMasks" },
	{ .val = KMIP_TAG_PUBLIC_PROTECTION_STORAGE_MASKS,
			.name = "PublicProtectionStorageMasks" },
	{ .val = KMIP_TAG_OBJECT_GROUPS, .name = "ObjectGroups" },
	{ .val = KMIP_TAG_OBJECT_TYPES, .name = "ObjectTypes" },
	{ .val = KMIP_TAG_CONSTRAINTS, .name = "Constraints" },
	{ .val = KMIP_TAG_CONSTRAINT, .name = "Constraint" },
	{ .val = KMIP_TAG_ROTATE_INTERVAL, .name = "RotateInterval" },
	{ .val = KMIP_TAG_ROTATE_AUTOMATIC, .name = "RotateAutomatic" },
	{ .val = KMIP_TAG_ROTATE_OFFSET, .name = "RotateOffset" },
	{ .val = KMIP_TAG_ROTATE_DATE, .name = "RotateDate" },
	{ .val = KMIP_TAG_ROTATE_GENERATION, .name = "RotateGeneration" },
	{ .val = KMIP_TAG_ROTATE_NAME, .name = "RotateName" },
	{ .val = KMIP_TAG_ROTATE_NAME_VALUE, .name = "RotateNameValue" },
	{ .val = KMIP_TAG_ROTATE_NAME_TYPE, .name = "RotateNameType" },
	{ .val = KMIP_TAG_ROTATE_LATEST, .name = "RotateLatest" },
	{ .val = KMIP_TAG_ASYNCHRONOUS_REQUEST, .name = "AsynchronousRequest" },
	{ .val = KMIP_TAG_SUBMISSION_DATE, .name = "SubmissionDate" },
	{ .val = KMIP_TAG_PROCESSING_STAGE, .name = "ProcessingStage" },
	{ .val = KMIP_TAG_ASYNCHRONOUS_CORRELATION_VALUES,
			.name = "AsynchronousCorrelationValues" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_types[] = {
	{ .val = KMIP_TYPE_STRUCTURE, .name = "Structure" },
	{ .val = KMIP_TYPE_INTEGER, .name = "Integer" },
	{ .val = KMIP_TYPE_LONG_INTEGER, .name = "LongInteger" },
	{ .val = KMIP_TYPE_BIG_INTEGER, .name = "BigInteger" },
	{ .val = KMIP_TYPE_ENUMERATION, .name = "Enumeration" },
	{ .val = KMIP_TYPE_BOOLEAN, .name = "Boolean" },
	{ .val = KMIP_TYPE_TEXT_STRING, .name = "TextString" },
	{ .val = KMIP_TYPE_BYTE_STRING, .name = "ByteString" },
	{ .val = KMIP_TYPE_DATE_TIME, .name = "DateTime" },
	{ .val = KMIP_TYPE_INTERVAL, .name = "Interval" },
	{ .val = KMIP_TYPE_DATE_TIME_EXTENDED, .name = "DateTimeExtended" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_operations[] = {
	{ .val = KMIP_OPERATION_CREATE, .name = "Create" },
	{ .val = KMIP_OPERATION_CREATE_KEY_PAIR, .name = "CreateKeyPair" },
	{ .val = KMIP_OPERATION_REGISTER, .name = "Register" },
	{ .val = KMIP_OPERATION_RE_KEY, .name = "ReKey" },
	{ .val = KMIP_OPERATION_DERIVE_KEY, .name = "DeriveKey" },
	{ .val = KMIP_OPERATION_CERTIFY, .name = "Certify" },
	{ .val = KMIP_OPERATION_RE_CERTIFY, .name = "ReCertify" },
	{ .val = KMIP_OPERATION_LOCATE, .name = "Locate" },
	{ .val = KMIP_OPERATION_CHECK, .name = "Check" },
	{ .val = KMIP_OPERATION_GET, .name = "Get" },
	{ .val = KMIP_OPERATION_GET_ATTRIBUTES, .name = "GetAttributes" },
	{ .val = KMIP_OPERATION_GET_ATTRIBUTE_LIST,
			.name = "GetAttributeList" },
	{ .val = KMIP_OPERATION_ADD_ATTRIBUTE, .name = "AddAttribute" },
	{ .val = KMIP_OPERATION_MODIFY_ATTRIBUTE, .name = "ModifyAttribute" },
	{ .val = KMIP_OPERATION_DELETE_ATTRIBUTE, .name = "DeleteAttribute" },
	{ .val = KMIP_OPERATION_OBTAIN_LEASE, .name = "ObtainLease" },
	{ .val = KMIP_OPERATION_GET_USAGE_ALLOCATION,
			.name = "GetUsageAllocation" },
	{ .val = KMIP_OPERATION_ACTIVATE, .name = "Activate" },
	{ .val = KMIP_OPERATION_REVOKE, .name = "Revoke" },
	{ .val = KMIP_OPERATION_DESTROY, .name = "Destroy" },
	{ .val = KMIP_OPERATION_ARCHIVE, .name = "Archive" },
	{ .val = KMIP_OPERATION_RECOVER, .name = "Recover" },
	{ .val = KMIP_OPERATION_VALIDATE, .name = "Validate" },
	{ .val = KMIP_OPERATION_QUERY, .name = "Query" },
	{ .val = KMIP_OPERATION_CANCEL, .name = "Cancel" },
	{ .val = KMIP_OPERATION_POLL, .name = "Poll" },
	{ .val = KMIP_OPERATION_NOTIFY, .name = "Notify" },
	{ .val = KMIP_OPERATION_PUT, .name = "Put" },
	{ .val = KMIP_OPERATION_RE_KEY_KEY_PAIR, .name = "ReKeyKeyPair" },
	{ .val = KMIP_OPERATION_DISCOVER_VERSIONS, .name = "DiscoverVersions" },
	{ .val = KMIP_OPERATION_ENCRYPT, .name = "Encrypt" },
	{ .val = KMIP_OPERATION_DECRYPT, .name = "Decrypt" },
	{ .val = KMIP_OPERATION_SIGN, .name = "Sign" },
	{ .val = KMIP_OPERATION_SIGNATURE_VERIFY, .name = "SignatureVerify" },
	{ .val = KMIP_OPERATION_MAC, .name = "MAC" },
	{ .val = KMIP_OPERATION_MAC_VERIFY, .name = "MACVerify" },
	{ .val = KMIP_OPERATION_RNG_RETRIEVE, .name = "RNGRetrieve" },
	{ .val = KMIP_OPERATION_RNG_SEED, .name = "RNGSeed" },
	{ .val = KMIP_OPERATION_HASH, .name = "Hash" },
	{ .val = KMIP_OPERATION_CREATE_SPLIT_KEY, .name = "CreateSplitKey" },
	{ .val = KMIP_OPERATION_JOIN_SPLIT_KEY, .name = "JoinSplitKey" },
	{ .val = KMIP_OPERATION_IMPORT, .name = "Import" },
	{ .val = KMIP_OPERATION_EXPORT, .name = "Export" },
	{ .val = KMIP_OPERATION_LOG, .name = "Log" },
	{ .val = KMIP_OPERATION_LOGIN, .name = "Login" },
	{ .val = KMIP_OPERATION_LOGOUT, .name = "Logout" },
	{ .val = KMIP_OPERATION_DELEGATE_LOGIN, .name = "DelegateLogin" },
	{ .val = KMIP_OPERATION_ADJUST_ATTRIBUTE, .name = "AdjustAttribute" },
	{ .val = KMIP_OPERATION_SET_ATTRIBUTE, .name = "SetAttribute" },
	{ .val = KMIP_OPERATION_SET_ENDPOINT_ROLE, .name = "SetEndpointRole" },
	{ .val = KMIP_OPERATION_PKS_11, .name = "PKCS_11" },
	{ .val = KMIP_OPERATION_INTEROP, .name = "Interop" },
	{ .val = KMIP_OPERATION_RE_PROVISION, .name = "ReProvision" },
	{ .val = KMIP_OPERATION_SET_DEFAULTS, .name = "SetDefaults" },
	{ .val = KMIP_OPERATION_SET_CONSTRAINTS, .name = "SetConstraints" },
	{ .val = KMIP_OPERATION_GET_CONSTRAINTS, .name = "GetConstraints" },
	{ .val = KMIP_OPERATION_QUERY_ASYNCHRONOUS_REQUESTS,
			.name = "QueryAsynchronousRequests" },
	{ .val = KMIP_OPERATION_PROCESS, .name = "Process" },
	{ .val = KMIP_OPERATION_PING, .name = "Ping" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_batch_error_cont_options[] = {
	{ .val = KMIP_BATCH_ERR_CONT_CONTINUE, .name = "Continue" },
	{ .val = KMIP_BATCH_ERR_CONT_STOP, .name = "Stop" },
	{ .val = KMIP_BATCH_ERR_CONT_UNDO, .name = "Undo" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_crypto_usage_masks[] = {
	{ .val = KMIP_CRY_USAGE_MASK_SIGN, .name = "Sign" },
	{ .val = KMIP_CRY_USAGE_MASK_VERIFY, .name = "Verify" },
	{ .val = KMIP_CRY_USAGE_MASK_ENCRYPT, .name = "Encrypt" },
	{ .val = KMIP_CRY_USAGE_MASK_DECRYPT, .name = "Decrypt" },
	{ .val = KMIP_CRY_USAGE_MASK_WRAP_KEY, .name = "WrapKey" },
	{ .val = KMIP_CRY_USAGE_MASK_UNWRAP_KEY, .name = "UnwrapKey" },
	{ .val = KMIP_CRY_USAGE_MASK_EXPORT, .name = "Export" },
	{ .val = KMIP_CRY_USAGE_MASK_MAC_GENERATE, .name = "MACGenerate" },
	{ .val = KMIP_CRY_USAGE_MASK_MAC_VERIFY, .name = "MACVerify" },
	{ .val = KMIP_CRY_USAGE_MASK_DERIVE_KEY, .name = "DeriveKey" },
	{ .val = KMIP_CRY_USAGE_MASK_CONTENT_COMMITMENT,
				.name = "ContentCommitmentNonRepudiation" },
	{ .val = KMIP_CRY_USAGE_MASK_KEY_AGREEMENT, .name = "KeyAgreement" },
	{ .val = KMIP_CRY_USAGE_MASK_CERTIFICATE_SIGN,
						.name = "CertificateSign" },
	{ .val = KMIP_CRY_USAGE_MASK_CLR_SIGN, .name = "CRLSign" },
	{ .val = KMIP_CRY_USAGE_MASK_GENERATE_CRYPTOGRAM,
					.name = "GenerateCryptogram" },
	{ .val = KMIP_CRY_USAGE_MASK_VALIDATE_CRYPTOGRAM,
					.name = "ValidateCryptogram" },
	{ .val = KMIP_CRY_USAGE_MASK_TRANSLATE_ENCRYPT,
					.name = "TranslateEncrypt" },
	{ .val = KMIP_CRY_USAGE_MASK_TRANSLATE_DECRYPT,
					.name = "TranslateDecrypt" },
	{ .val = KMIP_CRY_USAGE_MASK_TRANSLATE_WRAP, .name = "TranslateWrap" },
	{ .val = KMIP_CRY_USAGE_MASK_TRANSLATE_UNWRAP,
					.name = "TranslateUnwrap" },
	{ .val = KMIP_CRY_USAGE_MASK_AUTHENTICATE, .name = "Authenticate" },
	{ .val = KMIP_CRY_USAGE_MASK_UNRESTRICTED, .name = "Unrestricted" },
	{ .val = KMIP_CRY_USAGE_MASK_FPE_ENCRYPT, .name = "FPEEncrypt" },
	{ .val = KMIP_CRY_USAGE_MASK_FPE_DECRYPT, .name = "FPEDecrypt" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_result_statuses[] = {
	{ .val = KMIP_RESULT_STATUS_SUCCESS, .name = "Success" },
	{ .val = KMIP_RESULT_STATUS_OPERATION_FAILED,
				.name = "OperationFailed" },
	{ .val = KMIP_RESULT_STATUS_OPERATION_PENDING,
				.name = "OperationPending" },
	{ .val = KMIP_RESULT_STATUS_OPERATION_UNDONE,
				.name = "OperationUndone" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_result_reasons[] = {
	{ .val = KMIP_RESULT_REASON_ITEM_NOT_FOUND, .name = "ItemNotFound" },
	{ .val = KMIP_RESULT_REASON_RESPONSE_TOO_LARGE,
				.name = "ResponseTooLarge" },
	{ .val = KMIP_RESULT_REASON_AUTH_NOT_SUCCESSFUL,
				.name = "AuthenticationNotSuccessful" },
	{ .val = KMIP_RESULT_REASON_INVALID_MESSAGE, .name = "InvalidMessage" },
	{ .val = KMIP_RESULT_REASON_OPERATION_NOT_SUCCESSFUL,
				.name = "OperationNotSupported" },
	{ .val = KMIP_RESULT_REASON_MISSING_DATA, .name = "MissingData" },
	{ .val = KMIP_RESULT_REASON_INVALIUD_FIELD, .name = "InvalidField" },
	{ .val = KMIP_RESULT_REASON_FEATURE_NOT_SUPPORTED,
				.name = "FeatureNotSupported" },
	{ .val = KMIP_RESULT_REASON_OP_CANCELED_BY_REQUESTOR,
				.name = "OperationCanceledByRequeste" },
	{ .val = KMIP_RESULT_REASON_CRYPTOGRAPHIC_FAILURE,
				.name = "CryptographicFailure" },
	{ .val = KMIP_RESULT_REASON_ILLEGAL_OPERATION,
				.name = "IllegalOperation" },
	{ .val = KMIP_RESULT_REASON_PERMISSION_DENIED,
				.name = "PermissionDenied" },
	{ .val = KMIP_RESULT_REASON_OBJECT_ARCHIVED,
				.name = "ObjectArchived" },
	{ .val = KMIP_RESULT_REASON_INDEX_OUT_OF_BOUNDS,
				.name = "IndexOutOfBounds" },
	{ .val = KMIP_RESULT_REASON_APP_NAMESPACE_NOT_SUPPORTED,
				.name = "ApplicationNamespaceNotSupported" },
	{ .val = KMIP_RESULT_REASON_KEY_FORMAT_TYPE_NOT_SUPPORTED,
				.name = "KeyFormatTypeNotSupported" },
	{ .val = KMIP_RESULT_REASON_KEY_COMPRESSION_TYPE_NOT_SUPPORTED,
				.name = "KeyCompressionTypeNotSupported" },
	{ .val = KMIP_RESULT_REASON_ENCODING_OPTION_ERROR,
				.name = "EncodingOptionError" },
	{ .val = KMIP_RESULT_REASON_KEY_VALUE_NOT_PRESENT,
				.name = "KeyValueNotPresent" },
	{ .val = KMIP_RESULT_REASON_ATTESTATION_REQUIRED,
				.name = "AttestationRequired" },
	{ .val = KMIP_RESULT_REASON_ATTESTATION_FAILED,
				.name = "AttestationFailed" },
	{ .val = KMIP_RESULT_REASON_SENSITIVE, .name = "Sensitive" },
	{ .val = KMIP_RESULT_REASON_NOT_EXTRACTABLE, .name = "NotExtractable" },
	{ .val = KMIP_RESULT_REASON_OBJECT_ALREADY_EXISTS,
				.name = "ObjectAlreadyExists" },
	{ .val = KMIP_RESULT_REASON_INVALID_TICKET, .name = "InvalidTicket" },
	{ .val = KMIP_RESULT_REASON_USAGE_LIMIT_EXCEEDED,
				.name = "UsageLimitExceeded" },
	{ .val = KMIP_RESULT_REASON_NUMERIC_RANGE, .name = "NumericRange" },
	{ .val = KMIP_RESULT_REASON_INVALID_DATA_TYPE,
				.name = "InvalidDataType" },
	{ .val = KMIP_RESULT_REASON_READ_ONLY_ATTRIBUTE,
				.name = "ReadOnlyAttribute" },
	{ .val = KMIP_RESULT_REASON_MULTI_VALUED_ATTRIBUTE,
				.name = "MultiValuedAttribute" },
	{ .val = KMIP_RESULT_REASON_UNSUPPORTED_ATTRIBUTE,
				.name = "UnsupportedAttribute" },
	{ .val = KMIP_RESULT_REASON_ATTRIBUTE_INSTANCE_NOT_FOUND,
				.name = "AttributeInstanceNotFound" },
	{ .val = KMIP_RESULT_REASON_ATTRIBUTE_NOT_FOUND,
				.name = "AttributeNotFound" },
	{ .val = KMIP_RESULT_REASON_ATTRIBUTE_READ_ONLY,
				.name = "AttributeReadOnly" },
	{ .val = KMIP_RESULT_REASON_ATTRIBUTE_SINGLE_VALUED,
				.name = "AttributeSingleValued" },
	{ .val = KMIP_RESULT_REASON_BAD_CRYPTOGRAPHIC_PARAMETERS,
				.name = "BadCryptographicParameters" },
	{ .val = KMIP_RESULT_REASON_BAD_PASSWORD, .name = "BadPassword" },
	{ .val = KMIP_RESULT_REASON_CODEC_ERROR, .name = "CodecError" },
	{ .val = KMIP_RESULT_REASON_ILLEGAL_OBJECT_TYPE,
				.name = "IllegalObjectType" },
	{ .val = KMIP_RESULT_REASON_INCOMPATIBLE_CRYPTO_USAGE_MASK,
				.name = "IncompatibleCryptographicUsageMask" },
	{ .val = KMIP_RESULT_REASON_INTERNAL_SERVER_ERROR,
				.name = "InternalServerError" },
	{ .val = KMIP_RESULT_REASON_INVALID_ASYNC_CORRELATION_VALUE,
				.name = "InvalidAsynchronousCorrelationValue" },
	{ .val = KMIP_RESULT_REASON_INVALID_ATTRIBUTE,
				.name = "InvalidAttribute" },
	{ .val = KMIP_RESULT_REASON_INVALID_ATTRIBUTE_VALUE,
				.name = "InvalidAttributeValue" },
	{ .val = KMIP_RESULT_REASON_INVALID_CORRELATION_VALUE,
				.name = "InvalidCorrelationValue" },
	{ .val = KMIP_RESULT_REASON_INVALID_CSR, .name = "InvalidCSR" },
	{ .val = KMIP_RESULT_REASON_INVALID_OBJECT_TYPE,
				.name = "InvalidObjectType" },
	{ .val = KMIP_RESULT_REASON_KEY_WRAP_TYPE_NOT_SUPPORTED,
				.name = "KeyWrapTypeNotSupported" },
	{ .val = KMIP_RESULT_REASON_MISSING_INITIALIZATION_VECTOR,
				.name = "MissingInitializationVector" },
	{ .val = KMIP_RESULT_REASON_NOT_UNIQUE_NAME_ATTRIBUTE,
				.name = "NonUniqueNameAttribute" },
	{ .val = KMIP_RESULT_REASON_OBJECT_DESTROYED,
				.name = "ObjectDestroyed" },
	{ .val = KMIP_RESULT_REASON_OBJECT_NOT_FOUND,
				.name = "ObjectNotFound" },
	{ .val = KMIP_RESULT_REASON_NOT_AUTHORISED, .name = "NotAuthorised" },
	{ .val = KMIP_RESULT_REASON_SERVER_LIMIT_EXCEEDED,
				.name = "ServerLimitExceeded" },
	{ .val = KMIP_RESULT_REASON_UNKNOWN_ENUMERATION,
				.name = "UnknownEnumeration" },
	{ .val = KMIP_RESULT_REASON_UNKNOWN_MESSAGE_EXTENSION,
				.name = "UnknownMessageExtension" },
	{ .val = KMIP_RESULT_REASON_UNKNOWN_TAG, .name = "UnknownTag" },
	{ .val = KMIP_RESULT_REASON_UNSUPPORTED_CRYPTO_PARAMETERS,
				.name = "UnsupportedCryptographicParameters" },
	{ .val = KMIP_RESULT_REASON_UNSUPPORTED_PROTOCOL_VERSION,
				.name = "UnsupportedProtocolVersion" },
	{ .val = KMIP_RESULT_REASON_WRAPPING_OBJECT_ARCHIVED,
				.name = "WrappingObjectArchived" },
	{ .val = KMIP_RESULT_REASON_WRAPPING_OBJECT_DESTROYED,
				.name = "WrappingObjectDestroyed" },
	{ .val = KMIP_RESULT_REASON_WRAPPING_OBJECT_NOT_FOUND,
				.name = "WrappingObjectNotFound" },
	{ .val = KMIP_RESULT_REASON_WRONG_KEY_LIFECYCLE_STATE,
				.name = "WrongKeyLifecycleState" },
	{ .val = KMIP_RESULT_REASON_PROTECTION_STORAGE_UNAVAILABLE,
				.name = "ProtectionStorageUnavailable" },
	{ .val = KMIP_RESULT_REASON_PKCS_11_CODE_ERROR,
				.name = "PKCS_11CodecError" },
	{ .val = KMIP_RESULT_REASON_PKCS_11_INVALID_FUNCTION,
				.name = "PKCS_11InvalidFunction" },
	{ .val = KMIP_RESULT_REASON_PKCS_11_INVALID_INTERFACE,
				.name = "PKCS_11InvalidInterface" },
	{ .val = KMIP_RESULT_REASON_PRIVATE_PROT_STORAGE_UNAVAILABLE,
				.name = "PrivateProtectionStorageUnavailable" },
	{ .val = KMIP_RESULT_REASON_PUBLIC_PROT_STORAGE_UNAVAILABLE,
				.name = "PublicProtectionStorageUnavailable" },
	{ .val = KMIP_RESULT_REASON_UNKNOWN_OBJECT_GROUP,
				.name = "UnknownObjectGroup" },
	{ .val = KMIP_RESULT_REASON_CONSTRAINT_VIOLATION,
				.name = "ConstraintViolation" },
	{ .val = KMIP_RESULT_REASON_DUPLICATE_PROCESS_REQUEST,
				.name = "DuplicateProcessRequest" },
	{ .val = KMIP_RESULT_REASON_GENERAL_FAILURE, .name = "GeneralFailure" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_query_functions[] = {
	{ .val = KMIP_QUERY_OPERATIONS, .name = "QueryOperations" },
	{ .val = KMIP_QUERY_OBJECTS, .name = "QueryObjects" },
	{ .val = KMIP_QUERY_SERVER_INFORMATION,
				.name = "QueryServerInformation" },
	{ .val = KMIP_QUERY_APPLICATION_NAMESPACES,
				.name = "QueryApplicationNamespaces" },
	{ .val = KMIP_QUERY_EXTENSION_LIST, .name = "QueryExtensionList" },
	{ .val = KMIP_QUERY_EXTENSION_MAP, .name = "QueryExtensionMap" },
	{ .val = KMIP_QUERY_ATTESTATION_TYPES,
				.name = "QueryAttestationTypes" },
	{ .val = KMIP_QUERY_QUERY_RNGS, .name = "QueryRNGs" },
	{ .val = KMIP_QUERY_VALIDATIONS, .name = "QueryValidations" },
	{ .val = KMIP_QUERY_PROFILES, .name = "QueryProfiles" },
	{ .val = KMIP_QUERY_CAPABILITIES, .name = "QueryCapabilities" },
	{ .val = KMIP_QUERY_CLIENT_REGISTRATION_METHODS,
				.name = "QueryClientRegistrationMethods" },
	{ .val = KMIP_QUERY_DEFAULTS_INFORMATION,
				.name = "QueryDefaultsInformation" },
	{ .val = KMIP_QUERY_STORAGE_PROTECTION_MASKS,
				.name = "QueryStorageProtectionMasks" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_name_types[] = {
	{ .val = KMIP_NAME_TYPE_UNINTERPRETED_TEXT_STRING,
				.name = "UninterpretedTextString" },
	{ .val = KMIP_NAME_TYPE_URI, .name = "URI" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_alternate_name_types[] = {
	{ .val = KMIP_ALT_NAME_TYPE_UNINTERPRETED_TEXT_STRING,
				.name = "UninterpretedTextString" },
	{ .val = KMIP_ALT_NAME_TYPE_URI, .name = "URI" },
	{ .val = KMIP_ALT_NAME_TYPE_OBJECT_SERIAL_NUMBER,
				.name = "ObjectSerialNumber" },
	{ .val = KMIP_ALT_NAME_TYPE_EMAIL_ADDRESS, .name = "EmailAddress" },
	{ .val = KMIP_ALT_NAME_TYPE_DNS_NAME, .name = "DNSName" },
	{ .val = KMIP_ALT_NAME_TYPE_X_500_DISTINGUISHED_NAME,
				.name = "X_500DistinguishedName" },
	{ .val = KMIP_ALT_NAME_TYPE_IP_ADDRESS, .name = "IPAddress" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_unique_identifiers[] = {
	{ .val = KMIP_UNIQUE_ID_ID_PLACEHOLDER, .name = "IDPlaceholder" },
	{ .val = KMIP_UNIQUE_ID_CERTIFY, .name = "Certify" },
	{ .val = KMIP_UNIQUE_ID_CREATE, .name = "Create" },
	{ .val = KMIP_UNIQUE_ID_CREATE_KEY_PAIR, .name = "CreateKeyPair" },
	{ .val = KMIP_UNIQUE_ID_CREATE_KEY_PAIR_PRIVATE,
					.name = "CreateKeyPairPrivateKey" },
	{ .val = KMIP_UNIQUE_ID_CREATE_KEY_PAIR_PUBLIC,
					.name = "CreateKeyPairPublicKey" },
	{ .val = KMIP_UNIQUE_ID_CREATE_SPLIT_KEY, .name = "CreateSplitKey" },
	{ .val = KMIP_UNIQUE_ID_DERIVE_KEY, .name = "DeriveKey" },
	{ .val = KMIP_UNIQUE_ID_IMPORT, .name = "Import" },
	{ .val = KMIP_UNIQUE_ID_JOIN_SPLIT_KEY, .name = "JoinSplitKey" },
	{ .val = KMIP_UNIQUE_ID_LOCATE, .name = "Locate" },
	{ .val = KMIP_UNIQUE_ID_REGISTER, .name = "Register" },
	{ .val = KMIP_UNIQUE_ID_RE_KEY, .name = "ReKey" },
	{ .val = KMIP_UNIQUE_ID_RE_CERTIFY, .name = "ReCertify" },
	{ .val = KMIP_UNIQUE_ID_RE_KEY_KEY_PAIR, .name = "ReKeyKeyPair" },
	{ .val = KMIP_UNIQUE_ID_RE_KEY_KEY_PAIR_PRIVATE,
					.name = "ReKeyKeyPairPrivateKey" },
	{ .val = KMIP_UNIQUE_ID_RE_KEY_KEY_PAIR_PUBLIC,
					.name = "ReKeyKeyPairPublicKey" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_object_types[] = {
	{ .val = KMIP_OBJECT_TYPE_CERTIFICATE, .name = "Certificate" },
	{ .val = KMIP_OBJECT_TYPE_SYMMETRIC_KEY, .name = "SymmetricKey" },
	{ .val = KMIP_OBJECT_TYPE_PUBLIC_KEY, .name = "PublicKey" },
	{ .val = KMIP_OBJECT_TYPE_PRIVATE_KEY, .name = "PrivateKey" },
	{ .val = KMIP_OBJECT_TYPE_SPLIT_KEY, .name = "SplitKey" },
	{ .val = KMIP_OBJECT_TYPE_TEMPLATE, .name = "Template" },
	{ .val = KMIP_OBJECT_TYPE_SECRET_DATA, .name = "SecretData" },
	{ .val = KMIP_OBJECT_TYPE_OPAQUE_OBJECT, .name = "OpaqueObject" },
	{ .val = KMIP_OBJECT_TYPE_PGP_KEY, .name = "PGPKey" },
	{ .val = KMIP_OBJECT_TYPE_CERTIFICATE_REQUEST,
					.name = "CertificateRequest" },
	{ .val = 0, .name = NULL },
};


static const struct kmip_enum kmip_crypto_algos[] = {
	{ .val = KMIP_CRYPTO_ALGO_DES, .name = "DES" },
	{ .val = KMIP_CRYPTO_ALGO_3DES, .name = "3DES" },
	{ .val = KMIP_CRYPTO_ALGO_AES, .name = "AES" },
	{ .val = KMIP_CRYPTO_ALGO_RSA, .name = "RSA" },
	{ .val = KMIP_CRYPTO_ALGO_DSA, .name = "DSA" },
	{ .val = KMIP_CRYPTO_ALGO_ECDSA, .name = "ECDSA" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA1, .name = "HMAC_SHA1" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA224, .name = "HMAC_SHA224" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA256, .name = "HMAC_SHA256" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA384, .name = "HMAC_SHA384" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA512, .name = "HMAC_SHA512" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_MD5, .name = "HMAC_MD5" },
	{ .val = KMIP_CRYPTO_ALGO_DH, .name = "DH" },
	{ .val = KMIP_CRYPTO_ALGO_ECDH, .name = "ECDH" },
	{ .val = KMIP_CRYPTO_ALGO_ECMQV, .name = "ECMQV" },
	{ .val = KMIP_CRYPTO_ALGO_BLOWFISH, .name = "Blowfish" },
	{ .val = KMIP_CRYPTO_ALGO_CAMELLIA, .name = "Camellia" },
	{ .val = KMIP_CRYPTO_ALGO_CAST5, .name = "CAST5" },
	{ .val = KMIP_CRYPTO_ALGO_IDEA, .name = "IDEA" },
	{ .val = KMIP_CRYPTO_ALGO_MARS, .name = "MARS" },
	{ .val = KMIP_CRYPTO_ALGO_RC2, .name = "RC2" },
	{ .val = KMIP_CRYPTO_ALGO_RC4, .name = "RC4" },
	{ .val = KMIP_CRYPTO_ALGO_RC5, .name = "RC5" },
	{ .val = KMIP_CRYPTO_ALGO_SKIPJACK, .name = "SKIPJACK" },
	{ .val = KMIP_CRYPTO_ALGO_TWOFISH, .name = "Twofish" },
	{ .val = KMIP_CRYPTO_ALGO_EC, .name = "EC" },
	{ .val = KMIP_CRYPTO_ALGO_ONE_TIME_PAD, .name = "OneTimePad" },
	{ .val = KMIP_CRYPTO_ALGO_CHACHA20, .name = "ChaCha20" },
	{ .val = KMIP_CRYPTO_ALGO_POLY1305, .name = "Poly1305" },
	{ .val = KMIP_CRYPTO_ALGO_CHACHA20_POLY1305,
			.name = "ChaCha20Poly1305" },
	{ .val = KMIP_CRYPTO_ALGO_SHA3_224, .name = "SHA3_224" },
	{ .val = KMIP_CRYPTO_ALGO_SHA3_256, .name = "SHA3_256" },
	{ .val = KMIP_CRYPTO_ALGO_SHA3_384, .name = "SHA3_384" },
	{ .val = KMIP_CRYPTO_ALGO_SHA3_512, .name = "SHA3_512" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA3_224, .name = "HMAC_SHA3_224" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA3_256, .name = "HMAC_SHA3_256" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA3_384, .name = "HMAC_SHA3_384" },
	{ .val = KMIP_CRYPTO_ALGO_HMAC_SHA3_512, .name = "HMAC_SHA3_512" },
	{ .val = KMIP_CRYPTO_ALGO_SHAKE_128, .name = "SHAKE_128" },
	{ .val = KMIP_CRYPTO_ALGO_SHAKE_256, .name = "SHAKE_256" },
	{ .val = KMIP_CRYPTO_ALGO_ARIA, .name = "ARIA" },
	{ .val = KMIP_CRYPTO_ALGO_SEED, .name = "SEED" },
	{ .val = KMIP_CRYPTO_ALGO_SM2, .name = "SM2" },
	{ .val = KMIP_CRYPTO_ALGO_SM3, .name = "SM3" },
	{ .val = KMIP_CRYPTO_ALGO_SM4, .name = "SM4" },
	{ .val = KMIP_CRYPTO_ALGO_GOST_R34_10_2012, .name = "GOSTR34_10_2012" },
	{ .val = KMIP_CRYPTO_ALGO_GOST_R34_11_2012, .name = "GOSTR34_11_2012" },
	{ .val = KMIP_CRYPTO_ALGO_GOST_R34_13_2015, .name = "GOSTR34_13_2015" },
	{ .val = KMIP_CRYPTO_ALGO_GOST_28147_89, .name = "GOST28147_89" },
	{ .val = KMIP_CRYPTO_ALGO_XMSS, .name = "XMSS" },
	{ .val = KMIP_CRYPTO_ALGO_SPHINCS_256, .name = "SPHINCS_256" },
	{ .val = KMIP_CRYPTO_ALGO_MCELIECE, .name = "McEliece" },
	{ .val = KMIP_CRYPTO_ALGO_MCELIECE_6960119,
			.name = "McEliece_6960119" },
	{ .val = KMIP_CRYPTO_ALGO_MCELIECE_8192128,
			.name = "McEliece_8192128" },
	{ .val = KMIP_CRYPTO_ALGO_ED25519, .name = "Ed25519" },
	{ .val = KMIP_CRYPTO_ALGO_ED448, .name = "Ed448" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_certificate_types[] = {
	{ .val = KMIP_CERTIFICATE_TYPE_X_509, .name = "X_509" },
	{ .val = KMIP_CERTIFICATE_TYPE_PGP, .name = "PGP" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_states[] = {
	{ .val = KMIP_STATE_PRE_ACTIVE, .name = "PreActive" },
	{ .val = KMIP_STATE_ACTIVE, .name = "Active" },
	{ .val = KMIP_STATE_DEACTIVATED, .name = "Deactivated" },
	{ .val = KMIP_STATE_COMPROMISED, .name = "Compromised" },
	{ .val = KMIP_STATE_DESTROYED, .name = "Destroyed" },
	{ .val = KMIP_STATE_DESTROYED_COMPROMISED,
					.name = "DestroyedCompromised" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_protection_storage_masks[] = {
	{ .val = KMIP_PROT_STORAGE_MASK_SOFTWARE, .name = "Software" },
	{ .val = KMIP_PROT_STORAGE_MASK_HARDWARE, .name = "Hardware" },
	{ .val = KMIP_PROT_STORAGE_MASK_ON_PROCESSOR, .name = "OnProcessor" },
	{ .val = KMIP_PROT_STORAGE_MASK_ON_SYSTEM, .name = "OnSystem" },
	{ .val = KMIP_PROT_STORAGE_MASK_OFF_SYSTEM, .name = "OffSystem" },
	{ .val = KMIP_PROT_STORAGE_MASK_HYPERVISOR, .name = "Hypervisor" },
	{ .val = KMIP_PROT_STORAGE_MASK_OPERATING_SYSTEM,
						.name = "OperatingSystem" },
	{ .val = KMIP_PROT_STORAGE_MASK_CONTAINER, .name = "Container" },
	{ .val = KMIP_PROT_STORAGE_MASK_ON_PREMISES, .name = "OnPremises" },
	{ .val = KMIP_PROT_STORAGE_MASK_OFF_PREMISES, .name = "OffPremises" },
	{ .val = KMIP_PROT_STORAGE_MASK_SELF_MANAGED, .name = "SelfManaged" },
	{ .val = KMIP_PROT_STORAGE_MASK_OUTSOURCED, .name = "Outsourced" },
	{ .val = KMIP_PROT_STORAGE_MASK_VALIDATED, .name = "Validated" },
	{ .val = KMIP_PROT_STORAGE_MASK_SAME_JURISDICATION,
						.name = "SameJurisdiction" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_revoke_reasons[] = {
	{ .val = KMIP_REVOK_RSN_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_REVOK_RSN_KEY_COMPROMISE, .name = "KeyCompromise" },
	{ .val = KMIP_REVOK_RSN_CA_COMPROMISE, .name = "CACompromise" },
	{ .val = KMIP_REVOK_RSN_AFFILIATION_CHANGED,
					.name = "AffiliationChanged" },
	{ .val = KMIP_REVOK_RSN_SUPERSEDED, .name = "Superseded" },
	{ .val = KMIP_REVOK_RSN_CESSATION_OF_OPERATION,
					.name = "CessationOfOperation" },
	{ .val = KMIP_REVOK_RSN_PRIVILEGE_WITHDRAWN,
					.name = "PrivilegeWithdrawn" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_object_group_members[] = {
	{ .val = KMIP_OBJ_GROUP_MEMBER_FRESH, .name = "GroupMemberFresh" },
	{ .val = KMIP_OBJ_GROUP_MEMBER_DEFAULT, .name = "GroupMemberDefault" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_storage_status_masks[] = {
	{ .val = KMIP_STORAGE_STATUS_MASK_ONLINE, .name = "OnLineStorage" },
	{ .val = KMIP_STORAGE_STATUS_MASK_ARCHIVAL, .name = "ArchivalStorage" },
	{ .val = KMIP_STORAGE_STATUS_MASK_DESTTROYED,
			.name = "DestroyedStorage" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_key_format_types[] = {
	{ .val = KMIP_KEY_FORMAT_TYPE_RAW, .name = "Raw" },
	{ .val = KMIP_KEY_FORMAT_TYPE_OPAQUE, .name = "Opaque" },
	{ .val = KMIP_KEY_FORMAT_TYPE_PKCS_1, .name = "PKCS_1" },
	{ .val = KMIP_KEY_FORMAT_TYPE_PKCS_8, .name = "PKCS_8" },
	{ .val = KMIP_KEY_FORMAT_TYPE_X_509, .name = "X_509" },
	{ .val = KMIP_KEY_FORMAT_TYPE_EC_PRIVATE_KEY, .name = "ECPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_SYMMETRIC_KEY,
					.name = "TransparentSymmetricKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DSA_PRIVATE_KEY,
					.name = "TransparentDSAPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DSA_PUBLIC_KEY,
					.name = "TransparentDSAPublicKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_RSA_PRIVATE_KEY,
					.name = "TransparentRSAPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_RSA_PUBLIC_KEY,
					.name = "TransparentRSAPublicKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DH_PRIVATE_KEY,
					.name = "TransparentDHPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_DH_PUBLIC_KEY,
					.name = "TransparentDHPublicKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDSA_PRIVATE_KEY,
					.name = "TransparentECDSAPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDSA_PUBLIC_KEY,
					.name = "TransparentECDSAPublicKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDH_PRIVATE_KEY,
					.name = "TransparentECDHPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECDH_PUBLIC_KEY,
					.name = "TransparentECDHPublicKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECMQV_PRIVATE_KEY,
					.name = "TransparentECMQVPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_ECMQV_PUBLIC_KEY,
					.name = "TransparentECMQVPublicKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_EC_PRIVATE_KEY,
					.name = "TransparentECPrivateKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_TRANSPARENT_EC_PUBLIC_KEY,
					.name = "TransparentECPublicKey" },
	{ .val = KMIP_KEY_FORMAT_TYPE_PKCS_12, .name = "PKCS_12" },
	{ .val = KMIP_KEY_FORMAT_TYPE_PKCS_10, .name = "PKCS_10" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_key_compression_types[] = {
	{ .val = KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_UNCOMPRESSED,
				.name = "ECPublicKeyTypeUncompressed" },
	{ .val = KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_COMPRESSED_PRIME,
				.name = "ECPublicKeyTypeX9_62CompressedPrime" },
	{ .val = KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_COMPRESSED_CHAR2,
				.name = "ECPublicKeyTypeX9_62CompressedChar2" },
	{ .val = KMIP_KEY_COMPRESSION_TYPE_EC_PUBKEY_HYBID,
				.name = "ECPublicKeyTypeX9_62Hybrid" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_wrapping_methods[] = {
	{ .val = KMIP_WRAPPING_METHOD_ENCRYPT, .name = "Encrypt" },
	{ .val = KMIP_WRAPPING_METHOD_MAC_SIGN, .name = "MAC_sign" },
	{ .val = KMIP_WRAPPING_METHOD_ENCRYPT_THEN_MAC_SIGN,
					.name = "EncryptThenMAC_sign" },
	{ .val = KMIP_WRAPPING_METHOD_MAC_SIGN_THEN_ENCRYPT,
					.name = "MAC_signThenEncrypt" },
	{ .val = KMIP_WRAPPING_METHOD_TR_31, .name = "TR_31" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_key_wrap_types[] = {
	{ .val = KMIP_KEY_WRAP_TYPE_NOT_WRAPPED, .name = "NotWrapped" },
	{ .val = KMIP_KEY_WRAP_TYPE_AS_REGISTERED, .name = "AsRegistered" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_block_cipher_modes[] = {
	{ .val = KMIP_BLOCK_CIPHER_MODE_CBC, .name = "CBC" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_ECB, .name = "ECB" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_PCBC, .name = "PCBC" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_CFB, .name = "CFB" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_OFB, .name = "OFB" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_CTR, .name = "CTR" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_CMAC, .name = "CMAC" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_CCM, .name = "CCM" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_GCM, .name = "GCM" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_CBC_MAC, .name = "CBC_MAC" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_XTS, .name = "XTS" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_AES_KEY_WRAP_PADDING,
						.name = "AESKeyWrapPadding" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_NIST_KEY_WRAP, .name = "NISTKeyWrap" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_X9_102_AESKW, .name = "X9_102AESKW" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_X9_102_TDKW, .name = "X9_102TDKW" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_X9_102_AKW1, .name = "X9_102AKW1" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_X9_102_AKW2, .name = "X9_102AKW2" },
	{ .val = KMIP_BLOCK_CIPHER_MODE_AEAD, .name = "AEAD" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_padding_methods[] = {
	{ .val = KMIP_PADDING_METHOD_NONE, .name = "None" },
	{ .val = KMIP_PADDING_METHOD_OAEP, .name = "OAEP" },
	{ .val = KMIP_PADDING_METHOD_PKCS5, .name = "PKCS5" },
	{ .val = KMIP_PADDING_METHOD_SSL3, .name = "SSL3" },
	{ .val = KMIP_PADDING_METHOD_ZEROS, .name = "Zeros" },
	{ .val = KMIP_PADDING_METHOD_ANSI_X9_23, .name = "ANSIX9_23" },
	{ .val = KMIP_PADDING_METHOD_ISO_10126, .name = "ISO10126" },
	{ .val = KMIP_PADDING_METHOD_PKCS_1_5, .name = "PKCS1V1_5" },
	{ .val = KMIP_PADDING_METHOD_X9_31, .name = "X9_31" },
	{ .val = KMIP_PADDING_METHOD_PSS, .name = "PSS" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_hashing_algos[] = {
	{ .val = KMIP_HASHING_ALGO_MD2, .name = "MD2" },
	{ .val = KMIP_HASHING_ALGO_MD4, .name = "MD4" },
	{ .val = KMIP_HASHING_ALGO_MD5, .name = "MD5" },
	{ .val = KMIP_HASHING_ALGO_SHA_1, .name = "SHA_1" },
	{ .val = KMIP_HASHING_ALGO_SHA_224, .name = "SHA_224" },
	{ .val = KMIP_HASHING_ALGO_SHA_256, .name = "SHA_256" },
	{ .val = KMIP_HASHING_ALGO_SHA_384, .name = "SHA_384" },
	{ .val = KMIP_HASHING_ALGO_SHA_512, .name = "SHA_512" },
	{ .val = KMIP_HASHING_ALGO_RIPEMD_160, .name = "RIPEMD_160" },
	{ .val = KMIP_HASHING_ALGO_TIGER, .name = "Tiger" },
	{ .val = KMIP_HASHING_ALGO_WIRLPOOL, .name = "Whirlpool" },
	{ .val = KMIP_HASHING_ALGO_SHA_512_224, .name = "SHA_512_224" },
	{ .val = KMIP_HASHING_ALGO_SHA_512_256, .name = "SHA_512_256" },
	{ .val = KMIP_HASHING_ALGO_SHA_3_224, .name = "SHA3_224" },
	{ .val = KMIP_HASHING_ALGO_SHA_3_256, .name = "SHA3_256" },
	{ .val = KMIP_HASHING_ALGO_SHA_3_384, .name = "SHA3_384" },
	{ .val = KMIP_HASHING_ALGO_SHA_3_512, .name = "SHA3_512" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_key_role_types[] = {
	{ .val = KMIP_KEY_ROLE_TYPE_BDK, .name = "BDK" },
	{ .val = KMIP_KEY_ROLE_TYPE_CVK, .name = "CVK" },
	{ .val = KMIP_KEY_ROLE_TYPE_DEK, .name = "DEK" },
	{ .val = KMIP_KEY_ROLE_TYPE_KMAC, .name = "KMAC" },
	{ .val = KMIP_KEY_ROLE_TYPE_MKSMC, .name = "MKSMC" },
	{ .val = KMIP_KEY_ROLE_TYPE_MKSMI, .name = "MKSMI" },
	{ .val = KMIP_KEY_ROLE_TYPE_MKDAC, .name = "_MKDAC" },
	{ .val = KMIP_KEY_ROLE_TYPE_MKDN, .name = "MKDN" },
	{ .val = KMIP_KEY_ROLE_TYPE_MKCP, .name = "MKCP" },
	{ .val = KMIP_KEY_ROLE_TYPE_MKOTH, .name = "MKOTH" },
	{ .val = KMIP_KEY_ROLE_TYPE_KEK, .name = "KEK" },
	{ .val = KMIP_KEY_ROLE_TYPE_MAC16609, .name = "MAC16609" },
	{ .val = KMIP_KEY_ROLE_TYPE_MAC97971, .name = "MAC97971" },
	{ .val = KMIP_KEY_ROLE_TYPE_MAC97972, .name = "MAC97972" },
	{ .val = KMIP_KEY_ROLE_TYPE_MAC97973, .name = "MAC97973" },
	{ .val = KMIP_KEY_ROLE_TYPE_MAC97974, .name = "MAC97974" },
	{ .val = KMIP_KEY_ROLE_TYPE_MAC97975, .name = "MAC97975" },
	{ .val = KMIP_KEY_ROLE_TYPE_ZPK, .name = "ZPK" },
	{ .val = KMIP_KEY_ROLE_TYPE_PVKIBM, .name = "PVKIBM" },
	{ .val = KMIP_KEY_ROLE_TYPE_PVKPVV, .name = "PVKPVV" },
	{ .val = KMIP_KEY_ROLE_TYPE_PVKOTH, .name = "PVKOTH" },
	{ .val = KMIP_KEY_ROLE_TYPE_DUKPT, .name = "DUKPT" },
	{ .val = KMIP_KEY_ROLE_TYPE_IV, .name = "IV" },
	{ .val = KMIP_KEY_ROLE_TYPE_TRKBK, .name = "TRKBK" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_sinature_algos[] = {
	{ .val = KMIP_SIGNATURE_ALGO_MD2_WITH_RSA_ENCRYPTION,
					.name = "MD2WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_MD5_WITH_RSA_ENCRYPTION,
					.name = "MD5WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA_1_WITH_RSA_ENCRYPTION,
					.name = "SHA_1WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA_224_WITH_RSA_ENCRYPTION,
					.name = "SHA_244WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA_256_WITH_RSA_ENCRYPTION,
					.name = "SHA_256WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA_384_WITH_RSA_ENCRYPTION,
					.name = "SHA_384WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA_512_WITH_RSA_ENCRYPTION,
					.name = "SHA_512WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_RSASSA_PSS, .name = "RSASSA_PSS" },
	{ .val = KMIP_SIGNATURE_ALGO_DSA_WITH_SHA_1, .name = "DSAWithSHA_1" },
	{ .val = KMIP_SIGNATURE_ALGO_DSA_WITH_SHA_244,
					.name = "DSAWithSHA224" },
	{ .val = KMIP_SIGNATURE_ALGO_DSA_WITH_SHA_256,
					.name = "DSAWithSHA256" },
	{ .val = KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_1,
					.name = "ECDSAWithSHA_1" },
	{ .val = KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_224,
					.name = "ECDSAWithSHA224" },
	{ .val = KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_256,
					.name = "ECDSAWithSHA256" },
	{ .val = KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_384,
					.name = "ECDSAWithSHA384" },
	{ .val = KMIP_SIGNATURE_ALGO_ECDSA_WITH_SHA_512,
					.name = "ECDSAWithSHA512" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA3_256_WITH_RSA_ENCRYPTION,
					.name = "SHA3_256WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA3_385_WITH_RSA_ENCRYPTION,
					.name = "SHA3_384WithRSAEncryption" },
	{ .val = KMIP_SIGNATURE_ALGO_SHA3_512_WITH_RSA_ENCRYPTION,
					.name = "SHA3_512WithRSAEncryption" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_mask_generators[] = {
	{ .val = KMIP_MASK_GENERATOR_MGF1, .name = "MGF1" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_encoding_options[] = {
	{ .val = KMIP_ENCODING_OPTION_NO, .name = "NoEncoding" },
	{ .val = KMIP_ENCODING_OPTION_TTLV, .name = "TTLVEncoding" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_recommended_cuurves[] = {
	{ .val = KMIP_REC_CURVE_P_192, .name = "P192" },
	{ .val = KMIP_REC_CURVE_K_163, .name = "K163" },
	{ .val = KMIP_REC_CURVE_B_163, .name = "B163" },
	{ .val = KMIP_REC_CURVE_P_224, .name = "P224" },
	{ .val = KMIP_REC_CURVE_K_223, .name = "K223" },
	{ .val = KMIP_REC_CURVE_B_223, .name = "B223" },
	{ .val = KMIP_REC_CURVE_P_256, .name = "P256" },
	{ .val = KMIP_REC_CURVE_K_283, .name = "K283" },
	{ .val = KMIP_REC_CURVE_B_283, .name = "B283" },
	{ .val = KMIP_REC_CURVE_P_384, .name = "P384" },
	{ .val = KMIP_REC_CURVE_K_409, .name = "K409" },
	{ .val = KMIP_REC_CURVE_B_409, .name = "B409" },
	{ .val = KMIP_REC_CURVE_P_521, .name = "P521" },
	{ .val = KMIP_REC_CURVE_K_571, .name = "K571" },
	{ .val = KMIP_REC_CURVE_B_571, .name = "B571" },
	{ .val = KMIP_REC_CURVE_SECP112R1, .name = "Secp112r1" },
	{ .val = KMIP_REC_CURVE_SECP112R2, .name = "Secp112r2" },
	{ .val = KMIP_REC_CURVE_SECP128R1, .name = "Secp128r1" },
	{ .val = KMIP_REC_CURVE_SECP128R2, .name = "Secp128r2" },
	{ .val = KMIP_REC_CURVE_SECP160K1, .name = "Secp160k1" },
	{ .val = KMIP_REC_CURVE_SECP160R1, .name = "Secp160r1" },
	{ .val = KMIP_REC_CURVE_SECP160R2, .name = "Secp160r2" },
	{ .val = KMIP_REC_CURVE_SECP192K1, .name = "Secp192k1" },
	{ .val = KMIP_REC_CURVE_SECP224K1, .name = "Secp224k1" },
	{ .val = KMIP_REC_CURVE_SECP256K1, .name = "Secp256k1" },
	{ .val = KMIP_REC_CURVE_SECT113R1, .name = "Sect113r1" },
	{ .val = KMIP_REC_CURVE_SECT113R2, .name = "Sect113r2" },
	{ .val = KMIP_REC_CURVE_SECT131R1, .name = "Sect131r1" },
	{ .val = KMIP_REC_CURVE_SECT131R2, .name = "Sect131r2" },
	{ .val = KMIP_REC_CURVE_SECT163R1, .name = "Sect163r1" },
	{ .val = KMIP_REC_CURVE_SECT193R1, .name = "Sect193r1" },
	{ .val = KMIP_REC_CURVE_SECT193R2, .name = "Sect193r2" },
	{ .val = KMIP_REC_CURVE_SECT239K1, .name = "Sect239k1" },
	{ .val = KMIP_REC_CURVE_ANSIX9P192V2, .name = "Ansix9p192v2" },
	{ .val = KMIP_REC_CURVE_ANSIX9P192V3, .name = "Ansix9p192v3" },
	{ .val = KMIP_REC_CURVE_ANSIX9P239V1, .name = "Ansix9p239v1" },
	{ .val = KMIP_REC_CURVE_ANSIX9P239V2, .name = "Ansix9p239v2" },
	{ .val = KMIP_REC_CURVE_ANSIX9P239V3, .name = "Ansix9p239v3" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB163V1, .name = "Ansix9c2pnb163v1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB163V2, .name = "Ansix9c2pnb163v2" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB163V3, .name = "Ansix9c2pnb163v3" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB176V1, .name = "Ansix9c2pnb176v1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB191V1, .name = "Ansix9c2tnb191v1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB191V2, .name = "Ansix9c2tnb191v2" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB191V3, .name = "Ansix9c2tnb191v3" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB208W1, .name = "Ansix9c2pnb208w1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB239V1, .name = "Ansix9c2tnb239v1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB239V2, .name = "Ansix9c2tnb239v2" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB239V3, .name = "Ansix9c2tnb239v3" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB272W1, .name = "Ansix9c2pnb272w1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB304W1, .name = "Ansix9c2pnb304w1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB359V1, .name = "Ansix9c2tnb359v1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2PNB368W1, .name = "Ansix9c2pnb368w1" },
	{ .val = KMIP_REC_CURVE_ANSIX9C2TNB431R1, .name = "Ansix9c2tnb431r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP160R1, .name = "Brainpoolp160r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP160T1, .name = "Brainpoolp160t1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP192R1, .name = "Brainpoolp192r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP192T1, .name = "Brainpoolp192t1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP224R1, .name = "Brainpoolp224r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP224T1, .name = "Brainpoolp224t1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP256R1, .name = "Brainpoolp256r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP256T1, .name = "Brainpoolp256t1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP320R1, .name = "Brainpoolp320r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP320T1, .name = "Brainpoolp320t1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP384R1, .name = "Brainpoolp384r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP384T1, .name = "Brainpoolp384t1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP512R1, .name = "Brainpoolp512r1" },
	{ .val = KMIP_REC_CURVE_BRAINPOOLP512T1, .name = "Brainpoolp512t1" },
	{ .val = KMIP_REC_CURVE_CURVE25519, .name = "Curve25519" },
	{ .val = KMIP_REC_CURVE_CURVE448, .name = "Curve448" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_protection_levels[] = {
	{ .val = KMIP_PROTECTION_LEVEL_HIGH, .name = "High" },
	{ .val = KMIP_PROTECTION_LEVEL_LOW, .name = "Low" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_key_value_location_types[] = {
	{ .val = KMIP_KEY_VAL_LOC_TYPE_UNINTERPRETED_TEXT_STRING,
				.name = "UninterpretedTextString" },
	{ .val = KMIP_KEY_VAL_LOC_TYPE_URI, .name = "URI" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_link_types[] = {
	{ .val = KMIP_LINK_TYPE_CERTIFICATE, .name = "CertificateLink" },
	{ .val = KMIP_LINK_TYPE_PUBLIC_KEY, .name = "PublicKeyLink" },
	{ .val = KMIP_LINK_TYPE_PRIVATE_KEY, .name = "PrivateKeyLink" },
	{ .val = KMIP_LINK_TYPE_DERIVATION_BASE_OBJECT,
					.name = "DerivationBaseObjectLink" },
	{ .val = KMIP_LINK_TYPE_DERIVED_KEY, .name = "DerivedKeyLink" },
	{ .val = KMIP_LINK_TYPE_REPLACEMENT_OBJECT,
					.name = "ReplacementObjectLink" },
	{ .val = KMIP_LINK_TYPE_REPLACED_OBJECT, .name = "ReplacedObjectLink" },
	{ .val = KMIP_LINK_TYPE_PARENT, .name = "ParentLink" },
	{ .val = KMIP_LINK_TYPE_CHILD, .name = "ChildLink" },
	{ .val = KMIP_LINK_TYPE_PREVIOUS, .name = "PreviousLink" },
	{ .val = KMIP_LINK_TYPE_NEXT, .name = "NextLink" },
	{ .val = KMIP_LINK_TYPE_PKCS_12_CERTIFICATE,
					.name = "PKCS_12CertificateLink" },
	{ .val = KMIP_LINK_TYPE_PKCS_12_PASSWORD,
					.name = "PKCS_12PasswordLink" },
	{ .val = KMIP_LINK_TYPE_WRAPPING_KEY, .name = "WrappingKeyLink" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_client_registration_methods[] = {
	{ .val = KMIP_CLIENT_REG_METH_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_CLIENT_REG_METH_SERVER_PRE_GENERATED,
					.name = "ServerPreGenerated" },
	{ .val = KMIP_CLIENT_REG_METH_SERVER_ON_DEMAND,
					.name = "ServerOnDemand" },
	{ .val = KMIP_CLIENT_REG_METH_CLIENT_GENERATED,
					.name = "ClientGenerated" },
	{ .val = KMIP_CLIENT_REG_METH_CLIENT_REGISTERED,
					.name = "ClientRegistered" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_rng_algorithms[] = {
	{ .val = KMIP_RNG_ALGO_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_RNG_ALGO_FIPS_186_2, .name = "FIPS186_2" },
	{ .val = KMIP_RNG_ALGO_DRBG, .name = "DRBG" },
	{ .val = KMIP_RNG_ALGO_NRBG, .name = "NRBG" },
	{ .val = KMIP_RNG_ALGO_ANSI_X9_31, .name = "ANSIX9_31" },
	{ .val = KMIP_RNG_ALGO_ANSI_X9_62, .name = "ANSIX9_62" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_drbg_algorithms[] = {
	{ .val = KMIP_DRBG_ALGO_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_DRBG_ALGO_DUAL_EC, .name = "Dual_EC" },
	{ .val = KMIP_DRBG_ALGO_HASH, .name = "Hash" },
	{ .val = KMIP_DRBG_ALGO_HMAC, .name = "HMAC" },
	{ .val = KMIP_DRBG_ALGO_CTR, .name = "CTR" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_fips186_variations[] = {
	{ .val = KMIP_FIPS186_VARI_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_FIPS186_VARI_GP_X_ORIGINAL, .name = "GPxOriginal" },
	{ .val = KMIP_FIPS186_VARI_GP_X_CHANGE_NOTICE,
					.name = "GPxChangeNotice" },
	{ .val = KMIP_FIPS186_VARI_X_ORIGINAL, .name = "XOriginal" },
	{ .val = KMIP_FIPS186_VARI_X_CHANGE_NOTICE, .name = "XChangeNotice" },
	{ .val = KMIP_FIPS186_VARI_K_ORIGINAL, .name = "KOriginal" },
	{ .val = KMIP_FIPS186_VARI_K_CHANGE_NOTICE, .name = "KChangeNotice" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_validation_authority_types[] = {
	{ .val = KMIP_VALIDATION_AUTH_TYPE_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_VALIDATION_AUTH_TYPE_NIST_CMVP, .name = "NISTCMVP" },
	{ .val = KMIP_VALIDATION_AUTH_TYPE_COMMON_CRITERIA,
					.name = "CommonCriteria" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_validation_types[] = {
	{ .val = KMIP_VALIDATION_TYPE_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_VALIDATION_TYPE_HARDWARE, .name = "Hardware" },
	{ .val = KMIP_VALIDATION_TYPE_SOFTWARE, .name = "Software" },
	{ .val = KMIP_VALIDATION_TYPE_FIRMWARE, .name = "Firmware" },
	{ .val = KMIP_VALIDATION_TYPE_HYBRID, .name = "Hybrid" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_unwrap_modes[] = {
	{ .val = KMIP_UNWRAP_MODE_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_UNWRAP_MODE_PROCESSED, .name = "Processed" },
	{ .val = KMIP_UNWRAP_MODE_NOT_PROCESSED, .name = "NotProcessed" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_destroy_actions[] = {
	{ .val = KMIP_DESTROY_ACTION_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_DESTROY_ACTION_KEY_MATERIAL_DELETED,
					.name = "KeyMaterialDeleted" },
	{ .val = KMIP_DESTROY_ACTION_KEY_MATERIAL_SHREDDED,
					.name = "KeyMaterialShredded" },
	{ .val = KMIP_DESTROY_ACTION_META_DATA_DELETED,
					.name = "MetaDataDeleted" },
	{ .val = KMIP_DESTROY_ACTION_META_DATA_SHREDDED,
					.name = "MetaDataShredded" },
	{ .val = KMIP_DESTROY_ACTION_DELETED, .name = "Deleted" },
	{ .val = KMIP_DESTROY_ACTION_SHREDDED, .name = "Shredded" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_shredding_algorithms[] = {
	{ .val = KMIP_SHREDDING_ALGO_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_SHREDDING_ALGO_CRYPTOGRAPHIC, .name = "Cryptographic" },
	{ .val = KMIP_SHREDDING_ALGO_UNSUPPORTED, .name = "Unsupported" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_rng_modes[] = {
	{ .val = KMIP_RNG_MODE_UNSPECIFIED, .name = "Unspecified" },
	{ .val = KMIP_RNG_MODE_SHARED_INSTANTIATION,
					.name = "SharedInstantiation" },
	{ .val = KMIP_RNG_MODE_NON_SHARED_INSTANCIATION,
					.name = "NonSharedInstantiation" },
	{ .val = 0, .name = NULL },
};

static const struct kmip_enum kmip_profile_names[] = {
	{ .val = KMIP_PROFILE_BASELINE_SERVER_BASIC_KMIP_V1_2,
		.name = "BaselineServerBasicKMIPV1_2" },
	{ .val = KMIP_PROFILE_BASELINE_SERVER_TLS_V1_2_KMIP_V1_2,
		.name = "BaselineServerTLSV1_2KMIPV1_2" },
	{ .val = KMIP_PROFILE_BASELINE_CLIENT_BASIC_KMIP_V1_2,
		.name = "BaselineClientBasicKMIPV1_2" },
	{ .val = KMIP_PROFILE_BASELINE_CLIENT_TLS_V1_2_KMIP_V1_2,
		.name = "BaselineClientTLSV1_2KMIPV1_2" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_BASIC_KMIP_V1_2,
		.name = "CompleteServerBasicKMIPV1_2" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2_KMIP_V1_2,
		.name = "CompleteServerTLSV1_2KMIPV1_2" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_0,
		.name = "TapeLibraryClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_1,
		.name = "TapeLibraryClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_2,
		.name = "TapeLibraryClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_0,
		.name = "TapeLibraryServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_1,
		.name = "TapeLibraryServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_2,
		.name = "TapeLibraryServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_0,
		.name = "SymmetricKeyLifecycleClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_1,
		.name = "SymmetricKeyLifecycleClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_2,
		.name = "SymmetricKeyLifecycleClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_0,
		.name = "SymmetricKeyLifecycleServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_1,
		.name = "SymmetricKeyLifecycleServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_2,
		.name = "SymmetricKeyLifecycleServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_0,
		.name = "AsymmetricKeyLifecycleClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_1,
		.name = "AsymmetricKeyLifecycleClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_2,
		.name = "AsymmetricKeyLifecycleClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_0,
		.name = "AsymmetricKeyLifecycleServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_1,
		.name = "AsymmetricKeyLifecycleServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_2,
		.name = "AsymmetricKeyLifecycleServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT_KMIP_V1_2,
		.name = "BasicCryptographicClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER_KMIP_V1_2,
		.name = "BasicCryptographicServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT_KMIP_V1_2,
		.name = "AdvancedCryptographicClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER_KMIP_V1_2,
		.name = "AdvancedCryptographicServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT_KMIP_V1_2,
		.name = "RNGCryptographicClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER_KMIP_V1_2,
		.name = "RNGCryptographicServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_0,
		.name = "BasicSymmetricKeyFoundryClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_0,
		.name = "IntermediateSymmetricKeyFoundryClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_0,
		.name = "AdvancedSymmetricKeyFoundryClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_1,
		.name = "BasicSymmetricKeyFoundryClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_1,
		.name = "IntermediateSymmetricKeyFoundryClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_1,
		.name = "AdvancedSymmetricKeyFoundryClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_2,
		.name = "BasicSymmetricKeyFoundryClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_2,
		.name = "IntermediateSymmetricKeyFoundryClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_2,
		.name = "AdvancedSymmetricKeyFoundryClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_0,
		.name = "SymmetricKeyFoundryServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_1,
		.name = "SymmetricKeyFoundryServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_2,
		.name = "SymmetricKeyFoundryServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_0,
		.name = "OpaqueManagedObjectStoreClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_1,
		.name = "OpaqueManagedObjectStoreClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_2,
		.name = "OpaqueManagedObjectStoreClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_0,
		.name = "OpaqueManagedObjectStoreServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_1,
		.name = "OpaqueManagedObjectStoreServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_2,
		.name = "OpaqueManagedObjectStoreServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_0,
		.name = "SuiteBMinLOS_128ClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_1,
		.name = "SuiteBMinLOS_128ClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_2,
		.name = "SuiteBMinLOS_128ClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_0,
		.name = "SuiteBMinLOS_128ServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_1,
		.name = "SuiteBMinLOS_128ServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_2,
		.name = "SuiteBMinLOS_128ServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_0,
		.name = "SuiteBMinLOS_192ClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_1,
		.name = "SuiteBMinLOS_192ClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_2,
		.name = "SuiteBMinLOS_192ClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_0,
		.name = "SuiteBMinLOS_192ServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_1,
		.name = "SuiteBMinLOS_192ServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_2,
		.name = "SuiteBMinLOS_192ServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_0,
		.name = "StorageArrayWithSelfEncryptingDriveClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_1,
		.name = "StorageArrayWithSelfEncryptingDriveClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_2,
		.name = "StorageArrayWithSelfEncryptingDriveClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_0,
		.name = "StorageArrayWithSelfEncryptingDriveServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_1,
		.name = "StorageArrayWithSelfEncryptingDriveServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_2,
		.name = "StorageArrayWithSelfEncryptingDriveServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_0,
		.name = "HTTPSClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_1,
		.name = "HTTPSClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_2,
		.name = "HTTPSClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_0,
		.name = "HTTPSServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_1,
		.name = "HTTPSServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_2,
		.name = "HTTPSServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_JSON_CLIENT_KMIP_V1_0,
		.name = "JSONClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_JSON_CLIENT_KMIP_V1_1,
		.name = "JSONClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_JSON_CLIENT_KMIP_V1_2,
		.name = "JSONClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_JSON_SERVER_KMIP_V1_0,
		.name = "JSONServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_JSON_SERVER_KMIP_V1_1,
		.name = "JSONServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_JSON_SERVER_KMIP_V1_2,
		.name = "JSONServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_XML_CLIENT_KMIP_V1_0,
		.name = "XMLClientKMIPV1_0" },
	{ .val = KMIP_PROFILE_XML_CLIENT_KMIP_V1_1,
		.name = "XMLClientKMIPV1_1" },
	{ .val = KMIP_PROFILE_XML_CLIENT_KMIP_V1_2,
		.name = "XMLClientKMIPV1_2" },
	{ .val = KMIP_PROFILE_XML_SERVER_KMIP_V1_0,
		.name = "XMLServerKMIPV1_0" },
	{ .val = KMIP_PROFILE_XML_SERVER_KMIP_V1_1,
		.name = "XMLServerKMIPV1_1" },
	{ .val = KMIP_PROFILE_XML_SERVER_KMIP_V1_2,
		.name = "XMLServerKMIPV1_2" },
	{ .val = KMIP_PROFILE_BASELINE_SERVER_BASIC_KMIP_V1_3,
		.name = "BaselineServerBasicKMIPV1_3" },
	{ .val = KMIP_PROFILE_BASELINE_SERVER_TLS_V1_2_KMIP_V1_3,
		.name = "BaselineServerTLSV1_2KMIPV1_3" },
	{ .val = KMIP_PROFILE_BASELINE_CLIENT_BASIC_KMIP_V1_3,
		.name = "BaselineClientBasicKMIPV1_3" },
	{ .val = KMIP_PROFILE_BASELINE_CLIENT_TLS_V1_2_KMIP_V1_3,
		.name = "BaselineClientTLSV1_2KMIPV1_3" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_BASIC_KMIP_V1_3,
		.name = "CompleteServerBasicKMIPV1_3" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2_KMIP_V1_3,
		.name = "CompleteServerTLSV1_2KMIPV1_3" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_3,
		.name = "TapeLibraryClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_3,
		.name = "TapeLibraryServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_3,
		.name = "SymmetricKeyLifecycleClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_3,
		.name = "SymmetricKeyLifecycleServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_3,
		.name = "AsymmetricKeyLifecycleClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_3,
		.name = "AsymmetricKeyLifecycleServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT_KMIP_V1_3,
		.name = "BasicCryptographicClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER_KMIP_V1_3,
		.name = "BasicCryptographicServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT_KMIP_V1_3,
		.name = "AdvancedCryptographicClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER_KMIP_V1_3,
		.name = "AdvancedCryptographicServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT_KMIP_V1_3,
		.name = "RNGCryptographicClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER_KMIP_V1_3,
		.name = "RNGCryptographicServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_3,
		.name = "BasicSymmetricKeyFoundryClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_3,
		.name = "IntermediateSymmetricKeyFoundryClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_3,
		.name = "AdvancedSymmetricKeyFoundryClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_3,
		.name = "SymmetricKeyFoundryServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_3,
		.name = "OpaqueManagedObjectStoreClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_3,
		.name = "OpaqueManagedObjectStoreServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_3,
		.name = "SuiteBMinLOS_128ClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_3,
		.name = "SuiteBMinLOS_128ServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_3,
		.name = "SuiteBMinLOS_192ClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_3,
		.name = "SuiteBMinLOS_192ServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_3,
		.name = "StorageArrayWithSelfEncryptingDriveClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_3,
		.name = "StorageArrayWithSelfEncryptingDriveServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_3,
		.name = "HTTPSClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_3,
		.name = "HTTPSServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_JSON_CLIENT_KMIP_V1_3,
		.name = "JSONClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_JSON_SERVER_KMIP_V1_3,
		.name = "JSONServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_XML_CLIENT_KMIP_V1_3,
		.name = "XMLClientKMIPV1_3" },
	{ .val = KMIP_PROFILE_XML_SERVER_KMIP_V1_3,
		.name = "XMLServerKMIPV1_3" },
	{ .val = KMIP_PROFILE_BASELINE_SERVER_BASIC_KMIP_V1_4,
		.name = "BaselineServerBasicKMIPV1_4" },
	{ .val = KMIP_PROFILE_BASELINE_SERVER_TLS_V1_2_KMIP_V1_4,
		.name = "BaselineServerTLSV1_2KMIPV1_4" },
	{ .val = KMIP_PROFILE_BASELINE_CLIENT_BASIC_KMIP_V1_4,
		.name = "BaselineClientBasicKMIPV1_4" },
	{ .val = KMIP_PROFILE_BASELINE_CLIENT_TLS_V1_2_KMIP_V1_4,
		.name = "BaselineClientTLSV1_2KMIPV1_4" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_BASIC_KMIP_V1_4,
		.name = "CompleteServerBasicKMIPV1_4" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2_KMIP_V1_4,
		.name = "CompleteServerTLSV1_2KMIPV1_4" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_CLIENT_KMIP_V1_4,
		.name = "TapeLibraryClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_SERVER_KMIP_V1_4,
		.name = "TapeLibraryServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_4,
		.name = "SymmetricKeyLifecycleClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_4,
		.name = "SymmetricKeyLifecycleServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT_KMIP_V1_4,
		.name = "AsymmetricKeyLifecycleClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER_KMIP_V1_4,
		.name = "AsymmetricKeyLifecycleServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT_KMIP_V1_4,
		.name = "BasicCryptographicClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER_KMIP_V1_4,
		.name = "BasicCryptographicServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT_KMIP_V1_4,
		.name = "AdvancedCryptographicClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER_KMIP_V1_4,
		.name = "AdvancedCryptographicServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT_KMIP_V1_4,
		.name = "RNGCryptographicClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER_KMIP_V1_4,
		.name = "RNGCryptographicServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_4,
		.name = "BasicSymmetricKeyFoundryClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_4,
		.name = "IntermediateSymmetricKeyFoundryClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT_KMIP_V1_4,
		.name = "AdvancedSymmetricKeyFoundryClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER_KMIP_V1_4,
		.name = "SymmetricKeyFoundryServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT_KMIP_V1_4,
		.name = "OpaqueManagedObjectStoreClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER_KMIP_V1_4,
		.name = "OpaqueManagedObjectStoreServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_CLIENT_KMIP_V1_4,
		.name = "SuiteBMinLOS_128ClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_128_SERVER_KMIP_V1_4,
		.name = "SuiteBMinLOS_128ServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_CLIENT_KMIP_V1_4,
		.name = "SuiteBMinLOS_192ClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_SUITE_B_MINLOS_192_SERVER_KMIP_V1_4,
		.name = "SuiteBMinLOS_192ServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT_KMIP_V1_4,
		.name = "StorageArrayWithSelfEncryptingDriveClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER_KMIP_V1_4,
		.name = "StorageArrayWithSelfEncryptingDriveServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_HTTPS_CLIENT_KMIP_V1_4,
		.name = "HTTPSClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_HTTPS_SERVER_KMIP_V1_4,
		.name = "HTTPSServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_JSON_CLIENT_KMIP_V1_4,
		.name = "JSONClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_JSON_SERVER_KMIP_V1_4,
		.name = "JSONServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_XML_CLIENT_KMIP_V1_4,
		.name = "XMLClientKMIPV1_4" },
	{ .val = KMIP_PROFILE_XML_SERVER_KMIP_V1_4,
		.name = "XMLServerKMIPV1_4" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_BASIC,
		.name = "CompleteServerBasic" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER_TLS_V1_2,
		.name = "CompleteServerTLSV1_2" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_CLIENT,
		.name = "TapeLibraryClient" },
	{ .val = KMIP_PROFILE_TAPE_LIBRARY_SERVER,
		.name = "TapeLibraryServer" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_CLIENT,
		.name = "SymmetricKeyLifecycleClient" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_LIFECYCLE_SERVER,
		.name = "SymmetricKeyLifecycleServer" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_CLIENT,
		.name = "AsymmetricKeyLifecycleClient" },
	{ .val = KMIP_PROFILE_ASYMMETRIC_KEY_LIFECYCLE_SERVER,
		.name = "AsymmetricKeyLifecycleServer" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_CLIENT,
		.name = "BasicCryptographicClient" },
	{ .val = KMIP_PROFILE_BASIC_CRYPTOGRAPHIC_SERVER,
		.name = "BasicCryptographicServer" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_CLIENT,
		.name = "AdvancedCryptographicClient" },
	{ .val = KMIP_PROFILE_ADVANCED_CRYPTOGRAPHIC_SERVER,
		.name = "AdvancedCryptographicServer" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_CLIENT,
		.name = "RNGCryptographicClient" },
	{ .val = KMIP_PROFILE_RNG_CRYPTOGRAPHIC_SERVER,
		.name = "RNGCryptographicServer" },
	{ .val = KMIP_PROFILE_BASIC_SYMMETRIC_KEY_FOUNDRY_CLIENT,
		.name = "BasicSymmetricKeyFoundryClient" },
	{ .val = KMIP_PROFILE_INTERMEDIATE_SYMMETRIC_KEY_FOUNDRY_CLIENT,
		.name = "IntermediateSymmetricKeyFoundryClient" },
	{ .val = KMIP_PROFILE_ADVANCED_SYMMETRIC_KEY_FOUNDRY_CLIENT,
		.name = "AdvancedSymmetricKeyFoundryClient" },
	{ .val = KMIP_PROFILE_SYMMETRIC_KEY_FOUNDRY_SERVER,
		.name = "SymmetricKeyFoundryServer" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_CLIENT,
		.name = "OpaqueManagedObjectStoreClient" },
	{ .val = KMIP_PROFILE_OPAQUE_MANAGED_OBJECT_STORE_SERVER,
		.name = "OpaqueManagedObjectStoreServer" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_CLIENT,
		.name = "StorageArrayWithSelfEncryptingDriveClient" },
	{ .val = KMIP_PROFILE_STORAGE_ARRAY_WITH_SELF_ENCRYPTING_DRIVE_SERVER,
		.name = "StorageArrayWithSelfEncryptingDriveServer" },
	{ .val = KMIP_PROFILE_HTTPS_CLIENT,
		.name = "HTTPSClient" },
	{ .val = KMIP_PROFILE_HTTPS_SERVER,
		.name = "HTTPSServer" },
	{ .val = KMIP_PROFILE_JSON_CLIENT,
		.name = "JSONClient" },
	{ .val = KMIP_PROFILE_JSON_SERVER,
		.name = "JSONServer" },
	{ .val = KMIP_PROFILE_XML_CLIENT,
		.name = "XMLClient" },
	{ .val = KMIP_PROFILE_XML_SERVER,
		.name = "XMLServer" },
	{ .val = KMIP_PROFILE_AES_XTS_CLIENT,
		.name = "AESXTSClient" },
	{ .val = KMIP_PROFILE_AES_XTS_SERVER,
		.name = "AESXTSServer" },
	{ .val = KMIP_PROFILE_QUANTUM_SAFE_CLIENT,
		.name = "QuantumSafeClient" },
	{ .val = KMIP_PROFILE_QUANTUM_SAFE_SERVER,
		.name = "QuantumSafeServer" },
	{ .val = KMIP_PROFILE_PKCS_11_CLIENT,
		.name = "PKCS_11Client" },
	{ .val = KMIP_PROFILE_PKCS_11_SERVER,
		.name = "PKCS_11Server" },
	{ .val = KMIP_PROFILE_BASELINE_CLIENT,
		.name = "BaselineClient" },
	{ .val = KMIP_PROFILE_BASELINE_SERVER,
		.name = "BaselineServer" },
	{ .val = KMIP_PROFILE_COMPLETE_SERVER,
		.name = "CompleteServer" },
	{ .val = 0, .name = NULL },
};

struct kmip_enum_info {
	enum kmip_tag tag;
	const struct kmip_enum *enum_info;
	bool is_mask;
};

static const struct kmip_enum_info enum_info[] = {
	{ .tag = KMIP_TAG_ATTRIBUTE_REFERENCE, .enum_info = kmip_tags,
	  .is_mask = false },
	{ .tag = KMIP_TAG_OPERATION, .enum_info = kmip_operations,
	  .is_mask = false },
	{ .tag = KMIP_TAG_BATCH_ERROR_CONTINUATION_OPTION,
	  .enum_info = kmip_batch_error_cont_options, .is_mask = false },
	{ .tag = KMIP_TAG_CRYPTOGRAPHIC_USAGE_MASK,
	  .enum_info = kmip_crypto_usage_masks, .is_mask = true },
	{ .tag = KMIP_TAG_RESULT_STATUS,
	  .enum_info = kmip_result_statuses, .is_mask = false },
	{ .tag = KMIP_TAG_RESULT_REASON,
	  .enum_info = kmip_result_reasons, .is_mask = false },
	{ .tag = KMIP_TAG_QUERY_FUNCTION,
	  .enum_info = kmip_query_functions, .is_mask = false },
	{ .tag = KMIP_TAG_NAME_TYPE,
	  .enum_info = kmip_name_types, .is_mask = false },
	{ .tag = KMIP_TAG_ALTERNATE_NAME_TYPE,
	  .enum_info = kmip_alternate_name_types, .is_mask = false },
	{ .tag = KMIP_TAG_UNIQUE_IDENTIFIER,
	  .enum_info = kmip_unique_identifiers, .is_mask = false },
	{ .tag = KMIP_TAG_OBJECT_TYPE,
	  .enum_info = kmip_object_types, .is_mask = false },
	{ .tag = KMIP_TAG_CRYPTOGRAPHIC_ALGORITHM,
	  .enum_info = kmip_crypto_algos, .is_mask = false },
	{ .tag = KMIP_TAG_CERTIFICATE_TYPE,
	  .enum_info = kmip_certificate_types, .is_mask = false },
	{ .tag = KMIP_TAG_STATE,
	  .enum_info = kmip_states, .is_mask = false },
	{ .tag = KMIP_TAG_PROTECTION_STORAGE_MASK,
	  .enum_info = kmip_protection_storage_masks, .is_mask = true },
	{ .tag = KMIP_TAG_REVOCATION_REASON_CODE,
	  .enum_info = kmip_revoke_reasons, .is_mask = false },
	{ .tag = KMIP_TAG_OBJECT_GROUP_MEMBER,
	  .enum_info = kmip_object_group_members, .is_mask = false },
	{ .tag = KMIP_TAG_STORAGE_STATUS_MASK,
	  .enum_info = kmip_storage_status_masks, .is_mask = true },
	{ .tag = KMIP_TAG_KEY_FORMAT_TYPE,
	  .enum_info = kmip_key_format_types, .is_mask = false },
	{ .tag = KMIP_TAG_KEY_COMPRESSION_TYPE,
	  .enum_info = kmip_key_compression_types, .is_mask = false },
	{ .tag = KMIP_TAG_WRAPPING_METHOD,
	  .enum_info = kmip_wrapping_methods, .is_mask = false },
	{ .tag = KMIP_TAG_KEY_WRAP_TYPE,
	  .enum_info = kmip_key_wrap_types, .is_mask = false },
	{ .tag = KMIP_TAG_BLOCK_CIPHER_MODE,
	  .enum_info = kmip_block_cipher_modes, .is_mask = false },
	{ .tag = KMIP_TAG_PADDING_METHOD,
	  .enum_info = kmip_padding_methods, .is_mask = false },
	{ .tag = KMIP_TAG_HASHING_ALGORITHM,
	  .enum_info = kmip_hashing_algos, .is_mask = false },
	{ .tag = KMIP_TAG_MASK_GENERATOR_HASHING_ALGORITHM,
	  .enum_info = kmip_hashing_algos, .is_mask = false },
	{ .tag = KMIP_TAG_KEY_ROLE_TYPE,
	  .enum_info = kmip_key_role_types, .is_mask = false },
	{ .tag = KMIP_TAG_DIGITAL_SIGNATURE_ALGORITHM,
	  .enum_info = kmip_sinature_algos, .is_mask = false },
	{ .tag = KMIP_TAG_MASK_GENERATOR,
	  .enum_info = kmip_mask_generators, .is_mask = false },
	{ .tag = KMIP_TAG_ENCODING_OPTION,
	  .enum_info = kmip_encoding_options, .is_mask = false },
	{ .tag = KMIP_TAG_RECOMMENDED_CURVE,
	  .enum_info = kmip_recommended_cuurves, .is_mask = false },
	{ .tag = KMIP_TAG_PROTECTION_LEVEL,
	  .enum_info = kmip_protection_levels, .is_mask = false },
	{ .tag = KMIP_TAG_KEY_VALUE_LOCATION_TYPE,
	  .enum_info = kmip_key_value_location_types, .is_mask = false },
	{ .tag = KMIP_TAG_LINK_TYPE,
	  .enum_info = kmip_link_types, .is_mask = false },
	{ .tag = KMIP_TAG_CLIENT_REGISTRATION_METHOD,
	  .enum_info = kmip_client_registration_methods, .is_mask = false },
	{ .tag = KMIP_TAG_RNG_ALGORITHM,
	  .enum_info = kmip_rng_algorithms, .is_mask = false },
	{ .tag = KMIP_TAG_DRBG_ALGORITHM,
	  .enum_info = kmip_drbg_algorithms, .is_mask = false },
	{ .tag = KMIP_TAG_FIPS186_VARIANT,
	  .enum_info = kmip_fips186_variations, .is_mask = false },
	{ .tag = KMIP_TAG_VALIDATION_AUTHORITY_TYPE,
	  .enum_info = kmip_validation_authority_types, .is_mask = false },
	{ .tag = KMIP_TAG_VALIDATION_TYPE,
	  .enum_info = kmip_validation_types, .is_mask = false },
	{ .tag = KMIP_TAG_UNWRAP_MODE,
	  .enum_info = kmip_unwrap_modes, .is_mask = false },
	{ .tag = KMIP_TAG_DESTROY_ACTION,
	  .enum_info = kmip_destroy_actions, .is_mask = false },
	{ .tag = KMIP_TAG_SHREDDING_ALGORITHM,
	  .enum_info = kmip_shredding_algorithms, .is_mask = false },
	{ .tag = KMIP_TAG_RNG_MODE,
	  .enum_info = kmip_rng_modes, .is_mask = false },
	{ .tag = KMIP_TAG_PROFILE_NAME,
	  .enum_info = kmip_profile_names, .is_mask = false },
	{ .tag = 0, .enum_info = NULL, .is_mask = false },
};

static const struct kmip_enum kmip_v1_attribute_names[] = {
	{ .val = KMIP_TAG_UNIQUE_IDENTIFIER, .name = "Unique Identifier" },
	{ .val = KMIP_TAG_NAME, .name = "Name" },
	{ .val = KMIP_TAG_OBJECT_TYPE, .name = "Object Type" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_ALGORITHM,
				.name = "Cryptographic Algorithm" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_DOMAIN_PARAMETERS,
				.name = "Cryptographic Domain Parameters" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_LENGTH,
				.name = "Cryptographic Length" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_PARAMETERS,
				.name = "Cryptographic Parameters" },
	{ .val = KMIP_TAG_CERTIFICATE_TYPE, .name = "Certificate Type" },
	{ .val = KMIP_TAG_CERTIFICATE_IDENTIFIER,
				.name = "Certificate Identifier" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER, .name = "Certificate Issuer" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT, .name = "Certificate Subject" },
	{ .val = KMIP_TAG_DIGEST, .name = "Digest" },
	{ .val = KMIP_TAG_OPERATION_POLICY_NAME,
				.name = "Operation Policy Name" },
	{ .val = KMIP_TAG_CRYPTOGRAPHIC_USAGE_MASK,
				.name = "Cryptographic Usage Mask" },
	{ .val = KMIP_TAG_LEASE_TIME, .name = "Lease Time" },
	{ .val = KMIP_TAG_USAGE_LIMITS, .name = "Usage Limits" },
	{ .val = KMIP_TAG_STATE, .name = "State" },
	{ .val = KMIP_TAG_INITIAL_DATE, .name = "Initial Date" },
	{ .val = KMIP_TAG_ACTIVATION_DATE, .name = "Activation Date" },
	{ .val = KMIP_TAG_PROCESS_START_DATE, .name = "Process Start Date" },
	{ .val = KMIP_TAG_PROTECT_STOP_DATE, .name = "Protect Stop Date" },
	{ .val = KMIP_TAG_DEACTIVATION_DATE, .name = "Deactivation Date" },
	{ .val = KMIP_TAG_DESTROY_DATE, .name = "Destroy Date" },
	{ .val = KMIP_TAG_COMPROMISE_OCCURRENCE_DATE,
				.name = "Compromise Occurrence Date" },
	{ .val = KMIP_TAG_COMPROMIZE_DATE, .name = "Compromise Date" },
	{ .val = KMIP_TAG_REVOCATION_REASON, .name = "Revocation Reason" },
	{ .val = KMIP_TAG_ARCHIVE_DATE, .name = "Archive Date" },
	{ .val = KMIP_TAG_OBJECT_GROUP, .name = "Object Group" },
	{ .val = KMIP_TAG_LINK, .name = "Link" },
	{ .val = KMIP_TAG_APPLICATION_SPECIFIC_INFORMATION,
				.name = "Application Specific Information" },
	{ .val = KMIP_TAG_CONTACT_INFORMATION, .name = "Contact Information" },
	{ .val = KMIP_TAG_LAST_CHANGE_DATE, .name = "Last Change Date" },
	{ .val = KMIP_TAG_CUSTOM_ATTRIBUTE, .name = "Custom Attribute" },
	{ .val = KMIP_TAG_ALTERNATE_NAME, .name = "Alternative Name" },
	{ .val = KMIP_TAG_KEY_VALUE_PRESENT, .name = "Key Value Present" },
	{ .val = KMIP_TAG_KEY_VALUE_LOCATION, .name = "Key Value Location" },
	{ .val = KMIP_TAG_ORIGINAL_CREATION_DATE,
				.name = "Original Creation Date" },
	{ .val = KMIP_TAG_RANDOM_NUMBER_GENERATOR,
				.name = "Random Number Generator" },
	{ .val = KMIP_TAG_PKCS_12_FRIENDLY_NAME,
				.name = "PKCS#12 Friendly Name" },
	{ .val = KMIP_TAG_DESCRIPTION, .name = "Description" },
	{ .val = KMIP_TAG_COMMENT, .name = "Comment" },
	{ .val = KMIP_TAG_SENSITIVE, .name = "Sensitive" },
	{ .val = KMIP_TAG_ALWAYS_SENSITIVE, .name = "Always Sensitive" },
	{ .val = KMIP_TAG_EXTRACTABLE, .name = "Extractable" },
	{ .val = KMIP_TAG_NEVER_EXTRACTABLE, .name = "Never Extractable" },
	/*
	 * KMIP v2.x attribute names (just for reference, will most likely
	 * not appear in a v1.x attribute list
	 */
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_CN,
				.name = "Certificate Subject CN" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_O,
				.name = "Certificate Subject O" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_OU,
				.name = "Certificate Subject OU" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_EMAIL,
			.name = "Certificate Subject Email" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_C,
				.name = "Certificate Subject C" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_ST,
				.name = "Certificate Subject ST" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_L,
				.name = "Certificate Subject L" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_UID,
				.name = "Certificate Subject UID" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_SERIAL_NUMBER,
				.name = "Certificate Subject Serial Number" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_TITLE,
				.name = "Certificate Subject Title" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_DC,
				.name = "Certificate Subject DC" },
	{ .val = KMIP_TAG_CERTIFICATE_SUBJECT_DN_QUALIFIER,
				.name = "Certificate Subject DN Qualifier" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_CN,
				.name = "Certificate Issuer CN" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_O,
				.name = "Certificate Issuer O" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_OU,
				.name = "Certificate Issuer OU" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_EMAIL,
				.name = "Certificate Issuer Email" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_C,
				.name = "Certificate Issuer C" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_ST,
				.name = "Certificate Issuer ST" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_L,
				.name = "Certificate Issuer L" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_UID,
				.name = "Certificate Issuer UID" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_SERIAL_NUMBER,
				.name = "Certificate Issuer Serial Number" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_TITLE,
				.name = "Certificate Issuer Title" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_DC,
				.name = "Certificate Issuer DC" },
	{ .val = KMIP_TAG_CERTIFICATE_ISSUER_DN_QUALIFIER,
				.name = "Certificate Issuer DN Qualifier" },
	{ .val = KMIP_TAG_CERTIFICATE_TYPE, .name = "Certificate Type" },
	{ .val = KMIP_TAG_CERTIFICATE_LENGTH, .name = "Certificate Length" },
	{ .val = KMIP_TAG_DIGITAL_SIGNATURE_ALGORITHM,
				.name = "Digital Signature Algorithm" },
	{ .val = KMIP_TAG_FRESH, .name = "Fresh" },
	{ .val = KMIP_TAG_KEY_FORMAT_TYPE, .name = "Key Format Type" },
	{ .val = KMIP_TAG_NIST_KEY_TYPE, .name = "NIST Key Type" },
	{ .val = KMIP_TAG_OPAQUE_DATA_TYPE, .name = "Opaque Data Type" },
	{ .val = KMIP_TAG_PROTECTION_LEVEL, .name = "Protection Level" },
	{ .val = KMIP_TAG_PROTECTION_PERIOD, .name = "Protection Period" },
	{ .val = KMIP_TAG_PROTECTION_STORAGE_MASK,
				.name = "Protection Storage Mask" },
	{ .val = KMIP_TAG_QUANTUM_SAFE, .name = "Quantum Safe" },
	{ .val = KMIP_TAG_SHORT_UNIQUE_IDENTIFIER,
				.name = "Short Unique Identifier" },
	{ .val = KMIP_TAG_ATTRIBUTE, .name = "Vendor Attribute" },
	{ .val = KMIP_TAG_X_509_CERTIFICATE_IDENTIFIER,
				.name = "X.509 Certificate Identifier" },
	{ .val = KMIP_TAG_X_509_CERTIFICATE_ISSUER,
				.name = "X.509 Certificate Issuer" },
	{ .val = KMIP_TAG_X_509_CERTIFICATE_SUBJECT,
				.name = "X.509 Certificate Subject" },
	{ .val = 0, .name = NULL },
};

/**
 * Return the name of the enumeration value, or NULL if the value is unknown
 */
static const char *kmip_enum_name_by_value(const struct kmip_enum *info,
					   uint32_t val)
{
	unsigned int i;

	for (i = 0; info[i].name != NULL; i++) {
		if (info[i].val == val)
			return info[i].name;
	}

	return NULL;
}

/**
 * Return the enumeration value specified as enumeration name, or in hex
 * notation.
 */
int kmip_enum_value_by_name_or_hex(const struct kmip_enum *info,
				   const char *name, uint32_t *value)
{
	unsigned int i;
	int64_t val;
	int rc;

	if (name == NULL || value == NULL)
		return -EINVAL;

	for (i = 0; info[i].name != NULL; i++) {
		if (strcmp(info[i].name, name) == 0) {
			*value = info[i].val;
			return 0;
		}
	}

	rc = kmip_parse_hex_int(name, &val);
	if (rc != 0)
		return rc;

	*value = val;
	return 0;
}

/*
 * Get the name of an enumeration value belonging to the specified tag
 */
const char *kmip_enum_name_by_tag_value(enum kmip_tag tag, uint32_t val)
{
	int i;

	for (i = 0; enum_info[i].enum_info != NULL; i++) {
		if (enum_info[i].tag != tag)
			continue;

		return kmip_enum_name_by_value(enum_info[i].enum_info, val);
	}

	return NULL;
}

/**
 * Return the enumeration value belonging to the specified tag, specified as
 * enumeration name, or in hex notation.
 */
int kmip_enum_value_by_tag_name_or_hex(enum kmip_tag tag, const char *name,
				       uint32_t *value)
{
	int64_t val;
	int i, rc;

	if (name == NULL || value == NULL)
		return -EINVAL;

	for (i = 0; enum_info[i].enum_info != NULL; i++) {
		if (enum_info[i].tag != tag)
			continue;

		return kmip_enum_value_by_name_or_hex(enum_info[i].enum_info,
						      name, value);
	}

	rc = kmip_parse_hex_int(name, &val);
	if (rc != 0)
		return rc;

	*value = val;
	return 0;
}

/**
 * Return the name of the tag, or NULL if the tag is unknown
 */
const char *kmip_tag_name_by_tag(enum kmip_tag tag)
{
	return kmip_enum_name_by_value(kmip_tags, tag);
}

/**
 * Return the name or the hex representation of the tag
 */
const char *kmip_tag_name_or_hex_by_tag(enum kmip_tag tag, char tmp_buff[20])
{
	const char *str;

	str = kmip_enum_name_by_value(kmip_tags, tag);
	if (str == NULL) {
		sprintf(tmp_buff, "0x%06x", tag);
		str = tmp_buff;
	}
	return str;
}

/**
 * Return the tag value specified as tag name, or in hex notation. Returns 0
 * in case of an error.
 */
enum kmip_tag kmip_tag_by_name_or_hex(const char *name)
{
	uint32_t val;
	int rc;

	rc = kmip_enum_value_by_name_or_hex(kmip_tags, name, &val);
	if (rc != 0)
		return 0;

	return val;
}

/**
 * Return the name of the type, or NULL if the type is unknown
 */
const char *kmip_type_name_by_type(enum kmip_type type)
{
	return kmip_enum_name_by_value(kmip_types, type);
}

/**
 * Return the type value specified as type name, or in hex notation. Returns 0
 * in case of an error.
 */
enum kmip_type kmip_type_by_name_or_hex(const char *name)
{
	uint32_t val;
	int rc;

	rc = kmip_enum_value_by_name_or_hex(kmip_types, name, &val);
	if (rc != 0)
		return 0;

	return val;
}

/**
 * Return the KMIP v1.x attribute name of the tag, or NULL if the tag is unknown
 */
const char *kmip_v1_attr_name_by_tag(enum kmip_tag attr_tag)
{
	return kmip_enum_name_by_value(kmip_v1_attribute_names, attr_tag);
}

/**
 * Return the attribute tag value specified as KMIP v1.x attribute name.
 * Returns 0 in case of an error.
 */
enum kmip_tag kmip_attr_tag_by_v1_attr_name(const char *name)
{
	uint32_t val;
	int rc;

	rc = kmip_enum_value_by_name_or_hex(kmip_v1_attribute_names, name,
					    &val);
	if (rc != 0)
		return 0;

	return val;
}

/**
 * Returns true if the tag is a mask
 */
bool kmip_is_tag_mask(enum kmip_tag tag)
{
	int i;

	for (i = 0; enum_info[i].enum_info != NULL; i++) {
		if (enum_info[i].tag != tag)
			continue;

		return enum_info[i].is_mask;
	}

	return false;
}

/**
 * Returns the enumeration info for the specified tag, or NULL, if the tag
 * is not associated with an enumeration.
 */
const struct kmip_enum *kmip_enum_info_by_tag(enum kmip_tag tag)
{
	int i;

	for (i = 0; enum_info[i].enum_info != NULL; i++) {
		if (enum_info[i].tag != tag)
			continue;

		return enum_info[i].enum_info;
	}

	return NULL;
}
